/*
 * Copyright (C) 2008, 2009 Andrew Sichevoi.
 *
 * This file is part of Conler (http://thekondor.net/conler).
 *
 * Conler is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Conler is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Conler. If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>

#include <libxml/tree.h>
#include <libxml/xmlwriter.h>

#include <string.h>
#include <stdio.h>

#include "common.h"
#include "const.h"
#include "types.h"
#include "debug.h"

/* Continues iteration if it is not a desired node */
#define SKIP_NODE_PROCESSING(node, node_name) \
            if (xmlStrcmp((xmlChar *)node->name, BAD_CAST(node_name)))\
                continue;

#define SKIP_NODETYPE_PROCESSING(node, node_type) \
            if (node_type != node->type)\
                continue;

static xmlNodePtr get_element_node(const xmlNodePtr const, const char* const);
static AccessPoint_s* node_as_ap(const xmlDocPtr const, const xmlNodePtr const);
static Command_s* node_as_cmd(const xmlNodePtr const);
static gboolean write_ap_cfg(xmlTextWriterPtr, GSList*);


MaemoVersion get_maemo_version()
{
    FILE* version_file = g_fopen(OSSO_SOFTWARE_VERSION, "r");
    static const int BUF_SIZE = 64;
    char contents[BUF_SIZE];
    MaemoVersion version = UNKNOWN_VERSION;

    if (version_file) {
        if (fgets(contents, BUF_SIZE - 1, version_file)) {
            contents[BUF_SIZE] = 0;
            version = g_strrstr(contents, DIABLO_IDENT) ? DIABLO : CHINOOK;
        }

        fclose(version_file);
    }

    return version;
}


// TODO: return error code
// TODO: s/CFG_RETURN/R_CFG_RETURN/g
CfgReadResult read_cfg(const gchar* const filename, ConlerCfg_s* cfg)
{
#undef CFG_RETURN
#define CFG_RETURN(reason, result)  { DBG("cfg.err = %s\n", reason);\
                                      xmlFreeDoc(doc);  return result; }
    xmlNodePtr cur_node;
    xmlDocPtr doc;

    /* Preset base fields */
    cfg->version = NULL;
    cfg->access_points = NULL;

    doc = xmlParseFile(filename);
    if (!doc)
        return CFG_R_OPEN_ERR;

    cur_node = xmlDocGetRootElement(doc);
    if (!cur_node)
        CFG_RETURN("Empty document", CFG_R_EMPTF_ERR);

    if (xmlStrcmp(cur_node->name, BAD_CAST(ROOT_CFG_ELEMENT)))
        CFG_RETURN("Document wrong type", CFG_R_WRGT_ERR);

    /* Get version of configuration */
    cfg->version = (gchar *)xmlGetProp(cur_node, BAD_CAST(VERSION_CFG_ATTR));

    cur_node = get_element_node(cur_node, APS_CFG_ELEMENT);
    /* Return 'Ok' because there is nothing abnormal,
       but just for debugging purposes and correct memory
       management it is done via CFG_RETURN 
    */
    if (!cur_node)
        CFG_RETURN("No APs are defined", CFG_R_OK);

    /* Collect APs */
    for (cur_node = cur_node->xmlChildrenNode; cur_node; cur_node = cur_node->next) {
        SKIP_NODE_PROCESSING(cur_node, AP_CFG_ELEMENT);

        AccessPoint_s* ap = node_as_ap(doc, cur_node); // TODO: don't use 'doc'
        if (ap)
            APPEND2SLIST(cfg->access_points, ap);
    }

    xmlFreeDoc(doc);
    return CFG_R_OK;
}


// TODO: code error return
gboolean write_cfg(const gchar* const filename, const ConlerCfg_s* const cfg)
{
#undef W_CFG_RETURN_IF_FAIL
#define W_CFG_RETURN_IF_FAIL(rc) if (rc < 0) { xmlFreeTextWriter(writer);\
                                               xmlBufferFree(buf); return FALSE;}
    xmlTextWriterPtr writer;
    xmlBufferPtr buf;
    int rc;

    buf = xmlBufferCreate();
    g_return_val_if_fail(buf != NULL, FALSE);

    writer = xmlNewTextWriterMemory(buf, 0);
    if (!writer) {
	CFG_DBG("Can't make an instance of xmlTextWriterPtr");
        xmlBufferFree(buf);
        return FALSE;
    }

    rc = xmlTextWriterStartDocument(writer, NULL, "utf-8", NULL); 
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterWriteComment(writer,
                          BAD_CAST("This file is generated automatically. "
                                   "Please don't edit it manually, use a "
                                   "special gui wizard"));

    rc = xmlTextWriterStartElement(writer, BAD_CAST(ROOT_CFG_ELEMENT));
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterWriteAttribute(writer, BAD_CAST(VERSION_CFG_ATTR),
                                             BAD_CAST(cfg->version));
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterStartElement(writer, BAD_CAST(APS_CFG_ELEMENT));
    W_CFG_RETURN_IF_FAIL(rc);

    if (!write_ap_cfg(writer, cfg->access_points))
        W_CFG_RETURN_IF_FAIL(-1);

    /* Close 'access-points' element */
    rc = xmlTextWriterEndElement(writer);
    W_CFG_RETURN_IF_FAIL(rc);

    /* Close 'conler-config' element */
    rc = xmlTextWriterEndElement(writer);
    W_CFG_RETURN_IF_FAIL(rc);

    rc = xmlTextWriterEndDocument(writer);
    W_CFG_RETURN_IF_FAIL(rc);

    xmlFreeTextWriter(writer);

    // TODO: use xmlSaveFormatFile() or similar
    FILE* config = fopen(filename, "w");
    if (!config)
        W_CFG_RETURN_IF_FAIL(-1);

    xmlBufferDump(config, buf);
    fclose(config);

    xmlBufferFree(buf);

    return TRUE;
}

// TODO: config writing and reading approaches are not consistent
static gboolean write_ap_cfg(xmlTextWriterPtr writer, GSList* ap_list)
{
#define W_CFG_CHK_START_ELEMENT(writer, rc, name)		 \
    CFG_DBG("cfg.write: start element '%s'", name);		 \
    rc = xmlTextWriterStartElement(writer, BAD_CAST(name));      \
    g_return_val_if_fail(rc >= 0, FALSE);
#define W_CFG_CHK_END_ELEMENT(writer, rc)			 \
    CFG_DBG("cfg.write: end element");				 \
    rc = xmlTextWriterEndElement(writer);                        \
    g_return_val_if_fail(rc >= 0, FALSE);
#define W_CFG_CHK_WRITE_ATTRIBUTE(writer, rc, attr, value)	 \
    CFG_DBG("cfg.write: write attr '%s'", attr);                 \
    rc = xmlTextWriterWriteAttribute(writer, BAD_CAST(attr),     \
					     BAD_CAST(value));   \
    g_return_val_if_fail(rc >= 0, FALSE);

    int rc;

    for (GSList* iter = ap_list; iter; iter = g_slist_next(iter)) {
        // TODO: set name to 'unknown', set 'disabled' property to True
        AccessPoint_s* ap = AP_LIST_CAST(iter);
        if (!ap->name || !g_utf8_strlen(ap->name, -1)) {
            CFG_DBG("AP name must be set. Skipping.");
            continue;
        }

	W_CFG_CHK_START_ELEMENT(writer, rc, AP_CFG_ELEMENT);
	W_CFG_CHK_WRITE_ATTRIBUTE(writer, rc, NAME_CFG_ATTR, ap->name);

        if (ap->disabled) {
	    W_CFG_CHK_WRITE_ATTRIBUTE(writer, rc, DISABLED_CFG_ATTR, TRUE_BOOL);
        }

	W_CFG_CHK_START_ELEMENT(writer, rc, CMDS_CFG_ELEMENT);

        /* Write commands */
        for (GSList* cmd_iter = ap->commands;
                                        cmd_iter;
                                        cmd_iter = g_slist_next(cmd_iter)) {
	    Command_s* cmd = CMD_LIST_CAST(cmd_iter);

	    W_CFG_CHK_START_ELEMENT(writer, rc, CMD_CFG_ELEMENT);

	    if (cmd->disabled) {
		W_CFG_CHK_WRITE_ATTRIBUTE(writer, rc, DISABLED_CFG_ATTR,
					  TRUE_BOOL);
	    }

	    if (cmd->superuser_env) {
		W_CFG_CHK_WRITE_ATTRIBUTE(writer, rc, SUPERUSER_ENV_CFG_ATTR,
					  TRUE_BOOL);
	    }

	    rc = xmlTextWriterWriteString(writer, BAD_CAST(cmd->command));
	    g_return_val_if_fail(rc >= 0, FALSE);

	    /* Close 'command' element */
	    W_CFG_CHK_END_ELEMENT(writer, rc);
        }

        /* Close 'commands' element */
	W_CFG_CHK_END_ELEMENT(writer, rc);

        /* Close 'access-point' element */
	W_CFG_CHK_END_ELEMENT(writer, rc);
    }

    return TRUE;
}

void free_cfg(ConlerCfg_s* cfg)
{

    if (cfg->access_points) {
        g_slist_foreach(cfg->access_points, (GFunc)free_access_point, NULL);
        g_slist_free(cfg->access_points);
    }

    if (cfg->version)
        g_free(cfg->version);
}

static AccessPoint_s* node_as_ap(const xmlDocPtr const doc,\
                                 const xmlNodePtr const node)
{
    AccessPoint_s* ap = NULL;
    xmlNodePtr cur_node;
    static guint unknown_id = 0;

    gchar* name = (gchar *)xmlGetProp(node, BAD_CAST(NAME_CFG_ATTR));

    /* AccessPoint is not valid if:
     *  - no children
     *  - next is not a CMDS_CFG_ELEMENT
     */
    cur_node = get_element_node(node, CMDS_CFG_ELEMENT);
    if (!cur_node || xmlStrcmp(cur_node->name, BAD_CAST(CMDS_CFG_ELEMENT))) 
        return NULL;

    ap = g_new(AccessPoint_s, 1);
    g_assert(ap != NULL);
   
    if (!name) {
        ap->name = g_strdup_printf(UNKNOWN_AP_FMT, unknown_id++);
        ap->disabled = TRUE;
    }
    else
    {
        ap->name = name;
        ap->disabled = FALSE;
    }

    ap->commands = NULL;

    /* Check if AP is disabled */
    xmlChar* disabled = xmlGetProp(node, BAD_CAST(DISABLED_CFG_ATTR));
    if (disabled) {
        ap->disabled = !xmlStrcmp(disabled, BAD_CAST(TRUE_BOOL));
        xmlFree(disabled);
    }

    /* Collect AP's commands */
    for (cur_node = cur_node->xmlChildrenNode; cur_node;\
                                               cur_node = cur_node->next) {
        SKIP_NODE_PROCESSING(cur_node, CMD_CFG_ELEMENT);

        Command_s* cmd = node_as_cmd(cur_node);
        if (cmd)
            APPEND2SLIST(ap->commands, cmd);
    }

    return ap;
}

static Command_s* node_as_cmd(const xmlNodePtr const node)
{
    Command_s* cmd = NULL;

    if (!node)
        creturn(NULL, "read_cfg: cmd node is NULL");

    cmd = g_new(Command_s, 1);
    g_assert(cmd != NULL);

    cmd->command = (gchar *)xmlNodeGetContent(node);
    cmd->disabled = FALSE;
    cmd->superuser_env = FALSE;

    xmlChar* disabled = xmlGetProp(node, BAD_CAST(DISABLED_CFG_ATTR));
    if (disabled) {
        cmd->disabled = !xmlStrcmp(disabled, BAD_CAST(TRUE_BOOL));
        xmlFree(disabled);
    }

    xmlChar* superuser_env = xmlGetProp(node, BAD_CAST(SUPERUSER_ENV_CFG_ATTR));
    if (superuser_env) {
        cmd->superuser_env = !xmlStrcmp(superuser_env, BAD_CAST(TRUE_BOOL));
        xmlFree(superuser_env);
    }

    return cmd;
}


static xmlNodePtr get_element_node(const xmlNodePtr const node,\
                                   const char* const name)
{
    xmlNodePtr cur_node = NULL; 

    for (cur_node = node->children; cur_node; cur_node = cur_node->next) {
        SKIP_NODETYPE_PROCESSING(cur_node, XML_ELEMENT_NODE);

        if (!xmlStrcmp(cur_node->name, BAD_CAST(name)))
            return cur_node;
    }

    return NULL;
}

void free_command_list(GSList* cmd_list)
{
    if (!cmd_list)
        return;

    g_slist_foreach(cmd_list, (GFunc)free_command, NULL);
    g_slist_free(cmd_list);
}

void free_command(Command_s* cmd)
{
    if (!cmd)
        return;

    if (cmd->command)
        g_free(cmd->command);

    g_free(cmd);
}

Command_s* copy_command(Command_s* cmd)
{
    Command_s* new_cmd = g_new(Command_s, 1);
    g_assert(new_cmd != NULL);

    new_cmd->command = g_strdup(cmd->command);
    new_cmd->disabled = cmd->disabled;
    new_cmd->superuser_env = cmd->superuser_env;

    return new_cmd;
}

void free_access_point(AccessPoint_s* ap)
{
    if (!ap)
        return; 

    if (ap->name)
        g_free(ap->name);

    free_command_list(ap->commands);
    g_free(ap);
}

AccessPoint_s* copy_access_point(AccessPoint_s* ap)
{
    AccessPoint_s* new_ap = g_new(AccessPoint_s, 1);
    g_assert(new_ap != NULL);

    new_ap->name = g_strdup(ap->name);
    new_ap->disabled = ap->disabled;
    new_ap->commands = NULL;

    for (GSList* iter = ap->commands; iter; iter = g_slist_next(iter))
	APPEND2SLIST(new_ap->commands, copy_command(CMD_LIST_CAST(iter)));

    return new_ap;
}


// TODO: dump_config(), dump_AP()
#ifdef DEBUG
#undef BOOL2STR
#define BOOL2STR(b) (b ? "yes" : "no")

void dump_ap(gchar* annotation, AccessPoint_s* ap, gboolean print_commands)
{
    DBG("[dump] AP:");
    if (annotation)
	DBG(" (%s)", annotation);

    if (!ap) {
	DBG(" AP is NULL!");
	return;
    }

    DBG("  Name: %s", ap->name);
    DBG("  Disabled: %s", BOOL2STR(ap->disabled));
    DBG("  Commands: %s", BOOL2STR(ap->commands != NULL));

    if (print_commands)
	dump_commands(NULL, ap->commands);
}

void dump_commands(gchar* annotation, GSList* commands)
{
    DBG("[dump] Commands:");
    if (annotation)
	DBG(" (%s)", annotation);

    if (!commands) {
	DBG(" Command list is NULL!");
	return;
    }

    for (GSList* iter = commands; iter; iter = g_slist_next(iter)) {
	Command_s* cmd = CMD_LIST_CAST(iter);

	DBG(" disabled=%s, su=%s, '%s'", BOOL2STR(cmd->disabled),
	                                 BOOL2STR(cmd->superuser_env),
	                                 cmd->command);
    }
}
#undef BOOL2STR
#endif






