/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h> /* stderr, stdout */
#include <stdlib.h>
#include <string.h> /* strdup */
#include <unistd.h> /* pipe */
#include <glib.h>
#include <glib-object.h>
#include <libobd.h>
#include <poll.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>

#include "obd-thread.h"
#include "obd-sched.h"
#include "bt-serial.h"
#include "log.h"
#include "dbus.h"
#include "carmand-trip.h"

/* 
 * Defines the number of iterations befere the state machine
 * verify if the connection is working or not.
 */
#define MAX_NUM_OF_ERRS_BEFORE_BL 7

#define PIPE_READ 0
#define PIPE_WRITE 1

#define CHK_CONNECTION_PID 1
#define CHK_CONNECTION_INTERVAL 5
#define OBD_TIMEOUT 4

enum {
	TOBD_START_BT_CONNECTION = 1,
	TOBD_START_SERIAL_CONNECTION,
	TOBD_START_SIMULATOR,
	TOBD_START_DISCONNECTION,
	TOBD_CONNECTING_BT,
	TOBD_CONNECTING_SERIAL,
	TOBD_CONNECTED,
	TOBD_DISCONNECTED,
	TOBD_UNEXPECTED_DISCONNECTION,
	TOBD_QUIT
};

static GAsyncQueue *g_rtqueue = NULL;
static GSList *g_dtc_list = NULL;

static int obd_thread_status = TOBD_DISCONNECTED;
static GThread *obd_thread_loop;

static char *obd_bt_id = NULL;
static char *obd_serial_port = NULL;
static char *obd_simulator_file = NULL;

static void *libobd = NULL;

static int g_main_pipefd[2] = {0, 0};
static int g_pipefd[2] = {0, 0};
static struct pollfd g_pollfd;

static struct obd_sensor_data fresh_rpm;
static struct obd_sensor_data fresh_speed;

static int dtc_list_request = 0;
static int is_blocked = 0;

/*
 * Used to check about the possibility of a state transanction.
 *
 * FIXME: remove it or put in another place
 */
#define state(x, y) ((obd_thread_status == (x)) && (ns == (y)))

static inline int obd_thread_is_connected(void)
{
	return obd_thread_status == TOBD_CONNECTED;
}

static inline int obd_thread_is_connecting(void)
{
	return obd_thread_status == TOBD_START_BT_CONNECTION ||
		obd_thread_status == TOBD_START_SERIAL_CONNECTION ||
		obd_thread_status == TOBD_START_SIMULATOR ||
		obd_thread_status == TOBD_CONNECTING_SERIAL ||
		obd_thread_status == TOBD_CONNECTING_BT;
}

static inline int obd_thread_is_disconnected(void)
{
	return obd_thread_status == TOBD_DISCONNECTED ||
		obd_thread_status == TOBD_START_DISCONNECTION ||
		obd_thread_status == TOBD_UNEXPECTED_DISCONNECTION;
}

static int setup_libobd_simulator(void **libobd, char *file)
{
	int ret = 0;

	*libobd = obd_alloc();
	if (!*libobd) {
		ERROR("Sorry, cannot alocate obd data");
		goto err_obd;
	}

	if (strlen(file) > 0) {
		ret = obd_open_file(*libobd, file);
		if (ret < 0) {
			ERROR("problem with odb_open_file - %s", file);
			goto err_open_dev;
		}

	} else {
		ret = obd_open_simulator(*libobd);
		if (ret < 0) {
			ERROR("problem with odb_open_simulator");
			goto err_open_dev;
		}
	}

	return 0;

err_open_dev:
	obd_close(libobd);
err_obd:
	obd_dealloc(libobd);
	*libobd = NULL;
	return ret;
}

static int setup_libobd(void **obdstruct, char *rfcomm)
{
	int ret = 0;

	*obdstruct = obd_alloc();
	if (!*obdstruct) {
		ERROR("Sorry, cannot alocate obd data");
		goto err_obd;
	}

	ret = obd_open_dev(*obdstruct, rfcomm);
	if (ret < 0) {
		ERROR("Problem with odb_open_dev, using: %s", rfcomm);
		goto err_open_dev;
	}

	ret = obd_channel_setup(*obdstruct);
	if (ret < 0) {
		ERROR("Problems setuping the obd channel, using: %s", rfcomm);
		goto err_setup_channel;
	}

	return 0;

err_setup_channel:
err_open_dev:
	obd_close(*obdstruct);
	obd_dealloc(*obdstruct);
	*obdstruct = NULL;
err_obd:
	return ret;
}

static int block(int timeout)
{
	int poll_ret;
	char buf;
	int err;

	if (timeout < 0) {
		timeout = -1;
	} else {
		timeout = timeout * 1000;
	}

	DEBUG("Blocking - %d", timeout);

	/* FIXME: treat signal interrupt - EINTR */
	is_blocked = 1;
	poll_ret = poll(&g_pollfd, 1, timeout);
	err = errno;
	is_blocked = 0;

	if (poll_ret > 0) {
		int x = poll_ret;
		if (g_pollfd.revents == 0)
			goto unknown_reason;

		DEBUG("Unblocking - reading buffer... (events: %d/pollret: %d)",
						g_pollfd.revents, poll_ret);
		while (x > 0) {
			read(g_pipefd[PIPE_READ], &buf, 1);
			DEBUG("Unblocking  (%02d/%02d) - buffer: %c",
					(poll_ret - x) + 1, poll_ret, buf);
			x--;
		}
	} else if (poll_ret == 0) {
		DEBUG("Unblocking by timeout - %d", timeout);
	} else
		goto unknown_reason;

	return poll_ret;

unknown_reason:
	DEBUG("Unblocking... unknown reason, err: %s (%d)",
				strerror(err), err);
	return -1;
}

static int unblock(char db_char)
{
	DEBUG("unblock() called, reason: %c", db_char);
	return write(g_pipefd[PIPE_WRITE], &db_char, 1);
}

static int unblock_if_applicable(char db_char)
{

	if (!obd_thread_is_connected())
		return -1;

	if (!is_blocked)
		return -2;

	return unblock(db_char);
}

static int notify_main_thread(char data)
{
	return write(g_main_pipefd[1], &data, 1);
}

static int verify_connection(void)
{
	struct obd_sensor_data psd;

	INFO("Checking obd connection state");

	psd.pid = CHK_CONNECTION_PID;

	return obd_sendpid(libobd, &psd, OBD_TIMEOUT);
}

static int change_state_to(int ns, char *data)
{
	int notify_local = 0;
	int notify_main = 0;

	if state(TOBD_DISCONNECTED, TOBD_START_BT_CONNECTION) {
		obd_bt_id = strdup(data);

		notify_main = 1;
		notify_local = 1;
	} else if state(TOBD_DISCONNECTED, TOBD_START_SIMULATOR) {
		obd_simulator_file = strdup(data);

		notify_main = 1;
		notify_local = 1;
	} else if state(TOBD_CONNECTING_BT, TOBD_START_SERIAL_CONNECTION) {
		obd_serial_port = strdup(data);

		notify_local = 1;
	} else if (state(TOBD_START_BT_CONNECTION, TOBD_DISCONNECTED) ||
		state(TOBD_START_BT_CONNECTION, TOBD_QUIT) ||
		state(TOBD_CONNECTING_BT, TOBD_DISCONNECTED)) {

		free(obd_bt_id);
		obd_bt_id = NULL;

		notify_main = 1;
	} else if (state(TOBD_CONNECTING_SERIAL, TOBD_DISCONNECTED) ||
		state(TOBD_CONNECTED, TOBD_DISCONNECTED) ||
		state(TOBD_START_DISCONNECTION, TOBD_DISCONNECTED) ||
		state(TOBD_START_SIMULATOR, TOBD_DISCONNECTED) ||
		state(TOBD_UNEXPECTED_DISCONNECTION, TOBD_DISCONNECTED) ||
		state(TOBD_CONNECTING_SERIAL, TOBD_QUIT) ||
		state(TOBD_CONNECTED, TOBD_QUIT) ||
		state(TOBD_START_SIMULATOR, TOBD_QUIT) ||
		state(TOBD_START_DISCONNECTION, TOBD_QUIT)) {

		if (libobd) {
			obd_close(libobd);
			obd_dealloc(libobd);
		}

		if (obd_bt_id) {
			if (disconnect_service(obd_serial_port) < 0)
				ERROR("Problems removing bluetooth connection");
			free(obd_bt_id);
			obd_bt_id = NULL;
		}

		if (obd_serial_port) {
			free(obd_serial_port);
			obd_serial_port = NULL;
		}

		if (obd_simulator_file) {
			free(obd_simulator_file);
			obd_simulator_file = NULL;;
		}

		notify_main = 1;
	} else if (state(TOBD_CONNECTING_SERIAL, TOBD_CONNECTED) ||
		state(TOBD_START_SIMULATOR, TOBD_CONNECTED)) {

		notify_local = 1;
		notify_main = 1;
		obdsched_clear_bl();
	} else if (state(TOBD_CONNECTED, TOBD_UNEXPECTED_DISCONNECTION)) {

		notify_local = 1;
		notify_main = 1;
	} else if (state(TOBD_CONNECTED, TOBD_START_DISCONNECTION)) {

		notify_local = 1;
		/*notify_main = 1;*/
	} else if (state(TOBD_DISCONNECTED, TOBD_QUIT)) {

		notify_local = 1;
		/*notify_main = 1;*/
	} else if (state(TOBD_START_BT_CONNECTION, TOBD_CONNECTING_BT) ||
		state(TOBD_START_SERIAL_CONNECTION, TOBD_CONNECTING_SERIAL)) {
		/* nop */
	} else {
		goto unknown_state;
	}

	DEBUG("STATE CHANGED: from %d to %d",
			obd_thread_status,
			ns);

	obd_thread_status = ns;

	if (notify_main)
		notify_main_thread('C');

	if (notify_local)
		unblock('C');

	return 0;

unknown_state:
	INFO("Opps, unable to change from state %d to %d",
		obd_thread_status, ns);

	if (state(TOBD_START_BT_CONNECTION, TOBD_CONNECTING_BT) ||
		state(TOBD_CONNECTING_BT, TOBD_START_BT_CONNECTION)) {
		INFO("[Already started the \"connecting process\"...]");
	} else if state(TOBD_START_BT_CONNECTION, TOBD_START_BT_CONNECTION) {
		INFO("[hunf? from start_connection to start_connection? ...]");
	} else if state(TOBD_CONNECTED, TOBD_START_BT_CONNECTION) {
		INFO("[Already connected...]");
	}

	return -1;
}

static int calc_elapsed_time(struct timeval *stv, struct timeval *etv)
{
	if (etv->tv_usec > stv->tv_usec)
		return (etv->tv_usec - stv->tv_usec)/1000;
	else
		return (1000000 + etv->tv_usec - stv->tv_usec)/1000;
}

static struct obd_sensor_data *getnext_from_obd(int pid)
{
	struct obd_sensor_data *psd;
	struct timeval stv, etv;
	int ret;

	INFO("New sensor to check: %d", pid);

	psd = (struct obd_sensor_data *) malloc(
				sizeof(struct obd_sensor_data)
				);
	psd->pid = pid;

	gettimeofday(&stv, NULL);
	ret = obd_sendpid(libobd, psd, OBD_TIMEOUT);

	gettimeofday(&etv, NULL);
	DEBUG("Sensor: [0x%02x/%02d] retrieved in %03dms", pid, pid,
				calc_elapsed_time(&stv, &etv));

	if (ret < 0) {
		INFO("Connection time out when asking for the sensor: %d",
			pid);
		change_state_to(TOBD_UNEXPECTED_DISCONNECTION, NULL);
		/* FIXME: Leak? Returning invalid value? */
		goto none;
	}

	if (psd->valid & OBD_INVALID_DATA) {
		obdsched_add_bl(pid);
		/* FIXME: Leak? */
		psd = NULL;
		goto none;
	}

	if (pid == RPM_PID) {
		memcpy(&fresh_rpm, psd, sizeof(struct obd_sensor_data));
	} else if (pid == SPEED_PID) {
		memcpy(&fresh_speed, psd, sizeof(struct obd_sensor_data));
	}

none:
	return psd;
}

static void ml_start_bt_connection(void)
{
	char *serial;
	change_state_to(TOBD_CONNECTING_BT, NULL);

	INFO("Processing the start bt connection request"
		", trying to connect to: %s", obd_bt_id);

	if (setup_spp_connection(obd_bt_id, &serial) < 0) {
		ERROR("Problems connecting obd device");

		change_state_to(TOBD_DISCONNECTED, NULL);
	} else {
		INFO("Bluetooth connection is ok! rfcomm: %s",
			serial);

		change_state_to(TOBD_START_SERIAL_CONNECTION, serial);
		free(serial);
	}
}

static void ml_start_serial_connection(void)
{
	int ret = 0;

	INFO("Processing the start serial connection request, trying to"
		"connect to: %s", obd_serial_port);

	change_state_to(TOBD_CONNECTING_SERIAL, NULL);

	ret = setup_libobd(&libobd, obd_serial_port);
	if (ret < 0) {
		ERROR("Problems setuping obd");

		change_state_to(TOBD_DISCONNECTED, NULL);
	} else {
		INFO("OBD device is ready");

		change_state_to(TOBD_CONNECTED, NULL);
	}
}

static void ml_start_simulator(void)
{
	int ret;

	INFO("Processing the start simulator request, trying to"
		"connect to: %s", obd_simulator_file);

	ret = setup_libobd_simulator(&libobd, obd_simulator_file);
	if (ret < 0) {
		ERROR("Problems setuping obd");

		change_state_to(TOBD_DISCONNECTED, NULL);
	} else {
		INFO("OBD device is ready");

		change_state_to(TOBD_CONNECTED, NULL);
	}
}

static int list_dtc(void)
{
	struct obd_dtc_data dtc_data;
	int i;

	if (!libobd) {
		ERROR("libobd is not valid?");
		return -1;
	}

	if (obd_senddtc(libobd, &dtc_data, 5) < 0) {
		ERROR("Error gettting dtc list\n");
		return -2;
	}

	if (dtc_data.valid < 0) {
		ERROR("Got a not valid DTC reply\n");
	} else {
		DEBUG("DTC list retrieved, num of DTCs: %d", dtc_data.valid);
		for (i = 0; i < dtc_data.valid; i++) {
			DEBUG("DTC: \"%s\" ", dtc_data.code_list[i]);
			g_dtc_list = g_slist_append(g_dtc_list,
					g_strdup(dtc_data.code_list[i]));
		}
	}

	return 0;
}

static void ml_connected(void)
{
	int pid;
	int t;

	if (dtc_list_request) {
		if (!list_dtc())
			notify_main_thread('P');

		dtc_list_request = 0;
		return;
	}

	pid = obdsched_getnext();
	if (pid >= 0) {
		DEBUG("Checking pid: %d", pid);
		struct obd_sensor_data *psd = getnext_from_obd(pid);

		if (!psd)
			return;

		g_async_queue_push(g_rtqueue, psd);
		notify_main_thread('D');

		return;
	}

	t = obdsched_timenext();
	INFO("Connected, next sensor in %d seconds", t);

	if (t < 0 || t > CHK_CONNECTION_INTERVAL) {
		block(CHK_CONNECTION_INTERVAL);
		if (verify_connection())
			change_state_to(TOBD_UNEXPECTED_DISCONNECTION, NULL);
			return;
	}

	block(t);

	return;
}

static gboolean obd_thread_main_loop(void)
{
	for (;;) {
		switch (obd_thread_status) {
		case TOBD_DISCONNECTED:
			INFO("I'm disconnected...");
			block(-1);
			break;
		case TOBD_START_BT_CONNECTION:
			ml_start_bt_connection();
			break;
		case TOBD_START_SERIAL_CONNECTION:
			ml_start_serial_connection();
			break;
		case TOBD_START_SIMULATOR:
			ml_start_simulator();
			break;
		case TOBD_CONNECTED:
			ml_connected();
			break;
		case TOBD_START_DISCONNECTION:
		case TOBD_UNEXPECTED_DISCONNECTION:
			change_state_to(TOBD_DISCONNECTED, NULL);
			break;
		case TOBD_QUIT:
			DEBUG("Finishing OBD thread");
			return FALSE;
			break;
		default:
			ERROR("Unknown state: %d", obd_thread_status);
			block(-1);
			break;
		}
	}

	return TRUE;
}

/* 
 * Change the state of the thread_obd to START_CONNECT and wait for
 * the thread loop to call obd_thread_start_connect().
 * 
 * The schedule usually blocks when no sensor needs to be updated.
 * we can call schedule_unblock() to unblock it, and finnaly process
 * our connection requisition.
 *
 * @param char *addr Address of the wished bt device.
 *
 */
int obd_thread_connect_bt(char *addr)
{
	INFO("Rcvd a request to schedule a request to connect to a bt id: %s",
		addr);

	return change_state_to(TOBD_START_BT_CONNECTION, addr);
}

/*
 * Change the state of tobd to CONNECT_SERIAL.
 *
 * @param char *serial 
 *
 */
int obd_thread_connect_serial(char *serial)
{
	INFO("Rcvd a request to schedule a connection to a serial: %s",
		serial);

	return change_state_to(TOBD_START_SERIAL_CONNECTION, serial);
}

/*
 * Change the state of tobd to CONNECT_SERIAL.
 *
 * @param char *serial 
 *
 */
int obd_thread_start_simulator(char *file)
{
	INFO("Rcvd a request to schedule a simulator: %s",
		file);

	return change_state_to(TOBD_START_SIMULATOR, file);
}

/*
 * Change the state of the machine state to 
 * START_DISCONNECTION and ask the schedule
 * to unblock.
 *
 */
int obd_thread_disconnect(void)
{
	return change_state_to(TOBD_START_DISCONNECTION, NULL);
}

const char *obd_thread_connection_status(void)
{
	const char *connected = "Connected";
	const char *connecting = "Connecting";
	const char *disconnected = "Disconnected";

	if (obd_thread_is_connected()) {
		return connected;
	} else if (obd_thread_is_connecting()) {
		return connecting;
	} else if (obd_thread_is_disconnected()) {
		return disconnected;
	}

	return NULL;
}

static gboolean obd_watch(GIOChannel *io, GIOCondition cond, gpointer user_data)
{
	int fd;
	char buf;

	if (cond & G_IO_NVAL) {
		ERROR("obd_watch: NVAL");
		return FALSE;
	}

	if (cond & (G_IO_HUP | G_IO_ERR)) {
		ERROR("obd_watch: HUP");
		return FALSE;
	}

	fd = g_io_channel_unix_get_fd(io);
	if (read(fd, &buf, 1) < 0) {
		int err = errno;
		ERROR("read(): %s(%d)", strerror(err), err);
		return TRUE;
	}

	switch (buf) {
	struct obd_sensor_data *sensor_data;
	case 'D':
		sensor_data = obd_thread_get_next();
		DEBUG("Pid [0x%02x/%02d] Result[0]: %f (Valid: %d)",
				sensor_data->pid, sensor_data->pid,
				sensor_data->result[0],
				sensor_data->valid);

		if (sensor_data->valid & OBD_HAS_SECOND_VALUE)
			DEBUG("             Result[1]: %f",
					sensor_data->result[1]);
		else
			sensor_data->result[1] = 0;

		if ((sensor_data->valid & OBD_HAS_FIRST_VALUE_IMPERIAL) !=
					OBD_HAS_FIRST_VALUE_IMPERIAL) {
			sensor_data->rimperial[0] = sensor_data->result[0];
		}

		if ((sensor_data->valid & OBD_HAS_SECOND_VALUE_IMPERIAL) !=
					OBD_HAS_SECOND_VALUE_IMPERIAL) {
			sensor_data->rimperial[1] = sensor_data->result[1];
		}

		dbus_emit_obd_data(sensor_data->pid,
				sensor_data->result[0],
				sensor_data->result[1],
				sensor_data->rimperial[0],
				sensor_data->rimperial[1]);
		free(sensor_data);

		break;
	case 'C':
		DEBUG("State of the connection changed, now it is: %s",
			obd_thread_connection_status());
		dbus_emit_obd_status(obd_thread_connection_status());
		break;
	case 'P':
		DEBUG("Received a DTC List");

		dbus_emit_obd_dtc(g_dtc_list);

		g_slist_foreach(g_dtc_list, (GFunc) g_free, NULL);
		g_slist_free(g_dtc_list);
		g_dtc_list = NULL;

		break;
	default:
		DEBUG("I have no idea what i need to do!");
		break;
	}

	return TRUE;
}

static void obd_watch_destroy(gpointer user_data)
{
}

int obd_init(void)
{
	GIOChannel *io;
	int ret = 0;

	g_type_init();
	g_thread_init(NULL);
	obdsched_init();

	fresh_rpm.valid = 0;
	fresh_speed.valid = 0;

	if (!init_bt_serial()) {
		ERROR("Could not init the DBUS_BUS_SYSTEM");
		goto error;
	}

	if (pipe(g_main_pipefd)) {
		ret = errno;
		ERROR("Could not create the main pipe: %s (%d)",
			strerror(ret), ret);
		goto error;
	}

	if (pipe(g_pipefd)) {
		ret = errno;
		ERROR("Could not create the block pipe: %s (%d)",
			strerror(ret), ret);
		goto error;
	}

	g_rtqueue = g_async_queue_new();

	io = g_io_channel_unix_new(g_main_pipefd[0]);
	g_io_add_watch_full(io, G_PRIORITY_DEFAULT,
			G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
			obd_watch, NULL, obd_watch_destroy);
	g_io_channel_unref(io);

	g_pollfd.fd = g_pipefd[PIPE_READ];
	g_pollfd.events = POLLIN | POLLPRI;

	obd_thread_loop = g_thread_create ((GThreadFunc) obd_thread_main_loop,
						NULL, TRUE, NULL);
	return 0;

error:
	DEBUG("Problems initing the obd-thread");
	if (g_main_pipefd[PIPE_READ]) {
		close(g_pipefd[PIPE_READ]);
		g_pipefd[PIPE_READ] = 0;
	}

	if (g_main_pipefd[PIPE_WRITE]) {
		close(g_pipefd[PIPE_WRITE]);
		g_pipefd[PIPE_WRITE] = 0;
	}

	if (g_pipefd[PIPE_READ]) {
		close(g_pipefd[PIPE_READ]);
		g_pipefd[PIPE_READ] = 0;
	}

	if (g_pipefd[PIPE_WRITE]) {
		close(g_pipefd[PIPE_WRITE]);
		g_pipefd[PIPE_WRITE] = 0;
	}

	g_pollfd.fd = 0;
	g_pollfd.events = 0;

	return ret;
}

struct obd_sensor_data *obd_thread_get_next(void)
{
	struct obd_sensor_data *sensor_data;

	sensor_data = g_async_queue_pop(g_rtqueue);

	return sensor_data;
}

int obd_thread_add_time_sensor(int pid, int interval)
{
	int ret;
	ret = obdsched_add_tsensor(pid, interval);

	DEBUG("Unblock the thread? %d.", unblock_if_applicable('S'));
	unblock_if_applicable('S');

	return ret;
}

int obd_thread_del_time_sensor(int pointer)
{
	int ret;
	ret = obdsched_del_tsensor(pointer);

	return ret;
}

int obd_thread_add_round_sensor(int pid, int interval, int round)
{
	int ret;
	ret = obdsched_add_rsensor(pid, interval, round);

	DEBUG("Unblock the thread? %d.", unblock_if_applicable('S'));

	return ret;
}

int obd_thread_del_round_sensor(int pointer)
{
	int ret;
	ret = obdsched_del_rsensor(pointer);

	return ret;
}

int obd_thread_get_last_data(struct obd_sensor_data **rpm,
				struct obd_sensor_data **speed)
{
	*rpm = &fresh_rpm;
	*speed = &fresh_speed;

	if (!obd_thread_is_connected())
		return -1;

	return 0;
}

int obd_thread_list_dtc(void)
{
	if (!obd_thread_is_connected())
		return -1;

	if (dtc_list_request != 1) {
		dtc_list_request = 1;
		unblock('P');
		return 0;
	}

	return -2;
}

int obd_thread_exit(void)
{
	DEBUG("Disconnecting OBD devices...");
	change_state_to(TOBD_QUIT, NULL);

	DEBUG("Wating the thread join...");
	g_thread_join(obd_thread_loop);

	DEBUG("Exiting obdsched...");
	obdsched_exit();

	DEBUG("Exiting bt serial...");
	quit_bt_serial();

	return 1;
}

