/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <glib.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>

#include "obd-thread.h"
#include "obd-sched.h"
#include "log.h"

#define MAX_PERIOD 128

struct rsensor_t {
	int pid;
	unsigned int period;
	unsigned int offset;
};

struct tsensor_t {
	int pid;
	unsigned int period;
	time_t exp_time;
};

static GStaticMutex g_sched_mutex = G_STATIC_MUTEX_INIT;

static unsigned int g_round = 0;

static GSList *g_rlist = NULL; /* round roing list */
static GSList *g_tlist = NULL; /* time based list */
static GSList *g_prcur = NULL; /* pointer for current in the round */
static GHashTable *g_blist = NULL;

#define FAILS_BEFORE_BL 5

static int compare_time(const void *a, const void *b)
{
	const struct tsensor_t *pa = a;
	const struct tsensor_t *pb = b;

	return pa->exp_time - pb->exp_time;
}

static int bl_haspid(int pid)
{
	void *p;

	p = g_hash_table_lookup(g_blist, GINT_TO_POINTER(pid));
	if (p)
		DEBUG("This sensor already fails %d time", GPOINTER_TO_INT(p));

	if (p && GPOINTER_TO_INT(p) > FAILS_BEFORE_BL-1)
		return 1;

	return 0;
}

static int rl_haspid(struct tsensor_t *pts)
{
	struct rsensor_t *prs;
	GSList *next = g_rlist;

	while (next) {
		prs = next->data;
		if (prs->pid == pts->pid) {
			pts->exp_time = time(NULL) + pts->period;
			g_tlist = g_slist_sort(g_tlist, compare_time);
			DEBUG("Time scheduled pid: %d is on the rlist. Ignoring",
					pts->pid);
			return 1;
		}

		next = g_slist_next(next);
	}

	DEBUG("Time scheduled pid %d is not on the rlist, proceeding...",
			pts->pid);

	return 0;
}

static int get_next_exptime()
{
	struct tsensor_t *pts;
	GSList *next = g_tlist;
	int pid = -1;
	int delta = -1;

	if (!g_tlist)
		return -1;

	while (pid < 0 && next) {
		pts = next->data;
		if (!bl_haspid(pts->pid) && !rl_haspid(pts)) {
			pid = pts->pid;
			delta = pts->exp_time - time(NULL);
		}
		next = g_slist_next(next);
	}

	if (pid < 0)
		return -1;

	if (delta < 0)
		return 0;

	return delta;
}

static int get_num_round()
{
	struct rsensor_t *prs;
	GSList *next = g_rlist;
	int num = 0;

	if (!g_rlist)
		return 0;

	while (next) {
		prs = next->data;
		if (!bl_haspid(prs->pid))
			num++;
		next = g_slist_next(next);
	}
	return num;
}

static int get_nextround()
{
	int pid = -1;
	struct rsensor_t *prs;
	GSList *next;

	if (g_prcur == NULL)
		g_prcur = g_rlist;

	next = g_prcur;

	while (pid < 0) {
		prs = next->data;
		if (!bl_haspid(prs->pid) && g_round % prs->period == prs->offset) {
			g_prcur = next->next;
			pid = prs->pid;
		}
		next = g_slist_next(next);
		if (next == NULL) {
			g_round++;
			next = g_rlist;
		}
	}

	return pid;
}

static int get_nexttime()
{
	struct tsensor_t *pts;
	GSList *next = g_tlist;
	int pid = -1;

	if (!g_tlist)
		return -1;

	while (pid < 0 && next) {
		pts = next->data;
		if (!bl_haspid(pts->pid)) {
			pid = pts->pid;
			pts->exp_time = time(NULL) + pts->period;
			g_tlist = g_slist_sort(g_tlist, compare_time);
		}
		next = g_slist_next(next);
	}

	if (pid < 0)
		return -1;

	return pid;
}

int obdsched_add_tsensor(int pid, int period)
{
	struct tsensor_t *pts;

	if (period < 0 || period > 7200) {
		ERROR("Wrong value for period - %d", period);
		return -1;
	}

	pts = g_slice_new0(struct tsensor_t);
	if (pts == NULL) {
		ERROR("Could not alloc memory to tsensor_t data");
		return -1;
	}

	DEBUG("Adding time sensor %d with period: %d", pid, period);

	pts->period = period;
	pts->pid = pid;
	pts->exp_time = time(NULL) + period;

	g_static_mutex_lock(&g_sched_mutex);
	g_tlist = g_slist_insert_sorted(g_tlist, pts, compare_time);
	g_static_mutex_unlock(&g_sched_mutex);

	return (int) pts;
}

int obdsched_del_tsensor(int pointer)
{
	struct tsensor_t *pts;
	GSList *next = g_tlist;
	int num = 0;

	g_static_mutex_lock(&g_sched_mutex);
	while (next) {
		pts = next->data;
		if (pts == pointer) {
			g_tlist = g_slist_remove(g_tlist, pts);
			next = g_tlist;
			g_slice_free(struct tsensor_t, pts);
			num++;
			continue;
		}
		next = g_slist_next(next);
	}
	g_static_mutex_unlock(&g_sched_mutex);

	return num;
}

static int obdsched_clear_tsensor()
{
	struct tsensor_t *pts;
	GSList *next = g_tlist;
	int num = g_slist_length(g_tlist);

	g_static_mutex_lock(&g_sched_mutex);
	while (next) {
		pts = next->data;
		g_tlist = g_slist_remove(g_tlist, pts);
		g_slice_free(struct tsensor_t, pts);
		next = g_tlist;
	}
	g_static_mutex_unlock(&g_sched_mutex);

	return num;
}

int obdsched_add_rsensor(int pid, int period, int offset)
{
	struct rsensor_t *prs;

	if (period <= 0 || period > 120) {
		ERROR("Wrong value for period - %d", period);
		return -1;
	}

	if (period <= offset) {
		ERROR("Wrong offset number - %d", offset);
		return -1;
	}

	prs = g_slice_new0(struct rsensor_t);

	if (prs == NULL) {
		ERROR("Could not alloc memory to tsensor_t data");
		return -1;
	}

	DEBUG("Adding round sensor %d with period: %d and offset: %d",
			pid, period, offset);

	prs->pid = pid;
	prs->period = period;
	prs->offset = offset;

	g_static_mutex_lock(&g_sched_mutex);
	g_rlist = g_slist_append(g_rlist, prs);
	g_static_mutex_unlock(&g_sched_mutex);

	return (int) prs;

}

int obdsched_del_rsensor(int pointer)
{
	struct rsensor_t *prs;
	GSList *next = g_rlist;
	int num = 0;

	DEBUG("Removing round sensor: %d (%d removed/%d total)",
		pointer, num, g_slist_length(g_rlist));

	g_static_mutex_lock(&g_sched_mutex);
	while (next) {
		prs = next->data;
		if (prs == pointer) {
			if (g_prcur == next)
				g_prcur = next->next;
			g_rlist = g_slist_remove(g_rlist, prs);
			next = g_rlist;
			g_slice_free(struct rsensor_t, prs);
			num++;
			continue;
		}
		next = g_slist_next(next);
	}
	if (g_prcur == NULL) {
		g_prcur = g_rlist;
		g_round++;
	}
	g_static_mutex_unlock(&g_sched_mutex);

	DEBUG("Round sensor: %d removed. (%d removed/%d total)",
		pointer, num, g_slist_length(g_rlist));


	return num;
}

static int obdsched_clear_rsensor()
{
	struct rsensor_t *prs;
	GSList *next = g_rlist;
	int num = g_slist_length(g_rlist);

	g_static_mutex_lock(&g_sched_mutex);
	while (next) {
		prs = next->data;
		g_rlist = g_slist_remove(g_rlist, prs);
		g_slice_free(struct rsensor_t, prs);
		next = g_rlist;
	}
	g_prcur = NULL;
	g_round = 0;
	g_static_mutex_unlock(&g_sched_mutex);

	return num;
}

int obdsched_clear_all()
{
	int num;

	num = obdsched_clear_tsensor();
	num += obdsched_clear_rsensor();

	return num;
}

int obdsched_add_bl(int pid)
{
	int inc = 0;
	void *p;

	INFO("Adding sensor %d to black list.\n", pid);

	g_static_mutex_lock(&g_sched_mutex);
	p = g_hash_table_lookup(g_blist, GINT_TO_POINTER(pid));
	if (p)
		inc = GPOINTER_TO_INT(p);

	inc++;
	g_hash_table_insert (g_blist, GINT_TO_POINTER(pid), GINT_TO_POINTER(inc));
	g_static_mutex_unlock(&g_sched_mutex);

	return 0;
}

int obdsched_clear_bl()
{
	int num = g_hash_table_size(g_blist);

	g_static_mutex_lock(&g_sched_mutex);
	g_hash_table_remove_all(g_blist);
	g_static_mutex_unlock(&g_sched_mutex);

	return num;
}

int obdsched_getnext()
{
	int nextt, nround, ret;

	g_static_mutex_lock(&g_sched_mutex);

	nextt = get_next_exptime();
	nround = get_num_round();

	if (nextt < 0 || nextt > 0) {
		if (nround == 0) {/* no time no round */
			DEBUG("no round or time sensor");
			ret = -1;
		} else {
			ret = get_nextround();
			DEBUG("getting next round sensor: %03d", ret);
		}
	} else {
		ret = get_nexttime();
		DEBUG("getting next time: %03d", ret);
	}

	g_static_mutex_unlock(&g_sched_mutex);

	DEBUG("Get next: ret: %d, round: %d, time: %d", ret, nround, nextt);
	return ret;
}

int obdsched_timenext()
{
	int ret;

	g_static_mutex_lock(&g_sched_mutex);
	ret = get_next_exptime();
	g_static_mutex_unlock(&g_sched_mutex);

	return ret;
}

int obdsched_init()
{
	g_blist = g_hash_table_new(g_direct_hash, g_direct_equal);

	return 0;
}

int obdsched_exit()
{
	int num = 0;

	num = obdsched_clear_all();
	num += obdsched_clear_bl();

	g_hash_table_destroy(g_blist);

	return num;
}

