/*
 *
 *  carmand - Carman Daemon
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/un.h>
#include <unistd.h>
#include <wait.h>
#include <glib.h>
#include <stdint.h>
#include <time.h>

#include "dbus.h"
#include "log.h"

#define LOC_WRAPPER "/usr/bin/locationwrapper"
#define WRAPPER_SOCKET_NAME "\0/org/indt/location-wrapper"

#define GPS_NORMAL_MODE		0x00
#define GPS_SIMULATOR_MODE	0x01
#define GPS_CMD_CONNECT		0x02
#define GPS_CMD_DISCONNECT	0x04

enum {
	GPS_EVT_ERROR,
	GPS_EVT_DISCONNECTED,
	GPS_EVT_CONNECTED,
	GPS_EVT_FIXING,
	GPS_EVT_FIXED
};

typedef enum {
	GPS_DISCONNECTED,
	GPS_CONNECTING,
	GPS_FIXING,
	GPS_FIXED
} gps_status_t;

#define MAX_GPS_CMD_SIZE	32
typedef struct {
	guint8		type;
	guint32		size;
	guint8		data[0];
} gps_cmd_t, gps_evt_t;

static location_t last_gps_fix;
static pid_t chldpid = 0;
static int wsk = -1;
static guint watch = 0;
static gps_status_t gps_status = GPS_DISCONNECTED;
static guint8 gps_conn_type = GPS_SIMULATOR_MODE;

static gboolean wrapper_watch(GIOChannel *io,
			GIOCondition cond, gpointer user_data)
{
	uint8_t buf[148];
	gps_evt_t *evt = (gps_evt_t *) buf;
	location_t *fix;
	int ret, fd;

	if (cond & G_IO_NVAL) {
		ERROR("wrapper_watch: NVAL");
		return FALSE;
	}

	if (cond & (G_IO_HUP | G_IO_ERR)) {
		ERROR("wrapper_watch: HUP");
		return FALSE;
	}

	fd = g_io_channel_unix_get_fd(io);

	ret = read(fd, &buf, sizeof(buf));
	if(ret < sizeof(gps_evt_t))
		return TRUE;

	switch (evt->type) {
	case GPS_EVT_ERROR: /* Connection attempt failed */
	case GPS_EVT_DISCONNECTED:
		dbus_emit_gps_status("Disconnected");
		gps_status = GPS_DISCONNECTED;
		break;

	case GPS_EVT_CONNECTED:
		/*
		 * Workaround to allow disconnect before
		 * receive the first fixing data. It is
		 * also used to notify the UI that the
		 * GPS is already connected.
		 */
		gps_status = GPS_FIXING;
		dbus_emit_gps_status("Fixing");
		break;
	case GPS_EVT_FIXING:
		if (gps_status != GPS_FIXING) {
			dbus_emit_gps_status("Fixing");
			gps_status = GPS_FIXING;
		}
		break;
	case GPS_EVT_FIXED:
		if (gps_status != GPS_FIXED) {
			dbus_emit_gps_status("Fixed");
			gps_status = GPS_FIXED;
		}
		fix = (location_t *) evt->data;

		/* memcpy(&last_gps_fix, fix, sizeof(location_t)); */

		last_gps_fix.latitude = fix->latitude;
		last_gps_fix.longitude = fix->longitude;
		last_gps_fix.altitude = fix->altitude;
		last_gps_fix.mode = fix->mode;
		last_gps_fix.track = fix->track;
		last_gps_fix.speed = fix->speed;
		dbus_emit_gps_data(fix);
		break;
	}

	return TRUE;
}

static void wrapper_watch_destroy(gpointer user_data)
{
	wsk = -1;
	watch = 0;
}

static pid_t start_wrapper(const char *cmd)
{
	pid_t pid;

	pid = fork();
	if (pid < 0)
		return -errno;
	else if (pid == 0) {
		execv(cmd, NULL);
		exit(1);
	}

	if (waitpid(pid, NULL, WNOHANG))
		pid = -1;

	return pid;
}

int location_init(void)
{
	struct stat st;

	if (stat(LOC_WRAPPER, &st) < 0) {
		int err = errno;
		ERROR("stat(%s): %s(%d)",
				LOC_WRAPPER, strerror(err), err);
		return -err;
	}
	chldpid = start_wrapper(LOC_WRAPPER);
	if (chldpid < 0)
		return -1;

	last_gps_fix.latitude = 0;
	last_gps_fix.longitude = 0;
	last_gps_fix.altitude = 0;
	last_gps_fix.mode = 0;
	last_gps_fix.track = 0;
	last_gps_fix.speed = 0;


	return 0;
}

void location_exit(void)
{
	if (watch)
		g_source_remove(watch);

	if (chldpid > 0)
		kill(chldpid, SIGTERM);
}

static int wrapper_connect(void)
{
	GIOChannel *io;
	int sk, err;
	struct sockaddr_un addr = {
		AF_UNIX, WRAPPER_SOCKET_NAME
	};

	sk = socket(PF_LOCAL, SOCK_SEQPACKET, 0);
	if (sk < 0) {
		err = errno;
		return -err;
	}

	if (connect(sk, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		err = errno;
		close(sk);
		return -err;
	}

	io = g_io_channel_unix_new(sk);
	g_io_channel_set_close_on_unref(io, TRUE);
	watch = g_io_add_watch_full(io, G_PRIORITY_DEFAULT,
			G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
			wrapper_watch, NULL, wrapper_watch_destroy);
	g_io_channel_unref(io);

	return sk;
}

int location_connect(const char *addr)
{
	guint8 buf[MAX_GPS_CMD_SIZE];
	gps_cmd_t *cmd = (gps_cmd_t *) buf;
	gint32 size, dsize;
	int err;

	if (!addr)
		return -EINVAL;

	dsize = strlen(addr);
	size = sizeof(gps_cmd_t) + dsize;
	if (size >= MAX_GPS_CMD_SIZE)
		return -EINVAL;

	cmd->type = GPS_CMD_CONNECT;
	cmd->size = dsize;
	strcpy((char *) cmd->data, addr);
	if (wsk < 0) {
		err = wrapper_connect();
		if (err < 0) {
			ERROR("wrapper connection: %s(%d)", strerror(-err),
							-err);
			return err;
		}

		wsk = err;
	}

	if (gps_status != GPS_DISCONNECTED) {
		ERROR("GPS already connected or in progress");
		return -EALREADY;
	}

	if (strcmp("simulator", addr) == 0) {
		cmd->type |= GPS_SIMULATOR_MODE;
		gps_conn_type = GPS_SIMULATOR_MODE;
	} else
		gps_conn_type = GPS_NORMAL_MODE;

	if (write(wsk, cmd, size) != size) {
		err = errno;
		ERROR("write(): %s(%d)", strerror(err), err);
		return -err;
	}

	gps_status = GPS_CONNECTING;
	dbus_emit_gps_status("Connecting");

	return 0;
}

int location_disconnect()
{
	guint8 buf[MAX_GPS_CMD_SIZE];
	gps_cmd_t *cmd = (gps_cmd_t *) buf;
	size_t size;
	int err;

	if (wsk < 0) {
		ERROR("location wrapper not connected!");
		return -EIO;
	}

	if (gps_status == GPS_DISCONNECTED || gps_status == GPS_CONNECTING) {
		ERROR("GPS not connected!");
		return -EPERM;
	}

	cmd->type = GPS_CMD_DISCONNECT | gps_conn_type;
	cmd->size = 0;
	size = sizeof(gps_cmd_t);

	if (write(wsk, cmd, size) != size) {
		err = errno;
		ERROR("write(): %s(%d)", strerror(err), err);
		return -err;
	}

	gps_status = GPS_DISCONNECTED;

	return 0;
}

const char *location_status(void)
{
	switch (gps_status) {
	case GPS_FIXING:
		return "Fixing";
	case GPS_FIXED:
		return "Fixed";
	case GPS_CONNECTING:
		return "Connecting";
	case GPS_DISCONNECTED:
	default:
		return "Disconnected";
	}
}

int gps_get_last_data(location_t **fix)
{
	*fix = &last_gps_fix;

	if (gps_status != GPS_FIXED)
		return -1;

	return 0;
}
