/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *  
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <Python.h>

typedef struct {
	unsigned long rate;
	unsigned long start_time;
} CarmanDataHeader_t;

typedef struct {
	double lat;
	double lon;
	unsigned int speed:8;
	unsigned int rpm:16;
	unsigned int mask:8;
} CarmanData_t;

typedef enum {
	CARMAN_COORD,
	CARMAN_SPEED,
	CARMAN_RPM,
	CARMAN_MAX
} CarmanDataType_t;

static PyObject *py_list;
static time_t start_time;
static time_t stop_time;
static unsigned long rate;

static PyObject *_leave(PyObject *self, PyObject *args);
static PyObject *_fetchAll(PyObject *self, PyObject *args);
static PyObject *_getSpeed(PyObject *self, PyObject *args);
static PyObject *_getRPM(PyObject *self, PyObject *args);
static PyObject *_getLatLong(PyObject *self, PyObject *args);

PyMethodDef trip_methods[] = {
	{ "leave", _leave, METH_VARARGS,
		"Free all resources." },
	{ "fetchAll", _fetchAll, METH_VARARGS,
		"Fetch all values from a trip .dat file" },
	{ "getSpeed", _getSpeed, METH_VARARGS,
		"getSpeed(time0, time1, n) - Return n speeds" },
	{ "getRPM", _getRPM, METH_VARARGS,
		"getRPM(time0, time1, n) - Return n RPMs" },
	{ "getLatLong", _getLatLong, METH_VARARGS,
		"getLatLong(time) - Retrun lat/long" },

	{ NULL, NULL, 0, NULL }
};

DL_EXPORT(void)
inittrip(void)
{
	Py_InitModule ("trip", trip_methods);
	if (PyErr_Occurred ())
		Py_FatalError ("can't initialise module trip");
}

static PyObject *_leave(PyObject *self, PyObject *args)
{
	Py_DECREF(py_list);
	return Py_BuildValue("N",py_list);
}

static PyObject *_fetchAll(PyObject *self, PyObject *args)
{
	PyObject *py_data;
	CarmanDataHeader_t header;
	CarmanData_t *data;
	char *filename;
	FILE *input = NULL;

	if (!PyArg_ParseTuple(args, "s", &filename)) {
			PyErr_SetString(PyExc_Exception,
				"Problems to parse file name arg.");
			goto err;
	}

	py_list = PyList_New(0);

	input = fopen(filename , "rb");
	if (input == NULL) {
		PyErr_SetString(PyExc_Exception,
				"Error opening file data.");
		goto err;
	}

	if (!fread(&header, sizeof(CarmanDataHeader_t), 1, input)) {
		PyErr_SetString(PyExc_Exception,
				"Invalid file data.");
		goto err;
	}

	data = (CarmanData_t *) malloc(sizeof(CarmanData_t));

	while (fread(data, sizeof(CarmanData_t), 1, input)) {
		py_data = PyCObject_FromVoidPtr(data, NULL);

		if (PyList_Append(py_list, py_data) < 0) {
			PyErr_SetString(PyExc_Exception,
				"Problems appending item to the return list");
			goto err;
		}
		data = (CarmanData_t *) malloc(sizeof(CarmanData_t));
	}

	if (!feof(input)) {
		PyErr_SetString(PyExc_Exception,
				"Error reading file data.");
		goto err;
	}

	rate = header.rate;	/* milsec */
	start_time = header.start_time; /* seconds */
	stop_time = start_time + (rate * PyList_Size(py_list)); /* seconds */

	fclose(input);

	Py_INCREF(py_list);

	return Py_BuildValue("N",py_list);

err:
	if (input != NULL)
		fclose(input);

	return NULL;
}

static PyObject *getData(PyObject *self, PyObject *args, CarmanDataType_t type)
{
	PyObject* py_tuple;
	PyObject *py_data;
	CarmanData_t *data;
	time_t t0, tn;
	int size, t0tn_size;
	int t0_i, tn_i;
	int i, j, k, cnt;
	double value;
	double new_value;


	if (!PyArg_ParseTuple(args, "iii", &t0, &tn, &size)) {
		PyErr_SetString(PyExc_Exception,
			"Problems to parse time and num data args.");
		goto err;
	} else if (size < 1) {
		PyErr_SetString(PyExc_Exception,
			"Invalid size to new data's tuple.");
		goto err;
	} else if (tn < t0) {
		PyErr_SetString(PyExc_Exception,
			"Graphic's start time is greater then stop time.");
		goto err;
	} else if (t0 < start_time) {
		PyErr_SetString(PyExc_Exception,
			"The data start time is greater then graphic's start time.");
		goto err;
	} else if (tn > stop_time) {
		PyErr_SetString(PyExc_Exception,
			"Graphic's stop time is greater then data's stop time.");
		goto err;
	} else if (type != CARMAN_SPEED && type != CARMAN_RPM) {
		PyErr_SetString(PyExc_Exception,
				"Invalid type to getData.");
		goto err;
	}

	if (PyList_Size(py_list) < 1) {
		PyErr_SetString(PyExc_Exception,
				"Carman's data list is empty!.");
		goto err;
	}

	/* FIXME: Review where is the Py_DECREF */

	/* Data's index for graphic's start time */
	t0_i = (t0 - start_time) / rate;
	/* Data's index for graphic's stop time */
	tn_i = (tn - start_time) / rate;

	/* Number of samples in the time interval */
	t0tn_size = tn_i - t0_i;

	if (t0tn_size < size) {
		/* FIXME: Review this */
		size = t0tn_size;
	}

	/* New graphic's tuple */
	py_tuple = PyTuple_New(size);

	j = 0;
	cnt = 0;
	value = 0;
	new_value = 0;
	/* Walk the data's list in the time interval */
	for (i = 0; i < t0tn_size; i++) {
		cnt ++;

		py_data = PyList_GetItem(py_list, t0_i + i);
		data = PyCObject_AsVoidPtr(py_data);

		if (type == CARMAN_SPEED)
			new_value = data->speed;
		else
			new_value = data->rpm;

		/* Index to graphic's tuple */
		k = (size * i) / t0tn_size;
		if (k > j) {
			value /= (cnt - 1);

			py_data = PyFloat_FromDouble(value);
			if (PyTuple_SetItem(py_tuple, j, py_data)) {
				PyErr_SetString(PyExc_Exception,
						"Problems setting item to the return tuple.");
				goto err;
			}
			j++;
			value = 0;
			cnt = 1;
		}
		value += new_value;
	}

	value /= cnt;

	py_data = PyFloat_FromDouble(value);
	if (PyTuple_SetItem(py_tuple, j, py_data)) {
		PyErr_SetString(PyExc_Exception,
				"Problems setting item to the return tuple.");
		goto err;
	}

	return Py_BuildValue("N",py_tuple);

err:
	return NULL;
}

static PyObject *_getSpeed(PyObject *self, PyObject *args)
{
	return getData(self, args, CARMAN_SPEED);
}

static PyObject *_getRPM(PyObject *self, PyObject *args)
{
	return getData(self, args, CARMAN_RPM);
}

static PyObject *_getLatLong(PyObject *self, PyObject *args)
{
	PyObject* py_tuple;
	PyObject *py_data;
	CarmanData_t *data;
	time_t time;
	int t_i;


	if (!PyArg_ParseTuple(args, "i", &time)) {
		PyErr_SetString(PyExc_Exception,
				"Problems to the parse time arg.");
		goto err;
	} else if (time < start_time) {
		PyErr_SetString(PyExc_Exception,
			"The data start time is greater then graphic's time.");
		goto err;
	} else if (time > stop_time) {
		PyErr_SetString(PyExc_Exception,
				"Graphic's time is greater then data's stop time.");
		goto err;
	}

	if (PyList_Size(py_list) < 1) {
		PyErr_SetString(PyExc_Exception,
				"Carman's data list is empty!.");
		goto err;
	}

	/* FIXME: Review where is the Py_DECREF */

	/* Data's index for graphic's time */
	t_i = (time - start_time) / rate;

	/* New graphic's tuple */
	py_tuple = PyTuple_New(2);

	py_data = PyList_GetItem(py_list, t_i);
	data = PyCObject_AsVoidPtr(py_data);

	py_data = PyFloat_FromDouble(data->lat);
	if (PyTuple_SetItem(py_tuple, 0, py_data)) {
		PyErr_SetString(PyExc_Exception,
			"Problems setting latitude to the return tuple.");
		goto err;
	}

	py_data = PyFloat_FromDouble(data->lon);
	if (PyTuple_SetItem(py_tuple, 1, py_data)) {
		PyErr_SetString(PyExc_Exception,
			"Problems setting longitude to the return tuple.");
		goto err;
	}

	return Py_BuildValue("N",py_tuple);

err:
	return NULL;
}

