/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdlib.h>
#include <string.h> /* strdup */
#include <sys/time.h>
#include <time.h>

#include <dbus/dbus.h>
#include <glib.h>
#include <libobd.h>

#include "bt-serial.h"
#include "log.h"

/* Connection to the system bus */
DBusConnection *connection = NULL;
static char *busid = NULL;

static char *activate_serial_service()
{
	DBusMessage *msg, *reply;
	gchar *bus_id;
	gchar *service;
	DBusError derr;

	if (!connection) {
		ERROR("No connection to activate serial service,"
			" hum... did you init_bt_serial()");
		return NULL;
	}

	msg = dbus_message_new_method_call("org.bluez", "/org/bluez",
					"org.bluez.Manager", "ActivateService");
	if (!msg)
		return NULL;

	service = g_strdup("serial");
	dbus_message_append_args(msg, DBUS_TYPE_STRING, &service,
				DBUS_TYPE_INVALID);
	g_free(service);

	dbus_error_init(&derr);
	reply = dbus_connection_send_with_reply_and_block(connection, msg,
							-1, &derr);
	if (dbus_error_is_set(&derr)) {
		ERROR("DBus error: %s", derr.message);
		dbus_error_free(&derr);
		return NULL;
	}

	if (dbus_message_get_args(reply,  &derr,
				DBUS_TYPE_STRING, &bus_id,
				DBUS_TYPE_INVALID) == FALSE) {

		if (dbus_error_is_set(&derr)) {
			ERROR("DBus error: %s", derr.message);
			dbus_error_free(&derr);
		}

		return NULL;
	}

	busid = strdup(bus_id);

	dbus_message_unref(reply);

	return busid;
}

int setup_spp_connection(char *address, char **devpath)
{
	const char *dev;
	DBusMessage *msg, *reply;
	gchar *service;
	DBusError derr;

	dbus_error_init(&derr);
	msg = dbus_message_new_method_call(busid, "/org/bluez/serial",
			"org.bluez.serial.Manager", "ConnectService");
	service = g_strdup("spp");
	dbus_message_append_args(msg, DBUS_TYPE_STRING, &address,
			DBUS_TYPE_STRING, &service, DBUS_TYPE_INVALID);
	dbus_error_init(&derr);
	reply = dbus_connection_send_with_reply_and_block(connection, msg,
							-1, &derr);
	g_free(service);
	dbus_message_unref(msg);

	if (dbus_error_is_set(&derr)) {
		ERROR("DBus error: %s", derr.message);
		dbus_error_free(&derr);
		return -1;
	}

	if (dbus_message_get_args(reply, &derr, DBUS_TYPE_STRING, &dev,
				DBUS_TYPE_INVALID) == FALSE) {
		ERROR("Message err: %s", derr.message);
		dbus_error_free(&derr);
		return -1;
	}

	INFO("Bt connection is ok, new rfcomm: %s", dev);

	if (devpath)
		*devpath = g_strdup(dev);

	dbus_message_unref(reply);

	return 0;
}

int disconnect_service(gchar *address)
{
	DBusMessage *msg, *reply;
	DBusError derr;

	if (connection == NULL)
		return -1;

	dbus_error_init(&derr);
	msg = dbus_message_new_method_call(busid, "/org/bluez/serial",
			"org.bluez.serial.Manager", "DisconnectService");
	dbus_message_append_args(msg, DBUS_TYPE_STRING, &address,
				DBUS_TYPE_INVALID);
	reply = dbus_connection_send_with_reply_and_block(connection, msg,
							-1, &derr);

	if (dbus_error_is_set(&derr)) {
		ERROR("DisconnectService error: %s", derr.message);
		dbus_error_free(&derr);
		return -1;
	}

	dbus_message_unref(msg);

	if (reply == NULL)
		return -1;

	return 0;
}

gboolean init_bt_serial(void)
{
	/* get system bus */
	connection = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (connection == NULL) {
		ERROR("No connection");
		return FALSE;
	}

	busid = activate_serial_service();
	if (!busid) {
		ERROR("Could not activate serial service");
		return FALSE;
	}

	return TRUE;
}

void quit_bt_serial(void)
{
	dbus_connection_unref(connection);
	free(busid);
}

