/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <glib.h>
#include <dbus.h>
#include <time.h>
#include <sys/time.h>

#include <gdbus.h>

#include "obd-thread.h"
#include "dbus.h"
#include "location.h"
#include "carmand-trip.h"

#include "log.h"

#define CARMAND_DBUS_PATH "/org/indt/carmand"

#define CARMAN_IFACE_CONF	CARMAND_DBUS_BUS_NAME ".Configuration"
#define CARMAN_IFACE_GPS	CARMAND_DBUS_BUS_NAME ".GPS"
#define CARMAN_IFACE_OBD	CARMAND_DBUS_BUS_NAME ".OBD"
#define CARMAN_IFACE_TRIP	CARMAND_DBUS_BUS_NAME ".Trip"

#define ERROR_IFACE	CARMAND_DBUS_BUS_NAME ".Error"

#define CONFIG_PATH	"/home/user/.carman/"
#define CONFIG_FILE	CONFIG_PATH"carmand.conf"
#define GROUP_NAME	"CARMAN_DAEMON_SETTINGS"

#define TIME_PRECISION	3
#define TRIP_FOLDER		"/home/user/MyDocs/Carman/Trips"
#define MAPS_FOLDER		"/home/user/MyDocs/Carman/Maps"
#define TRACKS_FOLDER	"/home/user/MyDocs/Carman/Tracks"
#define CONNECTION_MODE	"normal"

/* Max number of selected sensors for trip recording */
#define MAX_TRIP_SENSORS	3

#define KEY_TIME_PRECISION		"time_precision"
#define KEY_TRIP_FOLDER			"trip_folder"
#define KEY_MAPS_FOLDER			"maps_folder"
#define KEY_TRACKS_FOLDER		"tracks_folder"
#define KEY_OBD_DEV				"obd_device"
#define KEY_GPS_DEV				"gps_device"
#define KEY_TRIP_SENSORS		"trip_sensors"
#define KEY_LAST_TRIP_FOLDER	"last_trip_folder"
#define KEY_CONNECTION_MODE		"connection_mode"

#define CARMAND_ERROR_NAME	"org.indt.Error"

/* FIXME: Some items of this enum like CarmandErrorMemAlloc is not used.
 * 	It will be used or I can remove it?
 */
typedef enum {
	CarmandErrorInvalidArguments,
	CarmandErrorFailed,
	CarmandErrorNoDeviceConfigured,
	CarmandErrorInvalidGpsDevice,
	CarmandErrorInvalidObdDevice,
	CarmandErrorGpsDeviceNotConnected,
	CarmandErrorObdDeviceNotConnected,
	CarmandErrorDiskFull,
	CarmandErrorPermissionDenied,
	CarmandErrorMemAlloc,
	CarmandErrorNoConfigFile
} CarmandErrorType;


struct CarmandConfig {
	int time_precision;
	char *trip_folder;
	char *maps_folder;
	char *tracks_folder;
	char *obd_device;
	char *gps_device;
	char *mode;	/* normal or simulator */
	char *last_trip_folder;
	int trip_sensors[MAX_TRIP_SENSORS];
};

static GKeyFile *config_file = NULL;
static struct CarmandConfig *config = NULL;
static DBusConnection *connection = NULL;
static GMainLoop *event_loop = NULL;

static int read_config_file()
{
	GKeyFile *key_file;
	GError *gerr = NULL;
	int ret = 0;

	if (config_file)
		g_key_file_free(config_file);

	config_file = g_key_file_new();

	if (g_key_file_load_from_file(config_file, CONFIG_FILE,
				G_KEY_FILE_NONE, &gerr))
		key_file = config_file;
	else {
		ERROR("error reading from configuration file.\n");
		ret = -EINVAL;
		return ret;
	}

	config->time_precision = g_key_file_get_integer(key_file,
					GROUP_NAME, KEY_TIME_PRECISION, NULL);
	if (config->time_precision == 0)
		config->time_precision = TIME_PRECISION;

	config->trip_folder = g_key_file_get_string(key_file, GROUP_NAME,
					KEY_TRIP_FOLDER, NULL);
	if (config->trip_folder == NULL)
		config->trip_folder  = strdup(TRIP_FOLDER);

	config->maps_folder = g_key_file_get_string(key_file, GROUP_NAME,
					KEY_MAPS_FOLDER, NULL);
	if (config->maps_folder == NULL)
		config->maps_folder  = strdup(MAPS_FOLDER);

	config->tracks_folder = g_key_file_get_string(key_file, GROUP_NAME,
					KEY_TRACKS_FOLDER, NULL);
	if (config->tracks_folder == NULL)
		config->tracks_folder  = strdup(TRACKS_FOLDER);

	config->obd_device = g_key_file_get_string(key_file, GROUP_NAME,
						KEY_OBD_DEV, NULL);
	if (config->obd_device == NULL)
		config->obd_device = strdup("none");

	config->gps_device = g_key_file_get_string(key_file, GROUP_NAME,
						KEY_GPS_DEV, NULL);
	if (config->gps_device == NULL)
		config->gps_device = strdup("none");

	if (g_key_file_has_key(key_file, GROUP_NAME, KEY_TRIP_SENSORS, NULL)) {
		int *trip_sensors[MAX_TRIP_SENSORS];

		*trip_sensors = g_key_file_get_integer_list(key_file,
				GROUP_NAME, KEY_TRIP_SENSORS,
				(void *) MAX_TRIP_SENSORS, NULL);

		memcpy(config->trip_sensors, trip_sensors,
				MAX_TRIP_SENSORS * sizeof(int));
	}

	config->mode = g_key_file_get_string(key_file, GROUP_NAME,
						KEY_CONNECTION_MODE, NULL);
	if (config->mode == NULL)
		config->mode = strdup(CONNECTION_MODE);

	config->last_trip_folder = g_key_file_get_string(key_file, GROUP_NAME,
					KEY_LAST_TRIP_FOLDER, NULL);

	return ret;
}

/*
 * Set default values for config object.
 * It should prevent UI from crashing if config file is not present.
 */
void set_config_default_values()
{
	if (config) {
		config->time_precision = TIME_PRECISION; /* in seconds*/
		config->trip_folder = g_strdup(TRIP_FOLDER);
		config->maps_folder = g_strdup(MAPS_FOLDER);
		config->tracks_folder = g_strdup(TRACKS_FOLDER);
		config->obd_device = g_strdup("none");
		config->gps_device = g_strdup("none");
		config->trip_sensors[0] = 13; /* 0D = Speed */
		config->trip_sensors[1] = 12; /* 0C = RPM */
		config->trip_sensors[2] = 4; /* 04 = Eng. Load */
		config->last_trip_folder = NULL;
		config->mode = g_strdup(CONNECTION_MODE);
	}
}

/* Free config object */
void config_free(struct CarmandConfig *config)
{
	if (!config)
		return;

	g_free(config->trip_folder);
	g_free(config->maps_folder);
	g_free(config->tracks_folder);
	g_free(config->obd_device);
	g_free(config->gps_device);
	g_free(config->mode);
	g_free(config->last_trip_folder);
	g_free(config);
}

/* Load configuration from file /etc/carman/carmand.conf, if exists */
static void config_init(void)
{
	/* When no configuration, instance global variable */
	if (config) {
		config_free(config);
		config = NULL;
	}
	config = g_new0(struct CarmandConfig, 1);

	if (read_config_file() == 0)
		return;

	/* If carmand.conf does not exist or config file read operation
	 * fails, fill config with default values.
	 */
	ERROR("Using configuration default values.\n");
	set_config_default_values();
}

static gboolean config_recording(const char *filename, const char *content,
				const char *mode, GError **error)
{
	FILE *file;
	GIOChannel *io_channel;
	GIOStatus result;

	file = fopen(filename, "r");
	result = G_IO_STATUS_NORMAL;
	io_channel = NULL;

	/* WTF: this UGLY hack is just to check if the file exists or not? */
	if (file)
		fclose(file);
	else {
		file = fopen(filename, "w+");
		if (file)
			fclose(file);
	}

	/* what's the point to open a file without be able to write on it? */
	/* it's sane? */
	if (strcmp(mode, "+") == 0 && strcmp( mode, "a" ) == 0
			&& strcmp( mode, "w" ) == 0 )
	{
		ERROR("Improper file mode to save");
		goto err;
	}

	io_channel = g_io_channel_new_file(filename, mode, error);

	if (!io_channel) {
		ERROR("Oh no, the io_channel does not exist, file: %s", filename);
		goto err;
	}

	result = g_io_channel_write_chars(io_channel, content, strlen(content), NULL, error);
	if (result != G_IO_STATUS_NORMAL) {
		g_io_channel_shutdown(io_channel, FALSE, error);
		if (io_channel) {
			ERROR("Where is the io_channel?");
			goto err_io_ch;
		}
	}

	if (!g_io_channel_shutdown(io_channel, TRUE, error)) {
		if (io_channel) {
			ERROR("Oh no, the io_channel is NULL");
			goto err_io_ch;
		}
	}

	g_io_channel_unref(io_channel);

	return TRUE;

err_io_ch:
	g_free(io_channel);
	io_channel = NULL;
	return FALSE;
err:
	ERROR("CARMAND: Error recording configuration file.\n");
	return FALSE;
}

/* Saves current configuration on /etc/carman/carmand.conf
 */
static int config_save()
{
	GKeyFile* key_file;
	char *config_data = NULL;
	int ret = 0;

	ret = g_mkdir_with_parents(CONFIG_PATH, S_IRWXU | S_IRWXG |
			S_IROTH | S_IXOTH);
	if (ret) {
		ret = -EINVAL;
		goto out;
	}

	if (!config_file)
		config_file = g_key_file_new();

	key_file = config_file;

	/* Configuration to be saved always come from memory */
	if (config) {
		g_key_file_set_integer(key_file, GROUP_NAME,
				KEY_TIME_PRECISION,
				config->time_precision);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_TRIP_FOLDER,
				config->trip_folder);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_MAPS_FOLDER,
				config->maps_folder);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_TRACKS_FOLDER,
				config->tracks_folder);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_OBD_DEV,
				config->obd_device);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_GPS_DEV,
				config->gps_device);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_LAST_TRIP_FOLDER,
				config->last_trip_folder);

		g_key_file_set_string(key_file, GROUP_NAME,
				KEY_CONNECTION_MODE,
				config->mode);

	} else {
		ret = -EINVAL;
		goto out;
	}

	config_data = (char *) g_key_file_to_data(config_file, NULL, NULL);

	if (!config_recording(CONFIG_FILE, config_data, "w+", NULL)) {
		DEBUG("CARMAND: Error saving configuration file\n");
		ret = -EINVAL;
		goto out;
	}

	DEBUG("Configuration file recorded with success.");
out:
	if (config_data) {
		free(config_data);
		config_data = NULL;
	}

	return ret;

}

/*
 * Common D-Bus carman errors
 */

static inline DBusMessage *not_ready(DBusMessage *msg, const char *str)
{
	return g_dbus_create_error(msg, ERROR_IFACE ".NotReady", str);
}

static inline DBusMessage *connection_failed(DBusMessage *msg, const char *str)
{
	return g_dbus_create_error(msg,
			ERROR_IFACE ".ConnectionAttemptFailed", str);
}

static inline DBusMessage *failed_errno(DBusMessage *msg, int err)
{
	return g_dbus_create_error(msg,
			ERROR_IFACE ".Failed", strerror(err));
}

static DBusMessage *get_time_precision(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_INT32, &config->time_precision,
			DBUS_TYPE_INVALID);
}

/* Returns current save path for stored trips */
static DBusMessage *get_trips_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->trip_folder,
			DBUS_TYPE_INVALID);
}

/* Returns current save path for stored maps */
static DBusMessage *get_maps_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->maps_folder,
			DBUS_TYPE_INVALID);
}

/* Returns current save path for stored tracks */
static DBusMessage *get_tracks_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->tracks_folder,
			DBUS_TYPE_INVALID);
}

/* Returns current address of configured OBD-II device */
static DBusMessage *get_obd_device(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->obd_device,
			DBUS_TYPE_INVALID);
}

/* Returns current address of configured GPS device */
static DBusMessage *get_gps_device(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->gps_device,
			DBUS_TYPE_INVALID);
}

/* Returns current execution mode for GPS and OBD */
static DBusMessage *get_mode(DBusConnection *connection,
			DBusMessage *message, void *user_data)
{
	return g_dbus_create_reply(message,
			DBUS_TYPE_STRING, &config->mode,
			DBUS_TYPE_INVALID);
}

/* Updates current time precision for trip data in seconds */
static DBusMessage *set_time_precision(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	int value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_INT32,
			&value,	DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (value < 1 || value > 60) {
		return g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				"Out of range");
	}

	config->time_precision = value;

	trip_set_precision(config->trip_folder, config->time_precision);

	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);
}

/* Updates save folder for stored trips */
static DBusMessage *set_trips_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	FILE *dir;
	char *value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_STRING,
			&value,	DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if ((strcmp(value, "") == 0) || (strlen(value) > 256))
		return g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				"Out of range");

	/* FIXME: Use stat. Missing check for write permission */
	dir = fopen(value, "r"); /* Check if value is a valid path */
	if (!dir)
		goto error;
	else
		fclose(dir);

	g_free(config->trip_folder);
	config->trip_folder = g_strdup(value);
	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);

error:
	return g_dbus_create_error(message,
			"org.indt.Error.InvalidArguments",
			"Invalid folder");
}

/* Updates save folder for stored maps */
static DBusMessage *set_maps_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	FILE *dir;
	char *value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_STRING,
			&value,	DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if ((strcmp(value, "") == 0) || (strlen(value) > 256))
		return g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				"Out of range");

	/* FIXME: Use stat. Missing check for write permission */
	dir = fopen(value, "r"); /* Check if value is a valid path */
	if (!dir)
		goto error;
	else
		fclose(dir);

	g_free(config->maps_folder);
	config->maps_folder = g_strdup(value);
	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);

error:
	return g_dbus_create_error(message,
			"org.indt.Error.InvalidArguments",
			"Invalid folder");
}

/* Updates save folder for stored tracks */
static DBusMessage *set_tracks_folder(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	FILE *dir;
	char *value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_STRING,
			&value,	DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if ((strcmp(value, "") == 0) || (strlen(value) > 256))
		return g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				"Out of range");

	/* FIXME: Use stat. Missing check for write permission */
	dir = fopen(value, "r"); /* Check if value is a valid path */
	if (!dir)
		goto error;
	else
		fclose(dir);

	g_free(config->tracks_folder);
	config->tracks_folder = g_strdup(value);
	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);

error:
	return g_dbus_create_error(message,
			"org.indt.Error.InvalidArguments",
			"Invalid folder");
}

/* Updates OBD-II device configuration */
static DBusMessage *set_obd_device(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr,
				DBUS_TYPE_STRING, &value,
				DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	g_free(config->obd_device);
	if (strcmp(value, "none") == 0 || strcmp(value, "") == 0)
		config->obd_device = g_strdup("none");
	else
		config->obd_device = g_strdup(value);

	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);
}

/*
 * Updates GPS device configuration.
 * Possible Values: "none" - No GPS configured
 * "internal" - OnBoard GPS device
 * "<address>" - Bluetooth GPS Device MAC address
 */
static DBusMessage *set_gps_device(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	const gchar *value;
	gchar *gps_device;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_STRING,
			&value, DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (strcmp(value, "none") == 0 || strcmp(value, "") == 0)
		gps_device = g_strdup("none");
	else {
		if (strcmp(value, "internal") == 0) {
			/* FIXME: Check why stat() did not work here */
			FILE *gps = fopen("/dev/pgps", "rw");
			if (!gps)
				goto error;
			else
				fclose(gps);
		}

		/* internal or Bluetooth address */
		gps_device = g_strdup(value);
	}

	g_free(config->gps_device);
	config->gps_device = gps_device;

	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);
error:
	return g_dbus_create_error(message,
			"org.indt.Error.InvalidArguments",
			"Invalid GPS device");
}

static DBusMessage *set_mode(DBusConnection *connection,
			DBusMessage *message, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	const gchar *value;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(message, &derr, DBUS_TYPE_STRING,
			&value, DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (strcmp(value, "normal") != 0 && strcmp(value, "simulator") != 0) {
		reply = g_dbus_create_error(message,
				"org.indt.Error.InvalidArguments",
				"Invalid mode");
		return reply;
	}

	g_free(config->mode);
	config->mode = g_strdup(value);

	return g_dbus_create_reply(message, DBUS_TYPE_INVALID);
}

static DBusMessage *save_configuration(DBusConnection *conn,
	DBusMessage *msg, void *user_data)
{
	if (config_save()) {
		return g_dbus_create_error(msg, "org.indt.Error.Failed",
			"Error saving configuration file");
	}

	g_dbus_emit_signal(conn, CARMAND_DBUS_PATH, CARMAN_IFACE_CONF,
		"ConfigurationSaved", DBUS_TYPE_INVALID);

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *discard_configuration(DBusConnection *conn,
	DBusMessage *msg, void *user_data)
{
	config_init();

	g_dbus_emit_signal(conn, CARMAND_DBUS_PATH, CARMAN_IFACE_CONF,
		"ConfigurationDiscarded", DBUS_TYPE_INVALID);

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

/*
 * Methods and signals that will be registered in Carman's DBUS Configuration Interface
 */
static GDBusMethodTable dbus_iface_conf_methods[] = {
	{ "SetTimePrecision",		"i",	"",		set_time_precision		},
	{ "SetTripsFolder",			"s",	"",		set_trips_folder		},
	{ "SetMapsFolder",			"s",	"",		set_maps_folder			},
	{ "SetTracksFolder",		"s",	"",		set_tracks_folder		},
	{ "SetOBDDevice",			"s",	"",		set_obd_device			},
	{ "SetGPSDevice",			"s",	"",		set_gps_device			},
	{ "SetMode",				"s",	"",		set_mode				},
	{ "GetTimePrecision",		"",		"i",	get_time_precision		},
	{ "GetTripsFolder",			"",		"s",	get_trips_folder		},
	{ "GetMapsFolder",			"",		"s",	get_maps_folder			},
	{ "GetTracksFolder",		"",		"s",	get_tracks_folder		},
	{ "GetOBDDevice",			"",		"s",	get_obd_device			},
	{ "GetGPSDevice",			"",		"s",	get_gps_device			},
	{ "GetMode",				"",		"s",	get_mode				},
	{ "SaveConfiguration",		"",     "",		save_configuration		},
	{ "DiscardConfiguration",	"",     "",		discard_configuration	},
	{ NULL, NULL, NULL, NULL }
};

static GDBusSignalTable dbus_iface_conf_signals[] = {
	{ "ConfigurationSaved",     "" },
	{ "ConfigurationDiscarded", "" },
	{ NULL, NULL }
};

/*
 * GPS Interface methods
 */
static DBusMessage *gpsconnect(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	const char *device;
	if (!config || !config->gps_device)
		return not_ready(msg, "GPS not available");

	if (strcmp("normal", config->mode) == 0)
		device = config->gps_device;
	else
		device = "simulator";

	if (location_connect(device) < 0)
		return connection_failed(msg, "GPS connection attempt failed");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *gpsdisconnect(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	int err = 0;
	err = location_disconnect();
	if (err < 0)
		return failed_errno(msg, -err);
	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *gpsstatus(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	const char *status;

	status = location_status();
	/* FIXME: API and control panel needs be reviewed */
	return g_dbus_create_reply(msg,
			DBUS_TYPE_STRING, &status,
			DBUS_TYPE_INVALID);
}

static GDBusMethodTable gps_methods[] = {
	{ "Connect",		"",	"",	gpsconnect	},
	{ "Disconnect",		"",	"",	gpsdisconnect	},
	{ "Status",		"",	"s",	gpsstatus	},
	{ NULL, NULL, NULL, NULL }
};

static GDBusSignalTable gps_signals[] = {
	{ "StatusChanged",	"s"		},
	{ "DataAvailable",      "iddddd"	},
	{ NULL, NULL }
};

static DBusMessage *obd_start_simulator(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *file = NULL;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(msg, &derr, DBUS_TYPE_STRING,
			&file, DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (obd_thread_start_simulator(file) < 0)
		return connection_failed(msg, "OBD simulator attempt failed");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_connect_bt(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	if (!config)
		return not_ready(msg, "OBD not available");

	if (!strcmp(config->mode, "simulator")) {
		if (obd_thread_start_simulator("") < 0) {
			return connection_failed(msg,
					"OBD simulator failed to start");
		} else {
			return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
		}
	}

	if (!config->obd_device)
		return not_ready(msg, "OBD not available");

	if (obd_thread_connect_bt(config->obd_device) < 0)
		return connection_failed(msg, "OBD connection attempt failed");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_connect_serial(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	DBusError derr;
	char *serial_addr = NULL;

	dbus_error_init(&derr);
	if (!dbus_message_get_args(msg, &derr, DBUS_TYPE_STRING,
			&serial_addr, DBUS_TYPE_INVALID)) {
		reply = g_dbus_create_error(msg,
				"org.indt.Error.InvalidArguments",
				(char *) derr.message);
		dbus_error_free(&derr);
		return reply;
	}

	if (obd_thread_connect_serial(serial_addr) < 0)
		return connection_failed(msg, "OBD connection attempt failed");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_disconnect(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	if (obd_thread_disconnect() < 0)
		return connection_failed(msg, "OBD disconnection attempt failed");

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_connection_status(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	const char *status;

	status = obd_thread_connection_status();

	return g_dbus_create_reply(msg, DBUS_TYPE_STRING, &status,
		DBUS_TYPE_INVALID);
}

static DBusMessage *obd_add_time_sensor(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	DBusError derr;
	int pid,interval,ret;

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
			DBUS_TYPE_INT32, &pid,
			DBUS_TYPE_INT32, &interval,
			DBUS_TYPE_INVALID)) {

		return g_dbus_create_error(msg,
			"org.indt.Error.InvalidArguments",
			(char *) derr.message);
	}

	ret = obd_thread_add_time_sensor(pid, interval);

	if (ret < 0)
		DEBUG("Problems adding the sensor: %d", pid);

	return g_dbus_create_reply(msg, DBUS_TYPE_INT32, &ret, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_del_sensor(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	DBusError derr;
	int ref, ret;

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
			DBUS_TYPE_INT32, &ref,
			DBUS_TYPE_INVALID)) {

		return g_dbus_create_error(msg,
			"org.indt.Error.InvalidArguments",
			(char *) derr.message);
	}

	ret = obd_thread_del_round_sensor(ref);

	if (ret <= 0)
		ret = obd_thread_del_time_sensor(ref);

	return g_dbus_create_reply(msg, DBUS_TYPE_INT32, &ret, DBUS_TYPE_INVALID);
}


static DBusMessage *obd_add_round_sensor(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	DBusError derr;
	int pid,interval,round,ret;

	dbus_error_init(&derr);

	if (!dbus_message_get_args(msg, &derr,
			DBUS_TYPE_INT32, &pid,
			DBUS_TYPE_INT32, &interval,
			DBUS_TYPE_INT32, &round,
			DBUS_TYPE_INVALID)) {

		return g_dbus_create_error(msg,
			"org.indt.Error.InvalidArguments",
			(char *) derr.message);
	}

	ret = obd_thread_add_round_sensor(pid, interval, round);

	if (ret < 0)
		DEBUG("Problems adding the sensor: %d", pid);

	return g_dbus_create_reply(msg, DBUS_TYPE_INT32, &ret, DBUS_TYPE_INVALID);
}

static DBusMessage *obd_get_dtc_list(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	int ret = obd_thread_list_dtc();

	if (ret == -1) {
		return not_ready(msg, "OBD is not connected");
	}

	if (ret == -2) {
		return not_ready(msg,
			"Retrieve the list of DTCs already scheduled.");
	}

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static GDBusMethodTable obd_methods[] = {
	{ "ConnectBT",		"",		"",  obd_connect_bt		},
	{ "ConnectSerial",	"s",		"",  obd_connect_serial		},
	{ "StartSimulator",	"s",		"",  obd_start_simulator	},
	{ "Disconnect",		"",		"",  obd_disconnect		},
	{ "Status",		"",		"s", obd_connection_status	},
	{ "AddTimeSensor",	"ii",		"i", obd_add_time_sensor	},
	{ "DelSensor",		"i",		"i", obd_del_sensor		},
	{ "AddRoundSensor",	"iii",		"i", obd_add_round_sensor	},
	{ "RequestDTCList",	"",		"", obd_get_dtc_list		},
	{ NULL, NULL, NULL, NULL }
};

static GDBusSignalTable obd_signals[] = {
	{ "StatusChanged",	"s"	},
	{ "DataAvailable",	"idddd"	},
	{ "DTCList",		"as"	},
	{ NULL, NULL }
};

static DBusMessage *trip_flush_data(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	trip_flush_fds();

	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_TRIP, "DataFlushed",
			DBUS_TYPE_INVALID);

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

void dbus_emit_trip_reseted(void)
{
	g_free(config->last_trip_folder);
	config->last_trip_folder = g_strdup(trip_actual_folder());
	config_save();

	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_TRIP, "TripReseted",
			DBUS_TYPE_INVALID);
}

static DBusMessage *trip_reset_trip(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	trip_reset(config->trip_folder, config->time_precision);

	return g_dbus_create_reply(msg, DBUS_TYPE_INVALID);
}

static DBusMessage *trip_actual_trip(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	const char *trip = NULL;

	trip = trip_actual_folder();

	if (!trip)
		return not_ready(msg, "Trip is not running");

	return g_dbus_create_reply(msg, DBUS_TYPE_STRING,
				&trip, DBUS_TYPE_INVALID);
}

static GDBusMethodTable trip_methods[] = {
	{ "FlushData",		"",	"",		trip_flush_data		},
	{ "ResetTrip",		"",	"",		trip_reset_trip		},
	{ "ActualTripFolder",	"",	"s",		trip_actual_trip	},
	{ NULL, NULL, NULL, NULL }
};

static GDBusSignalTable trip_signals[] = {
	{ "DataFlushed",	""	},
	{ "TripReseted",	""	},
	{ "DataAvailable",	"d"	},
	{ NULL, NULL }
};

static void carmandtray_exited(void *user_data)
{
	DBusConnection *session = user_data;

	INFO("Carman Panel disconnected from carman bus...");
	g_main_loop_quit(event_loop);
	dbus_connection_unref(session);
}

int dbus_init(DBusConnection *conn, GMainLoop *loop)
{
	DBusConnection *session;
	DBusError derr;
	dbus_uint32_t result;

	config_init();


	if (config->last_trip_folder &&
		!trip_restart(config->last_trip_folder, config->trip_folder)) {
		/* ok */
		DEBUG("Old trip restarted");
	} else {
		DEBUG("Creating a new trip...");

		if (g_mkdir_with_parents(config->trip_folder, S_IRWXU | S_IRWXG |
				S_IROTH | S_IXOTH)) {
			DEBUG("Error creating trip folder, using default.");
			g_free(config->trip_folder);
			config->trip_folder = g_strdup(TRIP_FOLDER);
		}

		if (trip_init(config->trip_folder, config->time_precision) < 0) {
			DEBUG("Problems creating a trip file");
			config_free(config);
			config = NULL;
			return -1;
		}

		DEBUG("Trip created");

		g_free(config->last_trip_folder);
		config->last_trip_folder = g_strdup(trip_actual_folder());
		config_save();
	}

	connection = dbus_connection_ref(conn);
	g_dbus_register_interface(connection, CARMAND_DBUS_PATH, CARMAN_IFACE_CONF,
			dbus_iface_conf_methods,
			dbus_iface_conf_signals, NULL, NULL, NULL);

	g_dbus_register_interface(connection, CARMAND_DBUS_PATH, CARMAN_IFACE_GPS,
			gps_methods,
			gps_signals, NULL, NULL, NULL);

	g_dbus_register_interface(connection, CARMAND_DBUS_PATH, CARMAN_IFACE_OBD,
			obd_methods,
			obd_signals, NULL, NULL, NULL);

	g_dbus_register_interface(connection, CARMAND_DBUS_PATH, CARMAN_IFACE_TRIP,
			trip_methods,
			trip_signals, NULL, NULL, NULL);
	/*
	 * Carman Daemon must activate Carman Panel application and
	 * track it's lifetime/connection exitting automatically
	 * when this condition is detected.
	 */

	dbus_error_init(&derr);
	session = g_dbus_setup_bus(DBUS_BUS_SESSION, NULL, &derr);
	if (session == NULL) {
		ERROR("session bus: %s", derr.message);
		dbus_error_free(&derr);
		return -1;
	}

	dbus_bus_start_service_by_name(session,
			"org.indt.carmandtray", 0, &result, &derr);
	if (dbus_error_is_set(&derr)) {
		ERROR("Can't start Carman Panel: %s", derr.message);
		dbus_error_free(&derr);
	} else {
		event_loop = loop;
		g_dbus_add_disconnect_watch(session, "org.indt.carmandtray",
				carmandtray_exited, session, NULL);
	}

	return 0;
}

void dbus_exit(void)
{
	if (!connection)
		return;

	g_dbus_unregister_interface(connection,
			CARMAND_DBUS_PATH, CARMAN_IFACE_OBD);

	g_dbus_unregister_interface(connection,
			CARMAND_DBUS_PATH, CARMAN_IFACE_GPS);

	g_dbus_unregister_interface(connection,
			CARMAND_DBUS_PATH, CARMAN_IFACE_CONF);

	g_dbus_unregister_interface(connection,
			CARMAND_DBUS_PATH, CARMAN_IFACE_TRIP);

	dbus_connection_unref(connection);

	trip_exit();
}

void dbus_emit_obd_status(const char *status)
{
	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_OBD, "StatusChanged",
			DBUS_TYPE_STRING, &status,
			DBUS_TYPE_INVALID);
}

void dbus_emit_obd_data(int pid, double value1, double value2,
		double value3, double value4)
{
	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_OBD, "DataAvailable",
			DBUS_TYPE_INT32, &pid,
			DBUS_TYPE_DOUBLE, &value1,
			DBUS_TYPE_DOUBLE, &value2,
			DBUS_TYPE_DOUBLE, &value3,
			DBUS_TYPE_DOUBLE, &value4,
			DBUS_TYPE_INVALID);
}

void dbus_emit_gps_status(const char *status)
{
	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_GPS, "StatusChanged",
			DBUS_TYPE_STRING, &status,
			DBUS_TYPE_INVALID);
}

void dbus_emit_trip_data(double data)
{
	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_TRIP, "DataAvailable",
			DBUS_TYPE_DOUBLE, &data,
			DBUS_TYPE_INVALID);
}

void dbus_emit_gps_data(location_t *gps)
{
	/*
	DEBUG("GPS DATA: ");
	DEBUG("      mode: %d", gps->mode);
	DEBUG("  latitude: %f", gps->latitude);
	DEBUG(" longitude: %f", gps->longitude);
	DEBUG("  altitude: %f", gps->altitude);
	DEBUG("     speed: %f", gps->speed);
	DEBUG("     track: %f", gps->track);
	*/
	g_dbus_emit_signal(connection, CARMAND_DBUS_PATH,
			CARMAN_IFACE_GPS, "DataAvailable",
			DBUS_TYPE_INT32, &gps->mode,
			DBUS_TYPE_DOUBLE, &gps->latitude,
			DBUS_TYPE_DOUBLE, &gps->longitude,
			DBUS_TYPE_DOUBLE, &gps->altitude,
			DBUS_TYPE_DOUBLE, &gps->speed,
			DBUS_TYPE_DOUBLE, &gps->track,
			DBUS_TYPE_INVALID);
}

void dbus_emit_obd_dtc(GSList *dtc)
{
	DBusMessage *reply;
	DBusMessageIter iter, array_iter;
	const char *str;
	GSList *l;

	reply = dbus_message_new_signal(CARMAND_DBUS_PATH,
				CARMAN_IFACE_OBD, "DTCList");

	if (!reply) {
		ERROR("Unable to create DTC d-bus signal message");
		return;
	}

	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
			DBUS_TYPE_STRING_AS_STRING, &array_iter);

	for (l = dtc; l; l = l->next) {
		str = l->data;
		dbus_message_iter_append_basic(&array_iter,
				DBUS_TYPE_STRING, &str);
	}

	dbus_message_iter_close_container(&iter, &array_iter);

	DEBUG("Emitting DTCList signal...");

	dbus_connection_send(connection, reply, NULL);
	dbus_message_unref(reply);

	return;
}

