/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand-panel.
 *
 *  carmand-panel is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand-panel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <gtk/gtkmain.h>
#include <gtk/gtklabel.h>
#include <gtk/gtk.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-program.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <hildon/hildon-note.h>

#include "marshal.h"

/* FIXME: Check if localization will be necessary */
/* localization includes */
/*#include <libintl.h>
#include <locale.h>*/

#include "carmand-logmsg.h"
#include "carmand-dbus.h"
#include "carmand-tray.h"
#include "carmand-settings.h"

G_DEFINE_TYPE(CarmandTray, carmand_tray, G_TYPE_OBJECT)

/* Trip Dialog */
static void trip_reset_toogle(GtkWidget *widget, gpointer data)
{
	CarmandTray *self = CARMAND_TRAY(data);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
		gtk_widget_grab_focus(GTK_WIDGET(self->trip_entry));

	gtk_widget_set_sensitive(self->trip_entry,
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)));
}

static void create_trip_dialog(CarmandTray *self)
{
	GtkSizeGroup *size_group;
	gchar *trips_folder;
	GDir *trip_data_dir;
	DBusError derr;
	gint file_count = 0;

	size_group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

	if (!self->trip_dialog) {
		self->trip_dialog = gtk_dialog_new_with_buttons("Trip Reset",
				GTK_WINDOW(self->window),
				GTK_DIALOG_DESTROY_WITH_PARENT |
				GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_OK,
				GTK_RESPONSE_OK,
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_CANCEL,
				NULL);
	}

	trips_folder = cpanel_dbus_str_method_call(self->dbus,
			DBUS_GETTRIPSFOLDER,
			CARMAND_DBUS_IFACE_CONFIGURATION, &derr);
	if (trips_folder == NULL) {
		carmand_vdebug("%s Can't get trips folder\n", __func__);
		return;
	}

	trip_data_dir = g_dir_open(trips_folder, 0, NULL);
	if (trip_data_dir) {
		while (g_dir_read_name(trip_data_dir))
			file_count++;
	}

	g_free(trips_folder);

	self->trip_check = gtk_check_button_new();
	self->trip_caption = hildon_caption_new(size_group, "Save existing data:",
			self->trip_check, NULL, 0);
	hildon_caption_set_child_expand(HILDON_CAPTION(self->trip_caption),
			FALSE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(self->trip_dialog)->vbox),
			self->trip_caption);

	g_signal_connect(G_OBJECT(self->trip_check), "toggled",
			G_CALLBACK(trip_reset_toogle), self);

	self->trip_entry = gtk_entry_new();
	gtk_entry_set_max_length((GtkEntry *) self->trip_entry, 30);
	self->trip_caption = hildon_caption_new(size_group, "Trip name:",
					self->trip_entry, NULL, 0);
	hildon_caption_set_child_expand(HILDON_CAPTION(self->trip_caption),
			FALSE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(self->trip_dialog)->vbox),
			self->trip_caption);

	gtk_widget_set_sensitive((GtkWidget *) self->trip_check,
			(gboolean)(file_count > 1));
	gtk_widget_set_sensitive(self->trip_entry, FALSE);

	gtk_widget_show_all(self->trip_dialog);
}

static void trip_reset_menu_cb(GtkWidget *menu_item, gpointer user_data)
{
	CarmandTray *self = CARMAND_TRAY(user_data);
	HildonNote *dialog;

	carmand_vdebug("%s\n", __func__);

	dialog = (HildonNote *) hildon_note_new_confirmation(NULL,
			"Reset current trip and start a new one?");
	hildon_note_set_button_texts(dialog, "Yes", "No");

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
		cpanel_dbus_method_call(self->dbus, DBUS_METHOD_TRIP_RESETRIP,
				CARMAND_DBUS_IFACE_TRIP);

	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void open_carman_menu_cb(GtkWidget *menu_item, gpointer user_data)
{
	carmand_vdebug("%s\n", __func__);
	/*
	   use "&" with the command to execute it in background
	   and don't wait for it!
	   */
	system("/usr/bin/carman-evas &");
}

static void trip_settings_menu_cb(GtkWidget *menu_item, gpointer user_data)
{
	CarmandTray *self = CARMAND_TRAY(user_data);

	carmand_vdebug("%s\n", __func__);

	carmand_settings_new(GTK_WINDOW(self->window));
}

static void quit_menu_cb(GtkWidget *menu_item, gpointer user_data)
{
	CarmandTray *self = CARMAND_TRAY(user_data);
	HildonNote *dialog;

	carmand_vdebug("%s\n", __func__);

	dialog = (HildonNote *) hildon_note_new_confirmation(NULL,
			"Stop trip recording and quit?");
	hildon_note_set_button_texts(dialog, "Yes", "No");

	/* TODO: Implement carmand dbus call method to self-destruction */
	/*if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		carmand_tray_call_dbus_method("Disconnect");*/

	gtk_widget_destroy(GTK_WIDGET(dialog));
	gtk_main_quit();
}

static void carmand_exitted(GObject *obj, GParamSpec *spec, gpointer user_data)
{
	CarmandTray *self = CARMAND_TRAY(user_data);

	carmand_vdebug("%s\n", __func__);

	gtk_main_quit();
}

static void create_menu(CarmandTray *self)
{
	gint row = 0;

	carmand_vdebug("%s\n", __func__);

	if (self->menu)
		return;

	self->menu = gtk_menu_new();

	self->menu_trip_reset =
		gtk_menu_item_new_with_label("Reset trip");
	self->menu_open_carman =
		gtk_menu_item_new_with_label("Open Carman");
	self->menu_trip_settings =
		gtk_menu_item_new_with_label("Carman settings");
	self->menu_quit = gtk_menu_item_new_with_label("Quit");

	g_signal_connect(G_OBJECT(self->menu_trip_reset), "activate",
			G_CALLBACK(trip_reset_menu_cb), self);
	g_signal_connect(G_OBJECT(self->menu_open_carman), "activate",
			G_CALLBACK(open_carman_menu_cb), self);
	g_signal_connect(G_OBJECT(self->menu_trip_settings), "activate",
			G_CALLBACK(trip_settings_menu_cb), self);
	g_signal_connect(G_OBJECT(self->menu_quit), "activate",
			G_CALLBACK(quit_menu_cb), self);

	row++;
	gtk_menu_attach(GTK_MENU(self->menu), self->menu_trip_reset,
			0, 1, row, row + 1);
	gtk_widget_show_all (self->menu_trip_reset);

	row++;
	gtk_menu_attach(GTK_MENU(self->menu), self->menu_open_carman,
			0, 1, row, row + 1);
	gtk_widget_show_all (self->menu_open_carman);

	row++;
	gtk_menu_attach(GTK_MENU(self->menu), self->menu_trip_settings,
			0, 1, row, row+1);
	gtk_widget_show_all (self->menu_trip_settings);

	row++;
	gtk_menu_attach(GTK_MENU(self->menu), self->menu_quit,
			0, 1, row, row + 1);
	gtk_widget_show_all (self->menu_quit);
}

static void icon_on_click_cb(GtkStatusIcon *status_icon,
					gpointer user_data)
{
	CarmandTray *self = CARMAND_TRAY(user_data);

	carmand_vdebug("%s\n", __func__);

	create_menu(self);
	gtk_menu_popup(GTK_MENU (self->menu), NULL, NULL,
			(GtkMenuPositionFunc)gtk_status_icon_position_menu,
			status_icon, 0, gtk_get_current_event_time());
}

static GtkStatusIcon *create_icon(CarmandTray *self)
{

	GtkStatusIcon *tray_icon;

	tray_icon = gtk_status_icon_new();
	g_signal_connect(G_OBJECT(tray_icon), "activate",
			G_CALLBACK(icon_on_click_cb), self);

	gtk_status_icon_set_from_file(tray_icon,
			"/usr/share/carman/images/carmand_tray_stopped.png");

	gtk_status_icon_set_visible(tray_icon, TRUE);

	return tray_icon;
}

static void response_dialog_cb(GtkWidget *widget, gpointer data)
{
	carmand_vdebug("%s\n", __func__);

	gtk_widget_hide_all(widget);
	gtk_widget_destroy(widget);
}

static void carmand_tray_show_message(const char *msg, const char *icon)
{
	GtkDialog *dialog;

	carmand_vdebug("%s\n", __func__);

	dialog = GTK_DIALOG(hildon_note_new_information_with_icon_name(NULL, msg, icon));

	g_signal_connect(G_OBJECT(dialog), "response",
			G_CALLBACK(response_dialog_cb), NULL);
	gtk_widget_show_all(GTK_WIDGET(dialog));
}

static DBusHandlerResult connection_filter(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	const gchar *name, *old, *new;
	CarmandTray *self;
	gboolean result;
	int mode;

	if (dbus_message_is_signal(msg,
		DBUS_INTERFACE_DBUS, "NameOwnerChanged")) {

		if (!dbus_message_get_args(msg, NULL,
					DBUS_TYPE_STRING, &name,
					DBUS_TYPE_STRING, &old,
					DBUS_TYPE_STRING, &new,
					DBUS_TYPE_INVALID)) {
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
		}

		/* Ignore service creations */
		if (*new != '\0')
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

		if (strcmp("org.indt.carmand", name) == 0) {
			self = (CarmandTray *) user_data;
			g_signal_emit_by_name(G_OBJECT(self),
					"notify::carmand_exitted", NULL);
		}
	}
	
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/* GLib API Functions */

static void carmand_tray_finalize(GObject *obj)
{
	CarmandTray *self = CARMAND_TRAY(obj);

	carmand_vdebug("%s\n", __func__);

	if (self->menu)
		gtk_widget_destroy(self->menu);

	if (self->icon)
		g_object_unref(self->icon);

	if (self->window)
		gtk_widget_destroy(GTK_WIDGET(self->window));

	if (self->program)
		g_object_unref(self->program);

	if (self->dbus) {
		dbus_bus_release_name(self->dbus->session_bus,
				"org.indt.carmandtray", NULL);
		g_object_unref(self->dbus);
	}

	G_OBJECT_CLASS(carmand_tray_parent_class)->finalize(obj);
}

static void carmand_tray_class_init(CarmandTrayClass *klass)
{
	GObjectClass *obj_class = G_OBJECT_CLASS(klass);

	carmand_vdebug("%s\n", __func__);

	obj_class->finalize = carmand_tray_finalize;
}

static void carmand_tray_init(CarmandTray *self)
{
	carmand_vdebug("%s\n", __func__);

	self->program = NULL;
	self->window = NULL;
	self->icon = NULL;
	self->menu = NULL;
	self->dbus = NULL;

	self->menu_trip_reset = NULL;
	self->menu_open_carman = NULL;
	self->menu_trip_settings = NULL;
	self->menu_quit = NULL;

	self->trip_caption = NULL;
	self->trip_dialog = NULL;
	self->trip_check = NULL;
	self->trip_entry = NULL;
}

CarmandTray *carmand_tray_new(void)
{
	CarmandTray *self;

	self = g_object_new(CARMAND_TYPE_TRAY, NULL);
	if (!self) {
		carmand_error("Failed to create Carmand Tray.\n");
		return NULL;
	}

	g_signal_connect(G_OBJECT(self), "notify::carmand_exitted",
			G_CALLBACK(carmand_exitted), self);

	/* Connections */
	self->dbus = carmand_dbus_new(self, NULL);
	if (!self->dbus) {
		g_object_unref(self);
		return NULL;
	}

	if (dbus_bus_request_name(self->dbus->session_bus,
				"org.indt.carmandtray",
			DBUS_NAME_FLAG_DO_NOT_QUEUE, NULL) !=
			DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
		carmand_error("Carman tray already running!\n");
		g_object_unref(self);
		return NULL;
	}

	if (!dbus_connection_add_filter(self->dbus->private_bus,
				connection_filter, self, NULL))
		carmand_error("Can't add D-Bus connection filter function\n");

	self->program = HILDON_PROGRAM(hildon_program_get_instance());

	self->window = HILDON_WINDOW(hildon_window_new());
	hildon_program_add_window(self->program, self->window);

	create_menu(self);

	self->icon = create_icon(self);

	return self;
}

int main(int argc, char *argv[])
{
	CarmandTray *tray;

	g_type_init();
	g_thread_init(NULL);
	gdk_threads_init();
	gtk_init(NULL, NULL);

	tray = carmand_tray_new();
	if (tray == NULL)
		exit(EXIT_FAILURE);

	gtk_main();

	g_object_unref(tray);

	return 0;
}
