/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand-panel.
 *
 *  carmand-panel is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand-panel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libosso.h>
#include <glib.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <conbtdialogs-dbus.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-program.h>

/* Python includes */
#include <python2.5/Python.h>
#include "check_repository.h"

/* Localization includes */
#include <libintl.h>
#include <locale.h>

#include "carmand-logmsg.h"
#include "carmand-dbus.h"
#include "carmand-settings.h"

#define CONFIG_FILE		"/home/user/.carman/carman.conf"
#define CONFIG_GROUP		"carman-evas"
#define CONFIG_KEY		"user-repository-plugins"

#define DEFAULT_PRECISION_ITER	0
#define HOME_FOLDER		"/home/user/MyDocs"
#define DEFAULT_TRIP_FOLDER	HOME_FOLDER"/Carman/Trips"
#define DEFAULT_MAPS_FOLDER	HOME_FOLDER"/Carman/Maps"
#define DEFAULT_TRACKS_FOLDER	HOME_FOLDER"/Carman/Tracks"
#define MAPS_REPO_FOLDER	"/home/user/.carman/user_map_repo/"
#define ADD_MAP_PLUGIN		1
#define DEL_MAP_PLUGIN		2
#define SIMULATOR_ON		"simulator"
#define SIMULATOR_OFF		"normal"

#define MOUNT_TABLE		"/proc/mounts"
#define INTERNAL_MMC		"/media/mmc2"
#define EXTERNAL_MMC		"/media/mmc1"

gboolean obd_connected = FALSE;
gboolean gps_connected = FALSE;
static void load_map_repo(CarmandSettings *self, GtkWidget *combo);

/* FIXME: Needed by signal cbs. Must use marshall on signal cb to pass
 * parent_data on registered signal callback.
 */
CarmandSettings *carmand_settings_self;

G_DEFINE_TYPE(CarmandSettings, carmand_settings, GTK_TYPE_DIALOG)

/* Hildon Plugin Execute function */
osso_return_t execute(osso_context_t *osso,
		gpointer data, gboolean user_activated)
{
	g_type_init();

	if (carmand_settings_new(data) == NULL)
		return OSSO_ERROR;

	return OSSO_OK;
}

void dialog_response(GtkWidget *widget, gpointer data)
{
	carmand_vdebug("%s:%d\n", __func__, __LINE__);
	gtk_widget_hide_all(widget);
	gtk_widget_destroy(widget);
}

static void show_message(GtkWindow *w, char *msg, char *icon)
{
	GtkDialog* dialog = NULL;

	carmand_message("%s\n", msg);

	dialog = GTK_DIALOG(hildon_note_new_information_with_icon_name(w,
					msg, icon));
	gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void clear_models(GSList *ldevs)
{
	GtkListStore *lstore;
	GSList *lmodels;

	carmand_vdebug("%s\n", __func__);

	for (lmodels = ldevs; lmodels; lmodels = lmodels->next) {
		lstore = GTK_LIST_STORE(lmodels->data);
		gtk_list_store_clear(lstore);
	}
}

static void add_bluetooth_device(GSList *ldevs,
			const gchar *address, const gchar *name)
{
	GSList *lmodels;
	GtkListStore *lstore;
	GtkTreeIter iter_combo;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	/* Append on both combo boxes: OBD and GPS */
	for (lmodels = ldevs; lmodels; lmodels = lmodels->next) {

		lstore = GTK_LIST_STORE(lmodels->data);

		gtk_list_store_append(lstore, &iter_combo);
		gtk_list_store_set(lstore, &iter_combo, DEVICE_ADDRESS,
				address, DEVICE_NAME, name, -1);
	}
}

gchar *check_map_plugin(gchar *path)
{
	gchar *desc, *ret;

	desc = (gchar *) load_repo(path);
	ret = g_strdup(desc);
	return ret;
}

/* Load and Save Settings */
static void load_default_values_callback(GtkWidget *w, gpointer user_data)
{
	CarmandSettings *self = CARMAND_SETTINGS(user_data);
	HildonNote* dialog;

	carmand_vdebug("%s\n", __func__);

	dialog = (HildonNote *) hildon_note_new_confirmation(NULL,
					_("Reset to default values?"));
	hildon_note_set_button_texts(dialog, "Yes", "No");

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		gtk_combo_box_set_active(GTK_COMBO_BOX(
				self->combo_time_precision), 0);
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_trips_folder), 0);
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_maps_folder), 0);
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_tracks_folder), 0);
	}
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

gchar *save_storage_combo(CarmandSettings *self, GtkComboBox *combo, gchar *folder)
{
	GtkListStore *lstore;
	GtkTreeIter iter;
	gchar *value, *path = NULL, *msg;
	gboolean error;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	if (gtk_combo_box_get_active_iter(combo, &iter) == FALSE)
		return NULL;

	lstore =
		GTK_LIST_STORE(gtk_combo_box_get_model(combo));
	gtk_tree_model_get(GTK_TREE_MODEL(lstore), &iter,
			STORAGE_DEVICE, &value, -1);

	path = g_strconcat(value, folder, NULL);

	error = g_mkdir_with_parents(path, 0755);

	if (error) {
		free(path);
		return NULL;
	}

	return path;
}

gboolean load_config_storage_combo(GtkComboBox *combo, gchar *folder)
{
	GtkTreeIter iter;
	GtkListStore *list_store;
	gchar *device;
	gboolean result = FALSE;

	list_store =
		GTK_LIST_STORE(gtk_combo_box_get_model((GtkComboBox *) combo));
	result = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter);

	while (result) {
		gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
				STORAGE_DEVICE, &device, -1);
		if (g_str_has_prefix(folder, device)) {
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combo), &iter);
			result = TRUE;
			break;
		}
		result = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store),
				&iter);
	}
	return result;
}


/* Read all widgets and send values to carmand.conf through DBUS */
static gboolean save_config_callback(GtkWidget *w, gpointer user_data)
{
	CarmandSettings *self = CARMAND_SETTINGS(user_data);
	GtkListStore *list_store;
	GtkTreeIter iter;
	gchar *value;
	gboolean result;
	gchar *time_precision;
	gint value_precision;
	gchar *errfield;
	gchar *errmsg;
	gchar *folder = NULL;

	carmand_vdebug("%s\n", __func__);

	time_precision = gtk_combo_box_get_active_text(GTK_COMBO_BOX(
					self->combo_time_precision));

	value_precision = atoi(time_precision);

	if (!method_call_int_arg(self->dbus, DBUS_SETPRECISION,
			value_precision)) {
		errfield = g_strdup(_("Trips precision"));
		goto dbus_error;
	}

	folder = save_storage_combo(self, self->combo_trips_folder,
			"/Carman/Trips");

	if ((folder != NULL) &&
			(method_call_string_arg(self->dbus,
					DBUS_SETTRIPSFOLDER,
					folder))) {
		free(folder);
		folder = NULL;
	} else {
		errfield = g_strdup(_("Trips location"));
		goto dbus_error;
	}

	folder = save_storage_combo(self, self->combo_maps_folder,
			"/Carman/Maps");

	if ((folder != NULL) &&
			(method_call_string_arg(self->dbus,
					DBUS_SETMAPSFOLDER,
					folder))) {
		free(folder);
		folder = NULL;
	} else {
		errfield = g_strdup(_("Maps location"));
		goto dbus_error;
	}

	folder = save_storage_combo(self, self->combo_tracks_folder,
			"/Carman/Tracks");

	if ((folder != NULL) &&
			(method_call_string_arg(self->dbus,
					DBUS_SETTRACKSFOLDER,
					folder))) {
		free(folder);
		folder = NULL;
	} else {
		errfield = g_strdup(_("Tracks location"));
		goto dbus_error;
	}

	if (gtk_combo_box_get_active_iter(
		GTK_COMBO_BOX(self->combo_obd_device), &iter)) {
		list_store = GTK_LIST_STORE(gtk_combo_box_get_model(
					GTK_COMBO_BOX(self->combo_obd_device)));
		gtk_tree_model_get(GTK_TREE_MODEL(list_store),
				&iter, DEVICE_ADDRESS, &value, -1);
		if (!method_call_string_arg(self->dbus, DBUS_SETOBD, value)) {
			g_free(value);
			value = NULL;
			errfield = g_strdup(_("OBD device"));
			goto dbus_error;
		}
	}

	g_free(value);
	value = NULL;

	if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(self->combo_gps_device),
				&iter) ) {
		list_store = GTK_LIST_STORE(gtk_combo_box_get_model(
					GTK_COMBO_BOX(self->combo_gps_device)));
		gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
				DEVICE_ADDRESS, &value, -1);
		if (!method_call_string_arg(self->dbus, DBUS_SETGPS, value)) {
			g_free(value);
			value = NULL;
			errfield = g_strdup(_("GPS device"));
			goto dbus_error;
		}
	}

	g_free(value);
	value = NULL;

	cpanel_dbus_method_call(self->dbus, DBUS_SAVECONF,
			CARMAND_DBUS_IFACE_CONFIGURATION);

	gtk_widget_destroy(GTK_WIDGET(self));

	return TRUE;

dbus_error:
	errmsg = g_strdup_printf(_("Failure to save: %s"), errfield);
	show_message(GTK_WINDOW(self), errmsg, GTK_STOCK_DIALOG_ERROR);

	if (errfield)
		g_free(errfield);
	if (errmsg)
		g_free(errmsg);

	return FALSE;
}

/* */
static GtkWidget *create_number_editor(GtkWidget *container,
		GtkSizeGroup *size_group, gchar *label, gint min, gint max)
{
	GtkWidget *number_editor;
	GtkWidget *caption;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	number_editor = hildon_number_editor_new(min, max);
	caption = hildon_caption_new(size_group, label, number_editor, NULL, 0);
	hildon_caption_set_child_expand(HILDON_CAPTION(caption), TRUE);

	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(caption));

	return number_editor;
}

/* */
static GtkWidget *create_entry(GtkWidget *container,
		GtkSizeGroup *size_group, char *label, int n_chars)
{

	GtkWidget *caption;
	GtkWidget *entry;

	carmand_vdebug("%s\n", __func__);

	entry = gtk_entry_new();
	caption = hildon_caption_new(size_group, label, entry, NULL, 0);
	hildon_caption_set_child_expand(HILDON_CAPTION(caption), TRUE);
	gtk_entry_set_width_chars(GTK_ENTRY(entry), n_chars);

	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(caption));

	return entry;

}

/* */
static void create_path_dialog(CarmandSettings *self, gpointer txt_entry)
{
	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	if (!self->dialog_path) {
		self->dialog_path = gtk_dialog_new_with_buttons(
				_("Folder Selection"),
				GTK_WINDOW(self),
				GTK_DIALOG_DESTROY_WITH_PARENT |
				GTK_DIALOG_NO_SEPARATOR, _("Select"),
				GTK_RESPONSE_OK, GTK_STOCK_CANCEL,
				GTK_RESPONSE_CANCEL, NULL);

		gtk_dialog_set_default_response(GTK_DIALOG(self->dialog_path),
				GTK_RESPONSE_OK);

		self->model_path = HILDON_FILE_SYSTEM_MODEL(
				g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
					"ref-widget", GTK_DIALOG(
						self->dialog_path)->vbox, NULL));
	}

	if (self->tree_filesel_path)
		gtk_widget_destroy(GTK_WIDGET(self->tree_filesel_path));

	self->tree_filesel_path = HILDON_FILE_SELECTION(
			hildon_file_selection_new_with_model(self->model_path));

	hildon_file_selection_hide_content_pane(self->tree_filesel_path);

	gtk_widget_set_size_request(GTK_WIDGET(self->tree_filesel_path),
			FOLDER_SELECTION_WIDTH, FOLDER_SELECTION_HEIGHT);

	gtk_widget_show(GTK_WIDGET(self->tree_filesel_path));

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(self->dialog_path)->vbox),
			GTK_WIDGET(self->tree_filesel_path));

	hildon_file_selection_set_current_folder(self->tree_filesel_path, \
			(GtkFilePath*)txt_entry, NULL );
}

gchar *file_selection_dialog(CarmandSettings *self, int operation)
{
	GtkWidget *dialog;
	GtkFileFilter *filter;
	gchar *filename = NULL;

	dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(self),
			GTK_FILE_CHOOSER_ACTION_OPEN);

	if (operation == ADD_MAP_PLUGIN) {
		filter = gtk_file_filter_new();
		gtk_file_filter_set_name(filter, "Python file");
		gtk_file_filter_add_pattern(filter, "*.py");
		gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialog), filter);
	} else if (operation == DEL_MAP_PLUGIN) {
		gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(dialog), TRUE);
		if (!gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(dialog),
				self->user_repo_dir))
			return NULL;
	}

	gtk_widget_show_all(GTK_WIDGET(dialog));

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
	}

	gtk_widget_destroy(dialog);

	return filename;
}

/* */
const gchar *show_path_dialog(CarmandSettings *self)
{

	GtkFilePath *file_path;
	GtkFileSystem *file_system;
	const gchar *selected_path;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	selected_path = NULL;
	if (!self->dialog_path)
		create_path_dialog(self, HOME_FOLDER);

	gtk_widget_show_all(self->dialog_path);

	if (gtk_dialog_run(
		GTK_DIALOG(self->dialog_path)) == GTK_RESPONSE_OK) {

		file_path = (GtkFilePath *) hildon_file_selection_get_current_folder(
				self->tree_filesel_path);
		file_system = _hildon_file_system_model_get_file_system(
				self->model_path);

		if (file_path && file_system) {
			selected_path =
				gtk_file_system_path_to_filename(file_system,
						file_path);
			gtk_file_path_free(file_path);
		}
	}

	gtk_widget_destroy(self->dialog_path);
	self->dialog_path = NULL;
	return selected_path;
}

/* GTK Signals Callbacks */

gboolean copy_repo_file(gchar *source, gchar *dest)
{
	int ret;

	ret = (int) copy_files(source, dest);

	if (ret == 1)
		return TRUE;
	else
		return FALSE;
}

/* self->add_repo_button callback */
static void add_repo_click_cb(GtkButton *button, gpointer data)
{
	CarmandSettings *self = CARMAND_SETTINGS(data);
	gchar *path;
	gchar *repo_name;
	gboolean error;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	error = g_mkdir_with_parents(self->user_repo_dir, 0755);

	path = file_selection_dialog(self, ADD_MAP_PLUGIN);

	if (path && !error) {
		repo_name = check_map_plugin(path);
		if (repo_name && strcmp(repo_name, "")) {
			if (copy_repo_file(path, self->user_repo_dir)) {
				load_map_repo(self, self->combo_repo);
				show_message(GTK_WINDOW(self),
						_("Map plugin successfully installed"),
						GTK_STOCK_DIALOG_INFO);
			} else
				show_message(GTK_WINDOW(self),
						_("Error copying plugin file"),
						GTK_STOCK_DIALOG_ERROR);
		}  else
			show_message(GTK_WINDOW(self),
				_("Invalid map plugin"),
				GTK_STOCK_DIALOG_ERROR);

		g_free(repo_name);
	}
}

/* self->del_repo_button callback */
static void del_repo_click_cb(GtkButton *button, gpointer data)
{
	CarmandSettings *self = CARMAND_SETTINGS(data);
	gchar *value;
	GtkListStore *list_store;
	GtkTreeIter iter;
	HildonNote *dialog;
	gchar *aux;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(self->combo_repo),
				&iter)) {
		list_store = GTK_LIST_STORE(gtk_combo_box_get_model(
					GTK_COMBO_BOX(self->combo_repo)));
		gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
				MAP_FILE, &value, -1);
		if (strcmp(value, "none") != 0) {
			dialog = (HildonNote *)
				hildon_note_new_confirmation(NULL,
						_("Confirm delete?"));
			if (gtk_dialog_run(GTK_DIALOG(dialog)) ==
					GTK_RESPONSE_OK) {
				g_unlink(value);
				aux = g_strdup_printf("%sc", value);
				g_unlink(aux);
				g_free(aux);
				aux = g_strdup_printf("%so", value);
				g_unlink(aux);
				g_free(aux);
				load_map_repo(self, self->combo_repo);
			}
			gtk_widget_destroy(GTK_WIDGET(dialog));
		} else
			show_message(GTK_WINDOW(self), _("Select a valid maps"
						" plugin repository to delete."),
					GTK_STOCK_DIALOG_ERROR);
		g_free(value);
		value = NULL;
	}
}

/* Rules */

static void pair_device_callback(GtkWidget *w, gpointer user_data)
{
	CarmandSettings *self = CARMAND_SETTINGS(user_data);
	CarmandDbus *dbus = self->dbus;
	GError *gerr = NULL;

	carmand_vdebug("%s %p\n", __func__, dbus->proxy_bluetooth);

	if (!dbus_g_proxy_call(dbus->proxy_bluetooth, CONBTDIALOGS_SEARCH_REQ,
				&gerr, G_TYPE_STRING, "", G_TYPE_STRING, "",
				G_TYPE_STRV, "", G_TYPE_STRING, "require",
				G_TYPE_INVALID, G_TYPE_INVALID)) {
		show_message(GTK_WINDOW(self),
			_("Error during the call of DBus Method (Bluetooth UI)"),
			GTK_STOCK_DIALOG_ERROR);
		carmand_vdebug("%s: %s\n", __func__, gerr->message);
		g_error_free(gerr);
	}
}

static void bonding_created_cb(DBusGProxy *object,
		const gchar *address, gpointer user_data)
{
	CarmandSettings *self = CARMAND_SETTINGS(user_data);
	CarmandDbus *dbus = self->dbus;
	GQuark quark_address;
	gchar *name = NULL;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	if (strcmp(address,"") != 0) {
		quark_address = g_quark_from_string(address);

		dbus_g_proxy_call(dbus->proxy_bluez, "GetRemoteAlias", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

		if (name == NULL) {
			dbus_g_proxy_call(dbus->proxy_bluez, "GetRemoteName",
					NULL, G_TYPE_STRING, address,
					G_TYPE_INVALID, G_TYPE_STRING, &name,
					G_TYPE_INVALID);
		}

		if (name == NULL)
			return;

		add_bluetooth_device(self->list_obd_gps_devs, address, name);
		g_free(name);
	}
}

static void bonding_removed_cb(DBusGProxy *object,
		const gchar *address, gpointer user_data)
{
	CarmandSettings *self = CARMAND_SETTINGS(user_data);
	GSList *list_models = self->list_obd_gps_devs;
	GtkListStore *list_store;
	GtkTreeIter iter_combo;
	gboolean result;
	gchar *address_stored;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	while (list_models) {

		list_store = GTK_LIST_STORE(list_models->data);

		result = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter_combo);
		while (result) {
			gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter_combo,
					DEVICE_ADDRESS, &address_stored, -1);

			if (strcmp(address_stored, address) == 0) {
				gtk_list_store_remove(list_store, &iter_combo);
				break;
			} else {
				result = gtk_tree_model_iter_next(
						GTK_TREE_MODEL(list_store),
						&iter_combo);
			}
		}
		list_models = g_slist_next(list_models);
	}

	if (gtk_combo_box_get_active(GTK_COMBO_BOX(self->combo_obd_device)) == -1)
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_obd_device), 0);

	if (gtk_combo_box_get_active(GTK_COMBO_BOX(self->combo_gps_device)) == -1)
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_gps_device), 0);
}

/* button_obd_conn callback */
static void connect_obd(GtkWidget* widget, gpointer data)
{
	CarmandSettings *self = CARMAND_SETTINGS(data);
	GtkListStore *lstore;
	GtkTreeIter iter;
	gchar *value;

	carmand_vdebug("%s\n", __func__);

	if (obd_connected == TRUE) {
		cpanel_dbus_method_call(self->dbus,
				DBUS_OBD_METHOD_DISCONNECTOBD,
				CARMAND_DBUS_IFACE_OBD);
		return;
	}


	/* Otherwise: Set the new OBD device and connect */
	if (gtk_combo_box_get_active_iter(
		GTK_COMBO_BOX(self->combo_obd_device), &iter) == FALSE)
		return;

	lstore = GTK_LIST_STORE(gtk_combo_box_get_model(
				GTK_COMBO_BOX(self->combo_obd_device)));
	gtk_tree_model_get(GTK_TREE_MODEL(lstore),
			&iter, DEVICE_ADDRESS, &value, -1);

	if (method_call_string_arg(self->dbus,
				DBUS_SETOBD, value) == FALSE) {
		show_message(GTK_WINDOW(self),
				_("Connection failed!"),
				GTK_STOCK_DIALOG_ERROR);
		g_free(value);
		return;
	}

	g_free(value);

	cpanel_dbus_method_call(self->dbus,
			DBUS_OBD_METHOD_CONNECTOBD,
			CARMAND_DBUS_IFACE_OBD);
}

/* combo_gps_device "changed" signal callback */
static void combo_gps_obd_changed(GtkComboBox *combo, gpointer data)
{
	GtkWidget *button = GTK_WIDGET(data);
	GtkListStore *list_store;
	GtkTreeIter iter;
	gchar *value;

	carmand_vdebug("%s\n", __func__);

	if (gtk_combo_box_get_active_iter(combo, &iter) == FALSE)
		return;

	list_store = GTK_LIST_STORE(gtk_combo_box_get_model(combo));
	gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
			DEVICE_ADDRESS, &value,	-1);

	if (strcmp(value, "none") == 0)
		gtk_widget_set_sensitive(button, FALSE);
	else
		gtk_widget_set_sensitive(button, TRUE);
}

static void combo_repo_changed(GtkComboBox *combo, gpointer data)
{
	GtkWidget *button = GTK_WIDGET(data);
	GtkListStore *list_store;
	GtkTreeIter iter;
	gchar *value;

	carmand_vdebug("%s\n", __func__);

	if (gtk_combo_box_get_active_iter(combo, &iter) == FALSE)
		return;

	list_store = GTK_LIST_STORE(gtk_combo_box_get_model(combo));
	gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
			REPO_NAME, &value, -1);

	if (strcmp(value, "None") == 0)
		gtk_widget_set_sensitive(button, FALSE);
	else
		gtk_widget_set_sensitive(button, TRUE);

}

/* button_gps_conn callback */
static void connect_gps(GtkWidget *widget, gpointer data)
{
	CarmandSettings *self = CARMAND_SETTINGS(data);
	GtkListStore *lstore;
	GtkTreeIter iter;
	gchar *value;

	carmand_vdebug("%s\n", __func__);

	if (gps_connected == TRUE) {
		cpanel_dbus_method_call(self->dbus,
				DBUS_GPS_METHOD_DISCONNECT,
				CARMAND_DBUS_IFACE_GPS);
		return;
	}


	/* Otherwise: Set the new GPS device and connect */
	if (gtk_combo_box_get_active_iter(
		GTK_COMBO_BOX(self->combo_gps_device), &iter) == FALSE)
		return;

	lstore = GTK_LIST_STORE(gtk_combo_box_get_model(
				GTK_COMBO_BOX(self->combo_gps_device)));
	gtk_tree_model_get(GTK_TREE_MODEL(lstore), &iter,
			DEVICE_ADDRESS, &value,	-1);

	if (method_call_string_arg(self->dbus,
			DBUS_SETGPS, value) == FALSE) {
		g_free(value);
		show_message(GTK_WINDOW(self),
				_("Connection failed!"),
				GTK_STOCK_DIALOG_ERROR);
		return;
	}

	g_free(value);

	cpanel_dbus_method_call(self->dbus,
		DBUS_GPS_METHOD_CONNECT, CARMAND_DBUS_IFACE_GPS);
}

/* Check Simulator checkbox toggled signal */
static void check_mode_cb(GtkWidget *widget, gpointer data)
{
	CarmandSettings *self = CARMAND_SETTINGS(data);
	GtkToggleButton *check_mode = GTK_TOGGLE_BUTTON(widget);
	const gchar *mode;
	gboolean active;

	carmand_vdebug("%s\n", __func__);

	active = gtk_toggle_button_get_active(check_mode);
	mode = (active ? SIMULATOR_ON: SIMULATOR_OFF);

	method_call_string_arg(self->dbus, DBUS_SETMODE, mode);
}

static void change_gps_status(const gchar *status)
{
	CarmandSettings *self = carmand_settings_self;
	DBusError derr;
	gchar *gps_device;

	carmand_vdebug("%s\n", __func__);

	gps_device = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETGPS,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);

	gtk_label_set_text(GTK_LABEL(self->label_gps_status), status);

	if (gps_device == NULL)
		return;

	if (strcmp(status,"Disconnected") == 0) {
		gtk_widget_set_sensitive(self->combo_gps_device, TRUE);
		if (strcmp(gps_device, "none") == 0)
			gtk_widget_set_sensitive(self->button_gps_conn, FALSE);
		else
			gtk_widget_set_sensitive(self->button_gps_conn, TRUE);

		gtk_button_set_label(GTK_BUTTON(self->button_gps_conn),
				"Connect");
		gps_connected = FALSE;
	} else if (strcmp(status, "Fixing") == 0) {
		gtk_widget_set_sensitive(self->combo_gps_device, FALSE);
		gtk_widget_set_sensitive(self->button_gps_conn,	TRUE);
		gtk_button_set_label(GTK_BUTTON(self->button_gps_conn),
				"Disconnect");
		gps_connected = TRUE;
	} else if (strcmp(status, "Fixed") == 0) {
		gtk_button_set_label(GTK_BUTTON(self->button_gps_conn),
				"Disconnect");
		gtk_widget_set_sensitive(self->button_gps_conn,	TRUE);
		gtk_widget_set_sensitive(self->combo_gps_device, FALSE);
		gps_connected = TRUE;
	} else if (strcmp(status, "Connecting") == 0) {
		gtk_widget_set_sensitive(self->combo_gps_device, FALSE);
		gtk_widget_set_sensitive(self->button_gps_conn,	FALSE);
		gps_connected = FALSE;
	}

	g_free(gps_device);
}

/**
 * Change ODB status
 * @param self CarmandSettings
 * @param status String status
 * @param connected True or False
 * @param sensitive True or False
 */
static void change_obd_status(const gchar *status)
{
	CarmandSettings *self = carmand_settings_self;
	DBusError derr;
	gchar *obd_device;

	carmand_vdebug("%s\n", __func__);

	obd_device = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETOBD,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);

	if (obd_device == NULL)
		return;

	if (strcmp(status, "Disconnected") == 0) {
		gtk_button_set_label(GTK_BUTTON(self->button_obd_conn),
				"Connect");
		gtk_widget_set_sensitive(self->combo_obd_device, TRUE);
		if (strcmp(obd_device, "none") == 0)
			gtk_widget_set_sensitive(self->button_obd_conn, FALSE);
		else
			gtk_widget_set_sensitive(self->button_obd_conn, TRUE);

		obd_connected = FALSE;
	} else if (strcmp(status, "Connected") == 0) {
		gtk_button_set_label(GTK_BUTTON(self->button_obd_conn),
				"Disconnect");
		gtk_widget_set_sensitive(self->combo_obd_device, FALSE);
		gtk_widget_set_sensitive(self->button_obd_conn, TRUE);
		obd_connected = TRUE;
	} else if (strcmp(status, "Connecting") == 0) {
		gtk_widget_set_sensitive(self->button_obd_conn, FALSE);
		gtk_widget_set_sensitive(self->combo_obd_device, FALSE);
		obd_connected = FALSE;
	}

	g_free(obd_device);
}

static void signal_gps_status_cb(DBusGProxy *proxy, gpointer data)
{
	carmand_vdebug("%s\n", __func__);

	if (data)
		change_gps_status((const gchar *) data);
}

static void signal_obd_status_cb(DBusGProxy *proxy, gpointer data)
{
	carmand_vdebug("%s\n", __func__);

	if (data)
		change_obd_status((const gchar *) data);
}

/**
 * Get conn status
 * @param self CarmandSettings
 */
static void get_conn_status(CarmandSettings *self)
{
	DBusError derr;
	gchar *status;

	carmand_vdebug("%s\n", __func__);

	dbus_error_init(&derr);
	status = cpanel_dbus_str_method_call(self->dbus,
			DBUS_GPS_METHOD_STATUS,
			CARMAND_DBUS_IFACE_GPS, &derr);

	if (status) {
		change_gps_status(status);
		g_free(status);
	} else {
		show_message(GTK_WINDOW(self), (gchar *) derr.message,
			GTK_STOCK_DIALOG_ERROR);
		dbus_error_free(&derr);
	}

	dbus_error_init(&derr);
	status = cpanel_dbus_str_method_call(self->dbus,
			DBUS_OBD_METHOD_STATUS,
			CARMAND_DBUS_IFACE_OBD, &derr);

	if (status) {
		change_obd_status(status);
		g_free(status);
	} else {
		show_message(GTK_WINDOW(self), (gchar *) derr.message,
			GTK_STOCK_DIALOG_ERROR);
		dbus_error_free(&derr);
	}
}

/**
 * Load config values
 * @param self CarmandSettings
 */
static void load_config_values(CarmandSettings *self)
{
	GtkListStore *list_store;
	GtkTreeIter iter_combo_obd;
	GtkTreeIter iter_combo_gps;
	DBusError derr;
	gboolean result;
	gchar *address;
	gchar *device;
	gint value_precision;

	gchar *trips_folder;
	gchar *maps_folder;
	gchar *tracks_folder;
	gchar *obd_device;
	gchar *gps_device;
	gchar *mode;

	carmand_vdebug("%s\n", __func__);

	trips_folder = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETTRIPSFOLDER,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);

	if (trips_folder) {
		if (!load_config_storage_combo(self->combo_trips_folder,
					trips_folder)) {
			show_message(GTK_WINDOW(self),
					"Trips storage device not found,\n"
					"using home folder",
					GTK_STOCK_DIALOG_INFO);
			gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_trips_folder), 0);
		}

		free(trips_folder);
	}

	maps_folder = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETMAPSFOLDER,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);
	if (maps_folder) {
		if (!load_config_storage_combo(self->combo_maps_folder,
					maps_folder)) {
			show_message(GTK_WINDOW(self),
					"Maps storage device not found,\n"
					"using home folder",
					GTK_STOCK_DIALOG_INFO);
			gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_maps_folder), 0);
		}

		free(maps_folder);
	}

	tracks_folder = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETTRACKSFOLDER,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);

	if (tracks_folder) {
		if (!load_config_storage_combo(self->combo_tracks_folder,
					tracks_folder)) {
			show_message(GTK_WINDOW(self),
					"Tracks storage device not found,\n"
					"using home folder",
					GTK_STOCK_DIALOG_INFO);
			gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_tracks_folder), 0);
		}

		free(tracks_folder);
	}

	/* Changing simulator mode checkbox status */
	mode = cpanel_dbus_str_method_call(self->dbus, DBUS_GETMODE,
			CARMAND_DBUS_IFACE_CONFIGURATION, &derr);

	if (mode) {
		gboolean active;
		if (strcmp(SIMULATOR_OFF, mode) == 0)
			active = FALSE;
		else
			active = TRUE;

		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(self->check_mode), active);

		g_free(mode);
	}

	/* check values to avoid hildon_number_editor_set_value assertion
	 * failed
	 */
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->combo_time_precision), "3");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->combo_time_precision), "5");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->combo_time_precision), "15");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->combo_time_precision), "30");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->combo_time_precision), "60");

	value_precision = cpanel_dbus_int_method_call(self->dbus,
			DBUS_GETPRECISION,
			CARMAND_DBUS_IFACE_CONFIGURATION);

	switch (value_precision) {
	case 3:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 0);
		break;
	case 5:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 1);
		break;
	case 15:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 2);
		break;
	case 30:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 3);
		break;
	case 60:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 4);
		break;
	default:
		gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_time_precision), 0);
		break;
	}

	obd_device = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETOBD,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);
	if (obd_device == NULL)
		goto dbus_error;

	if (strcmp(obd_device, "none") == 0) {
		gtk_widget_set_sensitive((GtkWidget *) self->button_obd_conn,
				FALSE);
	}
	list_store = GTK_LIST_STORE(gtk_combo_box_get_model(
				(GtkComboBox *) self->combo_obd_device));
	result = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store),
			&iter_combo_obd);

	while (result) {
		gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter_combo_obd,
				DEVICE_ADDRESS, &address, -1);
		if (strcmp(address, obd_device) == 0) {
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(
						self->combo_obd_device),
						&iter_combo_obd);
			break;
		}
		result = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store),
				&iter_combo_obd);
	}

	gps_device = cpanel_dbus_str_method_call(self->dbus,
				DBUS_GETGPS,
				CARMAND_DBUS_IFACE_CONFIGURATION, &derr);
	if (gps_device == NULL)
		goto dbus_error;

	if (strcmp(gps_device, "none") == 0) {
		gtk_widget_set_sensitive((GtkWidget *) self->button_gps_conn,
				FALSE);
	}

	list_store = GTK_LIST_STORE(gtk_combo_box_get_model(
				(GtkComboBox *) self->combo_gps_device));
	result = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store),
			&iter_combo_gps);
	while (result) {
		gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter_combo_gps,
				DEVICE_ADDRESS, &address, -1);
		if (strcmp(address, gps_device) == 0) {
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(
						self->combo_gps_device),
						&iter_combo_gps);
			break;
		}
		result = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store),
				&iter_combo_gps);
	}

	g_free(obd_device);
	g_free(gps_device);
	return;

dbus_error:
	show_message(GTK_WINDOW(self), (gchar *) derr.message,
			GTK_STOCK_DIALOG_ERROR);
	dbus_error_free(&derr);
	gtk_widget_destroy(GTK_WIDGET(self));
}

/**
 * Load internal GPS
 * @param self CarmandSettings
 */
static void load_internal_gps(CarmandSettings *self)
{
	FILE *file_gps_device;
	gboolean internal_gps = FALSE;
	GtkListStore* list_store_gps;
	GtkTreeIter iter_combo;

	carmand_vdebug("%s\n", __func__);

	file_gps_device = fopen("/dev/pgps", "r");
	if (file_gps_device) {
		internal_gps = TRUE;
		fclose(file_gps_device);
	}

	list_store_gps = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(
			self->combo_gps_device)));

	if (internal_gps) {
		gtk_list_store_insert (list_store_gps, &iter_combo, 0);
		gtk_list_store_set (list_store_gps, &iter_combo,
				DEVICE_ADDRESS, "internal", DEVICE_NAME,
				_("Internal GPS"), -1);
	}

	gtk_combo_box_set_active(GTK_COMBO_BOX(self->combo_gps_device), 0);
}

/**
 * Load None BT devices
 * @param combo Combo box to include "none" item
 */
static void load_none_devices(GtkWidget *combo)
{
	GtkListStore  *list_store;
	GtkTreeIter iter_combo;

	carmand_vdebug("%s\n", __func__);

	list_store =
		GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(combo)));

	gtk_list_store_insert(list_store, &iter_combo, 0);
	gtk_list_store_set(list_store, &iter_combo, DEVICE_ADDRESS, "none",
			DEVICE_NAME, _("None"), -1);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
}

/**
 * Load Bluetooth devices.
 * @param self CarmandSettings
 */
static gboolean load_bt_devices(CarmandSettings *self)
{
	CarmandDbus *dbus = self->dbus;
	gchar **devices;
	gchar *address, *name;
	GError *gerr = NULL;
	int i;

	carmand_vdebug("%s\n", __func__);

	if (dbus_g_proxy_call(dbus->proxy_bluez, "ListBondings",
				&gerr, G_TYPE_INVALID, G_TYPE_STRV,
				&devices, G_TYPE_INVALID) == FALSE) {

		carmand_error("Failed to list bondings: %s",
				gerr->message);
		g_error_free(gerr);
		return FALSE;
	}

	if (devices == NULL)
		return FALSE;

	for (i = 0, address = devices[i]; address; address = devices[++i]) {
		name = NULL;
		dbus_g_proxy_call(dbus->proxy_bluez, "GetRemoteAlias", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

		if (name == NULL) {
			dbus_g_proxy_call(dbus->proxy_bluez, "GetRemoteName",
					NULL, G_TYPE_STRING, address,
					G_TYPE_INVALID, G_TYPE_STRING, &name,
					G_TYPE_INVALID);
		}

		add_bluetooth_device(self->list_obd_gps_devs, address, name);
	}

	free(devices);

	return TRUE;
}

static void load_map_repo(CarmandSettings *self, GtkWidget *combo)
{
	GtkListStore  *list_store;
	GtkTreeIter iter_combo;
	GDir *dir_iter;
	gchar *file;
	gchar *fullpath;
	gchar *reponame;
	GError *error = NULL;
	char tmp_file[100];
	static int count = 0;

	carmand_vdebug("%s\n", __func__);

	list_store =
		GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(combo)));
	gtk_list_store_clear(list_store);

	dir_iter = g_dir_open(self->user_repo_dir, 0, &error);
	if (dir_iter) {
		while (file = (gchar *)g_dir_read_name(dir_iter)) {
			snprintf(tmp_file, 90, "/tmp/carman_pluging%d.py", count++);	
			fullpath = g_strconcat(self->user_repo_dir, "/", file, NULL);
			copy_repo_file(fullpath, tmp_file);
			reponame = check_map_plugin(tmp_file);
			if (reponame && strcmp(reponame, "")) {
				gtk_list_store_insert(list_store, &iter_combo, 0);
				gtk_list_store_set(list_store, &iter_combo,
						MAP_FILE, fullpath, REPO_NAME,
						reponame, -1);
			}
			g_remove(tmp_file);
			strcat(tmp_file, "c");
			g_remove(tmp_file);
			g_free(fullpath);
			g_free(reponame);
		}
		g_dir_close(dir_iter);
	}

	gtk_list_store_insert(list_store, &iter_combo, 0);
	gtk_list_store_set(list_store, &iter_combo, MAP_FILE, "none",
			REPO_NAME, _("None"), -1);
	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
}

gboolean check_mount_point(gchar *mount_point)
{
	int i;
	char *tok, *cursor, *line = NULL;
	FILE *file;
	size_t len = 0;
	gboolean result = FALSE;

	if (!MOUNT_TABLE)
		goto out;

	file = fopen(MOUNT_TABLE, "r");
	if (!file) {
		carmand_error("Error opening mount point.");
		goto out;
	}

	while (result == 0 && getline(&line, &len, file) > 0) {
		cursor = line;
		for (i = 0; i < 2; i++)
			tok = strsep(&cursor, " ");

		if (tok && strcmp(mount_point, tok) == 0) {
			result = TRUE;
		}

		free(line);
		line = NULL;
	}

	fclose(file);
out:
	return result;
}
/* Check if there are mounted mmc disks and insert them
 * into combo_trips_folder, combo_maps_folder and combo_tracks_folder
 */
static void load_storage_devs(GtkWidget *combo)
{
	GtkListStore *list_store;
	GtkTreeIter iter_combo;
	GError *error = NULL;

	carmand_vdebug("%s\n", __func__);

	list_store =
		GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(combo)));
	gtk_list_store_clear(list_store);

	if (check_mount_point(INTERNAL_MMC)) {
		gtk_list_store_insert(list_store, &iter_combo, 0);
		gtk_list_store_set(list_store, &iter_combo, STORAGE_DEVICE,
			INTERNAL_MMC, STORAGE_NAME, "Internal MMC", -1);
	}

	if (check_mount_point(EXTERNAL_MMC)) {
		gtk_list_store_insert(list_store, &iter_combo, 0);
		gtk_list_store_set(list_store, &iter_combo, STORAGE_DEVICE,
				EXTERNAL_MMC, STORAGE_NAME, "External MMC", -1);

	}

	gtk_list_store_insert(list_store, &iter_combo, 0);
	gtk_list_store_set(list_store, &iter_combo, STORAGE_DEVICE,
			HOME_FOLDER, STORAGE_NAME, "Home folder", -1);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
}

/**
 * Load configs
 * @param self CarmandSettings
 */
static gboolean load_config(CarmandSettings *self)
{
	gchar *value;
	GKeyFile *key_file;

	carmand_vdebug("%s\n", __func__);

	g_free(self->user_repo_dir);
	self->user_repo_dir = NULL;
	key_file = g_key_file_new();
	if (g_key_file_load_from_file(key_file, CONFIG_FILE, G_KEY_FILE_NONE,
				NULL))
		self->user_repo_dir = g_key_file_get_string(key_file,
				CONFIG_GROUP, CONFIG_KEY, NULL);
	if (!self->user_repo_dir)
		self->user_repo_dir = g_strdup(MAPS_REPO_FOLDER);
	g_key_file_free(key_file);

	clear_models(self->list_obd_gps_devs);
	clear_models(self->list_map_repo);
	clear_models(self->list_storage_devs);

	if (!load_bt_devices(self))
		goto fail;

	load_none_devices(self->combo_obd_device);
	load_none_devices(self->combo_gps_device);
	load_map_repo(self, self->combo_repo);
	load_storage_devs(self->combo_trips_folder);
	load_storage_devs(self->combo_maps_folder);
	load_storage_devs(self->combo_tracks_folder);
	load_internal_gps(self);
	load_config_values(self);

	get_conn_status(self);

	return TRUE;
fail:
	carmand_error("Failed to load configs.");
	return FALSE;
}


/* UI Functions */
static GtkWidget *create_map_service_combo(CarmandSettings *self, GtkWidget *container,
		GtkSizeGroup *size_group, char *label, int type)
{
	GtkWidget *caption;
	GtkWidget *combo_box;
	GtkListStore *st_combo;
	GtkCellRenderer *renderer;
	GtkTreeIter iter_combo;

	carmand_vdebug("%s\n", __func__);

	st_combo = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	combo_box = gtk_combo_box_new_with_model(GTK_TREE_MODEL(st_combo));

	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo_box), renderer, TRUE);

	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo_box),
			renderer, "markup", REPO_NAME, NULL);

	self->list_map_repo = g_slist_prepend(self->list_map_repo,
			st_combo);
	caption = hildon_caption_new(size_group, label, combo_box, NULL, 0);

	gtk_box_pack_start(GTK_BOX(container), GTK_WIDGET(caption), FALSE,
			FALSE, 0);

	return combo_box;
}

static GtkWidget *create_storage_combo(CarmandSettings *self, GtkWidget *container,
		GtkSizeGroup *size_group, char *label, int type)
{
	GtkWidget *caption;
	GtkWidget *combo_box;
	GtkListStore *st_combo;
	GtkCellRenderer *renderer;
	GtkTreeIter iter_combo;

	carmand_vdebug("%s\n", __func__);

	st_combo = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	combo_box = gtk_combo_box_new_with_model(GTK_TREE_MODEL(st_combo));

	gtk_combo_box_set_model(GTK_COMBO_BOX(combo_box), GTK_TREE_MODEL(
				st_combo));

	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo_box), renderer, TRUE);

	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo_box),
			renderer, "markup", STORAGE_NAME, NULL);

	self->list_storage_devs = g_slist_prepend(self->list_storage_devs,
			st_combo);
	caption = hildon_caption_new(size_group, label, combo_box, NULL, 0);

	gtk_box_pack_start(GTK_BOX(container), GTK_WIDGET(caption), FALSE,
			FALSE, 0);

	return combo_box;
}

/**
 * Create combo box: time_precision, GPS and OBD
 */
static GtkWidget *create_combo_box(CarmandSettings *self, GtkWidget *container,
		GtkSizeGroup *size_group, char *label, int type)
{
	GtkWidget *caption;
	GtkWidget *combo_box;
	GtkListStore *st_combo;
	GtkCellRenderer *renderer;

	carmand_vdebug("%s\n", __func__);

	st_combo = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	combo_box = gtk_combo_box_new_with_model(GTK_TREE_MODEL(st_combo));

	gtk_combo_box_set_model(GTK_COMBO_BOX(combo_box), GTK_TREE_MODEL(
				st_combo));

	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo_box), renderer, TRUE);

	if (type == 1) {
		gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo_box),
				renderer, "markup", DEVICE_NAME, NULL);
		self->list_obd_gps_devs = g_slist_prepend(self->list_obd_gps_devs,
				st_combo);
	} else {
		gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo_box), renderer,
				"markup", VALUE, NULL);
	}

	caption = hildon_caption_new(size_group, label, combo_box, NULL, 0);

	gtk_box_pack_start(GTK_BOX(container), GTK_WIDGET(caption), FALSE,
			FALSE, 0);

	return combo_box;
}

/**
 * Create devices page
 * @param self CarmandSettings
 * @param container Container to connections page
 */
static void create_connections_page(CarmandSettings *self, GtkWidget *container)
{
	GtkWidget *hbox;
	GtkSizeGroup *size_group;
	GtkWidget *label;
	GtkWidget *bt_label;

	carmand_vdebug("%s\n", __func__);

	size_group = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);

	/* OBD-II */
	hbox = gtk_hbox_new(0,0);
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(hbox));

	self->combo_obd_device = create_combo_box(self, hbox, size_group,
			_("OBD-II:"), 1);
	bt_label = gtk_label_new("Connect");
	self->button_obd_conn = gtk_button_new();
	gtk_widget_set_size_request(GTK_WIDGET(self->button_obd_conn),120,-1);
	gtk_container_add(GTK_CONTAINER(self->button_obd_conn), GTK_WIDGET(bt_label));
	g_signal_connect(G_OBJECT(self->button_obd_conn), "clicked",
			G_CALLBACK(connect_obd), self);

	gtk_box_pack_end(GTK_BOX(hbox), GTK_WIDGET(self->button_obd_conn),
			FALSE, FALSE, 2);

	/* GPS */
	hbox = gtk_hbox_new(0,0);
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(hbox));

	self->combo_gps_device = create_combo_box(self, hbox, size_group,
					_("GPS:"), 1);

	self->label_gps_status = gtk_label_new(_("Disconnected"));
	gtk_container_add(GTK_CONTAINER(hbox), GTK_WIDGET(self->label_gps_status));

	bt_label = NULL;
	self->button_gps_conn = gtk_button_new();
	bt_label = gtk_label_new("Connect");
	gtk_widget_set_size_request(GTK_WIDGET(self->button_gps_conn),120,-1);
	gtk_container_add(GTK_CONTAINER(self->button_gps_conn), GTK_WIDGET(bt_label));
	g_signal_connect(G_OBJECT(self->button_gps_conn), "clicked",
			G_CALLBACK(connect_gps), self);

	gtk_box_pack_end(GTK_BOX(hbox), GTK_WIDGET(self->button_gps_conn),
			FALSE, FALSE, 2);

	/* Combo Boxes signals */
	g_signal_connect(G_OBJECT(self->combo_gps_device), "changed",
			G_CALLBACK(combo_gps_obd_changed),
			self->button_gps_conn);

	g_signal_connect(G_OBJECT(self->combo_obd_device), "changed",
			G_CALLBACK(combo_gps_obd_changed),
			self->button_obd_conn);

	/* Simulator mode check box */
	hbox = gtk_hbox_new(0,0);
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(hbox));

	self->check_mode = gtk_check_button_new();
	label = hildon_caption_new(size_group, _("Simulator mode"),
			self->check_mode, NULL, 0);
	hildon_caption_set_child_expand(HILDON_CAPTION(label), FALSE);
	gtk_container_add(GTK_CONTAINER(hbox), GTK_WIDGET(label));

	g_signal_connect(G_OBJECT(self->check_mode), "toggled",
			G_CALLBACK(check_mode_cb), self);

	/* Pair new Device Button */
	hbox = gtk_hbox_new(0,0);
	gtk_container_add (GTK_CONTAINER(container), GTK_WIDGET(hbox));

	bt_label = NULL;
	bt_label =	gtk_label_new("Pair new device");
	self->button_pair_dev = gtk_button_new();
	gtk_widget_set_size_request(self->button_pair_dev,-1,50);
	gtk_container_add(GTK_CONTAINER(self->button_pair_dev),GTK_WIDGET(bt_label));
	gtk_box_pack_start(GTK_BOX(hbox), self->button_pair_dev, TRUE, FALSE, 0);

	g_signal_connect(G_OBJECT(self->button_pair_dev), "clicked",
			G_CALLBACK(pair_device_callback), self);
}

/**
 * Create trip page
 * @param self CarmandSettings
 * @param container Container to trip page
 */
static void create_trip_page(CarmandSettings *self, GtkWidget *container)
{

	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkSizeGroup *size_group;

	carmand_vdebug("%s\n", __func__);

	size_group = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);

	vbox = gtk_vbox_new(FALSE,0);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(vbox));

	self->combo_time_precision = create_combo_box(self, hbox, size_group,
						_("Precision:"), 0);

	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(gtk_label_new(_("seconds"))),
			FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(hbox), TRUE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	self->combo_trips_folder = create_storage_combo(self, hbox, size_group,
			_("Trips storage device:"), 0);

	gtk_box_pack_end(GTK_BOX(vbox), GTK_WIDGET(hbox), TRUE, FALSE, 2);
	hbox = gtk_hbox_new(0,0);
	gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(hbox), FALSE, FALSE, 0);
}

/**
 * Create map page
 * @param self CarmandSettings
 * @param container Container to map page
 */
static void create_map_page(CarmandSettings *self, GtkWidget *container)
{

	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkSizeGroup *size_group;

	carmand_vdebug("%s\n", __func__);

	size_group = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);

	vbox = gtk_vbox_new(FALSE,0);
	hbox = gtk_hbox_new(0,0);
	gtk_container_add (GTK_CONTAINER(container), GTK_WIDGET(vbox));

	self->combo_maps_folder = create_storage_combo(self, hbox, size_group,
			_("Maps storage device:"), 0);

	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(hbox), TRUE, FALSE, 1);
	hbox = gtk_hbox_new(0,0);

	self->combo_tracks_folder = create_storage_combo(self, hbox, size_group,
			_("Tracks storage device:"), 0);

	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(hbox), TRUE, FALSE, 1);

	hbox = NULL;
	hbox = gtk_hbox_new(0,0);
	gtk_container_add (GTK_CONTAINER(container), GTK_WIDGET(hbox));
	self->combo_repo = create_map_service_combo(self, hbox, size_group,
						_("Extra map services:"), 1);

	self->add_repo_button = gtk_button_new_with_label(_("Add"));
	gtk_box_pack_start(GTK_BOX(hbox),
			GTK_WIDGET(self->add_repo_button),
			FALSE, FALSE, 2);

	self->del_repo_button = gtk_button_new_with_label(_("Delete"));
	gtk_box_pack_start(GTK_BOX(hbox),
			GTK_WIDGET(self->del_repo_button),
			FALSE, FALSE, 2);

	/* Signals */
	g_signal_connect(G_OBJECT(self->add_repo_button),
			"clicked", G_CALLBACK(add_repo_click_cb),
			self);

	g_signal_connect(G_OBJECT(self->del_repo_button),
			"clicked", G_CALLBACK(del_repo_click_cb),
			self);

	g_signal_connect(G_OBJECT(self->combo_repo), "changed",
			G_CALLBACK(combo_repo_changed),
			self->del_repo_button);
}

/**
 * Create settings' widgets content
 * @param self CarmandSettings
 */
static void create_widgets(CarmandSettings *self)
{
	GtkNotebook *notebook_tabs;
	GtkWidget *vbox_connections;
	GtkWidget *vbox_tripsettings;
	GtkWidget *vbox_mapsettings;

	carmand_vdebug("%s\n", __func__);

	/* Create tabs for Settings and Sensors */
	notebook_tabs = GTK_NOTEBOOK(gtk_notebook_new());

	vbox_connections = gtk_vbox_new(0,0);
	vbox_tripsettings = gtk_vbox_new(0,0);
	vbox_mapsettings = gtk_vbox_new(0,0);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook_tabs),
			vbox_connections, gtk_label_new(_("Connections")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook_tabs),
			vbox_tripsettings, gtk_label_new(_("Trip")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook_tabs),
			vbox_mapsettings, gtk_label_new(_("Maps")));

	/* Create Devices Page */
	create_connections_page(self, vbox_connections);

	/* Create Settings Page */
	create_trip_page(self, vbox_tripsettings);

	/* Create Maps Page */
	create_map_page(self, vbox_mapsettings);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(self)->vbox),
			GTK_WIDGET(notebook_tabs));
}

/**
 * Struct to dbus callbacks implemented in settings.
 */
static struct carmand_dbus_cbs dbus_callbacks = {
	.gps_statuschanged = signal_gps_status_cb,
	.obd_statuschanged = signal_obd_status_cb,
	.bonding_created = bonding_created_cb,
	.bonding_removed = bonding_removed_cb,
};

/* GLib API Functions */

static void destroy(GtkObject *obj)
{
	carmand_vdebug("%s\n", __func__);

	GTK_OBJECT_CLASS(carmand_settings_parent_class)->destroy(obj);
}

static void finalize(GObject *obj)
{
	CarmandSettings *self = CARMAND_SETTINGS(obj);

	carmand_vdebug("%s\n", __func__);

	if (self->dbus) {
		dbus_bus_release_name(self->dbus->session_bus,
				"org.indt.carmandsettings", NULL);
		g_object_unref(self->dbus);
	}

	clear_models(self->list_obd_gps_devs);
	g_slist_free(self->list_obd_gps_devs);
	self->list_obd_gps_devs = NULL;

	clear_models(self->list_map_repo);
	g_slist_free(self->list_map_repo);
	self->list_map_repo = NULL;

	clear_models(self->list_storage_devs);
	g_slist_free(self->list_storage_devs);
	self->list_storage_devs = NULL;

	carmand_settings_self = NULL;

	G_OBJECT_CLASS(carmand_settings_parent_class)->finalize(obj);
}

static void carmand_settings_class_init(CarmandSettingsClass *klass)
{
	GObjectClass *obj_class = G_OBJECT_CLASS(klass);
	GtkObjectClass *gtkobj_class = GTK_OBJECT_CLASS(klass);

	carmand_vdebug("%s\n", __func__);

	obj_class->finalize = finalize;
	gtkobj_class->destroy = destroy;
}

static void carmand_settings_init(CarmandSettings *self)
{
	carmand_vdebug("%s\n", __func__);

	self->dbus = NULL;
	self->list_obd_gps_devs = NULL;
	self->list_map_repo = NULL;
	self->list_storage_devs = NULL;
}

CarmandSettings *carmand_settings_new(GtkWindow *parent)
{
	CarmandSettings *self;
	GtkWidget *button;

	carmand_vdebug("%s\n", __func__);

	if (!Py_IsInitialized()) {
		Py_OptimizeFlag = 1;
		Py_Initialize();
		initcheck_repository();
	}

	self = g_object_new(CARMAND_TYPE_SETTINGS, NULL);

	if (!self) {
		carmand_error("Failed to create Carmand Settings.\n");
		return NULL;
	}
	/* FIXME: Used on signal callbacks, should use marshall stuff */
	carmand_settings_self = self;

	/* Connections */

	self->dbus = carmand_dbus_new(self, &dbus_callbacks);
	if (!self->dbus) {
		gtk_widget_destroy(GTK_WIDGET(self));
		return NULL;
	}

	if (dbus_bus_request_name(self->dbus->session_bus,
				"org.indt.carmandsettings",
			DBUS_NAME_FLAG_DO_NOT_QUEUE, NULL) !=
			DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
		carmand_error("Carmand settings already running!\n");
		g_object_unref(self);
		return NULL;
	}

	/* Widgets */

	gtk_window_set_title(GTK_WINDOW(self), _("Carman Settings"));

	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(self), parent);

	gtk_window_set_modal(GTK_WINDOW(self), TRUE);
	gtk_window_set_destroy_with_parent(GTK_WINDOW(self), TRUE);

	create_widgets(self);

	/* Loading configs */
	if (!load_config(self)) {
		show_message(GTK_WINDOW(self), "Error loading carmand"
				" configuration",
				GTK_STOCK_DIALOG_ERROR);
		gtk_widget_destroy(GTK_WIDGET(self));
		return NULL;
	}

	/* Buttons */

	button = gtk_dialog_add_button(GTK_DIALOG(self), "OK", GTK_RESPONSE_OK);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(save_config_callback), self);

	button = gtk_dialog_add_button(GTK_DIALOG(self), _("Default"), 0);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(load_default_values_callback), self);

	button = gtk_dialog_add_button(GTK_DIALOG(self), "Cancel",
			GTK_RESPONSE_CLOSE);
	g_signal_connect_swapped(GTK_BUTTON(button), "clicked",
			G_CALLBACK(gtk_widget_destroy), GTK_WIDGET(self));

	gtk_window_set_default_size(GTK_WINDOW(self), 600, 225);

	gtk_widget_show_all(GTK_WIDGET(self));

	return self;
}
