/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 * 
 *  carmand-panel is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand-panel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <conbtdialogs-dbus.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>

#include "marshal.h"
#include "carmand-logmsg.h"
#include "carmand-dbus.h"

/* FIXME: review dgettext code */
#define APP_NAME            "carmand-panel"
#define _(String) dgettext (APP_NAME, String)

G_DEFINE_TYPE(CarmandDbus, carmand_dbus, G_TYPE_OBJECT)

/* Carmand-Panel DBus Functions */

void cpanel_dbus_method_call(CarmandDbus *dbus,
		const gchar *method, const gchar *interface)
{
	DBusMessage  *msg;

	carmand_vdebug("[%s:%d]\n", __func__, __LINE__);

	msg = dbus_message_new_method_call(CARMAND_DBUS_BUS_NAME,
				CARMAND_DBUS_OBJECT_PATH,
				interface, method);

	dbus_connection_send(dbus->private_bus, msg, NULL);

	dbus_connection_flush(dbus->private_bus);
	dbus_message_unref(msg);
}

gint cpanel_dbus_int_method_call(CarmandDbus *dbus,
		const gchar *method, const gchar *iface)
{
	DBusMessage *msg, *reply;
	DBusError derr;
	gint retval = 0;

	carmand_vdebug("%s\n", __func__);

	msg = dbus_message_new_method_call(CARMAND_DBUS_BUS_NAME,
				CARMAND_DBUS_OBJECT_PATH,
				iface, method);

	dbus_error_init(&derr);
	reply = dbus_connection_send_with_reply_and_block(dbus->private_bus,
			msg, -1, &derr);

	dbus_message_unref(msg);
	if (reply == NULL) {
		carmand_error("%s(): %s\n", method, derr.message);
		dbus_error_free(&derr);
		return FALSE;
	}

	dbus_message_get_args(reply, NULL, DBUS_TYPE_INT32, &retval,
			DBUS_TYPE_INVALID);

	dbus_message_unref(reply);

	return retval;
}

gchar *cpanel_dbus_str_method_call(CarmandDbus *dbus, const gchar *method,
					const gchar *iface, DBusError *derr)
{
	DBusMessage *msg, *reply;
	DBusPendingCall *pending;
	gchar *msg_ret = NULL;
	const gchar *ret;

	carmand_vdebug("%s\n", __func__);

	msg = dbus_message_new_method_call(CARMAND_DBUS_BUS_NAME,
				CARMAND_DBUS_OBJECT_PATH,
				iface, method);

	dbus_error_init(derr);
	reply = dbus_connection_send_with_reply_and_block(dbus->private_bus,
			msg, -1, derr);

	dbus_message_unref(msg);

	if (dbus_error_is_set(derr)) {
		return NULL;
	} else {
		dbus_message_get_args(reply, NULL, DBUS_TYPE_STRING, &ret,
				DBUS_TYPE_INVALID);
		msg_ret = g_strdup(ret);
		dbus_message_unref(reply);
	}

	return msg_ret;
}
/*
 * DBus method call with gchar as argument
 * @param dbus CarmandDbus object
 * @param method DBus method call
 * @param int method call argument
 * @returns TRUE if success or FALSE if fails
 */
gboolean method_call_string_arg(CarmandDbus *dbus,
			const gchar *method, const gchar *arg)
{
	DBusMessage *msg, *reply;
	DBusError derr;

	carmand_vdebug("%s\n", __func__);

	msg = dbus_message_new_method_call(CARMAND_DBUS_BUS_NAME,
			CARMAND_DBUS_OBJECT_PATH,
			CARMAND_DBUS_IFACE_CONFIGURATION,
			method);

	dbus_message_append_args(msg, DBUS_TYPE_STRING, &arg,
			DBUS_TYPE_INVALID);

	dbus_error_init(&derr);
	reply = dbus_connection_send_with_reply_and_block(dbus->private_bus,
			msg, -1, &derr);

	dbus_message_unref(msg);
	if (reply == NULL) {
		carmand_error("%s(): %s\n", method, derr.message);
		dbus_error_free(&derr);
		return FALSE;
	}

	dbus_message_unref(reply);

	return TRUE;
}

/*
 * DBus method call with int as argument
 * @param dbus CarmandDbus object
 * @param method DBus method call
 * @param int method call argument
 * @returns TRUE if success or FALSE if fails
 */
gboolean method_call_int_arg(CarmandDbus *dbus, const gchar *method, gint arg)
{
	DBusMessage *msg, *reply;
	DBusError derr;

	carmand_vdebug("%s\n", __func__);

	msg = dbus_message_new_method_call(CARMAND_DBUS_BUS_NAME,
			CARMAND_DBUS_OBJECT_PATH,
			CARMAND_DBUS_IFACE_CONFIGURATION,
			method);

	dbus_message_append_args(msg,
			DBUS_TYPE_INT32, &arg,
			DBUS_TYPE_INVALID);

	dbus_error_init(&derr);
	reply = dbus_connection_send_with_reply_and_block(dbus->private_bus,
			msg, -1, &derr);

	dbus_message_unref(msg);
	if (reply == NULL) {
		carmand_error("%s(): %s\n", method, derr.message);
		dbus_error_free(&derr);
		return FALSE;
	}

	dbus_message_unref(reply);

	return TRUE;
}

static int create_dbus_signals_callbacks(CarmandDbus *self)
{

	struct carmand_dbus_cbs *callbacks = self->callbacks;

	carmand_vdebug("%s\n", __func__);

	/* GPS */
	if (callbacks->gps_statuschanged) {
		dbus_g_proxy_add_signal(self->proxy_gps,
				SIGNAL_GPS_STATUS, G_TYPE_STRING,
				G_TYPE_INVALID);
		dbus_g_proxy_connect_signal(self->proxy_gps, SIGNAL_GPS_STATUS,
				G_CALLBACK(callbacks->gps_statuschanged),
				self->parent_data, NULL);
	}

	/* OBD */
	if (callbacks->obd_statuschanged) {
		dbus_g_proxy_add_signal(self->proxy_obd, SIGNAL_OBD_STATUS,
				G_TYPE_STRING, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal(self->proxy_obd, SIGNAL_OBD_STATUS,
				G_CALLBACK(callbacks->obd_statuschanged),
				self->parent_data, NULL);
	}

	/*  BlueZ */
	dbus_g_object_register_marshaller(marshal_VOID__STRING, G_TYPE_NONE,
			G_TYPE_STRING, G_TYPE_INVALID);

	if (callbacks->bonding_created) {
		dbus_g_proxy_add_signal(self->proxy_bluez, "BondingCreated",
				G_TYPE_STRING, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal(self->proxy_bluez, "BondingCreated",
				G_CALLBACK(callbacks->bonding_created),
				self->parent_data, NULL);
	}

	if (callbacks->bonding_removed) {
		dbus_g_proxy_add_signal(self->proxy_bluez, "BondingRemoved",
				G_TYPE_STRING, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal(self->proxy_bluez, "BondingRemoved",
				G_CALLBACK(callbacks->bonding_removed),
				self->parent_data, NULL);
	}

	return 0;
}

/* GLib API Functions */

static void carmand_dbus_finalize(GObject *obj)
{
	CarmandDbus *self = CARMAND_DBUS(obj);

	carmand_vdebug("%s\n", __func__);

	if (self->proxy_bluetooth)
		g_object_unref(self->proxy_bluetooth);

	if (self->proxy_gps)
		g_object_unref(self->proxy_gps);

	if (self->proxy_obd)
		g_object_unref(self->proxy_obd);

	if (self->proxy_bluez)
		g_object_unref(self->proxy_bluez);

	if (self->system_bus)
		dbus_g_connection_unref(self->system_bus);

	if (self->session_bus)
		dbus_connection_unref(self->session_bus);

	if (self->g_private_bus)
		dbus_g_connection_unref(self->g_private_bus);

	G_OBJECT_CLASS(carmand_dbus_parent_class)->finalize(obj);
}

static void carmand_dbus_class_init(CarmandDbusClass *klass)
{
	GObjectClass *obj_class = G_OBJECT_CLASS (klass);

	carmand_vdebug("%s\n", __func__);

	obj_class->finalize = carmand_dbus_finalize;
}

static void carmand_dbus_init(CarmandDbus *self)
{
	carmand_vdebug("%s\n", __func__);

	self->proxy_bluetooth = NULL;

	self->system_bus = NULL;

	self->private_bus = NULL;
	self->g_private_bus = NULL;

	self->proxy_gps = NULL;
	self->proxy_obd = NULL;

	self->proxy_bluez = NULL;
}

CarmandDbus *carmand_dbus_new(gpointer parent_data, struct carmand_dbus_cbs *callbacks)
{
	CarmandDbus *self;
	GError *gerr = NULL;
	DBusError derr;

	carmand_vdebug("%s\n", __func__);

	self = g_object_new(CARMAND_TYPE_DBUS, NULL);
	if (!self) {
		carmand_error("g_object_new(): " \
				"Can't create carmand D-Bus object!\n");
		return NULL;
	}

	self->system_bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &gerr);
	if (gerr != NULL) {
		carmand_error("system bus connection failed: %s\n", gerr->message);
		g_error_free(gerr);
		g_object_unref(self);
		return NULL;
	}

	dbus_error_init(&derr);
	self->session_bus = dbus_bus_get(DBUS_BUS_SESSION, &derr);
	if (dbus_error_is_set(&derr)) {
		carmand_error("session bus connection failed: %s\n", derr.message);
		return NULL;
	}

	self->g_private_bus = dbus_g_connection_open(
			"unix:path=/var/run/carmand-dbus-path", NULL);
	if (self->g_private_bus == NULL) {
		carmand_error("carmand bus not active!\n");
		g_object_unref(self);
		return NULL;
	}

	self->private_bus = dbus_g_connection_get_connection(
			self->g_private_bus);
	dbus_bus_register(self->private_bus, NULL);

	self->proxy_gps = dbus_g_proxy_new_for_name(self->g_private_bus,
			CARMAND_DBUS_BUS_NAME,
			CARMAND_DBUS_OBJECT_PATH,
			CARMAND_DBUS_IFACE_GPS);

	self->proxy_obd = dbus_g_proxy_new_for_name(self->g_private_bus,
			CARMAND_DBUS_BUS_NAME,
			CARMAND_DBUS_OBJECT_PATH,
			CARMAND_DBUS_IFACE_OBD);

	/* FIXME: Verify error condition and propagate to UI */
	self->proxy_bluez = dbus_g_proxy_new_for_name(self->system_bus,
					"org.bluez", "/org/bluez/hci0",
					"org.bluez.Adapter");

	/* FIXME: Verify error condition and propagate to UI */
	self->proxy_bluetooth = dbus_g_proxy_new_for_name(self->system_bus,
				CONBTDIALOGS_DBUS_SERVICE,
				CONBTDIALOGS_DBUS_PATH,
				CONBTDIALOGS_DBUS_INTERFACE);

	self->parent_data = parent_data;

	if (callbacks) {
		self->callbacks = callbacks;
		create_dbus_signals_callbacks(self);
	} else
		self->callbacks = NULL;

	return self;
}
