/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of infosharingd.
 *
 *  infoshargind is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  infosharingd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with infosharingd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * This code is based in BlueZ source code
 * Copyright (C) 2006-2007  Nokia Corporation
 * Copyright (C) 2004-2008  Marcel Holtmann <marcel@holtmann.org>
 * Copyright (C) 2005-2007  Johan Hedberg <johan.hedberg@nokia.com>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <glib.h>
#include <dbus/dbus.h>
#include <gdbus.h>

static void dbus_message_iter_append_variant(DBusMessageIter *iter,
						int type, void *val)
{
	DBusMessageIter value;
	DBusMessageIter array;
	char *sig;

	switch (type) {
	case DBUS_TYPE_STRING:
		sig = DBUS_TYPE_STRING_AS_STRING;
		break;
	case DBUS_TYPE_BYTE:
		sig = DBUS_TYPE_BYTE_AS_STRING;
		break;
	case DBUS_TYPE_INT16:
		sig = DBUS_TYPE_INT16_AS_STRING;
		break;
	case DBUS_TYPE_UINT16:
		sig = DBUS_TYPE_UINT16_AS_STRING;
		break;
	case DBUS_TYPE_INT32:
		sig = DBUS_TYPE_INT32_AS_STRING;
		break;
	case DBUS_TYPE_UINT32:
		sig = DBUS_TYPE_UINT32_AS_STRING;
		break;
	case DBUS_TYPE_BOOLEAN:
		sig = DBUS_TYPE_BOOLEAN_AS_STRING;
		break;
	case DBUS_TYPE_ARRAY:
		sig = DBUS_TYPE_ARRAY_AS_STRING DBUS_TYPE_STRING_AS_STRING;
		break;
	case DBUS_TYPE_OBJECT_PATH:
		sig = DBUS_TYPE_OBJECT_PATH_AS_STRING;
		break;
	default:
		printf("Could not append variant with type %d", type);
		return;
	}

	dbus_message_iter_open_container(iter, DBUS_TYPE_VARIANT, sig, &value);

	if (type == DBUS_TYPE_ARRAY) {
		int i;
		const char ***str_array = val;

		dbus_message_iter_open_container(&value, DBUS_TYPE_ARRAY,
			DBUS_TYPE_STRING_AS_STRING, &array);

		for (i = 0; (*str_array)[i]; i++)
			dbus_message_iter_append_basic(&array, DBUS_TYPE_STRING,
							&((*str_array)[i]));

		dbus_message_iter_close_container(&value, &array);
	} else
		dbus_message_iter_append_basic(&value, type, val);

	dbus_message_iter_close_container(iter, &value);
}

void dbus_message_iter_append_dict_entry(DBusMessageIter *dict,
					const char *key, int type, void *val)
{
	DBusMessageIter entry;

	dbus_message_iter_open_container(dict, DBUS_TYPE_DICT_ENTRY,
							NULL, &entry);

	dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &key);

	dbus_message_iter_append_variant(&entry, type, val);

	dbus_message_iter_close_container(dict, &entry);
}

/* This function get an dictionary from DBusMessage and iter all
 * elements inside a GHashTable.
 * Dictionary keys are string and values should be int, bool or string.
 */
GHashTable *
dbus_iter_hash_table(DBusMessageIter iter, DBusError *error)
{
	GHashTable *hash = g_hash_table_new(g_str_hash, g_str_equal);
	DBusMessageIter sub;

	dbus_message_iter_recurse(&iter, &sub);

	if (dbus_message_iter_get_arg_type(&sub) != DBUS_TYPE_DICT_ENTRY)
		goto error;

	do {
		DBusMessageIter subiter, subval;
		char *opt, *str_val;
		gint int_val;
		gboolean bool_val;

		dbus_message_iter_recurse(&sub, &subiter);

		dbus_message_iter_get_basic(&subiter, &opt);
		dbus_message_iter_next(&subiter);
		dbus_message_iter_recurse(&subiter, &subval);

		if (dbus_message_iter_get_arg_type(&subval) == DBUS_TYPE_STRING) {
			dbus_message_iter_get_basic(&subval, &str_val);
			g_hash_table_insert(hash, opt, str_val);
		} else if (dbus_message_iter_get_arg_type(&subval) == DBUS_TYPE_BOOLEAN) {
			dbus_message_iter_get_basic(&subval, &bool_val);
			g_hash_table_insert(hash, opt, bool_val);
		} else if (dbus_message_iter_get_arg_type(&subval) == DBUS_TYPE_INT32) {
			dbus_message_iter_get_basic(&subval, &int_val);
			g_hash_table_insert(hash, opt, int_val);
		}

	} while (dbus_message_iter_next(&sub));

	return hash;
error:
	g_hash_table_destroy(hash);
	return NULL;
}
