/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of infosharingd.
 *
 *  infosharingd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  infosharingd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with infosharingd. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "purple.h"
#include <libpurple/dbus-server.h>

#include <glib.h>

#include <errno.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>

#include "server.h"
#include "dbus.h"
#include "log.h"

#define PURPLE_GLIB_READ_COND  (G_IO_IN | G_IO_HUP | G_IO_ERR)
#define PURPLE_GLIB_WRITE_COND (G_IO_OUT | G_IO_HUP | G_IO_ERR | G_IO_NVAL)

/* UI info (name, version, website, dev website) hash table */
static GHashTable *ui_info = NULL;

static struct infosharingd_event event;

/* Hash table of PurpleConversations */
GHashTable *convs = NULL;

/* DBus connection */
static DBusConnection *connection = NULL;

/* Default account */
InfoShareAccount *carman_account = NULL;

/* Pending buddy authorization requests */
GHashTable *pending_requests = NULL;

/* Main loop */
GMainLoop *event_loop = NULL;

typedef struct _PurpleGLibIOClosure {
	PurpleInputFunction function;
	guint result;
	gpointer data;
} PurpleGLibIOClosure;

static void purple_glib_io_destroy(gpointer data)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);
	g_free(data);
}

static gboolean purple_glib_io_invoke(GIOChannel *source,
	GIOCondition condition,
	gpointer data)
{
	PurpleGLibIOClosure *closure = data;
	PurpleInputCondition purple_cond = 0;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (condition & PURPLE_GLIB_READ_COND)
		purple_cond |= PURPLE_INPUT_READ;
	if (condition & PURPLE_GLIB_WRITE_COND)
		purple_cond |= PURPLE_INPUT_WRITE;

	closure->function(closure->data,
		g_io_channel_unix_get_fd(source), purple_cond);

	return TRUE;
}

static guint glib_input_add(gint fd,
	PurpleInputCondition condition,
	PurpleInputFunction function,
	gpointer data)
{
	PurpleGLibIOClosure *closure = g_new0(PurpleGLibIOClosure, 1);
	GIOChannel *channel;
	GIOCondition cond = 0;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	closure->function = function;
	closure->data = data;

	if (condition & PURPLE_INPUT_READ)
		cond |= PURPLE_GLIB_READ_COND;
	if (condition & PURPLE_INPUT_WRITE)
		cond |= PURPLE_GLIB_WRITE_COND;

	channel = g_io_channel_unix_new(fd);
	closure->result = g_io_add_watch_full(channel,
		G_PRIORITY_DEFAULT, cond,
		purple_glib_io_invoke, closure, purple_glib_io_destroy);

	g_io_channel_unref(channel);
	return closure->result;
}

static PurpleEventLoopUiOps eventloop_ui_ops =
{
	g_timeout_add,
	g_source_remove,
	glib_input_add,
	g_source_remove,
	NULL,
#if GLIB_CHECK_VERSION(2,14,0)
	g_timeout_add_seconds,
#else
	NULL,
#endif
	/* padding */
	NULL,
	NULL,
	NULL
};

/* Someone we don't have on our list added us; prompt to add them. */
static void request_add(PurpleAccount *account, const char *remote_user,
	const char *id, const char *alias, const char *message)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	event.request_add(g_strdup(remote_user),
		g_strdup(alias), g_strdup(message));
}

/* Prompt for authorization when someone adds this account to their buddy
 * list.
 * @return a UI-specific handle, as passed to #close_account_request.
 */
static void *request_authorize(PurpleAccount *account, const char *remote_user,
	const char *id, const char *alias, const char *message, gboolean on_list,
	PurpleAccountRequestAuthorizationCb authorize_cb,
	PurpleAccountRequestAuthorizationCb deny_cb, void *user_data)
{
	PendingRequestAuthorization *request = NULL;
	static int handle;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == pending_requests)
		pending_requests = g_hash_table_new(g_str_hash, g_str_equal);

	request = g_hash_table_lookup(pending_requests, remote_user);
	if (NULL != request)
		/* buddy already waiting for authorization request */
		return &handle;

	request = malloc(sizeof(PendingRequestAuthorization));
	if (!request)
		/* Out of memory :( */
		return &handle;

	request->authorize_cb = authorize_cb;
	request->deny_cb = deny_cb;
	request->user_data = user_data;

	g_hash_table_insert(pending_requests, g_strdup(remote_user), request);

	event.request_authorize(g_strdup(remote_user),
		g_strdup(alias), g_strdup(message), on_list);

	return &handle;
}

static void
infosharingd_request_close(void *uihandle)
{
	purple_request_close(PURPLE_REQUEST_ACTION, uihandle);
}

static PurpleAccountUiOps account_ui_ops =
{
	NULL,              /* notify_added          */
	NULL,              /* status_changed        */
	request_add,       /* request_add           */
	request_authorize, /* request_authorize     */
	infosharingd_request_close, /* close_account_request */
	NULL,
	NULL,
	NULL,
	NULL
};

/* Requests from the user information needed to add a buddy to the buddy
 * list.
 * NOTE: This callback is called when user needs to add information about a
 * given buddy. On Carman design, there is no need for the user to supply that
 * information, so the buddy can be added using default parameters.
 */
static void request_add_buddy(PurpleAccount *account, const char *buddy_name,
	const char *buddy_group, const char *buddy_alias)
{
	PurpleBuddy *buddy = NULL;
	PurpleGroup *group = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == carman_account) || (NULL == carman_account->purple_acc))
		return;

	if (NULL == buddy_name)
		return;

	if (NULL == buddy_group)
		buddy_group = g_strdup("Carman");

	buddy = purple_find_buddy(account, buddy_name);
	if (NULL != buddy) {
		/* Hack! Remove buddy from account and blist so we can re-add
		 * it again :D */
		purple_account_remove_buddy(carman_account->purple_acc, buddy,
			purple_buddy_get_group(buddy));
		purple_blist_remove_buddy(buddy);
	}

	group = purple_find_group(buddy_group);
	if (NULL == group) {
		group = purple_group_new(buddy_group);
		purple_blist_add_group(group, NULL);
	}

	buddy = purple_buddy_new(account, buddy_name, buddy_alias);
	if (NULL == buddy)
		return;

	purple_blist_add_buddy(buddy, NULL, group, NULL);
	purple_account_add_buddy(account, buddy);
}

static PurpleBlistUiOps blist_ui_ops =
{
	NULL,              /* new_list          */
	NULL,              /* new_node          */
	NULL,              /* show              */
	NULL,              /* update            */
	NULL,              /* remove            */
	NULL,              /* destroy           */
	NULL,              /* set_visible       */
	request_add_buddy, /* request_add_buddy */
	NULL,              /* request_add_chat  */
	NULL,              /* request_add_group */
	NULL,
	NULL,
	NULL,
	NULL
};

static PurpleConversationUiOps conversation_ui_ops =
{
	NULL,     /* create_conversation  */
	NULL,     /* destroy_conversation */
	NULL,     /* write_chat           */
	NULL,     /* write_im             */
	NULL,     /* write_conv           */
	NULL,     /* chat_add_users       */
	NULL,     /* chat_rename_user     */
	NULL,     /* chat_remove_users    */
	NULL,     /* chat_update_user     */
	NULL,     /* present              */
	NULL,     /* has_focus            */
	NULL,     /* custom_smiley_add    */
	NULL,     /* custom_smiley_write  */
	NULL,     /* custom_smiley_close  */
	NULL,     /* send_confirm         */
	NULL,
	NULL,
	NULL,
	NULL
};

static void call_request_action(PurpleRequestActionCb callback,
			void *user_data, int default_action, void *handle)
{
	/* Call default action callback */
	if (callback)
		callback(user_data, default_action);

	/* Close request */
	purple_request_close(PURPLE_REQUEST_ACTION, handle);
}

/* Prompts the user for an action.
 * NOTE: This is a hack because this function doesn't need to have UI
 * interaction (e.g. Google certificates can be automatically accepted
 * from within infosharingd.
 */
static void *request_action(const char *title, const char *primary,
	const char *secondary, int default_action, PurpleAccount *account,
	const char *who, PurpleConversation *conv, void *user_data,
	size_t action_count, va_list actions)
{
	static int handle;
	const char *text = va_arg(actions, const char *);
	PurpleRequestActionCb callback = va_arg(actions, PurpleRequestActionCb);

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	call_request_action(callback, user_data, default_action, &handle);
	return &handle;
}

static PurpleRequestUiOps request_ui_ops =
{
	NULL,           /* request_input  */
	NULL,           /* request_choice */
	request_action, /* request_action */
	NULL,           /* request_fields */
	NULL,           /* request_file   */
	NULL,           /* close_request  */
	NULL,           /* request_folder */
	NULL,
	NULL,
	NULL,
	NULL
};

static void ui_init(void)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	/* Initialize UI components for all libpurple modules */
	purple_accounts_set_ui_ops(&account_ui_ops);
	purple_blist_set_ui_ops(&blist_ui_ops);
	purple_conversations_set_ui_ops(&conversation_ui_ops);
	purple_request_set_ui_ops(&request_ui_ops);
}

static void ui_quit(void)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL != ui_info)
		g_hash_table_destroy(ui_info);
}

static GHashTable *ui_get_info(void)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == ui_info)
		/* Create a new ui_info hash table and insert UI information
		 * inside it */
		ui_info = g_hash_table_new(g_str_hash, g_str_equal);

		g_hash_table_insert(ui_info, "name", INFOSHARING_NAME);
		g_hash_table_insert(ui_info, "version", INFOSHARING_VERSION);
		g_hash_table_insert(ui_info, "website", INFOSHARING_WEBSITE);
		g_hash_table_insert(ui_info, "dev_website",
			INFOSHARING_DEV_WEBSITE);

	return ui_info;
}

static PurpleCoreUiOps core_ui_ops =
{
	NULL,	/* prefs_init */
	NULL,	/* debug_init */
	ui_init,
	ui_quit,
	ui_get_info,
	NULL,
	NULL,
	NULL
};

static void init_libpurple(gboolean debug_enabled)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	/* Set a custom user directory */
	purple_util_set_user_dir(INFOSHARING_DIR);

	/* Enable/disable debug to stdout */
	purple_debug_set_enabled(debug_enabled);

	purple_core_set_ui_ops(&core_ui_ops);
	purple_eventloop_set_ui_ops(&eventloop_ui_ops);

	if (!purple_core_init(INFOSHARING_NAME)) {
		fprintf(stderr, "Libpurple initialization failed. Aborting...\n");
		abort();
	}

	/* Initialize blist */
	purple_set_blist(purple_blist_new());
	purple_blist_load();

	/* Disable libpurple's D-Bus server to avoid conflicts */
	purple_dbus_uninit();
}

static void buddy_signed_on_signal(PurpleBuddy *buddy)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	event.buddy_signed_on(g_strdup(purple_buddy_get_name(buddy)),
		g_strdup(purple_buddy_get_alias_only(buddy)));
}

static void buddy_signed_off_signal(PurpleBuddy *buddy)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	event.buddy_signed_off(g_strdup(purple_buddy_get_name(buddy)),
		g_strdup(purple_buddy_get_alias_only(buddy)));
}

static void signed_on_signal(PurpleConnection *gc, gpointer null)
{
	PurpleAccount *account = purple_connection_get_account(gc);

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	event.signed_on(g_strdup(purple_account_get_username(account)),
		g_strdup(purple_account_get_protocol_id(account)));
}

static void signed_off_signal(PurpleConnection *gc, gpointer null)
{
	PurpleAccount *account = purple_connection_get_account(gc);

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	event.signed_off(g_strdup(purple_account_get_username(account)),
		g_strdup(purple_account_get_protocol_id(account)));
}

static void connection_error_signal(PurpleConnection *gc,
	PurpleConnectionError err, const gchar *desc)
{
	PurpleAccount *account = purple_connection_get_account(gc);
	char *short_desc = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	switch (err) {
		case 0: short_desc = g_strdup("Network error\n");
			break;
		case 1: short_desc = g_strdup("Invalid username\n");
			break;
		case 2: short_desc = g_strdup("Authentication failed\n");
			break;
		case 3: short_desc = g_strdup("Authentication impossible\n");
			break;
		case 4: short_desc = g_strdup("No SSL support\n");
			break;
		case 5: short_desc = g_strdup("Encryption error\n");
			break;
		case 6: short_desc = g_strdup("Name in use\n");
			break;
		case 7: short_desc = g_strdup("Invalid settings\n");
			break;
		case 8: short_desc = g_strdup("SSL certificate not provided\n");
			break;
		case 9: short_desc = g_strdup("SSL certificate untrusted\n");
			break;
		case 10: short_desc = g_strdup("SSL certificate expired\n");
			break;
		case 11: short_desc = g_strdup("SSL certificate not activated\n");
			break;
		case 12: short_desc = g_strdup("SSL certificate hostname mismatch\n");
			break;
		case 13: short_desc = g_strdup("SSL certificate fingerprint mismatch\n");
			break;
		case 14: short_desc = g_strdup("SSL certificate self signed\n");
			break;
		case 15: short_desc = g_strdup("SSL certificate other error\n");
			break;
		case 16: short_desc = g_strdup("Other error\n");
			break;
		default:
			/* Oh no! Unknown error */
			break;
	}

	event.connection_error(g_strdup(purple_account_get_username(account)),
		g_strdup(purple_account_get_protocol_id(account)), short_desc,
		g_strdup(desc));
}

static gboolean receiving_im_msg_signal(PurpleAccount *account, char **sender,
	char **message, PurpleConversation *conv, PurpleMessageFlags *flags)\
{
	const char *alias;
	char *stripped = NULL;
	gboolean ret = FALSE;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	/* Get PurpleBuddy structure from account and buddy username */
	PurpleBuddy *buddy = purple_find_buddy(account, *sender);
	if (NULL == buddy)
		return ret;

	/* Get buddy alias, if available, or username otherwise */
	alias = buddy->server_alias? : (buddy->alias? : buddy->name);

	/* Strip HTML tags from the received message */
	stripped = purple_markup_strip_html(*message);

	ret = event.receiving_im_message(*sender, g_strdup(alias), stripped);

	/* This string must be g_freed() when finished with it */
	g_free(stripped);

	/* TRUE if the message should be canceled or FALSE otherwise */
	return ret;
}

static void init_signals(void)
{
	void *blist_handle = purple_blist_get_handle();
	void *conn_handle = purple_connections_get_handle();
	void *conv_handle = purple_conversations_get_handle();
	static int handle;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	/* A buddy signed on */
	purple_signal_connect(blist_handle,
		"buddy-signed-on", &handle,
		PURPLE_CALLBACK(buddy_signed_on_signal), NULL);

	/* A buddy signed off */
	purple_signal_connect(blist_handle,
		"buddy-signed-off", &handle,
		PURPLE_CALLBACK(buddy_signed_off_signal), NULL);

	/* An account has signed on */
	purple_signal_connect(conn_handle,
		"signed-on", &handle,
		PURPLE_CALLBACK(signed_on_signal), NULL);

	/* An account has signed off */
	purple_signal_connect(conn_handle,
		"signed-off", &handle,
		PURPLE_CALLBACK(signed_off_signal), NULL);

	/* An account got connection error */
	purple_signal_connect(conn_handle,
		"connection-error", &handle,
		PURPLE_CALLBACK(connection_error_signal), NULL);

	/* Received an IM message */
	purple_signal_connect(conv_handle,
		"receiving-im-msg", &handle,
		PURPLE_CALLBACK(receiving_im_msg_signal), NULL);
}

static int usage(char *app_name)
{
	printf("Usage: %s [--debug]\n", app_name);
	return -EINVAL;
}

void infosharingd_finalize(void *user_data)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	dbus_exit();

	g_main_loop_quit(event_loop);
}

static void sig_term(int sig)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	infosharingd_finalize(NULL);
}

int main(int argc, char *argv[])
{
	DBusError derr;
	GList *aux = NULL;
	struct sigaction sa;

	event_loop = g_main_loop_new(NULL, FALSE);

	if (argc > 2 || (argc == 2 && strcmp("--debug", argv[1])))
		return usage(argv[0]);

	carman_account = malloc(sizeof(InfoShareAccount));
	carman_account->ready = FALSE;
	carman_account->purple_acc = NULL;

	/* Ignore SIGCHLD signals from child processes created when a DNS
	 * resolution is needed */
	signal(SIGCHLD, SIG_IGN);

	sa.sa_handler = sig_term;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	/* Initialize D-Bus connection */
	dbus_error_init(&derr);

	connection = dbus_connection_open(CARMAND_BUS_PATH, &derr);
	if (dbus_error_is_set(&derr)) {
		printf("Can't open D-Bus connection: %s\n",
				derr.message);
		dbus_error_free(&derr);
		return -1;
	}

	dbus_connection_setup_with_g_main(connection, NULL);
	if (!dbus_init(connection)) {
		printf("Infosharingd D-Bus server is not running\n");
		return -1;
	}

	/* Populate infosharingd_event callbacks */
	dbus_init_events(&event);

	/* Initialize conversations hash table */
	convs = g_hash_table_new(g_str_hash, g_str_equal);

	/* Initialize libpurple core */
	if (argc == 2)
		init_libpurple(TRUE);
	else
		init_libpurple(FALSE);

	/* Get default account */
	aux = purple_accounts_get_all();
	if (0 != g_list_length(aux))
		/* Get first account from accounts list */
		carman_account->purple_acc = (PurpleAccount *) aux->data;

	/* Attach callbacks to libpurple signals */
	init_signals();

	g_main_loop_run(event_loop);

	return 0;
}
