#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import dbus

from common.carlog import ERROR
from models.dbusmodel import CarmandDbusModel, EDbusModel
from models.infosharingservice import InfoSharingService

DBUS_ADDRESS        = "unix:path=/var/run/carmand-dbus-path"
DBUS_BUS_NAME       = "org.indt.carmanplugin"
DBUS_OBJECT_PATH    = "/org/indt/carmand"
DBUS_IFACE_SERVICE  = "org.indt.carmanplugin.Service"

__PROTOCOLS__  = {'prpl-msn':    ('MSN',    'images/thumbnail-msn'),
                  'prpl-yahoo':  ('Yahoo',  'images/thumbnail-yahoo'),
                  'prpl-jabber': ('Jabber', 'images/thumbnail-jabber')}

__CARMAN_MSG__ = "Online from Carman"
__CARMAN_GROUP__ = "Carman"

class ISService(InfoSharingService):

    def __init__(self, parent):

        # Execute infosharing
        self.__dbusmodel = ISDBusModel()
        self.__dbusmodel.SetClientBusName(
                CarmandDbusModel().get_bus().get_unique_name())

        self.parent = parent

        # Connect callbacks
        self._connect_signals_callbacks(self.__dbusmodel.get_bus())

    def __set_account_active_status(self):
        try:
            self.__dbusmodel.SetAccountStatus('available', __CARMAN_MSG__)
            return True
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def _request_add_buddy(self, remote_user, alias, msg):
        self.parent._account_request_add_cb(remote_user, alias)

    def _request_authorize_buddy(self, remote_user, alias, msg, on_list_str):
        if on_list_str == 'False':
            on_list = False
        else:
            on_list = True

        self.parent._account_request_authorize_cb(remote_user, alias, on_list)

    def _connect_signals_callbacks(self, bus):
        """
        Connect signals to callbacks
        """
        bus.add_signal_receiver(self.parent._on_signed_on_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'SignedOn')

        bus.add_signal_receiver(self.parent._on_signed_off_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'SignedOff')

        bus.add_signal_receiver(self.parent._on_buddy_signed_on_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'BuddySignedOn')

        bus.add_signal_receiver(self.parent._on_buddy_signed_off_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'BuddySignedOff')

        bus.add_signal_receiver(self.parent._on_connection_error_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'ConnectionError')

        bus.add_signal_receiver(self.parent._received_msg_cb,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'ReceivingImMessage')

        bus.add_signal_receiver(self._request_authorize_buddy,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'RequestAuthorizeBuddy')

        bus.add_signal_receiver(self._request_add_buddy,
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
                dbus_interface = DBUS_IFACE_SERVICE,
                signal_name = 'RequestAddBuddy')

    def account_exists(self):
        try:
            return self.__dbusmodel.AccountExists()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def authorize_buddy(self, name, authorize):
        self.__dbusmodel.AuthorizeBuddy(name, authorize)

    def send_message(self, name, message):
        self.__dbusmodel.SendMsg(name, message)

    def add_account(self, acc_id):
        try:
            self.__dbusmodel.NewAccount(acc_id[0], acc_id[1])
            return True
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def account_connect(self):
        try:
            self.__dbusmodel.ConnectAccount()
            self.__set_account_active_status()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_disconnect(self):
        try:
            self.__dbusmodel.DisconnectAccount()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_enabled(self):
        try:
            return self.__dbusmodel.AccountIsEnabled()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def account_is_connected(self):
        try:
            return self.__dbusmodel.AccountIsConnected()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_connecting(self):
        try:
            return self.__dbusmodel.AccountIsConnecting()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_disconnected(self):
        try:
            return self.__dbusmodel.AccountIsDisconnected()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_account_password(self, password, remember):
        try:
            self.__dbusmodel.SetAccountPassword(password)
            self.__dbusmodel.SetAccountRemPwd(remember)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account_password(self):
        try:
            return self.__dbusmodel.GetAccountPassword()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account_info(self):
        password = ''
        try:
            password = self.__dbusmodel.GetAccountPassword()
        except Exception, e:
            password = ''

        try:
            info = {
                'username': self.__dbusmodel.GetAccountUsername(),
                'protocol_id': self.__dbusmodel.GetAccountProtocolID(),
                'password': password,
                'alias': self.__dbusmodel.GetAccountAlias(),
                'remember_password': self.__dbusmodel.GetAccountRemPwd()
            }

            prpl_opts = self.get_protocol_options(info['protocol_id'])
            if 'connect_server' in prpl_opts:
                prpl_opts['server'] = prpl_opts['connect_server']

            for item in prpl_opts:
                info[item] = prpl_opts[item]

            return info
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return {}

    def get_protocol_options(self, protocol_id):
        try:
            if self.account_exists():
                prpl_opts = self.__dbusmodel.GetAccountProtocolOptions()
            else:
                prpl_opts = self.__dbusmodel.GetProtocolOptions(protocol_id)
                # Hack for google talk accounts
                if protocol_id == 'prpl-jabber':
                    prpl_opts['connect_server'] = 'talk.google.com'
                    prpl_opts['server'] = prpl_opts['connect_server']

            return prpl_opts
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return {}

    def set_protocol_options(self, info):
        try:
            if 'port' in info:
                info['port'] = int(info['port'])
            self.__dbusmodel.SetAccountProtocolOptions(info)
            return True
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def get_accounts_active(self):
        try:
            if self.__dbusmodel.AccountExists():
                account = self.__dbusmodel.GetAccountUsername()
                prot_id = self.__dbusmodel.GetAccountProtocolID()
                return [(account, prot_id)]
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return []

    def get_buddies_online(self):
        try:
            buddy_list = []
            for buddy in set(self.__dbusmodel.GetCarmanBuddies()):
                if buddy != \
                        self.__dbusmodel.GetAccountUsername().split("/")[0]:
                    alias = self.get_buddy_alias(buddy)
                    if alias:
                        buddy_list.append((buddy, alias))
                    else:
                        buddy_list.append((buddy, ''))
                buddy_list.sort()

            return buddy_list
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return []

    def add_buddy(self, name, alias=None, group=None):
        try:
            if group == None:
                group = 'Carman'
            if alias == None:
                alias = ''
            self.__dbusmodel.AddBuddy(name, alias, group)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def remove_buddy(self, name):
        try:
            self.__dbusmodel.RemoveBuddy(name)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_buddy_alias(self, name):
        """
        Get buddy alias

        @param name   Buddy name
        """
        try:
            alias = self.__dbusmodel.GetBuddyAlias(name)
            if not alias:
                alias = ''
            return alias
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_buddy_alias(self, name, alias):
        try:
            self.__dbusmodel.SetBuddyAlias(name, alias)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account(self):
        try:
            username = self.__dbusmodel.GetAccountUsername()
            prot_id = self.__dbusmodel.GetAccountProtocolID()
            return (username, prot_id)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return None

    def remove_account(self):
        try:
            self.__dbusmodel.RemoveAccount()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_account_info(self, info):
        if 'password' in info:
            try:
                self.__dbusmodel.SetAccountPassword(info['password'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['password']

        if 'alias' in info and info['alias'] != '':
            try:
                self.__dbusmodel.SetAccountAlias(info['alias'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['alias']

        if 'remember_password' in info:
            try:
                self.__dbusmodel.SetAccountRemPwd(info['remember_password'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['remember_password']

        # Just a hack to jabber protocol
        # See funcs get_protocol_options and get_account_info
        if 'server' in info and info['protocol_id'] == 'prpl-jabber':
            info['connect_server'] = info['server']
            del info['server']

        if 'confirm' in info:
            del info['confirm']

        if 'username' in info:
            username = info['username']
            del info['username']
        else:
            username = ''

        if 'protocol_id' in info:
            protocol_id = info['protocol_id']
            del info['protocol_id']
        else:
            protocol_id = ''

        if len(info) > 0:
            if not self.set_protocol_options(info):
                return False

        if username != '':
            try:
                if not self.__dbusmodel.AccountExists() and not \
                        self.account_exists():
                    self.__dbusmodel.NewAccount(username, protocol_id)
                    return True
                else:
                    old_user = self.__dbusmodel.GetAccountUsername()
                    if old_user != username:
                        self.__dbusmodel.SetAccountUsername(username)
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False

        return True

    def get_protocols(self):
        return __PROTOCOLS__

class ISDBusModel(dbus.Interface):

    def __init__(self):

        dbus.set_default_main_loop(EDbusModel().get_main_loop())
        self._bus = dbus.bus.BusConnection(DBUS_ADDRESS)
        self._bus_object = self._bus.get_object(DBUS_BUS_NAME,
                                                DBUS_OBJECT_PATH)
        dbus.Interface.__init__(self, self._bus_object,
                                        DBUS_IFACE_SERVICE)

        self.__name_owner_changed_cbs = []
        self._bus.add_signal_receiver(self.__name_owner_changed_cb,
                            dbus_interface="org.freedesktop.DBus",
                            signal_name="NameOwnerChanged")

    def __name_owner_changed_cb(self, *data):
        for cb in self.__name_owner_changed_cbs:
            cb(*data)

    def add_name_owner_changed_cb(self, cb):
        if callable(cb) and cb not in self.__name_owner_changed_cbs:
            self.__name_owner_changed_cbs.append(cb)

    def del_name_owner_changed_cb(self, cb):
        if cb in self.__name_owner_changed_cbs:
            self.__name_owner_changed_cbs.remove(cb)

    def get_bus_object(self):
        return self._bus_object

    def get_bus(self):
        return self._bus
