/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of infosharingd.
 *
 *  infosharingd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  infosharingd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with infosharingd. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include <errno.h>
#include <string.h>

#include <glib.h>
#include "purple.h"

#include "server.h"
#include "commands.h"
#include "log.h"

/* Verifies if an account is enabled */
int account_is_enabled()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	return purple_account_get_enabled(carman_account->purple_acc,
		INFOSHARING_NAME);
}

/* Verifies if an account is connected */
int account_is_connected()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	return purple_account_is_connected(carman_account->purple_acc);
}

/* Verifies if an account is connecting */
int account_is_connecting()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	return purple_account_is_connecting(carman_account->purple_acc);
}

/* Verifies if an account is disconnected */
int account_is_disconnected()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	return purple_account_is_disconnected(carman_account->purple_acc);
}

/* Verify if account is set */
gboolean account_is_defined()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	return NULL != carman_account->purple_acc? TRUE : FALSE;
}

/* Verify if account exists */
int account_exists()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return FALSE;

        const char* username = purple_account_get_username(carman_account->purple_acc);
        const char* protocol_id = purple_account_get_protocol_id(carman_account->purple_acc);

	return NULL != purple_accounts_find(username, protocol_id) ? TRUE : FALSE;
}

/* Creates a new account */
int account_new(char *username, char *protocol_id)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL != carman_account->purple_acc)
		return -EEXIST;

	carman_account->purple_acc = purple_account_new(username, protocol_id);
	if (FALSE == account_exists())
		purple_accounts_add(carman_account->purple_acc);

	return 0;
}

/* Remove account, if any */
int account_remove()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	int exists = account_exists();

	if (exists == TRUE) {
		purple_accounts_delete(carman_account->purple_acc);
		carman_account->purple_acc = NULL;
	}

	/* Propagate error from account_exists or return 0 */
	return exists;
}

int set_account_username(char *username)
{
	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (NULL != username)
		purple_account_set_username(carman_account->purple_acc,
				username);
	return 0;
}

int set_account_alias(char *alias)
{
	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (NULL != alias)
		purple_account_set_alias(carman_account->purple_acc, alias);

	return 0;
}

int set_account_password(char *pwd)
{
	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (NULL != pwd)
		purple_account_set_password(carman_account->purple_acc, pwd);

	return 0;
}

int set_account_rem_pwd(gboolean value)
{
	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	purple_account_set_remember_password(carman_account->purple_acc, value);

	return 0;
}

int set_account_protocol_id(char *protocol_id)
{
	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (NULL != protocol_id)
		purple_account_set_protocol_id(carman_account->purple_acc,
				protocol_id);

	return 0;
}

char *get_account_username()
{
	if (NULL == carman_account->purple_acc)
		return NULL;

	return g_strdup(purple_account_get_username(carman_account->purple_acc));
}

char *get_account_protocol_id()
{
	if (NULL == carman_account->purple_acc)
		return NULL;

	return g_strdup(purple_account_get_protocol_id(carman_account->purple_acc));
}

char *get_account_alias()
{
	char *alias;

	if (NULL == carman_account->purple_acc)
		return NULL;

	alias = g_strdup(purple_account_get_alias(carman_account->purple_acc));
	if (NULL == alias)
		return g_strdup("");

	return alias;
}

const char *get_account_password()
{
	return (NULL != carman_account->purple_acc) ?
		purple_account_get_password(carman_account->purple_acc) : NULL;
}

gboolean get_account_rem_pwd()
{
	if (NULL == carman_account->purple_acc)
		return FALSE;

	return purple_account_get_remember_password(carman_account->purple_acc);
}

/* Set account status */
int account_set_active_status(char *type, char *message)
{
        PurpleStatusType *statustype = NULL;
        PurpleSavedStatus *savedstatus = NULL;

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (NULL != message)
		purple_account_set_status(carman_account->purple_acc, type,
			TRUE, "message", message, NULL);
	else
		purple_account_set_status(carman_account->purple_acc, type, TRUE, NULL);

	/* FIXME: Hack to keep status 'available' for all primitive status. */
	savedstatus = purple_savedstatus_find(type);
	if (NULL == savedstatus) {
		statustype = purple_account_get_status_type(carman_account->purple_acc, type);
		savedstatus = purple_savedstatus_new(NULL,
			purple_status_type_get_primitive(statustype));
		purple_savedstatus_set_title(savedstatus, type);
	}

	purple_savedstatus_set_message(savedstatus, message);
	purple_prefs_set_int("/purple/savedstatus/idleaway",
		purple_savedstatus_get_creation_time(savedstatus));

	return 0;
}
/* Connect account */
int account_connect()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	purple_account_set_enabled(carman_account->purple_acc,
			INFOSHARING_NAME, TRUE);

	purple_account_connect(carman_account->purple_acc);
	return 0;
}

/* Disconnect account */
int account_disconnect()
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (account_is_connected())
		purple_account_disconnect(carman_account->purple_acc);

	purple_account_set_enabled(carman_account->purple_acc,
			INFOSHARING_NAME, FALSE);

	return 0;
}

/* Adds a buddy to a given account */
int add_buddy(char *buddy_name, char *buddy_group, char *buddy_alias)
{
	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	if (strcmp(buddy_name, "") == 0)
		return -EINVAL;

	/* Get PurpleConnection structure from account */
	PurpleConnection *gc =
		purple_account_get_connection(carman_account->purple_acc);

	/* Certify that there is an open connection for the given account
	 * before add buddy request */
	if ((NULL != gc) && g_list_find(purple_connections_get_all(), gc)) {
		purple_blist_request_add_buddy(carman_account->purple_acc,
			buddy_name, buddy_group, buddy_alias);
		return 0;
	} else
		return -EEXIST;
}

/* Removes a buddy from a given account */
int remove_buddy(char *buddy_name)
{
	PurpleBuddy *buddy = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	buddy = purple_find_buddy(carman_account->purple_acc, buddy_name);
	if (NULL != buddy) {
		purple_account_remove_buddy(carman_account->purple_acc, buddy,
			purple_buddy_get_group(buddy));
		purple_blist_remove_buddy(buddy);
		return 0;
	} else
		return -EINVAL;
}

/* Verify if buddy exists */
int buddy_exists(char *buddy_name)
{
	PurpleBuddy *buddy = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	buddy = purple_find_buddy(carman_account->purple_acc, buddy_name);
	if (NULL != buddy)
		return TRUE;
	else
		return FALSE;
}

/* Return a list of all buddies from an account */
GSList *get_buddies()
{
	GSList *buddies = purple_find_buddies(carman_account->purple_acc, NULL);
	GSList *bnames = NULL, *aux = NULL;
	PurpleBuddy *b;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == buddies)
		return NULL;

	for (aux = buddies; NULL != aux; aux = aux->next) {
			b = (PurpleBuddy *)aux->data;
			bnames = g_slist_append(bnames, b->name);
	}

	/* Returns a single linked list of buddies if account exists, or
	 * NULL otherwise */
	return bnames;
}

/* Return a list of carman online buddies from an account */
GSList *get_carman_buddies_online()
{
	GSList *buddies = purple_find_buddies(carman_account->purple_acc, NULL);
	GSList *bnames = NULL, *aux = NULL;
	const PurpleBuddy *buddy;
	const PurpleStatus *status;
	const PurplePresence *presence;
	const char *msg;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == buddies)
		return NULL;

	for (aux = buddies; NULL != aux; aux = aux->next) {
		presence = purple_buddy_get_presence((PurpleBuddy *)aux->data);
		if (purple_presence_is_online(presence)) {
			status = purple_presence_get_active_status(presence);
			msg = purple_status_get_attr_string(status, "message");
			if (msg && strcmp(msg, "Online from Carman") == 0) {
				buddy = (PurpleBuddy *)aux->data;
				bnames = g_slist_append(bnames, buddy->name);
			}
		}
	}

	/* Returns a single linked list of carman online buddies if account
	 * exists, or NULL otherwise */
	return bnames;
}

/* Returns a buddy presence (TRUE if available, FALSE otherwise) */
int get_buddy_presence(char *buddy_name)
{
	PurpleBuddy *buddy = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	buddy = purple_find_buddy(carman_account->purple_acc, buddy_name);
	if (NULL != buddy)
		return purple_presence_is_available(purple_buddy_get_presence(buddy));
	else
		return -EINVAL;
}

/* Get buddy alias */
const char *get_buddy_alias(char *buddy_name)
{
	PurpleBuddy *buddy = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return NULL;

	buddy = purple_find_buddy(carman_account->purple_acc, buddy_name);
	if (NULL != buddy)
		return purple_buddy_get_alias_only(buddy);
	else
		return NULL;
}

/* Set buddy alias */
int set_buddy_alias(char *buddy_name, char *buddy_alias)
{
	PurpleBuddy *buddy = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	buddy = purple_find_buddy(carman_account->purple_acc, buddy_name);
	if (NULL != buddy) {
		purple_blist_alias_buddy(buddy, buddy_alias);
		serv_alias_buddy(buddy);
		return 0;
	} else
		return -EINVAL;
}

/* Check if the given protocol id is usable by libpurple */
gboolean protocol_exists(char *protocol_id)
{
	GList *protocols = purple_plugins_get_protocols();
	GList *aux = NULL;
	PurplePlugin *iter = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	for (aux = protocols; NULL != aux; aux = aux->next)
		iter = aux->data;
		if (iter->info && !strcmp(protocol_id, iter->info->id))
			return TRUE;

	return FALSE;
}

GHashTable *get_protocol_options(char *protocol_id)
{
	PurpleAccountOption *option = NULL;
	PurplePlugin *plugin = NULL;
	PurplePluginProtocolInfo *prpl_info = NULL;
	PurplePrefType type;
	GList *iter = NULL;
	GHashTable *prpl_options = g_hash_table_new(g_str_hash, g_str_equal);
	char *label_name;
	char *setting;
	char *str_value;
	int int_value;
	gboolean bool_value;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	plugin = purple_plugins_find_with_id(protocol_id);
	if (NULL == plugin)
		return NULL;

	prpl_info = PURPLE_PLUGIN_PROTOCOL_INFO(plugin);

	iter = prpl_info->protocol_options;

	while (iter) {
		option = (PurpleAccountOption *)iter->data;
		type = purple_account_option_get_type(option);
		label_name = g_strdup(purple_account_option_get_text(option));
		setting = g_strdup(purple_account_option_get_setting(option));

		if (PURPLE_PREF_STRING == type) {
			str_value = (char *) purple_account_option_get_default_string(option);
			g_hash_table_insert(prpl_options, setting, str_value);

		} else if (PURPLE_PREF_INT == type) {
			int_value = purple_account_option_get_default_int(option);
			g_hash_table_insert(prpl_options, setting, (gpointer) int_value);

		} else if (PURPLE_PREF_BOOLEAN == type) {
			bool_value = purple_account_option_get_default_bool(option);
			g_hash_table_insert(prpl_options, setting, (gpointer) bool_value);

		} else if (PURPLE_PREF_STRING_LIST == type) {
			str_value = (char *) purple_account_option_get_default_list_value(option);
			g_hash_table_insert(prpl_options, setting, (gpointer) str_value);
		}

		iter = iter->next;
	}

	return prpl_options;
}


/* Get protocol options from an account protocol */
GHashTable *get_account_protocol_options()
{
	PurpleAccountOption *option = NULL;
	PurplePlugin *plugin = NULL;
	PurplePluginProtocolInfo *prpl_info = NULL;
	PurplePrefType type;
	GList *iter = NULL;
	GHashTable *prpl_options = g_hash_table_new(g_str_hash, g_str_equal);
	char *label_name;
	char *setting;
	char *str_value;
	int int_value;
	gboolean bool_value;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return NULL; /* ENOENT */

	plugin = purple_plugins_find_with_id(carman_account->purple_acc->protocol_id);
	if (NULL == plugin)
		return NULL;

	prpl_info = PURPLE_PLUGIN_PROTOCOL_INFO(plugin);

	iter = prpl_info->protocol_options;

	while (iter) {
		option = (PurpleAccountOption *)iter->data;
		type = purple_account_option_get_type(option);
		label_name = g_strdup(purple_account_option_get_text(option));
		setting = g_strdup(purple_account_option_get_setting(option));

		if (PURPLE_PREF_STRING == type) {
			str_value = (char *) purple_account_option_get_default_string(option);
			str_value = (char *) purple_account_get_string(carman_account->purple_acc,
				setting, str_value);
			g_hash_table_insert(prpl_options, setting, str_value);

		} else if (PURPLE_PREF_INT == type) {
			int_value = purple_account_option_get_default_int(option);
			int_value = purple_account_get_int(carman_account->purple_acc,
					setting, int_value);
			g_hash_table_insert(prpl_options, setting, int_value);

		} else if (PURPLE_PREF_BOOLEAN == type) {
			bool_value = purple_account_option_get_default_bool(option);
			bool_value = purple_account_get_bool(carman_account->purple_acc,
					setting, bool_value);
			g_hash_table_insert(prpl_options, setting, bool_value);

		} else if (PURPLE_PREF_STRING_LIST == type) {
			str_value = (char *) purple_account_option_get_default_list_value(option);
			str_value = (char *) purple_account_get_string(carman_account->purple_acc,
				setting, str_value);
			g_hash_table_insert(prpl_options, setting, str_value);
		}

		iter = iter->next;
	}

	return prpl_options;
}

/* Call internal libpurple callback according to user selection */
int request_authorize_buddy_cb(char *name, gboolean authorize)
{
	PendingRequestAuthorization *request = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if ((NULL == pending_requests) ||
			(NULL == carman_account->purple_acc))
		return -EINVAL;

	request = g_hash_table_lookup(pending_requests, name);
	if (!request)
		return -ENOENT;

	if (authorize) {
		request->authorize_cb(request->user_data);
		purple_blist_request_add_buddy(carman_account->purple_acc,
			name, NULL, NULL);
	} else
		request->deny_cb(request->user_data);

	g_hash_table_remove(pending_requests, name);

	return 0;
}

/* Set protocol options from a given hash table */
int set_account_protocol_options(GHashTable *prpl_options)
{
	PurpleAccountOption *option = NULL;
	PurplePlugin *plugin = NULL;
	PurplePluginProtocolInfo *prpl_info = NULL;
	PurplePrefType type;
	GList *iter = NULL;
	char *label_name;
	char *setting, *str_val;
	gboolean bool_val;
	gint int_val;
	gpointer value;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	plugin = purple_plugins_find_with_id(
		purple_account_get_protocol_id(carman_account->purple_acc));
	prpl_info = PURPLE_PLUGIN_PROTOCOL_INFO(plugin);

	iter = prpl_info->protocol_options;
	while (iter) {
		option = (PurpleAccountOption *)iter->data;
		type = purple_account_option_get_type(option);
		label_name = g_strdup(purple_account_option_get_text(option));
		setting = g_strdup(purple_account_option_get_setting(option));

		if (PURPLE_PREF_STRING == type) {
			if (g_hash_table_lookup_extended(prpl_options,
						setting, NULL, (gpointer) &str_val))
				purple_account_set_string(carman_account->purple_acc,
						setting, str_val);
		}
		if (PURPLE_PREF_INT == type) {
			if (g_hash_table_lookup_extended(prpl_options,
						setting, NULL, &int_val))
				purple_account_set_int(carman_account->purple_acc,
					setting, int_val);
		}
		if (PURPLE_PREF_BOOLEAN == type) {
			if (g_hash_table_lookup_extended(prpl_options,
						setting, NULL, &bool_val)){
				purple_account_set_bool(carman_account->purple_acc,
					setting, bool_val);
			}
		}

		/* TODO: Infosharingd does not support string list as protocol
		 * option. */
		iter = iter->next;
	}

	return 0;
}

/* Sends an IM message to a conversation */
int send_im_message(char *name, char *message)
{
	PurpleBuddy *buddy;
	gpointer value = NULL;

	infosharingd_debug("%s:%d\n", __func__, __LINE__);

	if (NULL == carman_account->purple_acc)
		return -ENOENT;

	/* Get PurpleBuddy structure from buddy name */
	buddy = purple_find_buddy(carman_account->purple_acc, name);

	if (NULL == buddy)
		return -EINVAL;

	/* Lookup into current conversations to see if there is already a
	 * conversation for this buddy. */
	value = g_hash_table_lookup(convs, buddy->name);

	if (NULL == value) {
		/* If no conversation is found, create a new one and add it to
		 * conversations hash table */
		value = PURPLE_CONV_IM(
			purple_conversation_new(PURPLE_CONV_TYPE_IM,
				buddy->account, buddy->name));
		g_hash_table_insert(convs, buddy->name, value);
	}

	/* Send IM type message to a conversation */
	purple_conv_im_send(value, message);

	return 0;
}
