#!/usr/bin/env sh

#
# Carman Cleanup
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Gustavo Barbieri <gustavo.barbieri@openbossa.org>
#         Eduardo Lima (Etrunko) <eduardo.lima@indt.org.br>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#


APT_GET="apt-get -y --force-yes -q"
DPKG="dpkg --abort-after 999999"
BASENAME=`which basename`
PACKAGES="carman \
carman-sdl \
carman-scan \
carman-gtk \
carman-common \
obdlib
"
CLEAN_PATHS="\
 /home/user/.carman \
 /usr/share/carman \
 /etc/carman/
 "

die()
{
    echo "### ERROR: $*"
    exit 1
}

info()
{
    echo "### INFO: $*"
}

warning()
{
    echo "### WARN: $*"
}

important()
{
    echo "### IMPORTANT: $*"
}

report_exit()
{
    if [ $1 -eq 0 ]; then
        STATUS="OK"
    else
        STATUS="FAILED, code=$1"
    fi
    echo "### STATUS: $STATUS"
}

report_cmd()
{
    sh -c "$*"
    ret=$?
    report_exit $ret
    return $ret
}

get_software_raw()
{
    cut -d_ -f2 /etc/osso_software_version
}

get_software()
{
    HW=$1
    VER=`get_software_raw`
    case $HW in
        770)
            case $VER in
                2006SE)
                    echo "gregale"
                    ;;
                2007HACKER)
                    echo "hackeredition"
                    ;;
                *)
                    echo "UNKNOWN"
                    ;;
            esac
            ;;
        N800)
            case $VER in
                2007SE)
                    echo "bora"
                    ;;
                2008SE)
                    echo "chinook"
                    ;;
				DIABLO)
					echo "diablo"
					;;
                *)
                    echo "UNKNOWN"
                    ;;
            esac
            ;;
        N810)
            case $VER in
                2008SE)
                    echo "chinook"
                    ;;
                DIABLO)
                    echo "diablo"
                    ;;
                *)
                    echo "UNKNOWN"
                    ;;
            esac
            ;;
        *)
            echo "UNKNOWN"
            ;;
    esac
}

get_hardware_raw()
{
    grep product /proc/component_version | \
        sed 's/^[^ ]\+[ ]\+\([a-zA-Z0-9]\+\)/\1/g'
}

get_hardware()
{
    N=`get_hardware_raw`
    case $N in
        SU-18)
            echo "770"
            ;;
        RX-34)
            echo "N800"
            ;;
        RX-44)
            echo "N810"
            ;;
        *)
            echo "UNKNOWN"
            ;;
    esac
}

plat_info_report()
{
    important "Platform Information to Report"
    important "HARDWARE: `get_hardware_raw`"
    important "SOFTWARE: `get_software_raw`"
}

apt_clean()
{
    report_cmd $APT_GET clean
}

apt_remove()
{
    report_cmd $APT_GET --purge remove $*
}

apt_f_install()
{
    report_cmd $APT_GET -f install
}

dpkg_configure_pending()
{
    report_cmd $DPKG --force-all --configure -a
}

clean_paths()
{
	check_carman_version
	ret=$?
	if [ $ret -eq 0 ];then
		echo "### REMOVING CARMAN'S FOLDERS"
		for p in $CLEAN_PATHS; do
			if [ -d $p ]; then
				report_cmd rm -fr $p
			fi
		done
	fi
}


check_running(){
	apt_worker_pid=`pidof apt-worker`
	test ! -z "$apt_worker_pid" && die "Application Installer is running."
	export HARDWARE=`get_hardware`
	export SOFTWARE=`get_software $HARDWARE`
	if [ $HARDWARE = "UNKNOWN" ] || [ $SOFTWARE = "UNKNOWN" ]; then
		plat_info_report
		die "Unsupported platform, please report"
	fi
}

check_carman_version()
{
	if [ -d /etc/carman/ ]; then
		grep -r carman-sdl /etc/carman
		ret=$?
		if [ $? -eq 0 ]; then
			return 0
		fi
	fi
	return 1
}

clean_packs()
{
	apt_clean
	apt_f_install
	dpkg_configure_pending
}

## Script init
check_running
clean_packs
for pack in $PACKAGES; do
{
	echo "### REMOVING: $pack"
	apt_remove $pack
}
done
clean_paths
exit $?
