/*
 * Carman Cleanup
 * Copyright (C) 2008 Instituto Nokia de Tecnologia
 * Author: Gustavo Barbieri <gustavo.barbieri@openbossa.org>
 *         Eduardo Lima (Etrunko) <eduardo.lima@indt.org.br>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "ui-impl.h"
#include <string.h>
#include <stdlib.h>

#if USE_HILDON
#include <libosso.h>
#include <hildon/hildon-banner.h>
#endif

static const char *base_packages[] =
{
	"carman",
	"carman-sdl",
	"carman-scan",
	"carman-gtk",
	"carman-common",
	"obdlib"
};
#define PACKAGES_SIZE 6

struct tag_desc
{
	const char *str;
	const char *color;
};

static const struct tag_desc tags_descs[] =
{
	{"ERROR", "Red"},
	{"INFO", "Green"},
	{"WARN", "OrangeRed"},
	{"IMPORTANT", "Blue"},
	{"COMMAND", "DarkMagenta"},
	{"COMMAND_STATUS", "Magenta"},
};
#define TAGS_DESCS_SIZE (sizeof(tags_descs)/sizeof(struct tag_desc))

struct app
{
	char *executable;
	GtkTextView *textview;
	GtkTextBuffer *textbuffer;
	struct app_ui
	{
		GtkWidget *window;
		GtkWidget *run;
		GtkWidget *close;
		GtkWidget *log;
		GtkWidget *dialog;
		GtkListStore *model;
		GtkTreeIter *iter;
	} ui;
	GtkTextTag *tags[TAGS_DESCS_SIZE];
	char *argv[3];
	GPid child_pid;
	struct app_fds
	{
		int out;
		int err;
		FILE *log;
	} fds;
	struct apps_iochannels
	{
		GIOChannel *out;
		GIOChannel *err;
	} iochannels;
	struct app_event_sources
	{
		guint out;
		guint err;
		guint child;
	} event_sources;
};

#define LINE_START_MARKER "### "
#define LINE_START_MARKER_SIZE (sizeof(LINE_START_MARKER) - 1)
#define LOG_PATH "/home/user/.carman_cleanup.log"

static GtkTextTag *
get_tag_for_line(GtkTextTag **tags, const char *line, int *offset)
{
	const char *sep;
	int i;

	*offset = 0;
	if (!line || strncmp(line, LINE_START_MARKER, LINE_START_MARKER_SIZE) != 0)
		return NULL;
	line += LINE_START_MARKER_SIZE;

	sep = strchr(line, ':');
	if (!sep)
		return NULL;

	*offset = sep - line;
	if (*offset < 0) {
		*offset = 0;
		return NULL;
	}

	for (i = 0; i < TAGS_DESCS_SIZE; i++)
		if (strncmp(line, tags_descs[i].str, *offset) == 0) {
			*offset += LINE_START_MARKER_SIZE + 1; /* include ':' */
			return tags[i];
		}

	*offset = 0;
	return NULL;
}

static gboolean
check_install(gpointer data)
{
	GtkTreeIter iter;
	struct app *app = data;
	char *cmd;
	int i;

#if USE_HILDON
	GtkWidget *banner;

	banner = hildon_banner_show_animation(GTK_WIDGET(app->ui.window), NULL, "Check packages");
#endif
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->ui.model), &iter);

	for (i = 0; i < PACKAGES_SIZE; i++) {
		cmd = g_strdup_printf("dpkg -l | grep -e '^i.  %s \\+0.6-3 '", base_packages[i]);

		if (system(cmd) == 0) {
			gtk_list_store_set(GTK_LIST_STORE(app->ui.model), &iter, 1, "installed", -1);
			gtk_widget_set_sensitive(app->ui.run, 1);
		} else
			gtk_list_store_set(GTK_LIST_STORE(app->ui.model), &iter, 1, "not installed", -1);

		gtk_tree_model_iter_next(GTK_TREE_MODEL(app->ui.model), &iter);
		g_free(cmd);
	}

#if USE_HILDON
	gtk_widget_destroy(banner);
#endif
	return FALSE;
}

static gboolean
update_list(struct app *app, const char *line)
{
	char *cmd;
	GdkColor color;
	gboolean ret = FALSE;

	cmd = g_malloc0(7);

	strncpy(cmd, line + LINE_START_MARKER_SIZE,	6);
	if (strcmp(cmd, "REMOVI") == 0) {
		if (app->ui.iter == NULL) {
			app->ui.iter = g_malloc(sizeof(GtkTreeIter));
			gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->ui.model), app->ui.iter);
		} else {
			if (!gtk_tree_model_iter_next(GTK_TREE_MODEL(app->ui.model), app->ui.iter))
				app->ui.iter = NULL;
		}

		gdk_color_parse("black", &color);
		gtk_list_store_set(GTK_LIST_STORE(app->ui.model), app->ui.iter, 1,
				"Removing", 2, &color, -1);
		ret = TRUE;

	}

	if (strcmp(cmd, "STATUS") == 0) {
		if (app->ui.iter != NULL) {
			if (strncmp(line + LINE_START_MARKER_SIZE + 8, "OK", 2) == 0) {
				gdk_color_parse("#006400", &color);
				gtk_list_store_set(GTK_LIST_STORE(app->ui.model), app->ui.iter, 1,
						"OK", 2, &color, -1);
			} else {
				gdk_color_parse("#ab0000", &color);
				gtk_list_store_set(GTK_LIST_STORE(app->ui.model), app->ui.iter, 1,
						"FAIL", 2, &color, -1);
			}
		}
		ret = TRUE;
	}

	g_free(cmd);

	return ret;
}

static void
add_line(struct app *app, const char *line)
{
	GtkTextIter end_itr;
	GtkTextTag *tag;
	int offset;

	gtk_text_buffer_get_end_iter(app->textbuffer, &end_itr);
	tag = get_tag_for_line(app->tags, line, &offset);

	if (update_list(app, line))
		return;

	if (!tag)
		gtk_text_buffer_insert(app->textbuffer, &end_itr, line, -1);
	else {
		gtk_text_buffer_insert_with_tags(app->textbuffer, &end_itr,
				line, offset, tag, NULL);

		gtk_text_buffer_insert(app->textbuffer, &end_itr, line + offset,
				-1);
	}

	gtk_text_view_scroll_to_iter(app->textview, &end_itr,
			0.0, FALSE, 0.0, 0.0);
	if (app->fds.log)
		fprintf(app->fds.log, line);
}

static void
close_iochannel(GIOChannel **chan)
{
	g_io_channel_shutdown(*chan, 0, NULL);
	g_io_channel_unref(*chan);
	*chan = NULL;
}

static void
remove_source(guint *id)
{
	g_source_remove(*id);
	*id = 0;
}

static void
cb_stopped(struct app *app)
{
	gtk_widget_set_sensitive(app->ui.close, 1);
}

static void
cb_child(GPid pid, gint status, gpointer data)
{
	struct app *app = data;

	app->child_pid = -1;

	remove_source(&app->event_sources.out);
	remove_source(&app->event_sources.err);
	remove_source(&app->event_sources.child);

	close_iochannel(&app->iochannels.out);
	close_iochannel(&app->iochannels.err);

	cb_stopped(app);
}

static gboolean
cb_input(GIOChannel *chan, GIOCondition cond, struct app *app)
{
	GIOStatus status;
	GError *error;
	char *str;

	str = NULL;
	error = NULL;
	status = g_io_channel_read_line(chan, &str, NULL, NULL, &error);
	add_line(app, str);
	g_free(str);

	return status == G_IO_STATUS_NORMAL;
}

static gboolean
cb_input_out(GIOChannel *chan, GIOCondition cond, gpointer data)
{
	struct app *app = data;
	gboolean r;

	r = cb_input(chan, cond, app);
	if (!r)
		app->event_sources.out = 0;

	return r;
}

static gboolean
cb_input_err(GIOChannel *chan, GIOCondition cond, gpointer data)
{
	struct app *app = data;
	gboolean r;

	r = cb_input(chan, cond, app);
	if (!r)
		app->event_sources.err = 0;

	return r;
}

static int
run(struct app *app)
{
	GSpawnFlags flags;
	GError *error;
	gboolean r;
	GIOCondition cond;

	app->argv[0] = "carman-cleanup-launcher";
	app->argv[1] = app->executable;
	app->argv[2] = NULL;

	flags = G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD;
	error = NULL;
	r = g_spawn_async_with_pipes("/tmp", app->argv, NULL, flags, NULL, NULL,
			&app->child_pid,
			NULL, &app->fds.out, &app->fds.err,
			&error);

	if (!r && error) {
		char buf[1024];

		snprintf(buf, sizeof(buf),
				LINE_START_MARKER "ERROR: Could not start executable: %s\n",
				error->message);
		add_line(app, buf);

		g_error_free(error);
		return 0;
	}

	app->event_sources.child = g_child_watch_add(app->child_pid, cb_child, app);

	app->iochannels.out = g_io_channel_unix_new(app->fds.out);
	app->iochannels.err = g_io_channel_unix_new(app->fds.err);

	cond = G_IO_IN | G_IO_ERR;
	app->event_sources.out =
		g_io_add_watch(app->iochannels.out, cond, cb_input_out, app);
	app->event_sources.err =
		g_io_add_watch(app->iochannels.err, cond, cb_input_err, app);

	gtk_widget_set_sensitive(app->ui.run, 0);
	gtk_widget_set_sensitive(app->ui.close, 0);

	return 1;
}

static void
cb_run(GtkWidget *btn, struct app *app)
{
	if (app->child_pid >= 0)
		return;

	gtk_text_buffer_set_text(app->textbuffer, "", -1);
	run(app);
}

static void
cb_view_log(GtkWidget *btn, struct app *app)
{
	gtk_dialog_run(GTK_DIALOG(app->ui.dialog));
	gtk_widget_hide(app->ui.dialog);

}

static void
warn_dialog(GtkWidget *parent, const char *msg)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new_with_markup(GTK_WINDOW(parent),
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_WARNING,
			GTK_BUTTONS_CLOSE,
			NULL);
	gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(dialog), msg);
	gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
}

static gboolean
do_quit(struct app *app)
{
	if (app->child_pid < 0) {
		gtk_main_quit();
		return 0;
	}

	warn_dialog(app->ui.window,
			"<big><b>Still processing.</b></big>\n"
			"You cannot close application yet.\n"
			"<b>It is dangerous, may corrupt your system.</b>");
	return 1;
}

static gboolean
cb_close(GtkWidget *wid, struct app *app)
{
	return do_quit(app);
}

static GtkWidget *
create_buttons(struct app *app)
{
	GtkWidget *hbox;

	hbox = gtk_hbutton_box_new();

	app->ui.close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	GTK_WIDGET_SET_FLAGS(app->ui.close, GTK_CAN_DEFAULT);
	g_signal_connect(app->ui.close, "clicked", G_CALLBACK(cb_close), app);

	app->ui.run = gtk_button_new_from_stock(GTK_STOCK_EXECUTE);
	gtk_widget_set_sensitive(app->ui.run, 0);
	g_signal_connect(app->ui.run, "clicked", G_CALLBACK(cb_run), app);

	app->ui.log = gtk_button_new_with_label("Log");
	g_signal_connect(app->ui.log, "clicked", G_CALLBACK(cb_view_log), app);

	gtk_box_pack_start(GTK_BOX(hbox), app->ui.close, 0, 1, 0);
	gtk_box_pack_start(GTK_BOX(hbox), app->ui.run, 0, 1, 0);
	gtk_box_pack_start(GTK_BOX(hbox), app->ui.log, 0, 1, 0);

	return hbox;
}

static void
create_columns(GtkTreeView *list)
{
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;

	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Package");

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, renderer, 1);
	gtk_tree_view_column_set_attributes(col, renderer, "text", 0, NULL);
	gtk_tree_view_column_set_min_width(col, 450);
	gtk_tree_view_column_set_alignment(col, 0.5);
	gtk_tree_view_append_column(list, col);

	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Status");

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, renderer, 1);
	gtk_tree_view_column_set_attributes(col, renderer, "text", 1, "foreground-gdk", 2, NULL);
	g_object_set(G_OBJECT(renderer), "xalign", 0.5, NULL);
	gtk_tree_view_column_set_alignment(col, 0.5);
	gtk_tree_view_append_column(list, col);
}

static GtkWidget *
create_list(struct app *app)
{
	GtkWidget *list;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	int i;

	list = gtk_tree_view_new();
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_NONE);

	create_columns(GTK_TREE_VIEW(list));
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list), TRUE);

	app->ui.model = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING, GDK_TYPE_COLOR);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(app->ui.model));

	g_object_unref(app->ui.model);

	for (i = 0; i < PACKAGES_SIZE; i++) {
		gtk_list_store_append(GTK_LIST_STORE(app->ui.model), &iter);
		gtk_list_store_set(GTK_LIST_STORE(app->ui.model), &iter, 0, base_packages[i], -1);
	}

	return list;
}

static GtkTextView *
create_textview(struct app *app)
{
	GtkTextView *tv;
	PangoFontDescription *font_desc;
	int i;

	tv = GTK_TEXT_VIEW(gtk_text_view_new());
	gtk_text_view_set_wrap_mode(tv, GTK_WRAP_NONE);
	gtk_text_view_set_editable(tv, 0);

	font_desc = pango_font_description_from_string("Mono");
	gtk_widget_modify_font(GTK_WIDGET(tv), font_desc);
	pango_font_description_free(font_desc);

	app->textbuffer = gtk_text_view_get_buffer(tv);

	for (i = 0; i < TAGS_DESCS_SIZE; i++) {
		const struct tag_desc *desc = tags_descs + i;
		app->tags[i] = gtk_text_buffer_create_tag(app->textbuffer,
				desc->str,
				"foreground", desc->color,
				"weight", PANGO_WEIGHT_BOLD,
				NULL);
	}

	return tv;
}

static GtkWidget *
create_dialog(struct app *app)
{
	GtkWidget *dialog;
	GtkWidget *scrolledwindow;

	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
			GTK_SHADOW_ETCHED_IN);
	gtk_container_add(GTK_CONTAINER(scrolledwindow), GTK_WIDGET(app->textview));

	dialog = gtk_dialog_new();
	gtk_window_set_default_size(GTK_WINDOW(dialog), 600, 300);
	gtk_window_set_title(GTK_WINDOW(dialog), "Log View");
	gtk_window_set_modal(GTK_WINDOW(dialog), 1);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), scrolledwindow);

	gtk_widget_show_all(scrolledwindow);

	return dialog;
}

static GtkWidget *
create_contents(struct app *app)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *buttons;
	GtkWidget *list;

	vbox = gtk_vbox_new(0, 5);

	label = gtk_label_new("This program will fully remove old packages of carman (0.6x)");

	list = create_list(app);

	app->textview = create_textview(app);
	app->ui.dialog = create_dialog(app);
	buttons = create_buttons(app);

	gtk_box_pack_start(GTK_BOX(vbox), label, 0, 1, 1);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), 0, 1, 13);
	gtk_box_pack_start(GTK_BOX(vbox), list, 1, 1, 1);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), 0, 1, 0);
	gtk_box_pack_start(GTK_BOX(vbox), buttons, 0, 0, 0);

	return vbox;
}

static gboolean
cb_quit(GtkWidget *wid, GdkEvent *event, struct app *app)
{
	return do_quit(app);
}

static int
usage(int argc, char *argv[])
{
	g_printerr("Usage:\n"
			"\t%s <executable>\n"
			"\n",
			argv[0]);
	return 0;
}

int
main(int argc, char *argv[])
{
	UiProgram *program;
	GtkWidget *contents;
	struct app app;

#if USE_HILDON
	osso_context_t *osso_context;
#endif

	if (argc < 2)
		return usage(argc, argv);

	app.executable = argv[1];
	app.fds.log = fopen(LOG_PATH,"w");
	if (!app.fds.log)
		printf("not open %s\n", LOG_PATH);
	app.child_pid = -1;

	gtk_init(&argc, &argv);

#if USE_HILDON
	osso_context = osso_initialize("br.org.indt.carman_cleanup", "0.1.0", TRUE, NULL);
#endif

	program = program_new();
	app.ui.window = create_window();
	g_set_application_name("Carman Cleanup");
	gtk_window_set_title(GTK_WINDOW(app.ui.window), "Carman Cleanup");
	program_add_window(program, app.ui.window);

	contents = create_contents(&app);
	gtk_container_add(GTK_CONTAINER(app.ui.window), contents);
	gtk_container_set_border_width(GTK_CONTAINER(app.ui.window), 10);
	gtk_widget_grab_focus(app.ui.close);
	gtk_widget_grab_default(app.ui.close);

	gtk_widget_show_all(app.ui.window);
	g_signal_connect(app.ui.window, "delete_event", G_CALLBACK(cb_quit), &app);

	g_timeout_add(100, check_install, &app);

	gtk_main();

	if (app.fds.log)
		fclose(app.fds.log);

#if USE_HILDON
	osso_deinitialize(osso_context);
#endif

	return 0;
}
