# Canola2 UPnP Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import logging

from terra.core.task import Task

from terra.core.manager import Manager
from terra.core.model import Model, ModelFolder
from terra.core.threaded_model import ThreadedModelFolder
from terra.core.task import Task
from terra.core.plugin_prefs import PluginPrefs
from terra.utils.encoding import to_utf8 as decode

from twisted.internet import glib2reactor
reactor = glib2reactor.install()

__all__ = ("SharedModelFolder",)


log = logging.getLogger("plugins.shared")

mger = Manager()
PluginDefaultIcon = mger.get_class("Icon/Plugin")
ModelAudio = mger.get_class("Model/Media/Audio")
ModelVideo = mger.get_class("Model/Media/Video/Local")
ModelImage = mger.get_class("Model/Media/Image")
CanolaError = mger.get_class("Model/Notify/Error")

control_point = None

chunk_size = 10
search_timeout = 100

class UPnPModelAudio(ModelAudio):
    terra_type = "Model/Media/Audio/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelAudio.__init__(self, name, None)

        self.parent = parent
        self.title = name
        self.uri = item.uri
        self.rating = None

        self.artist = ""
        self.genre = ""
        self.album = ""
        self.cover = None

class UPnPModelMusic(UPnPModelAudio):
    terra_type = "Model/Media/Audio/Shared/UPnP/Music"

    def __init__(self, name, parent, item):
        UPnPModelAudio.__init__(self, name, parent, item)

        if item.artist:
            self.artist = decode(item.artist)
        if item.genre:
            self.genre = decode(item.genre)
        if item.album:
            self.album = decode(item.album)

        self.cover = self.get_cover()

    def load_cover_path(self):
        canola_prefs = PluginPrefs("settings")
        try:
            path = canola_prefs["cover_path"]
        except KeyError:
            path = canola_prefs["cover_path"] = \
                os.path.expanduser("~/.canola/covers/")
            canola_prefs.save()

        if not os.path.exists(path):
            os.makedirs(path)
        return path

    def get_cover(self):
        path = self.load_cover_path()
        artist = self.artist
        album = self.album
        cover = "%s/%s/cover.jpg" % (artist, album)
        cover = os.path.join(path, cover)
        return cover



class UPnPModelVideo(ModelVideo):
    terra_type = "Model/Media/Video/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelVideo.__init__(self, name, None)

        self.parent = parent
        self.title = name
        self.uri = item.uri
        self.thumbnail = None


class UPnPModelImage(ModelImage):
    terra_type = "Model/Media/Image/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelImage.__init__(self, name, None)

        self.parent = parent
        self.path = item.uri


class UPnPModelFolder(ThreadedModelFolder):
    terra_type = "Model/Folder/Shared/UPnP"

    def __init__(self, name, parent, device, control_point):
        ThreadedModelFolder.__init__(self, name, parent)
        self.parent = parent
        self.device = device
        self.control_point = control_point

    def insert_new_items(self, items):
        from brisa.upnp.didl.DIDLLite import Container, AudioItem
        from brisa.upnp.didl.DIDLLite import VideoItem, ImageItem

        buffer = []
        for item in items:
            upnp_item = None
            name = decode(item.title)

            if not name:
                name = 'NO NAME'
            if isinstance(item, ImageItem):
                upnp_item = UPnPModelImage(name, self, item)
            elif isinstance(item, VideoItem):
                upnp_item = UPnPModelVideo(name, self, item)
            elif type(item) == AudioItem:
                upnp_item = UPnPModelAudio(name, self, item)
            elif isinstance(item, AudioItem):
                upnp_item = UPnPModelMusic(name, self, item)
            elif isinstance(item, Container):
                UPnPModelFolder(name, self, item, self.control_point)
                continue
            else:
                log.warning("UPnP type %s not match, item name:%s",
                            type(item), name)
                continue

            if len(self.children) < chunk_size:
                self.append(upnp_item)
            else:
                buffer.append(upnp_item)

            if len(buffer) > chunk_size:
                self.extend(buffer)
                buffer = []

        if len(buffer):
            self.extend(buffer)

    def do_load(self):
        device_id = self.device.id

        start = 0
        items = self.control_point.browse_items(device_id, start, chunk_size)
        log.info("%d UPnP Items found!", len(items))
        while items and len(items):
            self.insert_new_items(items)

            if len(items) != chunk_size or self.must_die:
                break

            start += chunk_size
            items = self.control_point.browse_items(device_id, start, chunk_size)

    def do_unload(self):
        log.info("%d UPnP items removed", len(self.children))
        ThreadedModelFolder.do_unload(self)


class UPnPDeviceModelFolder(UPnPModelFolder):
    terra_type = "Model/Folder/Shared/UPnP/Device"

    def __init__(self, name, parent, upnp_device, control_point):
        UPnPModelFolder.__init__(self, name, parent, upnp_device, control_point)
        self.parent = parent
        self.upnp_device = upnp_device
        self.control_point = control_point
        self.device.id = 0

    def do_load(self):
        self.control_point.current_server = self.upnp_device
        UPnPModelFolder.do_load(self)

    def do_unload(self):
        log.info("%d UPnP items removed", len(self.children))
        self.control_point.current_server = None
        UPnPModelFolder.do_unload(self)

class UPnPSearchDevicesIcon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Shared/UPnPSearchDevices"
    icon = "icon/main_item/upnp_search"
    plugin = "upnp"


class UPnPSearchDevicesModelFolder(ThreadedModelFolder, Task):
    terra_type = "Model/Folder/Task/Shared/UPnPSearchDevices"
    terra_task_type = "Task/Shared/UPnP"

    icon = "upnp_search"

    def __init__(self, parent):
        Task.__init__(self)
        ThreadedModelFolder.__init__(self, "UPnP Devices", parent)
        self.parent = parent
        self._timeout_inform_end = None
        self._setup_network()
        self.is_daemon = True

    def _check_network(self, network):
        if network.status == 0:
            self.state_reason = CanolaError("Network is down.")
        else:
            self.state_reason = None

    def _setup_network(self):
        self._network = Manager().get_status_notifier("Network")
        self._check_network(self._network)
        self._network.add_listener(self._check_network)

    def new_device_found(self, device):
        for item in self.children:
            if item.upnp_device.udn == device.udn:
                log.info("UPnP Device ja encontrado %s", device.friendly_name)
                return

        UPnPDeviceModelFolder(decode(device.friendly_name), self, device, self.upnp)
        if not self.is_loaded:
            self.inform_loaded()

    def del_device(self, device):
        for item in self.children:
            if item.upnp_device.udn == device:
                item.state_reason = CanolaError("Server is going down.")
                self.remove(item)
                return

    def start_search(self):
        self.upnp.start_device_search()
        return False

    def must_quit(self):
        if self.must_die:
            log.info("main loop quit")
            self.upnp.stop_device_search()
            if self.main_loop:
                self.main_loop.quit()
                self.main_loop = None
            if self._timeout_inform_end is not None:
                import gobject
                gobject.source_remove(self._timeout_inform_end)
                self._timeout_inform_end = None
            return False

        return True

    def do_load(self):
        import gobject
        gobject.threads_init()

        self.upnp = new_upnp_control_point(self)

        self.upnp.new_device_cb = self.new_device_found
        self.upnp.del_device_cb = self.del_device

        def _timeout_inform_end():
            if not self.is_loaded:
                self.inform_loaded()
            self._timeout_inform_end = None
            return False

        self.main_loop = gobject.MainLoop()
        gobject.idle_add(self.start_search)
        gobject.idle_add(self.must_quit)
        self._timeout_inform_end = None

        self.main_loop.run()

    def do_unload(self):
        log.info("%d UPnP devices removed", len(self.children))
        ThreadedModelFolder.do_unload(self)


def new_upnp_control_point(obj):
    global control_point

    if control_point:
        return control_point

    from brisa.control_point.control_point import ControlPoint

    class UPnPControlPoint(ControlPoint):
        current_server_device = None
        current_renderer_device = None
        current_media_id = None
        control_point_manager = None

        def __init__(self, parent):
            ControlPoint.__init__(self, self)
            self.new_device_cb = None
            self.del_device_cb = None

        def start_device_search(self):
            log.info("UPnP starting device search")
            self.start_search(search_timeout, reset=True)

        def stop_device_search(self):
            log.info("UPnP stopping device search")
            self.stop_search()

        def browse_items(self, device_id, start, count):
            ret = []
            try:
                items = self.browse(device_id, 'BrowseDirectChildren',
                                    '*', str(start), str(count), '')
            except:
                return []

            return items

        def on_new_media_server(self, device):
            if self.new_device_cb:
                self.new_device_cb(device)

        def on_del_media_server(self, device):
            if self.del_device_cb:
                self.del_device_cb(device)

        def on_new_media_renderer(self, device):
            log.info("UPnP Control Point, on_new_media_renderer")

        def on_del_media_renderer(self, device):
            log.info("UPnP Control Point, on_del_media_renderer")

    log.info("Creating new UPnP Control Point")
    control_point = UPnPControlPoint(obj)
    return control_point


