# Canola2 Remember The Milk Plugin
# Authors: Andrey Popelo <andrey@popelo.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import evas
import edje
import ecore

from terra.core.manager import Manager
from terra.core.controller import Controller
from terra.ui.base import PluginThemeMixin, EdjeWidget
from terra.core.terra_object import TerraObject
from terra.ui.screen import Screen

from model import SearchModelFolder


manager = Manager()
GeneralActionButton = manager.get_class("Widget/ActionButton")
BaseListController = manager.get_class("Controller/Folder")
BaseRowRenderer =  manager.get_class("Widget/RowRenderer")
ResizableRowRenderer = manager.get_class("Widget/ResizableRowRenderer")
EntryDialogModel = manager.get_class("Model/EntryDialog")
OptionsControllerMixin = manager.get_class("OptionsControllerMixin")
OptionsController = manager.get_class("OptionsController")
OptionsFolderController = manager.get_class("Controller/Options/Folder")
CanolaError = manager.get_class("Model/Notify/Error")


class OptionsControllerMixinEx(OptionsControllerMixin):
    def options_model_get(self):
        try:
            return self.model.options_model_get(self)
        except:
            return None

#class RTMTaskController(BaseListController):
#    terra_type = "Controller/RTMTask"
#
#    def __init__(self, model, canvas, parent):
#        BaseListController.__init__(self, model, canvas, parent)
#        err = CanolaError("Task properties editing temporary disabled, sorry")
#        self.parent.show_notify(err)


class ListController(BaseListController, OptionsControllerMixinEx):
    """Remember The Milk Navigation List.

    This is the initial list of Remember The Milk plugin. It shows all
    navigation possibilities, like Search for tasks, Today tasks, Lists, Smart
    lists ...

    This list is identical to Canola Text List, with a special treatment for
    the Search item, that popup a Search Modal before go to the next screen.

    @see: BaseListController
    """
    terra_type = "Controller/Folder/Task/Apps/RTM"

    def __init__(self, model, canvas, parent):
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixinEx.__init__(self)
        if model.not_logged_message:
            self.view.part_text_set("message", model.not_logged_message)

    # show throbber if model is being updated
    def _update_ui(self, model):
        self._check_model_updated()

    def _check_model_updated(self):
        if self.model.is_loaded:
            self.view.model_updated()
        else:
            self.view.throbber_start()
            self.model.callback_loaded = self._model_updated

    def _model_updated(self, model):
        self.view.throbber_stop()
        self.view.model_updated()
        model.callback_updated = None


    def cb_on_clicked(self, view, index):
        model = self.model.children[index]

        if type(model) is not SearchModelFolder:
            BaseListController.cb_on_clicked(self, view, index)
            return

        def do_search(ignored, text):
            if text is not None:
                model.query = text
                BaseListController.cb_on_clicked(self, view, index)

        dialog = EntryDialogModel("Search for tasks", "Enter keywords:",
                                    answer_callback=do_search)
        self.parent.show_notify(dialog)

class CompleteButton(EdjeWidget, TerraObject):
    group = "widget/list/rtm/complete_button"
    terra_type = "Widget/RTM/CompleteButton"

    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, self.group, parent, theme)

        self.button_complete_pressed_cb = None

    # TODO: theme_changed

    def _call_if_exists(self, cb, args=()):
        if cb is not None:
            cb(*args)

    def on_button_complete_pressed_set(self, cb):
        self.button_complete_pressed_cb = cb

    @edje.decorators.signal_callback("button,complete,pressed", "")
    def _on_button_complete_pressed(self, *ignore):
        self._call_if_exists(self.button_complete_pressed_cb)

class CompleteButtonRTM(PluginThemeMixin, CompleteButton):
    plugin = "rtm"


class GeneralRowRenderer(PluginThemeMixin, BaseRowRenderer):
    """Remember The Milk Base List Item Renderer.

    This renderer is applied on ServiceController. Providing
    a personalized list item for RTM tasks.
    """
    plugin = "rtm"

    def __init__(self, parent, theme=None):
        BaseRowRenderer.__init__(self, parent, theme)
        self.bg_selection = self.PluginEdjeWidget("widget/list/bg_selection")
        self.part_swallow("selection", self.bg_selection)

        self.complete_button = CompleteButtonRTM(self)
        self.complete_button.on_button_complete_pressed_set(self.cb_complete_pressed)
        self.part_swallow("complete_button", self.complete_button)

        self.hide_icons()

    def theme_changed(self, end_callback=None):
        def cb(*ignored):
            self.part_swallow("selection", self.bg_selection)
            self.part_swallow("complete_button", self.complete_button)
            if end_callback is not None:
                end_callback(self)

        self.bg_selection.theme_changed()
        self.complete_button.theme_changed()
        BaseRowRenderer.theme_changed(self, cb)

    def value_set(self, model):
        """Apply the model properties to the renderer."""
        if not model or model is self._model:
            return

        if not hasattr(model, "selected_state"):
            model.selected_state = False

        self.hide_icons()
        self._model = model
        self.show_icon()

        self.part_text_set("text", model.name)
        if model.completed:
            self.part_text_set("due", model.completed_formatted)
        else:
            self.part_text_set("due", model.due_formatted)
        self.part_text_set("list", model.list_name)

    def hide_icons(self):
        self.signal_emit("hide,icon_high", "")
        self.signal_emit("hide,icon_normal", "")
        self.signal_emit("hide,icon_low", "")
        self.signal_emit("hide,icon_completed", "")

    def show_icon(self):
        if self._model.completed:
            self.signal_emit("show,icon_completed", "")
        elif self._model.priority == "1":
            self.signal_emit("show,icon_high", "")
        elif self._model.priority == "2":
            self.signal_emit("show,icon_normal", "")
        elif self._model.priority == "3":
            self.signal_emit("show,icon_low", "")

    def cb_complete_pressed(self, *ignored):
        self.hide_icons()
        if not self._model.completed:
            self._model.complete()
            self.part_text_set("due", self._model.completed_formatted)

            #def cb_hide(*ignored):
            #    if self._model and self._model.completed:
            #        self._model.parent.children.remove(self._model)
            #ecore.timer_add(1.5, cb_hide)
        else:
            self._model.uncomplete()
            self.part_text_set("due", self._model.due_formatted)
        self.show_icon()

    @evas.decorators.del_callback
    def __on_delete(self):
        """Free internal data on delete."""
        self.bg_selection.delete()
        self.complete_button.delete()

class ResizableRowRendererWidget(GeneralRowRenderer, ResizableRowRenderer):
    """Remember The Milk Base List Item Renderer for Selected Items.

    This renderer is very similar with RowRendererWidget. The diference
    is the select animation that it starts.
    """
    row_group="list_item_rtm_resizeable"

    def __init__(self, parent, theme=None):
        GeneralRowRenderer.__init__(self, parent, theme)


class RowRendererWidget(GeneralRowRenderer):
    row_group="list_item_rtm"
    resizable_row_renderer = ResizableRowRendererWidget


# TODO more documentation
class ServiceController(BaseListController, OptionsControllerMixinEx):
    """Remember The Milk services controller.

    """
    terra_type = "Controller/Folder/Task/Apps/RTM/Service"
    row_renderer = RowRendererWidget

    def __init__(self, model, canvas, parent):
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixinEx.__init__(self)
        self.model.callback_notify = self._show_notify

    def _show_notify(self, err):
        """Popup a modal with a notify message."""
        self.parent.show_notify(err)

def row_value_set(self, v):
    self.part_text_set("text", v.name)
RowRendererFactory = manager.get_class("Widget/SimpleRowRendererFactory")
class ListsServiceController(ServiceController):
    """Remember The Milk list service controller."""
    terra_type = "Controller/Folder/Task/Apps/RTM/Service/Lists"
    row_renderer = RowRendererFactory(row_value_set, "list_item_text")


#
# Full screen task view
#
class TaskDetailsView(PluginThemeMixin, Screen):
    view_name = "task_details_view"
    plugin = "rtm"
    def __init__(self, evas, main_window):
        Screen.__init__(self, evas, "rtm/task/details", main_window, "Task details")
        self.canvas = evas

class TaskDetailsController(Controller, OptionsControllerMixinEx):
    #terra_type = "Controller/RTMTask"
    terra_type = "Controller/Folder/Task/Apps/RTM/RTMTask"

    def __init__(self, model, evas, parent):
        Controller.__init__(self, model, evas, parent)
        self.view = TaskDetailsView(evas, self.parent.view)
        OptionsControllerMixinEx.__init__(self)

        self.update_ui()

    def update_ui(self):
        self.view.part_text_set("name", self.model.name)
        self.view.part_text_set("property", "List: " + self.model.list_name)
        self.view.part_text_set("property2", "Priority: " + self.model.priority_formatted)
        self.view.part_text_set("property3", "Due date: " + self.model.due_formatted)

    def back(self):
        # TODO: refresh rtm tasks list
        self.parent.back()

    def go_home(self):
        self.parent.go_home()

    def delete(self):
        self.view.delete()
