# Canola2 Remember The Milk Plugin
# Authors: Andrey Popelo <andrey@popelo.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import logging
import webbrowser
import etk
import ecore
import evas.decorators

from terra.core.manager import Manager
from terra.core.threaded_func import ThreadedFunction
from terra.ui.throbber import EtkThrobber
from terra.ui.modal import Modal, ModalMessage

# ModalMessageConfirm is not available in older versions of Canola
try:
    from terra.ui.modal import ModalMessageConfirm
except:
    ModalMessageConfirm = None

from renderers import CheckListSingleRenderer

manager = Manager()
network = manager.get_status_notifier("Network")

ModalController = manager.get_class("Controller/Modal")
MixedListController = manager.get_class("Controller/Settings/Folder/MixedList")
UsernamePasswordModal = \
                    manager.get_class("Widget/Settings/UsernamePasswordModal")
MessageModal = manager.get_class("Widget/Settings/MessageModal")
CheckListPanel = manager.get_class("Widget/CheckListPanel")
CheckListPanelController = manager.get_class("Controller/Settings/CheckedFolder")
OptionsFolderController = manager.get_class("Controller/Options/Folder")
PanelButtonWidget = manager.get_class("Widget/PanelButton")


log = logging.getLogger("plugins.canola-rtm.options")

class ModalThrobber(Modal):
    def __init__(self, parent, message, show_cancel=True, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, "", theme, hborder, vborder)

        self.message = message
        self._init_embed()

    def _init_embed(self):
        self.embed = etk.Embed(self.evas)

        self.throbber = EtkThrobber(self.message)
        self.throbber.show()
        self.throbber.start()

        self.throbber_align = \
            etk.Alignment(0.5, 0.35, 0.0, 0.0, child=self.throbber)
        self.throbber_align.show()

        self.embed.add(self.throbber_align)
        self.embed.show()
        
        self.contents_set(self.embed.object)

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.embed.destroy()


###########################################################
# Settings
###########################################################

class RTMSettingsController(MixedListController):
    terra_type = "Controller/Settings/Folder/InternetMedia/RTM"

    def __init__(self, model, canvas, parent):
        MixedListController.__init__(self, model, canvas, parent)
        self.window = parent.window

class UserPassController(ModalController):
    terra_type = "Controller/Settings/Folder/InternetMedia/RTM/UserPass"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.parent_controller = parent
        self.model = model

        # if network connection is not available
        if not bool(network and network.status > 0.0):
            msg = "You need to be connected to the Internet."
            self.view = ModalMessage(self.parent.view, msg, hborder=120, vborder=120)
            self.view.callback_escape = self.close
            self.view.callback_clicked = self.close
            self.view.show()
            return

        # show a modal with ok button saying that browser will now open
        msg = "<small>Web browser will now open.<br>" \
              "Please, login with your username and password inside " \
              "a browser window and press \"OK\".</small>"

        # if ModalMessageConfirm is not available - use MessageModal
        if ModalMessageConfirm:
            self.view = ModalMessageConfirm(parent.view, msg, hborder=60,
                    vborder=90)

            self.view.callback_clicked_yes = self._on_ok_clicked
            self.view.callback_clicked_no = self.close
            self.view.callback_escape = self.close
        else:
            self.view = MessageModal(parent, self.model.name, msg,
                                     label_ok=" Cancel ",
                                     label_cancel="  OK  ")
            self.view.callback_ok_clicked = self.close
            self.view.callback_cancel_clicked = self._on_ok_clicked
            self.view.callback_escape = self.close

        self.view.show()

        def open_browser(session):
            # open web browser
            webbrowser.open(self.model.rclient.auth_url)

        # avoiding interface lock
        ThreadedFunction(None, open_browser, None).start()

    def close(self):
        def cb(*ignored):
            self.parent_controller.view.list.redraw_queue()
            self.back()
        self.view.hide(end_callback=cb)

    def _on_ok_clicked(self):
        self.view.delete()

        # if failed to login - exit with error message
        if not self.model.rclient.token:
            self.view = ModalMessage(
                self.parent.view, "Failed to login", hborder=120, vborder=120)
            self.view.callback_escape = self.close
            self.view.callback_clicked = self.close
            self.view.show()
            return

        def login():
            self.model.execute()

        def login_finished(exception, retval):
            if exception is None:
                self.model.callback_on_login()
                self.close()
            else:
                self.view.delete()
                self.view = ModalMessage(
                    self.parent.view, "Sorry, an unknown error has occured",
                    hborder=120, vborder=120)
                self.view.callback_escape = self.close
                self.view.callback_clicked = self.close
                self.view.show()
                log.error(exception)

        ThreadedFunction(login_finished, login).start()

        self.view = ModalThrobber(self.parent.view,
                                   "  Synchronizing...",
                                   show_cancel=False,
                                   hborder=220, vborder=120)
        self.view.show()

    def delete(self):
        self.view.delete()
        self.view = None
        self.model = None

class HomeScreenSettingsController(CheckListPanelController):
    terra_type = "Controller/Settings/Folder/InternetMedia/RTM/HomeScreen"
    header_text = "Choose which folders to show on plugin home screen:"

    def __init__(self, model, canvas, parent):
        CheckListPanelController.__init__(self, model, canvas, parent)

    # for header text
    def _setup_view(self):
        title = self.model.name
        self.view = CheckListPanel(self.parent.window, title,
                                   self.model.children, self.row_renderer, None,
                                   self.header_text)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back

    def callback_item_data(self, model):
        return (model.name, model.checked)

class ResetDBController(ModalController):
    terra_type = "Controller/Settings/Action/InternetMedia/RTM/ResetDB"
    message_confirm = "<small>This action will erase your entire local " \
                      "Remember The Milk database. Proceed anyway?</small>"
    message = "<small>Database erased</small>"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        model.callback_ready = self._finish

        # if ModalMessageConfirm is not available - use MessageModal
        if ModalMessageConfirm:
            self.view = ModalMessageConfirm(
                parent.view, self.message_confirm, hborder=60,
                vborder=120)
            self.view.callback_escape = self.close
            self.view.callback_clicked_yes = self._start
            self.view.callback_clicked_no = self.close
        else:
            self.view = MessageModal(parent, model.title,
                                     self.message_confirm,
                                     label_ok=" Cancel ",
                                     label_cancel="  OK  ")
            self.view.callback_escape = self.close
            self.view.callback_ok_clicked = self.close
            self.view.callback_cancel_clicked = self._start

        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back()
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()

    def _start(self):
        self.view.hide()
        self.model.execute()

    def _finish(self):
        self.view.delete()
        self.view = ModalMessage(
            self.parent.view, self.message, hborder=60, vborder=120)
        self.view.callback_escape = self.close
        self.view.callback_clicked = self.close
        self.view.show()


###########################################################
# Options
###########################################################

class HomeScreenOptionsController(OptionsFolderController):
    terra_type = "Controller/Options/Folder/Apps/RTM"

    def __init__(self, model, canvas, parent):
        OptionsFolderController.__init__(self, model, canvas, parent)
        model.callback_close = self.back

class RTMTaskOptionsController(MixedListController):
    terra_type = "Controller/Options/Folder/Apps/RTM/RTMTask"

    def __init__(self, model, canvas, parent):
        MixedListController.__init__(self, model, canvas, parent)
        # increase list row height
        self.view.list.scroll_contents.row_height = 80;

        # XXX "Add" button is drawn on the right, while "OK" buttons are drawn
        # on the left - move everything to one side
        self.view.button_add(label="dummy", show=False)
        if model.updating:
            self.view.button_add(label="  Save  ", func=self.callback_update)
        else:
            self.view.button_add(label="  Add  ", func=self.callback_add)

    def _cb_use(self, model):
        self.use(model)

    def callback_update(self, button):
        if self.model.callback_update:
            self.model.callback_update()
        self.back()

    def callback_add(self, button):
        if self.model.callback_add:
            self.model.callback_add()

        def cb(*ignored):
            self.parent.back()
        self.back(end_callback=cb)


# Single Check list panels
# ------------------------

class CheckListSinglePanel(CheckListPanel):
    def __init__(self, main_window, title, elements, theme=None, header_text=""):
        CheckListPanel.__init__(self, main_window, title, elements,
                                CheckListSingleRenderer,
                                theme=theme, header_text=header_text)
        self.callback_ok = None
        self._setup_buttons()

    def _setup_buttons(self):
        ok = PanelButtonWidget(self.evas, "  OK  ",
                               clicked_callback=self._cb_ok_clicked,
                               parent=self)
        self._setup_button_box(right=ok)

    def _cb_ok_clicked(self, *ignored):
        if self.callback_ok is not None:
            self.callback_ok()

class CheckListSinglePanelController(CheckListPanelController):
    header_text = ""

    def __init__(self, model, canvas, parent):
        CheckListPanelController.__init__(self, model, canvas, parent)
        def mark_selected(*args, **kargs):
            for i, m in enumerate(self.model.children):
                if m.selected:
                    self.model.current = i
                    break

            self.view.redraw_item(self.model.current)
            return False

        ecore.idler_add(mark_selected)

    def _setup_view(self):
        title = self.model.name
        self.view = CheckListSinglePanel(self.parent.window, title,
                                        self.model.children,
                                        header_text=self.header_text)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back
        self.view.callback_ok = self.cb_on_ok

    def cb_on_clicked(self, view, index):
        old_index = self.model.current
        self.model.current = index

        if old_index is not None:
            self.view.redraw_item(old_index)
        self.view.redraw_item(index)

    def cb_on_ok(self):
        self.back()

class RTMListNameController(CheckListSinglePanelController):
    header_text = "Choose the list where you want to store your task:"
    terra_type = "Controller/Options/Folder/Apps/RTM/RTMTask/RTMListName"

    def cb_on_ok(self):
        self.model.list_id = self.model.children[self.model.current].id
        self.model.list_name = self.model.children[self.model.current].name
        self.back()

class PriorityController(CheckListSinglePanelController):
    header_text = "Choose task priority:"
    terra_type = "Controller/Options/Folder/Apps/RTM/RTMTask/Priority"

    def cb_on_ok(self):
        self.model.priority = self.model.children[self.model.current].priority
        self.model.update_priority_name()
        self.back()

class SortByController(CheckListSinglePanelController):
    terra_type = "Controller/Options/Folder/Apps/RTM/RTMTasks/SortBy"

    def cb_on_ok(self):
        self.model.sort_by = \
            self.model.children[self.model.current].sort_by

        def cb(*ignored):
            self.parent.back()
        self.back(end_callback=cb)
