# -*- coding: utf-8 -*-
# Canola2 Remember The Milk Plugin
# Authors: Andrey Popelo <andrey@popelo.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import logging
import urllib2
import socket
import re
from datetime import datetime

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

from terra.core.manager import Manager
from terra.core.task import Task
from terra.core.model import ModelFolder, Model
from terra.core.threaded_model import ThreadedModelFolder
from terra.core.threaded_func import ThreadedFunction
from terra.core.plugin_prefs import PluginPrefs

from client import RemoteClient
from utils import Timezone
from localclient import LocalClient
from renderers import LabeledEntryItemRenderer, PropertyValueItemRenderer

RTM_API_KEY = "1e6e489de9374b43ba280ab9741d290c"
RTM_SECRET  = "c7197d30f722247d"

manager = Manager()
prefs   = PluginPrefs("rtm")
rclient = RemoteClient(RTM_API_KEY, RTM_SECRET, prefs.get("token"))
client  = LocalClient(manager.canola_db, prefs, rclient)

PluginDefaultIcon = manager.get_class("Icon/Plugin")
CanolaError = manager.get_class("Model/Notify/Error")
OptionsModelFolder = manager.get_class("Model/Options/Folder")
OptionsActionModel = manager.get_class("Model/Options/Action")
SettingsModelFolder = manager.get_class("Model/Settings/Folder")
SettingsActionModel = manager.get_class("Model/Settings/Action")
MixedListItemDual = \
                manager.get_class("Model/Settings/Folder/MixedList/Item/Dual")
ItemRenderer = manager.get_class("Renderer/EtkList/Item")

log  = logging.getLogger("plugins.canola-rtm.model")



class FakeCanolaDB(object):
    def __init__(self):
        self.connection = sqlite.connect(manager.canola_db.filename)
        self.connection.text_factory = str

    def execute(self, stmt, *args):
        rows = self.connection.execute(stmt, *args).fetchall()
        # Commit if doing anything but a 'select' statement
        if not stmt[0] in ('S', 's'):
            log.debug("Executing SQL: %s", stmt)
            self.connection.commit()
        return rows

class Icon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Apps/RTM"
    icon = "icon/main_item/rtm"
    plugin = "rtm"

class RTMTaskModel(Model):
#class RTMTaskModel(ModelFolder):
    """Base class for Remember The Milk task models."""
    #terra_type = "Model/RTMTask"
    terra_type = "Model/Folder/Task/Apps/RTM/RTMTask"

    def __init__(self, name="", task=None, selected=False, parent=None):
        # ensure all task properties are set before doing PARENT.__init__
        self.reset_props()
        if task:
            self.update_props(task)
        Model.__init__(self, name, parent)
        #ModelFolder.__init__(self, name, parent)

    def format_priority(self, priority):
        if priority == "N":
            return "None"
        elif priority == "1":
            return "1 – High"
        elif priority == "2":
            return "2 – Medium"
        elif priority == "3":
            return "3 – High"

    def format_date(self, date):
        # assuming today is Monday Aug 12, then dates should be shown like this:
        # Aug 11     (for yesterday)
        # 08:00 AM   (for today with time set)
        # Today      (for today without time)
        # Tuesday    (for tomorrow)
        # Wednesday
        # ...
        # Aug 19     (for next Monday)
        # 19/12/2010 (for dates not in this year)

        if not date:
            return ""

        timezone  = prefs.get("timezone")
        offset    = prefs.get("timezone_offset", 0)

        tz   = Timezone(offset, timezone)
        now  = datetime.now(Timezone(0, 'UTC'))
        time = datetime.strptime(date, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=Timezone(0, 'UTC'))
        now  = now.astimezone(tz)
        time = time.astimezone(tz)
        delta = time - now

        # Today
        if time.day == now.day and time.month == now.month and \
           time.year == now.year:
            # without time
            if time.hour == 0 and time.minute == 0:
                formatted = "Today"
            # with time
            else:
                if prefs.get("timeformat") == "0":
                    formatted = time.strftime("%I:%M %p") # AM/PM
                else:
                    formatted = time.strftime("%H:%M") # 24-hour

        # date within one week
        elif delta.days >= 0 and delta.days < 6:
            formatted = time.strftime("%A")

        # not this year
        elif time.year <> now.year:
            if prefs.get("dateformat") == "1":
                formatted = time.strftime("%m/%d/%Y") # American
            else:
                formatted = time.strftime("%d/%m/%Y") # European

        # all other dates for this year
        else:
            formatted = time.strftime("%b %d")

        return formatted

    def complete(self):
        if self.completed:
            return
        rsp = client.tasks.complete(timeline=0,
                list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id)
        self.update_props_from_rsp(rsp)

    def uncomplete(self):
        if not self.completed:
            return
        rsp = client.tasks.uncomplete(timeline=0,
                list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id)
        self.update_props_from_rsp(rsp)

    def moveTo(self, to_list_id):
        if self.list_id == to_list_id:
            return
        rsp = client.tasks.moveTo(timeline=0,
                from_list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id, to_list_id=to_list_id)
        self.update_props_from_rsp(rsp)

    def setDueDate(self, due):
        if self.due == due:
            return
        rsp = client.tasks.setDueDate(timeline=0,
                list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id, due=due, parse="1")
        self.update_props_from_rsp(rsp)

    def setName(self, name):
        if self.name == name:
            return
        rsp = client.tasks.setName(timeline=0,
                list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id, name=name)
        self.update_props_from_rsp(rsp)
        # update task name manually
        if rsp and rsp.stat == "ok":
            self.name = name

    def setPriority(self, priority):
        if self.priority == priority:
            return
        rsp = client.tasks.setPriority(timeline=0,
                list_id=self.list_id, taskseries_id=self.taskseries_id,
                task_id=self.id, priority=priority)
        self.update_props_from_rsp(rsp)

    def reset_props(self):
        self.id                  = ""
        self.taskseries_id       = ""
        self.list_id             = ""
        self.list_name           = ""
        self.due                 = ""
        self.due_formatted       = ""
        self.has_due_time        = ""
        self.added               = ""
        self.added_formatted     = ""
        self.completed           = ""
        self.completed_formatted = ""
        self.deleted             = ""
        self.priority            = "N"
        self.priority_formatted  = self.format_priority(self.priority)
        self.postponed           = ""
        self.estimate            = ""

    def update_props_from_rsp(self, rsp):
        if rsp and rsp.stat == "ok":
            task               = rsp.list.taskseries.task
            task.list_id       = rsp.list.id
            task.taskseries_id = rsp.list.taskseries.id

            # XXX: list name and task name are not updated here
            self.update_props(task)

    def update_props(self, task):
        self.id                  = task.id
        self.name                = task.name if hasattr(task, "name") else self.name
        self.taskseries_id       = task.taskseries_id if hasattr(task, "taskseries_id") else self.taskseries_id
        self.list_id             = task.list_id if hasattr(task, "list_id") else self.list_id
        self.list_name           = task.list_name if hasattr(task, "list_name") else self.list_name
        self.due                 = task.due
        self.due_formatted       = self.format_date(task.due)
        self.has_due_time        = task.has_due_time
        self.added               = task.added
        self.added_formatted     = self.format_date(task.added)
        self.completed           = task.completed
        self.completed_formatted = self.format_date(task.completed)
        self.deleted             = task.deleted
        self.priority            = task.priority
        self.priority_formatted  = self.format_priority(task.priority)
        self.postponed           = task.postponed
        self.estimate            = task.estimate

    def options_model_get(self, controller):
        return RTMTaskOptionsModelFolder(None, controller, self)

class SelectItemModel(Model):
    """Base class for models which are used as items in CheckList model"""
    def __init__(self, name, selected=False, parent=None):
        Model.__init__(self, name, parent)
        self.selected = selected

class RTMListModel(SelectItemModel):
    """Base class for Remember The Milk list models."""
    terra_type = "Model/RTMList"

    def __init__(self, name, list, selected=False, parent=None):
        SelectItemModel.__init__(self, name, selected, parent)
        self.id = list.id
        self.deleted = list.deleted
        self.locked = list.locked
        self.archived = list.archived
        self.position = list.position
        self.smart = list.smart
        self.sort_order = list.sort_order
        self.filter = list.filter


class ServiceModelFolder(ModelFolder):
    terra_type = "Model/Folder/Task/Apps/RTM/Service"
    empty_msg = "No tasks found"

    def __init__(self, name, parent):
        ModelFolder.__init__(self, name, parent)
        self.callback_notify = None

    def do_load(self):
        del self.children[:]

        try:
            self.do_search()
        except (socket.gaierror, urllib2.URLError), e:
            emsg = "Unable to connect to server.<br>" + \
                "Check your connection and try again."
            log.error(e)
            if self.callback_notify:
                self.callback_notify(CanolaError(emsg))
        except Exception, e:
            emsg = "Sorry, an unknown error has occured."
            log.error(e)
            if self.callback_notify:
                self.callback_notify(CanolaError(emsg))

    def do_search(self):
        raise NotImplementedError("must be implemented by subclasses")

    def do_unload(self):
        del self.children[:]

    @classmethod
    def sort_tasks(self, lst, by="priority"):
        def by_name(x, y):
            if x.name > y.name:
                return 1
            elif x.name == y.name:
                return 0
            else:
                return -1

        def by_priority(x, y):
            if x.priority > y.priority:
                return 1
            elif x.priority == y.priority: # sort by due
                # tasks with no due will go to the end
                if not x.due:
                    return 1
                elif x.due > y.due:
                    return 1
                elif x.due == y.due: # sort by name
                    return by_name(x, y)
                else:
                    return -1
            else:
                 return -1

        def by_due(x, y):
            # tasks with no due will go to the end
            if not x.due:
                return 1
            if x.due > y.due:
                return 1
            elif x.due == y.due: # sort by priority
                if x.priority > y.priority:
                    return 1
                elif x.priority == y.priority: # sort by name
                    return by_name(x, y)
                else:
                    return -1
            else:
                 return -1

        if by == "priority":
            lst.sort(cmp=by_priority)
        elif by == "due":
            lst.sort(cmp=by_due)
        elif by == "name":
            lst.sort(cmp=by_name)

    @classmethod
    def sort_lists(self, lst):
        def by_position(x, y):
            if x.position > y.position:
                return 1
            elif x.position == y.position: # sort by name
                if x.name > y.name:
                    return 1
                elif x.name == y.name:
                    return 0
                else:
                    return -1
            else:
                 return -1

        lst.sort(cmp=by_position)

    @classmethod
    def process_tasks_response(self, rsp, parent=None):
        tasks = []
        lst = []
        if not rsp or not rsp.tasks or not rsp.tasks.list:
            return lst
        try:
            for list in rsp.tasks.list:
                for taskseries in list.taskseries:
                    for task in taskseries.task:
                        task.name = taskseries.name
                        task.taskseries_id = taskseries.id
                        task.list_id = list.id
                        task.list_name = list.name
                        tasks.append(task)

            sort_by = prefs.get("tasks_sort_by", "priority")
            ServiceModelFolder.sort_tasks(tasks, sort_by)

            for task in tasks:
                lst.append( RTMTaskModel(task.name, task, parent=parent) )

        except Exception, e:
            log.error(e)
        return lst

    @classmethod
    def process_lists_response(self, rsp, parent=None):
        lst = []
        if not rsp or not rsp.lists or not rsp.lists.list:
            return lst
        try:
            for list in rsp.lists.list:
                lst.append( RTMListModel(list.name, list, parent=parent) )
        except Exception, e:
            log.error(e)
        return lst


##############################################################################
# Remember The Milk Remote Service Models
##############################################################################

class TasksOptionsMixin(object):
    def options_model_get(self, controller):
        #return None
        return RTMTasksOptionsModelFolder(None, controller)

class InboxModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the Inbox option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Inbox"

    def do_search(self):
        rsp = client.tasks.getList(filter='list:Inbox AND status:incomplete')
        return self.process_tasks_response(rsp, self)

class TodayModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the Today option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Today"

    def do_search(self):
        self.name = "Today,  " + datetime.now().strftime("%B %d")
        rsp = client.tasks.getList(filter='due:today AND status:incomplete')
        return self.process_tasks_response(rsp, self)

    def do_unload(self):
        del self.children[:]
        self.name = "Today"

class TomorrowModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the Tomorrow option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Tomorrow"

    def do_search(self):
        rsp = client.tasks.getList(filter='due:tomorrow AND status:incomplete')
        return self.process_tasks_response(rsp, self)

class OverdueModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the Overdue option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Overdue"

    def do_search(self):
        rsp = client.tasks.getList(filter='dueBefore:today AND status:incomplete')
        return self.process_tasks_response(rsp, self)

class AllTasksModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the All tasks option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/AllTasks"

    def do_search(self):
        rsp = client.tasks.getList(filter='status:incomplete')
        return self.process_tasks_response(rsp, self)

class ListsModelFolder(ServiceModelFolder):
    """This model implements the Lists option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Lists"

    def do_search(self):
        rsp = client.lists.getList()
        lists = self.process_lists_response(rsp)
        ServiceModelFolder.sort_lists(lists)
        result = []
        for list in lists:
            result.append( ListModelFolder(list.name, list, self) )
        return result

class ListModelFolder(ServiceModelFolder, TasksOptionsMixin):
    """This model implements the List item."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/List"

    def __init__(self, name, list, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self.list = list

    def do_search(self):
        if self.list.smart == "1":
            # XXX: not a good solution to exclude completed tasks by default
            if re.search("(status:\\s*(?i)completed|completed(Before|After|Within)?:)", self.list.filter):
                filter = self.list.filter
            else:
                filter="status:incomplete AND %s" % self.list.filter
            rsp = client.tasks.getList(filter=filter)
        else:
            rsp = client.tasks.getList(filter='list:%s AND status:incomplete' %
                                       self.list.name)
        return self.process_tasks_response(rsp, self)

class SearchModelFolder(ServiceModelFolder):
    """This model implements the Search option."""
    terra_type = "Model/Folder/Task/Apps/RTM/Service/Search"

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self.query = None

    def do_search(self):
        if self.query <> None:
            # XXX: not a good solution to exclude completed tasks by default
            if re.search("(status:\\s*(?i)completed|completed(Before|After|Within)?:)", self.query):
                filter=self.query
            else:
                filter="status:incomplete AND %s" % self.query
            rsp = client.tasks.getList(filter=filter)
            self.query = None

        return self.process_tasks_response(rsp, self)


##############################################################################
# Main Remember The Milk Model
##############################################################################

class MainModelFolder(ModelFolder, Task):
    """Main Remember The Milk Model.

    This is the main remember the milk model. It initializes all other models.
    """
    terra_type = "Model/Folder/Task/Apps/RTM"
    terra_task_type = "Task/Folder/Task/Apps/RTM"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "Remember the milk", parent)
        self.not_logged_message = None

    def options_model_get(self, controller):
        if prefs.get("user_username"):
            return HomeScreenOptionsModelFolder(None, controller)
        else:
            return None

    def do_sync(self):
        def sync():
            # create a separate connection to Canola db
            db = FakeCanolaDB()
            try:
                self.client = LocalClient(db, prefs, rclient)
                self.client.sync(client.timeline)

                # TODO: allow user to edit this value in settings
                sync_timeout = prefs.get("sync_timeout", 30) # in minutes
                rsp = self.client.time.parse('now +%d minutes' % sync_timeout)
                if rsp and rsp.stat == "ok":
                    prefs["next_sync"] = getattr(rsp.time, '$t')
                    prefs.save()
            except sqlite.OperationalError, e:
                log.error("sqlite error: " + str(e))
            except Exception, e:
                log.error(e)
            finally:
                db.connection.close()

        def sync_finished(exception, retval):
            if not self.is_loading:
                return
            if exception is not None:
                log.error(exception)
            self.inform_loaded()

        self.is_loading = True
        ThreadedFunction(sync_finished, sync).start()

    def do_load(self):
        """Initialize base remember the milk models, like Today, Smart
        lists, Search and others.
        """
        if not prefs.get("user_username"):
            self.not_logged_message = "You are not logged in.<br>"\
                "Log in on Settings > Internet media > Remember The Milk"
            return

        # request timeline from server in background
        ThreadedFunction(None, lambda: client.timeline).start()

        if prefs.get("homescreen_show_inbox", False):
            InboxModelFolder("Inbox", self)
        if prefs.get("homescreen_show_today", True):
            TodayModelFolder("Today", self)
        if prefs.get("homescreen_show_tomorrow", True):
            TomorrowModelFolder("Tomorrow", self)
        if prefs.get("homescreen_show_overdue", True):
            OverdueModelFolder("Overdue", self)
        if prefs.get("homescreen_show_alltasks", False):
            AllTasksModelFolder("All tasks", self)
        ListsModelFolder("Lists", self)
        SearchModelFolder("Search", self)

        now       = getattr(client.time.parse('now').time, '$t')
        next_sync = prefs.get("next_sync")

        if not next_sync or next_sync == "0" or next_sync < now and \
           not self.is_loading:
            self.do_sync()


##############################################################################
# Remember The Milk Option Models
##############################################################################

# Check list panel models
# ------------------------
class CheckListModelFolder(ModelFolder):
    """Base class for CheckList models."""
    title = ""

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.renderer = PropertyValueItemRenderer(text_func=self.get_title,
                value_func=self.get_value, item_click=self.on_clicked)

    def get_title(self, row):
        return self.title

    def get_value(self, row):
        return ""

    def on_clicked(self, row, list):
        self.callback_use(self)

    def do_unload(self):
        self.current = None
        ModelFolder.do_unload(self)

# Labeled entry items
# -------------------
class MixedListItemLabeledEntry(ModelFolder):
    terra_type = "Model/Settings/Folder/MixedList/Item/LabeledEntry"

    def __init__(self, parent=None, label="", text=""):
        ModelFolder.__init__(self, label, parent)
        self.renderer = LabeledEntryItemRenderer(label=label, text=text)

    def get_text(self):
        return self.renderer.get_text()

    def set_text(self, text):
        self.renderer.set_text(text)

    def do_load(self):
        pass


#
# Options for tasks listing view
#
class RTMTasksOptionsModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM/RTMTasks"
    title = "Tasks"

    def __init__(self, parent, screen_controller=None):
        OptionsModelFolder.__init__(self, parent, screen_controller)

    def do_load(self):
        SortByModelFolder(self)
        RefreshOptionsModel(self)

class SortByModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM/RTMTasks/SortBy"
    title = "Sort by..."
    sort_by = ""

    def do_load(self):
        SortByPriorityOptionsModel(self)
        SortByDueOptionsModel(self)
        SortByNameOptionsModel(self)

    def do_unload(self):
        if self.sort_by:
            prefs["tasks_sort_by"] = self.sort_by
            prefs.save()
            self.screen_controller.model.do_load()

        del self.children[:]

class SortByOptionsModel(OptionsActionModel):
    default_state = False
    sort_by = ""

    def __init__(self, parent=None):
        OptionsActionModel.__init__(self, parent)
        if prefs.get("tasks_sort_by", "priority") == self.sort_by:
            self.selected = True
        else:
            self.selected = False
class SortByPriorityOptionsModel(SortByOptionsModel):
    terra_type = "Model/Options/Action/Apps/RTM/RTMTasks/SortBy/Priority"
    name = "Priority"
    sort_by = "priority"
class SortByDueOptionsModel(SortByOptionsModel):
    terra_type = "Model/Options/Action/Apps/RTM/RTMTasks/SortBy/Due"
    name = "Due date"
    sort_by = "due"
class SortByNameOptionsModel(SortByOptionsModel):
    terra_type = "Model/Options/Action/Apps/RTM/RTMTasks/SortBy/Name"
    name = "Name"
    sort_by = "name"

class RefreshOptionsModel(OptionsActionModel):
    terra_type = "Model/Options/Action/Apps/RTM/RTMTasks/Refresh"
    name = "Refresh"

    def execute(self):
        self.parent.screen_controller.model.do_load()
        #self.parent.screen_controller.view.loaded()


class HomeScreenOptionsModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM"
    title = "Tasks"

    def __init__(self, parent, screen_controller=None):
        OptionsModelFolder.__init__(self, parent, screen_controller)
        self.callback_close = None

    def do_load(self):
        RTMTaskOptionsModelFolder(self)
        SyncOptionsModel(self)

class RTMTaskOptionsModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM/RTMTask"
    title = "Add task"

    def __init__(self, parent, screen_controller=None, task=None):
        OptionsModelFolder.__init__(self, parent, screen_controller)
        if task:
            self.updating = True
        else:
            self.updating = False
        self.task = task

    def callback_add(self):
        self.new_name     = self.name_model.get_text()
        self.new_due      = self.due_model.get_text()
        self.new_list_id  = self.list_model.get_text()
        self.new_priority = self.priority_model.get_text()

        rsp = client.tasks.add(name=self.new_name, list_id=self.new_list_id, timeline=0)

        if rsp and rsp.stat == "ok":
            self.task.update_props_from_rsp(rsp)

            if self.new_due:
                self.task.setDueDate(self.new_due)

            if self.new_priority <> "N":
                self.task.setPriority(self.new_priority)

            self.screen_controller.model.do_sync()
            self.screen_controller.model.notify_model_changed()

        # TODO: inform user if due date cannot be parsed

    def callback_update(self):
        self.new_name     = self.name_model.get_text()
        self.new_due      = self.due_model.get_text()
        self.new_list_id  = self.list_model.get_text()
        self.new_priority = self.priority_model.get_text()

        self.task.setName(self.new_name)
        self.task.setDueDate(self.new_due)
        self.task.moveTo(self.new_list_id)
        self.task.setPriority(self.new_priority)

        # TODO: inform user if due date cannot be parsed
        # TODO: start sync in bg

        # refresh everything
        self.screen_controller.update_ui()
        self.do_unload()
        self.do_load()

    def do_load(self):
        self.title = "Task properties"

        if not self.updating:
            self.task = RTMTaskModel()

        self.name_model     = MixedListItemLabeledEntry(self, "Task name:", self.task.name)
        self.due_model      = MixedListItemLabeledEntry(self, "Due date:", self.task.due_formatted)
        self.list_model     = RTMListNameModelFolder(self.task.list_id, self)
        self.priority_model = PriorityModelFolder(self.task.priority, self)

class RTMListNameModelFolder(CheckListModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM/RTMTask/RTMListName"
    title = "List"

    def __init__(self, list_id="", parent=None):
        CheckListModelFolder.__init__(self, parent)

        self.list_id = list_id or prefs.get("defaultlist")
        rsp = client.lists.getList()
        self.lists = ServiceModelFolder.process_lists_response(rsp)
        ServiceModelFolder.sort_lists(self.lists)
        self.load_data()

    def load_data(self):
        for list in self.lists:
            if (self.list_id and list.id == self.list_id) or \
               (not self.list_id and list.name == "Inbox"):
                RTMListModel(list.name, list, selected=True, parent=self)
                self.list_name = list.name
            elif list.archived == "0" and list.deleted == "0" and \
                 list.smart == "0" and list.name <> "Sent":
                RTMListModel(list.name, list, parent=self)

    def get_text(self):
        return self.list_id

    def get_value(self, row):
        return self.list_name

    def do_load(self):
        del self.children[:]
        self.load_data()

class PriorityModelFolder(CheckListModelFolder):
    terra_type = "Model/Options/Folder/Apps/RTM/RTMTask/Priority"
    title = "Priority"

    def __init__(self, priority="N", parent=None):
        CheckListModelFolder.__init__(self, parent)
        self.priority = priority
        self.load_data()

    def get_text(self):
        return self.priority

    def get_value(self, row):
        return self.priority_name

    def update_priority_name(self):
        if self.priority == "1":
            self.priority_name = self.high.name
        elif self.priority == "2":
            self.priority_name = self.medium.name
        elif self.priority == "3":
            self.priority_name = self.low.name
        elif self.priority == "N":
            self.priority_name = self.none.name

    def load_data(self):
        self.high   = PriorityModel("1 – High", priority="1", parent=self)
        self.medium = PriorityModel("2 – Medium", priority="2", parent=self)
        self.low    = PriorityModel("3 – Low", priority="3", parent=self)
        self.none   = PriorityModel("None", priority="N", parent=self)

        if self.priority == "1":
            self.high.selected = True
        elif self.priority == "2":
            self.medium.selected = True
        elif self.priority == "3":
            self.low.selected = True
        elif self.priority == "N":
            self.none.selected = True

        self.update_priority_name()

    def do_load(self):
        del self.children[:]
        self.load_data()

class PriorityModel(SelectItemModel):
    def __init__(self, name, priority="N", selected=False, parent=None):
        SelectItemModel.__init__(self, name, selected, parent)
        self.priority = priority


class SyncOptionsModel(OptionsActionModel):
    terra_type = "Model/Options/Action/Apps/RTM/RTMTasks/Sync"
    name = "Sync"

    def execute(self):
        self.screen_controller.model.do_sync()
        self.screen_controller.model.notify_model_changed()
        if self.parent.callback_close:
            self.parent.callback_close()


##############################################################################
# Remember The Milk Settings Models
##############################################################################

class RTMSettingsModelFolder(SettingsModelFolder):
    # TODO: Folder/InternetMedia or Folder/Apps maybe ?
    terra_type = "Model/Settings/Folder/InternetMedia/RTM"
    title = "Remember The Milk"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        UserPassOptionsModel(self)
        HomeScreenSettingsModelFolder(self)
        ResetDBModel(self)

class UserPassOptionsModel(MixedListItemDual):
    terra_type = "Model/Settings/Folder/InternetMedia/RTM/UserPass"
    title = "Login to Remember The Milk"

    def __init__(self, parent=None):
        MixedListItemDual.__init__(self, parent)
        # TODO: check if network is down
        self.callback_on_login = self.on_login
        self.rclient = rclient

    def get_title(self):
        username = prefs.get("user_username", None)
        if not username:
            return "Login to RTM"
        else:
            return "Logged as %s" % username

    def get_left_button_text(self):
        username = prefs.get("user_username", None)
        if not username:
            return "Log on"
        else:
            return "Log off"

    def get_right_button_text(self):
        return "Change user"

    def on_clicked(self):
        username = prefs.get("user_username", None)
        if not username:
            self.callback_use(self)

    def on_left_button_clicked(self):
        username = prefs.get("user_username", None)
        if not username:
            self.callback_use(self)
        else:
            self.logoff()

        self.callback_update(self)
        #self.callback_killall()

    def on_right_button_clicked(self):
        self.logoff()
        self.callback_use(self)
        self.callback_update(self)
        #self.callback_killall()

    def on_login(self):
        client.db.reconnect()

    def logoff(self):
        prefs["user_id"] = None
        prefs["user_username"] = None
        prefs["user_fullname"] = None
        prefs["token"] = None
        prefs["last_sync"] = 0
        prefs.save()
        client.reset_db()

    # this method is run in separate thread
    def execute(self):
        prefs["token"]         = self.rclient.token
        prefs["user_id"]       = self.rclient.user.id
        prefs["user_username"] = self.rclient.user.username
        prefs["user_fullname"] = self.rclient.user.fullname

        # fetch and save settings
        rsp = self.rclient.settings.getList()
        if rsp:
            prefs['timezone']    = rsp.settings.timezone
            prefs['dateformat']  = rsp.settings.dateformat
            prefs['timeformat']  = rsp.settings.timeformat
            prefs['defaultlist'] = rsp.settings.defaultlist
            prefs['language']    = rsp.settings.language

        prefs.save()

        # create a separate connection to Canola db
        db = FakeCanolaDB()
        try:
            self.client = LocalClient(db, prefs, self.rclient)
            self.client.reset_db()
            self.client.sync()

            # get timezone offset
            rsp = self.client.timezones.getList( prefs.get("timezone") )
            if rsp and rsp.stat == "ok":
                offset = int(rsp.timezones.timezone[0].offset)
                prefs["timezone_offset"] = offset
                prefs.save()

        except sqlite.OperationalError, e:
            log.error("sqlite error: " + str(e))
        except Exception, e:
            log.error(e)
        finally:
            db.connection.close()

class HomeScreenSettingsModelFolder(CheckListModelFolder):
    terra_type = "Model/Settings/Folder/InternetMedia/RTM/HomeScreen"
    title = "Customize home screen"

    def do_load(self):
        InboxSettingsModel(parent=self)
        TodaySettingsModel(parent=self)
        TomorrowSettingsModel(parent=self)
        OverdueSettingsModel(parent=self)
        AllTasksSettingsModel(parent=self)

    def do_unload(self):
        for folder in self.children:
            prefs[folder.pref_name] = folder.checked
        prefs.save()

        del self.children[:]

class HomeScreenItemModel(SettingsActionModel):
    pref_name = ""
    default_state = True

    def __init__(self, parent=None):
        SettingsActionModel.__init__(self, parent)
        self.checked = prefs.get(self.pref_name, self.default_state)

    def execute(self):
        self.checked = not self.checked

class InboxSettingsModel(HomeScreenItemModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/HomeScreen/Inbox"
    name = "Inbox"
    pref_name = "homescreen_show_inbox"
    default_state = False

class TodaySettingsModel(HomeScreenItemModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/HomeScreen/Today"
    name = "Today"
    pref_name = "homescreen_show_today"

class TomorrowSettingsModel(HomeScreenItemModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/HomeScreen/Tomorrow"
    name = "Tomorrow"
    pref_name = "homescreen_show_tomorrow"

class OverdueSettingsModel(HomeScreenItemModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/HomeScreen/Overdue"
    name = "Overdue"
    pref_name = "homescreen_show_overdue"

class AllTasksSettingsModel(HomeScreenItemModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/HomeScreen/AllTasks"
    name = "All tasks"
    pref_name = "homescreen_show_alltasks"
    default_state = False

class ResetDBModel(SettingsActionModel):
    terra_type = "Model/Settings/Action/InternetMedia/RTM/ResetDB"
    title = "Reset database"

    def __init__(self, parent=None):
        Model.__init__(self, self.name, parent)
        self.renderer = ItemRenderer(text_func=self.get_title,
                item_click=self.on_clicked)
        self.callback_ready = None

    def get_title(self, row):
        return self.title

    def on_clicked(self, row, list):
        self.callback_use(self)

    def execute(self):
        log.info("Forcing RTM database resetting.")

        client.reset_db()
        prefs["last_sync"] = 0
        prefs.save()

        log.info("RTM DB resetting finished.")

        if self.callback_ready:
            self.callback_ready()
