# -*- coding: utf-8 -*-
# Canola2 Remember The Milk Plugin
# Authors: Andrey Popelo <andrey@popelo.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

from datetime import datetime, timedelta
import time
import re
import logging
import pickle

from client import DottedDict
from utils import Timezone, ExpressionParser
from terra.utils.encoding import to_utf8
from terra.core.manager import Manager

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

manager = Manager()
network = manager.get_status_notifier("Network")

log = logging.getLogger("plugins.canola-rtm.localclient")

class ArgumentError(Exception):
    def __init__(self, argument):
        self.argument = argument
    def __str__(self):
        return "Required argument for the method not specified: %s" % self.argument

class CanolaDBAdaptor(object):
    """This class provides a bit simpler interface to CanolaDB.
    
    You need to subclass it, set 'table' property and adjust 'stmt_*'
    properties if needed."""

    table       = ''
    stmt_create = ''
    stmt_drop   = 'DROP TABLE IF EXISTS %s'
    stmt_select = 'SELECT * FROM %s'
    stmt_insert = 'INSERT INTO %s (%s) VALUES (%s)'
    stmt_update = 'UPDATE %s SET %s WHERE id=?'
    stmt_delete = 'DELETE FROM %s'

    def __init__(self, canola_db):
        self.db = canola_db

    def _create_table(self, table=None, stmt_create=None):
        table = table or self.table
        stmt_create = stmt_create or self.stmt_create
        self.db.execute(stmt_create % table)

    def _drop_table(self, table=None, stmt_drop=None):
        table = table or self.table
        stmt_drop = stmt_drop or self.stmt_drop
        self.db.execute(stmt_drop % table)

    def _reset_db(self, table=None, stmt_create=None, stmt_drop=None):
        self._drop_table(table, stmt_create)
        self._create_table(table, stmt_drop)

    def _select(self, where=None, table=None):
        """The 'where' parameter should be a list with two items.
          where = [stmt, values]
           string---^       ^---list
        e.g.:
          where = ["id=? AND name LIKE ?", [1, "test"]]"""
        table = table or self.table
        stmt = self.stmt_select % table
        if where:
            stmt += ' WHERE ' + where[0]
            return self.db.execute(stmt, where[1])
        else:
            return self.db.execute(stmt)

    def _insert(self, item, table=None):
        """The 'item' parameter should be a dictionary.
        e.g.:
          item = {"id":1, "name":"test"}"""
        table = table or self.table
        columns = ', '.join(item.keys())
        values = ', '.join( '?' for value in item.values() )
        return self.db.execute(self.stmt_insert % (table, columns, values),
                               item.values())

    def _update(self, id, item, table=None):
        """The 'item' parameter should be a dictionary.
        e.g.:
          item = {"id":1, "name":"test"}"""
        table = table or self.table
        pairs = ', '.join("%s=?" % key for key in item.keys())
        values = item.values() + [id]
        return self.db.execute(self.stmt_update % (table, pairs), values)

    def _delete(self, where=None, table=None):
        """The 'where' parameter should be a list with two items.
          where = [stmt, values]
           string---^       ^---list
        e.g.:
          where = ["id=? AND name LIKE ?", [1, "test"]]"""
        table = table or self.table
        stmt = self.stmt_delete % table
        if where:
            stmt += ' WHERE ' + where[0]
            return self.db.execute(stmt, where[1])
        else:
            return self.db.execute(stmt)

class LocalService(CanolaDBAdaptor):
    """Base class for all local services.
    
    LocalService implements methods of the Remember The Milk API locally.
    Specific LocalService implements methods of one RTM API categories, like
    rtm.tasks, rtm.lists, ...
    
    API method is implemented as a usual method of LocalService class. If a
    specific API method is not implemented - call will be redirected to
    RemoteClient."""

    errors = { "105" : "Service currently unavailable",
               "320" : "list_id invalid or not provided",
               "340" : "taskseries_id/task_id invalid or not provided",
               "3000": "List name provided is invalid"
             }

    def __init__(self, name, db, rclient, sync_queue, prefs, redirect=True):
        CanolaDBAdaptor.__init__(self, db)
        self.name = name
        self.rclient = rclient
        self.sync_queue = sync_queue
        self.prefs = prefs
        self.redirect = redirect

    def __getattr__(self, attr):
        if self.redirect and self.rclient:
            # method not implemented locally - redirect call to RemoteClient
            return self.rclient.__getattr__(self.name).__getattr__(attr)
        else:
            raise NotImplementedError("Method '%s' is not implemented in local \
                    '%s' service." % (attr, self.name))

    def _sync(self, timeline):
        raise NotImplementedError("must be implemented by subclasses")

    def _check_network(self):
        return bool(network and network.status > 0.0)

    def _list_from_rows(self, rows, table=None):
        items = []
        for row in rows:
            items.append( self._item_from_row(row, table) )
        return items

    def _dict_from_object(self, obj, exclude=()):
        dict = {}
        for name, value in obj.__dict__.iteritems():
            if name[0] <> '_' and name not in exclude:
                dict[name] = value
        return dict

    def _item_from_row(self, row, table=None):
        raise NotImplementedError("must be implemented by subclasses")

    def _rsp_after_select(self, items, table=None):
        raise NotImplementedError("must be implemented by subclasses")

    def _rsp_after_update(self, items, table=None):
        raise NotImplementedError("must be implemented by subclasses")

    def _rsp_error(self, code):
        rsp = {"stat":"fail", "err": {"code":code, "msg":self.errors[code]}}
        return DottedDict('ROOT', rsp)

class SyncQueue(CanolaDBAdaptor):
    table       = 'rtm_sync_queue'
    stmt_create = """CREATE TABLE IF NOT EXISTS %s
                     (
                        stamp VARCHAR PRIMARY KEY,
                        service VARCHAR,
                        method VARCHAR,
                        args VARCHAR,
                        type VARCHAR,
                        date VARCHAR
                     )"""

    def __init__(self, db, rclient, lclient):
        CanolaDBAdaptor.__init__(self, db)
        self.rclient = rclient
        self.lclient = lclient
        self.obsolete_args = {}

    def add(self, service, method, args, stamp=None, type="rclient_exec"):
        stamp = stamp or self.stamp(service, method, args)
        now = datetime.now(Timezone(0, 'UTC')).strftime("%Y-%m-%dT%H:%M:%SZ")
        args = pickle.dumps(args)
        try:
            # if queue already contains the same command - delete it. new
            # command should be added to the end
            if self.has_item(stamp):
                self._delete(["stamp=?", [stamp]])

            self._insert({"stamp":stamp, "service":service, "method":method,
                          "args":args, "type":type, "date":now})
        except sqlite.IntegrityError:
            # if stamp is not unique - just don't add this row to db == don't
            # duplicate commands that are already in queue
            pass

    def delete(self, service, method, args, stamp=None, type="rclient_exec"):
        stamp = stamp or self.stamp(service, method, args)
        self._delete(["stamp=? AND type=?", [stamp, type]])

    def stamp(self, service, method, args):
        stamp = service + method
        for key in sorted(args.keys()):
            if key == "timeline": # don't include timeline in stamp
                continue
            stamp += str(args[key])
        return stamp

    def has_item(self, stamp):
        if self._select(["stamp=?", [stamp]]):
            return True
        else:
            return False

    def arg_make_obsolete(self, key, oldvalue, newvalue):
        self.obsolete_args[key] = {}
        self.obsolete_args[key][oldvalue] = newvalue

    def arg_is_obsolete(self, key, oldvalue):
        return (key in self.obsolete_args) and (oldvalue in self.obsolete_args[key])

    def arg_get_new_value(self, key, oldvalue):
        return self.obsolete_args[key][oldvalue]

    def args_update(self, old_args, new_args):
        for key, oldvalue in old_args.iteritems():
            self.arg_make_obsolete(key, oldvalue, new_args[key])

    def args_apply_updated(self, old_args):
        for key, oldvalue in old_args.iteritems():
            if self.arg_is_obsolete(key, oldvalue):
                old_args[key] = self.arg_get_new_value(key, oldvalue)

    def execute(self, timeline):
        rsp = None
        self.obsolete_args = {}

        # Some argument values may become obsolete during sync queue execution
        # (task_id, taskseries_id, list_id, ...) but methods in sync queue will
        # try to use old values. args_update() method is an interface to
        # substitute old values with new ones on the fly during sync queue
        # execution

        for command in self._select():
            command = self._item_from_row(command)
            service = command["service"]
            method  = command["method"]
            args    = pickle.loads(command["args"])
            type    = command["type"]

            # args_update() may have been called during execution - update old
            # args with new values
            self.args_apply_updated(args)

            if type == "rclient_exec":
                args["timeline"] = timeline
                rsp = self.rclient.__getattr__(service).__getattr__(method).__call__(**args)
                # FIXME: what if server returns error ("Service currentely
                # unavailable" for example) ?
            elif type == "lclient_hook":
                args["rsp"] = rsp # pass previous rsp as a parameter
                getattr(self.lclient.__getattr__(service), method)(**args)

        self._delete()

    def _item_from_row(self, row):
        return {"stamp"  : row[0],
                "service": row[1],
                "method" : row[2],
                "args"   : row[3],
                "type"   : row[4],
                "date"   : row[5]}

class FilterExpressionParser(ExpressionParser):
    # syntax definition for the parser
    # ! order is important
    syntax = [
                # group, eg: "(...)"
                { 'start': '\\s*\(',
                  'end': '\)\\s*',
                  'type': 'group'
                },

                # propval, eg: "hasNotes:true"
                # with optional NOT modifier
                { 'start': '\\s*([Nn][Oo][Tt])?\\s*([a-z][a-zA-Z]*): *("(.+?)"|[^ ()]+)\\s*',
                  'type': 'propval',
                  'create': lambda m: (m.group(2), m.group(4) or m.group(3), m.group(1))
                },

                # operator, eg: "AND", "OR"
                { 'start': '\\s*(?i)(AND|OR)\\s*', # case insensitive
                  'type': 'operator',
                  'create': lambda m: m.group(1)
                },

                # text
                # with optional NOT modifier
                # if nothing else matched - match all characters until special
                # character found
                { 'start': '\\s*([Nn][Oo][Tt])?\\s*([ ()]?[^ ()]*)',
                  'type': 'text',
                  'create': lambda m: (m.group(2), m.group(1))
                }
             ]

class TasksService(LocalService):
    table = "rtm_tasks"
    table_lists = "rtm_lists"
    table_taskseries = "rtm_taskseries"
    supported_filters = ("name", "list", "priority", "status", "due",
                         "dueBefore", "dueAfter", "completed",
                         "completedBefore", "completedAfter", "added",
                         "addedBefore", "addedAfter", "requiresSync")

    stmt_create = """CREATE TABLE IF NOT EXISTS %s
                     (
                        id INTEGER PRIMARY KEY,
                        id_taskseries INTEGER,
                        due VARCHAR,
                        has_due_time INTEGER,
                        added VARCHAR,
                        completed VARCHAR,
                        deleted VARCHAR,
                        priority VARCHAR,
                        postponed INTEGER,
                        estimate VARCHAR,
                        requires_sync INTEGER
                     )"""

    stmt_create_taskseries = """CREATE TABLE IF NOT EXISTS %s
                     (
                        id INTEGER PRIMARY KEY,
                        id_list INTEGER,
                        created VARCHAR,
                        modified VARCHAR,
                        name VARCHAR,
                        source VARCHAR,
                        url VARCHAR,
                        location_id VARCHAR,
                        requires_sync INTEGER
                     )"""

    def __init__(self, name, db, rclient, sync_queue, prefs, time_service, redirect=True):
        LocalService.__init__(self, name, db, rclient, sync_queue, prefs, redirect)
        self.time = time_service
        self.filter_parser = FilterExpressionParser()

    def add(self, **kwargs):
        try:
            timeline = kwargs["timeline"]
            name     = kwargs["name"]
            list_id  = kwargs["list_id"] if "list_id" in kwargs else None
            parse    = kwargs["parse"] if "parse" in kwargs else "0"
        except KeyError, e:
            raise ArgumentError(e)

        # TODO: add parse support
        # FIXME: check network
        if parse == '1' and self.rclient:
            return self.rclient.tasks.add(timeline=timeline, name=name,
                    list_id=list_id, parse=parse)

        # set default list_id
        if not list_id:
            list_id = self.prefs.get("defaultlist")
            if not list_id:
                list_id = self._list_from_rows(
                                self._select(("name=?", ['Inbox']), self.table_lists),
                                self.table_lists
                                )[0]['id']

        now = getattr(self.time.parse('now').time, '$t')

        self.prefs["taskseries_id"] = self.prefs.get("taskseries_id", 0) - 1
        self.prefs.save()
        newtaskseries = { "id": str(self.prefs.get("taskseries_id")),
                          "id_list": list_id,
                          "created": now,
                          "modified": now,
                          "name": name,
                          "source": "api",
                          "url": "",
                          "location_id": "",
                          "requires_sync": "1" }

        self.prefs["task_id"] = self.prefs.get("task_id", 0) - 1
        self.prefs.save()
        newtask = { "id": str(self.prefs.get("task_id")),
                    "id_taskseries": str(self.prefs.get("taskseries_id")),
                    "due": "",
                    "has_due_time": "0",
                    "added": now,
                    "completed": "",
                    "deleted": "",
                    "priority": "N",
                    "postponed": "0",
                    "estimate": "",
                    "requires_sync": "1" }

        self._insert(newtaskseries, self.table_taskseries)
        self._insert(newtask)

        # add entry to SyncQueue
        self.sync_queue.add(self.name, "add", kwargs)
        kwargs["list_id"]       = list_id
        kwargs["taskseries_id"] = newtask["id_taskseries"]
        kwargs["task_id"]       = newtask["id"]
        self.sync_queue.add(self.name, "updateID", kwargs, type="lclient_hook")

        return self._rsp_after_update(list_id, newtaskseries, newtask)

    # this method is not supported by official RTM API
    # it is used as a hook for SyncQueue
    def updateID(self, **kwargs):
        try:
            list_id        = kwargs["list_id"]
            taskseries_id  = kwargs["taskseries_id"]
            task_id        = kwargs["task_id"]
            rsp            = kwargs["rsp"]
        except KeyError, e:
            raise ArgumentError(e)

        if rsp and rsp.stat == "ok":
            ntaskseries_id = rsp.list.taskseries.id
            ntask_id       = rsp.list.taskseries.task.id

            # FIXME: list_id, taskseries_id ? what if there are tasks
            # with the same id ?
            self._update(task_id, {"id":ntask_id, "id_taskseries":ntaskseries_id})
            self._update(taskseries_id, {"id":ntaskseries_id}, self.table_taskseries)

            old_args = { "taskseries_id": taskseries_id,
                         "task_id"      : task_id }
            new_args = { "taskseries_id": ntaskseries_id,
                         "task_id"      : ntask_id }
            self.sync_queue.args_update(old_args, new_args)

    def setDueDate(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            list_id       = kwargs["list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
            due           = kwargs["due"] if "due" in kwargs else None
            has_due_time  = kwargs["has_due_time"] if "has_due_time" in kwargs else "0"
            parse         = kwargs["parse"] if "parse" in kwargs else "0"
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        if not due:
            due = ''
        elif parse == "1":
            rsp = self.time.parse(due)
            due = getattr(rsp.time, '$t')
            if rsp.time.precision == "time":
                has_due_time = "1"

        # FIXME: list_id, taskseries_id ? what if there are tasks
        # with the same id ?
        self._update(task_id, {"due":due, "has_due_time":has_due_time, "requires_sync":"1"})

        # add entry to SyncQueue
        stamp = self.name + "setDueDate" + str(list_id) + str(task_id) + str(taskseries_id)
        self.sync_queue.add(self.name, "setDueDate", kwargs, stamp)

        # update changed properties before sending response
        tasks[0]["due"]          = due
        tasks[0]["has_due_time"] = has_due_time

        # and generate response
        return self._rsp_after_update(list_id, taskseries[0], tasks[0])

    def setName(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            list_id       = kwargs["list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
            name          = kwargs["name"]
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        # FIXME: list_id, taskseries_id ? what if there are tasks
        # with the same id ?
        self._update(taskseries_id, {"name":name, "requires_sync":"1"}, self.table_taskseries)

        # add entry to SyncQueue
        stamp = self.name + "setName" + str(list_id) + str(task_id) + str(taskseries_id)
        self.sync_queue.add(self.name, "setName", kwargs, stamp)

        # update changed properties before sending response
        taskseries[0]["name"] = name

        return self._rsp_after_update(list_id, taskseries[0], tasks[0])

    def setPriority(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            list_id       = kwargs["list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
            priority      = kwargs["priority"] if "priority" in kwargs else "N"
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        # FIXME: list_id, taskseries_id ? what if there are tasks
        # with the same id ?
        self._update(task_id, {"priority":priority, "requires_sync":"1"})

        # add entry to SyncQueue
        stamp = self.name + "setPriority" + str(list_id) + str(task_id) + str(taskseries_id)
        self.sync_queue.add(self.name, "setPriority", kwargs, stamp)

        # update changed properties before sending response
        tasks[0]["priority"] = priority

        return self._rsp_after_update(list_id, taskseries[0], tasks[0])

    def complete(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            list_id       = kwargs["list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        # FIXME: list_id, taskseries_id ? what if there are tasks
        # with the same id ?
        completed = getattr(self.time.parse('now').time, '$t')
        self._update(task_id, {"completed":completed, "requires_sync":"1"})

        # add entry to SyncQueue
        stamp = self.sync_queue.stamp(self.name, "uncomplete", kwargs)
        if self.sync_queue.has_item(stamp):
            self.sync_queue.delete(self.name, "uncomplete", kwargs)
        else:
            self.sync_queue.add(self.name, "complete", kwargs)

        # update changed properties before sending response
        tasks[0]["completed"] = completed

        return self._rsp_after_update(list_id, taskseries[0], tasks[0])

    def uncomplete(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            list_id       = kwargs["list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        # FIXME: list_id, taskseries_id ? what if there are tasks
        # with the same id ?
        self._update(task_id, {"completed":"", "requires_sync":"1"})

        # add entry to SyncQueue
        stamp = self.sync_queue.stamp(self.name, "complete", kwargs)
        if self.sync_queue.has_item(stamp):
            self.sync_queue.delete(self.name, "complete", kwargs)
        else:
            self.sync_queue.add(self.name, "uncomplete", kwargs)

        # update changed properties before sending response
        tasks[0]["completed"] = ""

        return self._rsp_after_update(list_id, taskseries[0], tasks[0])

    def moveTo(self, **kwargs):
        try:
            timeline      = kwargs["timeline"]
            from_list_id  = kwargs["from_list_id"]
            to_list_id    = kwargs["to_list_id"]
            taskseries_id = kwargs["taskseries_id"]
            task_id       = kwargs["task_id"]
        except KeyError, e:
            raise ArgumentError(e)

        tasks, taskseries, lists = self._check_before_update(from_list_id, taskseries_id, task_id)

        if not lists:
            return self._rsp_error("320")
        if not taskseries or not tasks:
            return self._rsp_error("340")

        # FIXME: from_list_id, task_id ?
        self._update(taskseries_id, {"id_list":to_list_id, "requires_sync":"1"}, self.table_taskseries)

        # add entry to SyncQueue
        self.sync_queue.add(self.name, "moveTo", kwargs)

        return self._rsp_after_update(to_list_id, taskseries[0], tasks[0])

    def getList(self, list_id=None, filter=None):
        # if filter not supported locally - bypass request to RemoteClient
        if filter:
            parsed_filter = self.filter_parser.parse(filter)

            if self._filter_supported(parsed_filter):
                tasks, taskseries, lists = self._filters_apply(parsed_filter)
            elif self.rclient:
                return self.rclient.tasks.getList(list_id=list_id, filter=filter)
            # TODO: else return error code

        else:
            where = None
            if list_id:
                where = ["id_list=?", [list_id]]

            tasks, taskseries, lists = self._select_tasks(taskseries_where=where)

        return self._rsp_after_select(tasks, taskseries, lists)

    def _check_before_update(self, list_id, taskseries_id, task_id):
        t_where  = ["id=? AND id_taskseries=?", [task_id, taskseries_id]]
        ts_where = ["id=? AND id_list=?", [taskseries_id, list_id]]
        l_where  = ["id=?", [list_id]]

        return self._select_tasks(tasks_where=t_where,
                                  taskseries_where=ts_where,
                                  lists_where=l_where)


    def _select_tasks(self, tasks_where=None, taskseries_where=None, \
            lists_where=None):

        def remove_unnecessary(list, samples, field):
            result = []
            for item in list:
                for sample in samples:
                    if sample['id'] == item[field]:
                        result.append(item)
                        break
            return result

        # select tasks
        where = tasks_where
        tasks = self._list_from_rows(self._select(where))

        # gather unique taskseries ids
        taskseries_ids = {}
        for task in tasks:
            taskseries_ids[task["id_taskseries"]] = task["id_taskseries"]

        # select corresponding taskseries
        where = ["id IN (%s)" % ', '.join('?' for id in taskseries_ids.values()),
                 taskseries_ids.values()]
        if taskseries_where:
            where[0] += " AND (%s)" % taskseries_where[0]
            where[1] += taskseries_where[1]
        taskseries = self._list_from_rows(
                self._select(where, self.table_taskseries),
                self.table_taskseries)
        if taskseries_where:
            # remove tasks which don't correspond to found taskseries
            tasks = remove_unnecessary(tasks, taskseries, "id_taskseries")

        # gather unique list ids
        list_ids = {}
        for taskserie in taskseries:
            list_ids[taskserie["id_list"]] = taskserie["id_list"]

        # select corresponding lists
        where = ["id IN (%s)" % ', '.join('?' for id in list_ids.values()),
                 list_ids.values()]
        if lists_where:
            where[0] += " AND (%s)" % lists_where[0]
            where[1] += lists_where[1]
        lists = self._list_from_rows(self._select(where, self.table_lists),
                self.table_lists)
        if lists_where:
            # remove taskseries which don't correspond to found lists
            taskseries = remove_unnecessary(taskseries, lists, "id_list")

        return (tasks, taskseries, lists)

    def _reset_db(self, table=None, stmt_create=None, stmt_drop=None):
        log.debug("Resetting tasks db")
        self._drop_table()
        self._create_table()

        # reset local id counter
        self.prefs["task_id"] = 0
        self.prefs.save()

        log.debug("Resetting taskseries db")
        self._drop_table(self.table_taskseries)
        self._create_table(self.table_taskseries, self.stmt_create_taskseries)

        # reset local id counter
        self.prefs["taskseries_id"] = 0
        self.prefs.save()

    def _sync(self, timeline):
        if not self.rclient:
            return

        log.debug('Synchronizing tasks')

        # fetch tasks, which were added/changed on server since last sync
        last_sync = self.prefs.get("last_sync", "0")
        rsp = self.rclient.tasks.getList(last_sync=last_sync)

        new_tasks = []
        new_tasks_ids = []
        new_taskseries = []
        new_taskseries_ids = []
        deleted_tasks_ids = []
        deleted_taskseries_ids = []
        for list in rsp.tasks.list:
            if hasattr(list, "taskseries"):
                for taskseries in list.taskseries:
                    taskseries.id_list = list.id
                    new_taskseries.append(taskseries)
                    new_taskseries_ids.append(taskseries.id)
                    for task in taskseries.task:
                        if not task.deleted: # don't store deleted tasks
                            task.id_taskseries = taskseries.id
                            new_tasks.append(task)
                            new_tasks_ids.append(task.id)
                        else:
                            deleted_tasks_ids.append(task.id)

            if hasattr(list, "deleted"):
                for taskseries in list.deleted.taskseries:
                    deleted_taskseries_ids.append(taskseries.id)
                    for task in taskseries.task:
                        deleted_tasks_ids.append(task.id)

        if new_taskseries_ids or deleted_taskseries_ids:
            # delete all taskseries which may already exist locally
            # + delete all taskseries which were deleted on server
            ids = new_taskseries_ids + deleted_taskseries_ids
            where = ["id IN (%s)" % ', '.join('?' for id in ids), ids]
            self._delete(where, self.table_taskseries)

        if new_tasks_ids or deleted_tasks_ids:
            # delete all tasks which may already exist locally
            # + delete all tasks which were deleted on server
            ids = new_tasks_ids + deleted_tasks_ids
            where = ["id IN (%s)" % ', '.join('?' for id in ids), ids]
            self._delete(where)

        # insert new taskseries
        for taskseries in new_taskseries:
            taskseries.requires_sync = '0'
            self._insert( self._dict_from_object(taskseries, 
                                                 ("tags", "participants",
                                                  "notes", "task", "rrule")),
                          self.table_taskseries )

        # insert new tasks
        for task in new_tasks:
            task.requires_sync = '0'
            self._insert( self._dict_from_object(task) )

        # update last_sync date and time
        self.prefs['last_sync'] = getattr(self.time.parse('now').time, '$t')
        self.prefs.save()

    def _filter_supported(self, parse_tree):
        for item in parse_tree:
            if item['type'] == 'group':
                if not self._filter_supported(item['value']):
                    return False

            elif item['type'] == 'propval':
                if item['value'][0] not in self.supported_filters:
                    return False

        return True

    def _filters_apply(self, parse_tree):
        prev = None
        operator = ""

        # tasks, taskseries, lists
        t, ts, l = [], [], []

        for item in parse_tree:
            type = item['type']
            value = item['value']

            if type == 'group':
                t, ts, l = self._filters_apply(value)

            elif type == 'propval':
                t, ts, l = self._filter_apply(value[0], value[1], value[2])

            elif type == 'text':
                t, ts, l = self._filter_apply("name", value[0], value[1])

            elif type == 'operator':
                operator = value.lower()
                continue

            if prev:
                if operator == 'and':
                    t, ts, l = self._tasks_equal((t, ts, l), prev)

                elif operator == 'or':
                    t, ts, l = self._tasks_merge((t, ts, l), prev)

                else:
                    # use 'and' by default
                    t, ts, l = self._tasks_equal((t, ts, l), prev)

            prev = (t, ts, l)

        return (t, ts, l)

    def _filter_apply(self, prop, val, modifier):
        tasks_where = None
        taskseries_where = None
        lists_where = None

        if prop == "name":
            taskseries_where = ["name LIKE ?", ["%"+val.lower()+"%"]]

        elif prop == "list":
            lists_where = ["name LIKE ?", [val.lower()]]

        elif prop == "priority":
            if val.lower() == "none":
                val = "N"
            tasks_where = ["priority=?", [val]]

        elif prop == "status":
            if val == "completed":
                tasks_where = ["completed!=''", ()]
            elif val == "incomplete":
                tasks_where = ["completed=''", []]

        elif prop == "due":
            if val.lower() == "never":
                tasks_where = ["due=''", []]
            else:
                time = self.time.parse(val).time
                precision = time.precision
                val = getattr(time, '$t')
                if precision == 'time':
                    tasks_where = ["due=?", [val]]
                elif precision == 'date':
                    next_day = datetime.strptime(val, "%Y-%m-%dT%H:%M:%SZ") + timedelta(hours=24)
                    next_day = next_day.strftime("%Y-%m-%dT%H:%M:%SZ")
                    tasks_where = ["due>=? AND due<?", [val,next_day]]

        elif prop == "dueBefore":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["due<? AND due!=?", [val,""]]

        elif prop == "dueAfter":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["due>? AND due!=?", [val,""]]

        elif prop == "completed":
            time = self.time.parse(val).time
            precision = time.precision
            val = getattr(time, '$t')
            if precision == 'time':
                tasks_where = ["completed=?", [val]]
            elif precision == 'date':
                next_day = datetime.strptime(val, "%Y-%m-%dT%H:%M:%SZ") + timedelta(hours=24)
                next_day = next_day.strftime("%Y-%m-%dT%H:%M:%SZ")
                tasks_where = ["completed>=? AND completed<?", [val,next_day]]

        elif prop == "completedBefore":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["completed<? AND completed!=?", [val,""]]

        elif prop == "completedAfter":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["completed>? AND completed!=?", [val,""]]

        elif prop == "added":
            time = self.time.parse(val).time
            precision = time.precision
            val = getattr(time, '$t')
            if precision == 'time':
                tasks_where = ["added=?", [val]]
            elif precision == 'date':
                next_day = datetime.strptime(val, "%Y-%m-%dT%H:%M:%SZ") + timedelta(hours=24)
                next_day = next_day.strftime("%Y-%m-%dT%H:%M:%SZ")
                tasks_where = ["added>=? AND added<?", [val,next_day]]

        elif prop == "addedBefore":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["added<?", [val]]

        elif prop == "addedAfter":
            val = getattr(self.time.parse(val).time, '$t')
            tasks_where = ["added>?", [val]]

        elif prop == "requiresSync":
            if val.lower() == 'true':
                val = '1'
            elif val.lower() == 'false':
                val = '0'
            tasks_where = ["requires_sync=?", [val]]

        if modifier:
            if tasks_where:
                tasks_where[0] = modifier + " " + tasks_where[0]
            if taskseries_where:
                taskseries_where[0] = modifier + " " + taskseries_where[0]
            if lists_where:
                lists_where[0] = modifier + " " + lists_where[0]

        return self._select_tasks(tasks_where, taskseries_where, lists_where)

    def _tasks_equal(self, tuple1, tuple2):
        tasks1, taskseries1, lists1 = tuple1
        tasks2, taskseries2, lists2 = tuple2
        tasks, taskseries, lists = [], [], []

        def equal(list1, list2):
            result = []
            for item1 in list1:
                for item2 in list2:
                    if item1['id'] == item2['id']:
                        result.append(item1)
                        break
            return result

        tasks = equal(tasks1, tasks2)
        taskseries = equal(taskseries1, taskseries2)
        lists = equal(lists1, lists2)

        return tasks, taskseries, lists

    def _tasks_merge(self, tuple1, tuple2):
        tasks1, taskseries1, lists1 = tuple1
        tasks2, taskseries2, lists2 = tuple2

        def merge(list1, list2):
            result = list2
            for item1 in list1:
                found = False
                for item2 in list2:
                    if item1['id'] == item2['id']:
                        found = True
                        break
                if not found:
                    result.append(item1)
            return result

        tasks = merge(tasks1, tasks2)
        taskseries = merge(taskseries1, taskseries2)
        lists = merge(lists1, lists2)

        return tasks, taskseries, lists

    def _rsp_after_select(self, tasks, taskseries, lists):
        # TODO: add tags, participants, notes
        rsp_list = []
        for list in lists:
            rsp_taskseries = []
            for taskserie in taskseries:
                if taskserie["id_list"] == list["id"]:
                    rsp_task = []
                    for task in tasks:
                        if task["id_taskseries"] == taskserie["id"]:
                            rsp_task.append(task)

                    taskserie["task"] = rsp_task
                    rsp_taskseries.append(taskserie)

            list["taskseries"] = rsp_taskseries
            rsp_list.append(list)

        rsp = { "stat":"ok", "tasks":{"list":rsp_list} }
        return DottedDict('ROOT', rsp)

    def _rsp_after_update(self, list_id, taskseries, task):
        # TODO: add tags, participants, notes
        rsp_taskseries = { "id": taskseries["id"],
                           "created": taskseries["created"],
                           "modified": taskseries["modified"],
                           "name": taskseries["name"],
                           "source": taskseries["source"],
                           "task": task
                         }
        rsp_list = { "id":list_id, "taskseries":rsp_taskseries }
        rsp = { "stat":"ok", "list":rsp_list }
        return DottedDict('ROOT', rsp)

    def _item_from_row(self, row, table=None):
        if table == self.table_taskseries:
            return { "id": str(row[0]),
                     "id_list": str(row[1]),
                     "created": str(row[2]),
                     "modified": str(row[3]),
                     "name": to_utf8(row[4]),
                     "source": str(row[5]),
                     "url": str(row[6]),
                     "location_id": str(row[7]),
                     "requires_sync": str(row[8]) }
        # XXX: i don't like this
        elif table == self.table_lists:
            return ListsService._item_from_row(row)
        else:
            return { "id": str(row[0]),
                     "id_taskseries": str(row[1]),
                     "due": str(row[2]),
                     "has_due_time": str(row[3]),
                     "added": str(row[4]),
                     "completed": str(row[5]),
                     "deleted": str(row[6]),
                     "priority": str(row[7]),
                     "postponed": str(row[8]),
                     "estimate": str(row[9]),
                     "requires_sync": str(row[10]) }

class ListsService(LocalService):
    table = "rtm_lists"

    stmt_create = """CREATE TABLE IF NOT EXISTS %s
                     (
                        id INTEGER PRIMARY KEY,
                        name VARCHAR,
                        deleted INTEGER,
                        locked INTEGER,
                        archived INTEGER,
                        position INTEGER,
                        smart INTEGER,
                        sort_order INTEGER,
                        filter VARCHAR,
                        requires_sync INTEGER
                     )"""

    def getList(self):
        return self._rsp_after_select(
                self._list_from_rows(self._select()) )

    def add(self, timeline, name, filter=None):
        # Name cannot be Inbox or Sent.
        if name == "Inbox" or name == "Sent":
            return self._rsp_error("3000")

        self.prefs["list_id"] = self.prefs.get("list_id", 0) - 1
        self.prefs.save()
        newlist = { "id": str(self.prefs.get("list_id")),
                    "name": name,
                    "deleted": "0",
                    "locked": "0",
                    "archived": "0",
                    "position": "0",
                    "smart": "0",
                    "sort_order": "0",
                    "filter": "0",
                    "requires_sync": "1" }

        if self.rclient and self._check_network():
            rsp = self.rclient.lists.add(timeline=timeline, name=name,
                                            filter=filter)
            if rsp and rsp.stat == "ok":
                newlist['id'] = rsp.list.id
                newlist["requires_sync"] = "0"
            else:
                rsp = self._rsp_after_update(newlist)
        else:
            rsp = self._rsp_after_update(newlist)

        self._insert(newlist)

        return rsp

    def archive(self, timeline, list_id):
        return self._update_property(timeline, list_id, "archived", "1",
                "archive")

    def delete(self, timeline, list_id):
        return self._update_property(timeline, list_id, "deleted", "1",
                "delete")

    def setName(self, timeline, list_id, name):
        # Name cannot be Inbox or Sent.
        if name == "Inbox" or name == "Sent":
            return self._rsp_error("3000")

        return self._update_property(timeline, list_id, "name", name, "setName")

    def unarchive(self, timeline, list_id):
        return self._update_property(timeline, list_id, "archived", "0",
                "unarchive")

    def _update_property(self, timeline, list_id, propname, propvalue, method):
        self._update(list_id, {propname:propvalue})

        # if a network connection available
        if self.rclient and self._check_network():
            # TODO: check if this works
            method = self.rclient.lists.__getattr__(method)
            # update list on server
            rsp = method(timeline=timeline, list_id=list_id)
            if rsp and rsp.stat == "ok":
                return rsp

        # if it was changed only locally
        # mark it as "requires_sync"
        self._update(list_id, {"requires_sync":"1"})

        # and generate response from local db
        rows = self._select(("id=?", [list_id]))
        return self._rsp_after_update( self._item_from_row(rows[0]) )

    def _reset_db(self, table=None, stmt_create=None, stmt_drop=None):
        log.debug("Resetting lists db")
        self._drop_table()
        self._create_table()

        # create two default lists ("Inbox" and "Sent") and mark them as
        # "requires_sync"
        newlist = { "id": "-1",
                    "name": "Sent",
                    "deleted": "0",
                    "locked": "1",
                    "archived": "0",
                    "position": "0",
                    "smart": "0",
                    "sort_order": "0",
                    "filter": "0",
                    "requires_sync": "1" }
        self._insert(newlist)

        newlist["id"] = "-2"
        newlist["name"] = "Inbox"
        self._insert(newlist)

        # reset local id counter
        self.prefs["list_id"] = -3
        self.prefs.save()

    def _sync(self, timeline):
        if not self.rclient:
            return

        log.debug('Synchronizing lists')

        # XXX: it doesn't delete anything from db for now, just adds

        # fetch data from server
        remote_rsp = self.rclient.lists.getList()
        # fetch data from local db
        local_rsp = self.getList()

        # compare remote and local lists
        for rlst in remote_rsp.lists.list:
            found = False
            for llst in local_rsp.lists.list:
                # local list doesn't have the same id as on server, but it is
                # the same list
                if rlst.name == llst.name and llst.requires_sync == "1":
                    self._update(llst.id, {"id":rlst.id, "requires_sync":"0"})
                    llst.id = rlst.id
                    llst.requires_sync = "0"

                if rlst.id == llst.id:
                    # properties of a list might have changed remotely
                    if llst.requires_sync == "0":
                        id = llst.id
                        # find which properties changed and update them
                        if rlst.name <> llst.name:
                            self._update(id, {"name":rlst.name})
                        if rlst.deleted <> llst.deleted:
                            self._update(id, {"deleted":rlst.deleted})
                        if rlst.archived <> llst.archived:
                            self._update(id, {"archived":rlst.archived})
                        if rlst.position <> llst.position:
                            self._update(id, {"position":rlst.position})
                        if rlst.smart <> llst.smart:
                            self._update(id, {"smart":rlst.smart})
                        if rlst.sort_order <> llst.sort_order:
                            self._update(id, {"sort_order":rlst.sort_order})
                        if hasattr(rlst, "filter") and \
                           rlst.filter <> llst.filter:
                            self._update(id, {"filter":rlst.filter})

                    found = True
                    break


            # list from server was not found in local db
            if not found:
                # add it to local db
                rlst.filter = "0"
                rlst.requires_sync = "0"
                self._insert( self._dict_from_object(rlst) )

        # compare local and remote lists
        for llst in local_rsp.lists.list:
            found = False
            for rlst in remote_rsp.lists.list:
                if rlst.id == llst.id:
                    # if properties of a list were changed locally
                    if llst.requires_sync == "1":
                        tml = self.rclient.timeline
                        id = llst.id

                        # find out which properties changed and update them
                        if rlst.name <> llst.name:
                            self.rclient.lists.setName(timeline=tml, list_id=id,
                                    name=llst.name)
                        if rlst.deleted <> llst.deleted:
                            self.rclient.lists.delete(timeline=tml, list_id=id)
                        if rlst.archived == "0" and llst.archived == "1":
                            self.rclient.lists.archive(timeline=tml, list_id=id)
                        if rlst.archived == "1" and llst.archived == "0":
                            self.rclient.lists.unarchive(timeline=tml,
                                    list_id=id)

                    found = True
                    break

            # list was deleted from server
            if not found and llst.requires_sync == "0":
                # delete it from local db
                rsp = self._delete(["id=?", [llst.id]])

            # list from local db was not found on server
            elif not found and llst.deleted == "0" and llst.requires_sync == "1":
                # add it to the server
                rsp = self.rclient.lists.add(timeline=timeline,
                                            name=llst.name)
                if rsp and rsp.stat == "ok":
                    # update all other properties of a list on server
                    if llst.archived == "1":
                        self.rclient.lists.archive(timeline=timeline,
                                list_id=llst.id)

                    # update local list with newely recieved id from server
                    update_id = llst.id
                    llst.id = rsp.list.id
                    llst.requires_sync = "0"
                    self._update(update_id, self._dict_from_object(llst))

    def _rsp_after_update(self, items, table=None):
        rsp = { "stat": "ok",
                "transaction": {"id":"0", "undoable":"0"},
                "list": items }
        return DottedDict('ROOT', rsp)

    def _rsp_after_select(self, items, table=None):
        rsp = { "stat":"ok", "lists":{"list":items} }
        return DottedDict('ROOT', rsp)

    @classmethod
    def _item_from_row(self, row, table=None):
        return { "id": str(row[0]),
                 "name": to_utf8(row[1]),
                 "deleted": str(row[2]),
                 "locked": str(row[3]),
                 "archived": str(row[4]),
                 "position": str(row[5]),
                 "smart": str(row[6]),
                 "sort_order": str(row[7]),
                 "filter": to_utf8(row[8]),
                 "requires_sync": str(row[9]) }

class TimeService(LocalService):
    def __init__(self, name, db, rclient, sync_queue, prefs, timezones_service, redirect=True):
        LocalService.__init__(self, name, db, rclient, sync_queue, prefs, redirect)
        self.timezones = timezones_service

    def parse(self, text, timezone=None, dateformat='1'):
        def generate_response(time, precision='date'):
            rsp = { "stat":"ok", "time":{"precision":precision,"$t":time} }
            return DottedDict('ROOT', rsp)

        text      = text.strip().lower()
        timezone  = timezone or self.prefs.get("timezone", "UTC")
        offset    = self.prefs.get("timezone_offset")
        precision = 'date'

        if offset == None:
            # get timezone offset
            rsp = self.timezones.getList(timezone)
            if rsp and rsp.stat == "ok":
                #offset = int(rsp.timezones.timezone[0].current_offset)
                offset = int(rsp.timezones.timezone[0].offset)
                self.prefs["timezone_offset"] = offset
                self.prefs.save()
            else:
                offset = 0

        tz = Timezone(offset, timezone)

        if text == "now":
            time = datetime.now(tz)
            precision = 'time'
        elif text == "today":
            time = datetime.now(tz)
            time = time.replace(hour=0, minute=0, second=0, microsecond=0)
        elif text == "tomorrow":
            time = datetime.now(tz) + timedelta(days=1)
            time = time.replace(hour=0, minute=0, second=0, microsecond=0)
        elif text == "yesterday":
            time = datetime.now(tz) - timedelta(days=1)
            time = time.replace(hour=0, minute=0, second=0, microsecond=0)
        elif self.rclient and self._check_network():
            # bypass request to RemoteClient
            return self.rclient.time.parse(text=text, timezone=timezone,
                                            dateformat=dateformat)
        else:
            time = datetime.now(tz)
            precision = 'time'

        # convert to UTC
        time = time.astimezone(Timezone(0, 'UTC'))
        time = time.strftime("%Y-%m-%dT%H:%M:%SZ")

        return generate_response(time, precision)

    def _reset_db(self, table=None, stmt_create=None, stmt_drop=None):
        self.prefs["timezone_offset"] = None
        self.prefs.save()

    def _sync(self, timeline):
        pass

class TimezonesService(LocalService):
    table = "rtm_timezones"

    stmt_create = """CREATE TABLE IF NOT EXISTS %s
                     (
                        id INTEGER PRIMARY KEY,
                        name VARCHAR,
                        dst INTEGER,
                        offset INTEGER,
                        current_offset INTEGER
                     )"""

    def getList(self, name=None):
        where = None
        if name:
            where = ["name=?", [name]]
        return self._rsp_after_select(
                self._list_from_rows(self._select(where)) )

    def _reset_db(self, table=None, stmt_create=None, stmt_drop=None):
        log.debug("Resetting timezones db")
        self._drop_table()
        self._create_table()

    def _sync(self, timeline):
        if not self.rclient:
            return

        log.debug('Synchronizing timezones')

        if not self._select():
            for timezone in self.rclient.timezones.getList().timezones.timezone:
                self._insert( self._dict_from_object(timezone) )

    def _rsp_after_select(self, items, table=None):
        if items:
            return DottedDict('ROOT', {"stat":"ok", "timezones":{"timezone": items}})
        else:
            return self._rsp_error("105")

    def _item_from_row(self, row, table=None):
        return { "id": str(row[0]),
                 "name": str(row[1]),
                 "dst": str(row[2]),
                 "offset": str(row[3]),
                 "current_offset": str(row[4]) }


class LocalClient(object):
    """A client to Remember The Milk data, which is stored locally.

    Local client provides the same interface to RTM API methods as RemoteClient,
    so it can be used instead of RemoteClient transparentely. It also uses
    RemoteClient for data synchronization and redirects a method call to
    RemoteClient in case if it's not implemented locally.

    You can access RTM API methods using dot-notation.
    Eg.
                rtm = LocalClient( ... )
                rtm.tasks.getList()
       Client --^   ^     ^-- Method
                    └- Category
    """


    def __init__(self, db, prefs, rclient=None, redirect=True):
        self.db = db
        self.prefs = prefs
        self.rclient = rclient
        self.redirect = redirect
        self.services = {}
        self._timeline = None

        sq        = SyncQueue(db, rclient, self)
        lists     = ListsService('lists', db, rclient, sq, prefs, redirect)
        timezones = TimezonesService('timezones', db, rclient, sq, prefs, redirect)
        time      = TimeService('time', db, rclient, sq, prefs, timezones, redirect)
        tasks     = TasksService('tasks', db, rclient, sq, prefs, time, redirect)

        self.sync_queue = sq
 
        self.register_service(lists)
        self.register_service(tasks)
        self.register_service(time)
        self.register_service(timezones)

    def __getattr__(self, attr):
        # if service implemented locally
        if attr in self.services:
            # transfer control to local service
            return self.services[attr]
        elif self.redirect and self.rclient and self._check_network():
            # redirect method call to RemoteClient
            return self.rclient.__getattr__(attr)
        else:
            raise NotImplementedError("Service '%s' is not supported by LocalClient." % attr)

    @property
    def timeline(self):
        if not self._timeline and self._check_network():
               rsp = self.rclient.timelines.create()
               if rsp and rsp.stat == "ok":
                   self._timeline = rsp.timeline
        return self._timeline

    def reset_db(self):
        self.prefs['last_sync'] = 0
        self.prefs.save()

        self.sync_queue._reset_db()

        for name, service in self.services.iteritems():
            service._reset_db()

    def sync(self, timeline=None):
        if not self._check_network():
            return
        timeline = timeline or self.timeline
        if not timeline:
            return

        self.sync_queue.execute(timeline)

        for name, service in self.services.iteritems():
            service._sync(timeline)

    def register_service(self, service):
        self.services[service.name] = service

    def _check_network(self):
        return bool(network and network.status > 0.0)
