# -*- coding: utf-8 -*-
# Canola2 Remember The Milk Plugin
# Authors: Andrey Popelo <andrey@popelo.com>
#
# Based on 
#   Python module for Remember The Milk API
#   (http://intellectronica.net/python-rtm) 
#   originally created by Tom Berger <tom.berger@gmail.com>
# and
#   Official Python API Kit for Remember The Milk
#   (http://repo.or.cz/w/pyrtm.git)
#   originally created by Sridhar Ratnakumar <sridhar.ratna@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import logging
import socket
from threading import Lock

from urllib import urlencode
from urllib2 import urlopen, URLError
from md5 import md5

from terra.utils.encoding import to_utf8

log = logging.getLogger("plugins.canola-rtm.client")


class Method(object):
    """Remember The Milk API method.
    
    This class represents an RTM API method. Together with Client class and
    Category class it allows to access RTM API methods using dot-notation.
    Eg.
                rtm.tasks.getList()
       Client --^   ^     ^-- Method
                    └- Category
    """

    def __init__(self, client, category_name, method_name):
        self.client = client
        self.category_name = category_name
        self.method_name = method_name

    def __call__(self, **kwargs):
        return self.client.get(
            method='rtm.%s.%s' % (self.category_name, self.method_name),
            auth_token=self.client.token,
            **kwargs)


class Category(object):
    """Remember The Milk API category.
    
    This class represents an RTM API category. Together with Client class and
    Method class it allows to access RTM API methods using dot-notation.
    Eg.
                rtm.tasks.getList()
       Client --^   ^     ^-- Method
                    └- Category
    """

    def __init__(self, client, category_name):
        self.client = client
        self.category_name = category_name

    def __getattr__(self, attr):
        return Method(self.client, self.category_name, attr)


class RemoteClient(object):
    """Remember The Milk Backend.

    This class provides an interface to manage data on Remember The Milk
    server.
    You can access RTM API methods using dot-notation.
    Eg.
                rtm = RemoteClient( ... )
                rtm.tasks.getList()
       Client --^   ^     ^-- Method
                    └- Category
    """

    URL_SERVICE_REST = "http://api.rememberthemilk.com/services/rest/"
    URL_SERVICE_AUTH = "http://www.rememberthemilk.com/services/auth/"
    TIMEOUT          = 10 # in seconds

    def __init__(self, api_key, secret, token=None):
        self.api_key    = api_key
        self.secret     = secret
        self._token     = token
        self._frob      = None
        self._user      = None
        self._request_queue = []

    def __getattr__(self, attr):
        return Category(self, attr)

    def sign(self, params):
        data = self.secret + ''.join(
            k + str(params[k]) for k in sorted(params.keys()))

        return md5(data).hexdigest()

    def fetch(self, url, **kwargs):
        """
        This method tends to be thread safe (in the meaning that results from
        the server will be returned in the same order the requests were sent),
        but generally speaking it is not completely thread safe. 
        """

        # create lock
        lock = Lock()
        lock.acquire()
        # register it in request queue
        self._request_queue.append(lock)

        try:
            # if it has a parent
            if len(self._request_queue) > 1:
                # get parent lock
                parent_lock = self._request_queue[-2]
                # sleep until parent works
                parent_lock.acquire()

            # do work

            if kwargs:
                url = url + '?' + urlencode(kwargs)

            # for Python 2.5
            tmt = socket.getdefaulttimeout()
            socket.setdefaulttimeout(self.TIMEOUT)
            rsp = urlopen(url).read()
            socket.setdefaulttimeout(tmt)

        finally:
            # deregister from request queue
            self._request_queue.pop(0)
            # and release lock
            lock.release()

        return rsp

    def get(self, **params):
        # remove keys with empty values
        empty = []
        for key in params:
            if not params[key]:
                empty.append(key)
        for key in empty:
            del params[key]

        params['api_key'] = self.api_key
        params['format'] = 'json'
        params['api_sig'] = self.sign(params)

        i = 0
        while True:
            if i > 4:
                log.error("Failed to fetch data from server after 4 retries. Bad connection?")
                return None
            i += 1
            try:
                log.debug("RTM request: %s" % str(params))
                json = self.fetch(self.URL_SERVICE_REST, **params)
                log.debug("RTM response: %s" % json)
                data = DottedDict('ROOT', eval(json, {}, {}))
                break
            except URLError, e:
                log.debug_warning("Failed to fetch data from server. URLError: %s. Retrying %d" % (str(e), i))
            except Exception, e:
                log.debug_warning("Not all data fetched from server. Error: %s. Retrying %d" % (str(e), i))

        return data.rsp

    @property
    def auth_url(self):
        self._token = None
        self._frob  = None
        self._user  = None
        params = {
            'api_key': self.api_key,
            'perms'  : 'delete',
            'frob'   : self.frob
            }
        params['api_sig'] = self.sign(params)
        url = self.URL_SERVICE_AUTH + '?' + urlencode(params)
        log.debug("RTM auth url: %s" % url)
        return url

    @property
    def token(self):
        if not self._token:
            rsp = self.get(method='rtm.auth.getToken', frob=self.frob)
            if rsp and rsp.stat == "ok":
                self._token = rsp.auth.token
                self._user  = rsp.auth.user
                log.debug("RTM token: %s" % self._token)
                log.debug("RTM user: id=%s, name=%s, fullname=%s" %
                        (self._user.id, self._user.username, self._user.fullname))
            elif rsp and rsp.stat == "fail":
                log.error("RTM error %s: %s" % (rsp.err.code, rsp.err.msg))
        return self._token

    @property
    def frob(self):
        if not self._frob:
            rsp = self.get(method='rtm.auth.getFrob')
            if rsp and rsp.stat == "ok":
                self._frob = rsp.frob
                log.debug("RTM frob: %s" % self._frob)
            elif rsp and rsp.stat == "fail":
                log.error("RTM error %s: %s" % (rsp.err.code, rsp.err.msg))
        return self._frob

    @property
    def user(self):
        if not self._user:
            token = self.token
            if token:
                rsp = self.get(method='rtm.auth.checkToken', auth_token=token)
                if rsp and rsp.stat == "ok":
                    self._user  = rsp.auth.user
                    log.debug("RTM user: id=%s, name=%s, fullname=%s" %
                            (self._user.id, self._user.username, self._user.fullname))
                elif rsp and rsp.stat == "fail":
                    log.error("RTM error %s: %s" % (rsp.err.code, rsp.err.msg))
            else:
                return None
        return self._user


class DottedDict(object):
    "Make dictionary items accessible via the object-dot notation."

    def __init__(self, name, dictionary):
        self._name = name

        if type(dictionary) is dict:
            for key, value in dictionary.items():
                if type(value) is dict:
                    value = DottedDict(key, value)
                if type(value) is str and '\\' in value:
                    # convert special backslash escaped chars
                    value = value.replace('\\/', '/')
                    value = value.replace('"', '\\"')
                    value = to_utf8( eval('u"'+value+'"') )
                elif type(value) in (list, tuple) and key != 'tag':
                    value = [DottedDict('%s_%d' % (key, i), item)
                             for i, item in self.indexed(value)]
                setattr(self, key, value)

    # It always returns just one item.
    # This allows to iterate over nested DottedDict objects without additional
    # checks.
    def next(self):
        if self._iterate is True:
            self._iterate = False
        else:
            raise StopIteration
        return self

    def __iter__(self):
        self._iterate = True
        return self

    def __repr__(self):
        children = [c for c in dir(self) if not c.startswith('_')]
        return 'DottedDict <%s> : %s\n' % (
            self._name,
            ', '.join(children))

    def indexed(self, seq):
        index = 0
        for item in seq:
            yield index, item
            index += 1
