# Canola2 IM Plugin
# Authors: Thiago Borges Abdnur <bolaum@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import pdb

import logging

from terra.core.manager import Manager
from terra.core.model import ModelStatus
from terra.core.controller import Controller
from terra.ui.window import StatusIcon
from terra.ui.base import PluginThemeMixin

from backend import Backend
from chat import ChatDialogModel
from model import SetStatusFolderModel

manager = Manager()
be = Backend()

OptionsController = manager.get_class("OptionsController")

log = logging.getLogger("plugins.canola-im.status")

class IMStatusModel(ModelStatus):
    terra_type = "Model/Status/IM"

    def __init__(self):
        ModelStatus.__init__(self, "IM")


class IMStatusIcon(PluginThemeMixin, StatusIcon):
    plugin = "im"

    def __init__(self, main_window, icon, theme=None):
        StatusIcon.__init__(self, main_window, icon, theme=None)
        self.showing = True
        self._animating = False
        self.hide_icon()

    def show_icon(self):
        if self.showing:
            return
        self._parent_widget.add_status_icon(self)
        self.show()
        self.showing = True

    def hide_icon(self):
        if not self.showing:
            return
        self.animate = False
        self._parent_widget.del_status_icon(self)
        self.hide()
        self.showing = False

    def animating_get(self):
        return self._animating

    def set_status_icon(self, status):
        if status == 'dnd':
            status = 'busy'
        self.signal_emit("status,%s" % status, "")

    def animating_set(self, val):
        if val == self._animating:
            return # State didn't change
        self._animating = val
        if val:
            self.signal_emit("state,jumping_on", "")
        else:
            self.signal_emit("state,jumping_off", "")

    animate = property(animating_get, animating_set)


class IMStatusController(Controller):
    terra_type = "Controller/Status/IM"

    queue = []
    accounts = []

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.network_status = manager.get_status_notifier("Network")
        self.options_model = SetStatusFolderModel(None, self.parent)
        self.view = IMStatusIcon(parent.view, "im")
        self.view.signal_callback_add("mouse,clicked,*", "*",
                                    self._cb_on_mouse_down)
        be.callback_connected = self._update_connection
        be.callback_new_message = self._new_message
        be.callback_message_read = self._messages_read
        be.callback_self_status_changed = self._status_changed_cb
        self.network_status.add_listener(self._network_listener_cb)
        self.view.set_status_icon(be.status)
        self._chat = None

    def _network_listener_cb(self, network):
        if network.status == 0.0:
            log.debug("Network is down")
            be.disconnect_accounts_cb()

    def _status_changed_cb(self, status):
        self.view.set_status_icon(status)

    def _new_message(self, contact):
        if contact in self.queue:
            return
        self.queue.append(contact)
        self.view.animate = True

    def _messages_read(self, contact):
        if contact not in self.queue or len(self.queue) == 0:
            return # No enqueued messages for contact
        self.queue.remove(contact)
        if len(self.queue) == 0:
            self.view.animate = False

    def _cb_on_mouse_down(self, *ignored):
        if len(self.queue) == 0:
            self.options() # Show status options
            return # No enqueued messages
        # get first contact in queue with buffered messages
        contact = self.queue[0]
        # open chat dialog
        self._chat = ChatDialogModel("Chat with %s" % contact.alias, contact)
        self.parent.show_notify(self._chat)
        if not self.queue and not self.accounts:
            self.view.hide_icon()

    def options(self, leave_callback=None):
        if not self.options_model:
            return

        def cb(controller, *ignored):
            def cb_finished(*ignored):
                self.parent.view.signal_emit("events,unblock", "")
            self.parent.use_options(controller, cb_finished)

        self.parent.view.signal_emit("events,block", "")
        oc = OptionsController(self.options_model, self.evas,
                               self.parent, self, end_callback=cb,
                               leave_callback=leave_callback)

    def _update_connection(self, account, connected):
        if connected:
            self.accounts.append(account)
        else:
            if account in self.accounts:
                self.accounts.remove(account)
                if self._chat and not self._chat.closed and \
                  account == self._chat.contact.parent:
                    self._chat.account_offline()
        if not self.queue and not self.accounts:
            self.view.hide_icon()
        else:
            self.view.show_icon()

    def delete(self):
        self.view.delete()
