# Canola2 IM Plugin
# Authors: Thiago Borges Abdnur <bolaum@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import logging
import pdb
import bisect

from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.core.task import Task

from backend import Account, Backend
from constants import status_identifiers
from db import IMDataBase

manager = Manager()
be = Backend()
db = IMDataBase()

PluginDefaultIcon = manager.get_class("Icon/Plugin")
OptionsFolderModel = manager.get_class("Model/Options/Folder")

log = logging.getLogger("plugins.canola-im.model")

class Icon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Apps/IM"
    icon = "icon/main_item/im"
    plugin = "im"

##############################################################################
# Contact model folder
##############################################################################
class ContactModelFolder(ModelFolder, Task):
    """Contact Model Folder

    Model for entries in contact list.
    """
    terra_type = "Model/Folder/Task/Apps/IM/Account/Contact"
    terra_task_type = "Task/Folder/Task/Apps/IM/Account/Contact"

    empty_msg = "Not implemented yet"

    def __init__(self, contact, parent=None):
        self.contact = contact
        self.icon = None
        self.avatar = contact.avatar
        name = contact.alias
        self.update_status()
        ModelFolder.__init__(self, name, parent)

    def __cmp__(self, other):
        c1 = self.contact
        c2 = other.contact
        def by_status():
            statuses = { 'available' : 0, 'away' : 1, 'brb' : 1, 'dnd' : 2,
                     'busy' : 2, 'xa' : 3, 'hidden' : 4, 'offline' : 4}
            if c1.status == c2.status:
                return cmp(c1.alias.lower(), c2.alias.lower())
            else:
                return cmp(statuses[c1.status], statuses[c2.status])
        if self.parent.child_order == "bystatus":
            # Order by status
            return by_status()
        else:
            # Order by name
            return cmp(c1.alias.lower(), c2.alias.lower())

    def _select_icon(self, status):
        if status == "available":
            self.icon = "available"
        elif status == "away":
            self.icon = "away"
        elif status == "xa":
            self.icon = "xaway"
        elif status in ("brb", "busy", "dnd"):
            self.icon = "busy"
        elif status in ("hidden", "offline", "error", "unknown"):
            self.icon = "offline"
        else:
            self.icon = None

    def update_status(self):
        contact = self.contact
        self._select_icon(contact.status)
        status = status_identifiers[contact.status]["str"]
        self.name = contact.alias
        self.desc = contact.status_message
        self.activatable = status != "Offline"

    def update_avatar(self):
        self.avatar = self.contact.avatar

    def do_load(self):
        pass


##############################################################################
# Account model folder
##############################################################################
class AccountModelFolder(ModelFolder, Task):
    """Account Model Folder

    Model for entries in Main IM Model.
    """
    terra_type = "Model/Folder/Task/Apps/IM/Account"
    terra_task_type = "Task/Folder/Task/Apps/IM/Account"

    empty_msg = "No contacts available"

    def __init__(self, account, parent=None):
        self.account = account
        self.name = account.username
        self.icon = account.protocol_ref.lower()
        self.avatar = None
        self.children_offline = []
        self.children = []
        self.callback_offline = None
        self.set_connection_status()
        # Recovering prefered children order from DB
        self.child_order = db.get_account_option(account, 'contacts_order')
        if not self.child_order:
            self.child_order = "byname"
        ModelFolder.__init__(self, self.name, parent)

    def set_connection_status(self):
        if self.account.is_connected():
            self.desc = 'Connected'
            self.activatable = True
        else:
            self.desc = 'Not connected'
            self.activatable = False
            if self.callback_offline:
                self.callback_offline()

    def do_load(self):
        if not self.account.contacts_fetched:
            self.is_loading = True
            self.account.callback_contacts_fetched = self.load_contacts
        else:
            self.load_contacts()

    def load_contacts(self):
        account = self.account
        for contact in account.get_contacts():
            if contact.online:
                ContactModelFolder(contact, self)
        self.account.callback_status_update = self.contact_status_updated
        self.account.callback_contact_added = self.contact_added_cb
        self.account.callback_avatar_update = self.contact_avatar_updated
        if self.is_loading:
            self.inform_loaded()
        log.debug("Showing %d contacts" % len(self.children))

    def contact_added_cb(self, contact):
        if contact.online:
            ContactModelFolder(contact, self)

    def append(self, item):
        position = bisect.insort(self.children, item)

    def contact_status_updated(self, contact):
        if self.is_loading:
            return
        child = self.child_get_by_contact(contact)
        if child:
            if not contact.online:
                self.remove(child)
            else:
                self.remove(child)
                child.update_status()
                self.append(child)
        elif contact.online:
                ContactModelFolder(contact, self)

    def contact_avatar_updated(self, contact):
        if self.is_loading:
            return
        child = self.child_get_by_contact(contact)
        if child:
            child.update_avatar()
            self.notify_model_changed()

    def child_get_by_contact(self, contact):
        for child in self.children:
            if contact == child.contact:
                return child
        return  None

    def children_sort_by_name(self):
        self.child_order = "byname"
        self.children.sort()
        db.set_account_option(self.account, 'contacts_order', self.child_order)

    def children_sort_by_status(self):
        self.child_order = "bystatus"
        self.children.sort()
        db.set_account_option(self.account, 'contacts_order', self.child_order)

    def do_unload(self):
        ModelFolder.do_unload(self)
        self.account.callback_contacts_fetched = None
        self.account.callback_status_update = None
        self.account.callback_contact_added = None
        self.account.callback_avatar_update = None

    def is_removable_child(self, child):
        return False

    def on_remove_child(self, child):
        pass



##############################################################################
# Main IM Model
##############################################################################

class MainModelFolder(ModelFolder, Task):
    """Main IM Model.

    This is the main IM model. It initializes all other models.
    """
    terra_type = "Model/Folder/Task/Apps/IM"
    terra_task_type = "Task/Folder/Task/Apps/IM"

    empty_msg = "No accounts set. Use the button below to set one."

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "IM", parent)

    def do_load(self):
        for account in be.accounts:
            AccountModelFolder(account, self)

    def _create_model_from_account(self, account):
        AccountModelFolder(account, self)

    def child_get_by_account(self, account):
        for child in self.children:
            if account == child.account:
                return child
        return None

    def is_removable_child(self, child):
        return True

    def on_remove_child(self, child):
        if child.account.is_connected():
            child.account.disconnect()
        self.children.remove(child)
        be.remove_account(child.account)
        self.notify_model_changed()


##############################################################################
# IM Options Models
##############################################################################

class ProtocolFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/AddAccount/LoginInfo"

class BonjourFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/AddAccount/BonjourInfo"

class AddAccountOptionsFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/AddAccount"
    title = "Add an account"
    header_text = "Select a network:"

    def do_load(self):
        for net in be.get_available_networks():
            if net == 'Bonjour':
                BonjourFolderModel(self).name = net
            else:
                ProtocolFolderModel(self).name = net

class SetStatusFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/Account/Options/SetStatus"
    notify_status_changed = None
    title = "Set status"

    def do_load(self):
        statuses = [ "available", "away", "dnd", "offline" ]
        for status in statuses:
            model = OptionsFolderModel(self)
            model.name = status_identifiers[status]["str"]
            model.status = status

class SetOrderFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/Account/Options/SetOrder"

    def do_load(self):
        OptionsFolderModel(self).name = "by name"
        OptionsFolderModel(self).name = "by status"

class AccountOptionsFolderModel(OptionsFolderModel):
    terra_type = "Model/Options/Folder/Apps/IM/Account/Options"
    title = "Options"

    def __init__(self, account, parent, screen_controller=None):
        OptionsFolderModel.__init__(self, parent,
            screen_controller=screen_controller)
        self.account = account

    def do_load(self):
        SetStatusFolderModel(self).name = "Set status"
        SetOrderFolderModel(self).name = "Contacts order"

