# Canola2 IM Plugin
# Authors: Thiago Borges Abdnur <bolaum@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import pdb

import logging
import os

import etk
import evas.decorators
import textwrap
import elementary
elementary.init()

from terra.core.controller import Controller
from terra.core.manager import Manager
from terra.ui.window import NotifyWindow
from terra.ui.base import PluginThemeMixin

manager = Manager()
Notify = manager.get_class("Model/Notify")

log = logging.getLogger("plugins.canola-im.chat")

class ChatDialogModel(Notify):
    terra_type = "Model/Dialog/Apps/IM/Chat"

    def __init__(self, title, contact, answer_callback=None):
        Notify.__init__(self, "Chat", title, answer_callback)
        self._changed_callbacks = []
        if contact.log:
            self._messages = list(contact.log)
        else:
            self._messages = []
        self.contact = contact
        self.my_contact = contact.parent.my_contact
        self.closed = False
        for msg in contact.get_buffered_messages():
            self.add_contact_message(msg)
        if not contact.parent.is_connected():
            self.account_offline()
        else:
            self.offline = False

    def changed_callback_add(self, func):
        self._changed_callbacks.append(func)

    def changed_callback_del(self, func):
        self._changed_callbacks.remove(func)

    def notify_model_changed(self):
        for f in self._changed_callbacks:
            f(self)

    def add_message(self, id, alias, msg):
        self._messages.append((id, alias, msg))
        self.notify_model_changed()

    def add_self_message(self, msg):
        self.add_message(self.my_contact.id, self.my_contact.alias, msg)

    def add_contact_message(self, msg):
        self.add_message(self.contact.id, self.contact.alias, msg)

    def get_messages(self):
        return self._messages

    def account_offline(self):
        self.offline = True
        self.add_message("system", '', "Account was disconnected.")

    def user_offline(self):
        self.offline = True
        self.add_message("system", '', "%s went offline." % self.contact.alias)

class ChatDialogView(PluginThemeMixin, NotifyWindow):
    plugin = "im"

    def __init__(self, main_window, title, ids, theme=None):
        NotifyWindow.__init__(self, main_window, "chat_dialog", "", theme)

        self.ids_tags = {}
        for id in ids:
            self.ids_tags[id] = "<color%d>" % len(self.ids_tags)
        self.callback_send_clicked = None
        self.callback_close_clicked = None

        self.signal_callback_add("btn_ok,clicked",
                                 "", self.cb_send_clicked)
        self.signal_callback_add("btn_cancel,clicked",
                                 "", self.cb_close_clicked)
        self.on_key_down_add(self._on_key_down)

        self.part_text_set("title", title.encode('utf-8'))

        # Set theme (there should be a better way to do it)
        themes_dir = manager.terra_config.themes_dir
        elementary.c_elementary.theme_overlay_add(
           os.path.abspath(themes_dir + "/default-im.edj"))

        # Create a box #############
        bx = elementary.Box(self)
        bx.size_hint_weight_set(1.0, 1.0)
        bx.size_hint_align_set(-1.0, -1.0)
        bx.show()

        sc = elementary.Scroller(self)
        sc.size_hint_weight_set(1.0, 1.0)
        sc.size_hint_align_set(-1.0, -1.0)
        bx.pack_end(sc)

        # Create textbox for conversation
        en = elementary.Entry(self)
        en.line_wrap_set(True)
        en.editable_set(False)
        en.entry_set("")
        en.size_hint_weight_set(1.0, 1.0)
        en.size_hint_align_set(-1.0, -1.0)
        en.show()
        en.on_resize_add(self.cb_chatbox_scroll_down)

        sc.content_set(en)
        sc.bounce_set(False, True)
        sc.policy_set(elementary.ELM_SCROLLER_POLICY_OFF,
            elementary.ELM_SCROLLER_POLICY_AUTO)
        sc.show()
        self.chatbox = bx
        self.chatbox_en = en
        self.chatbox_sc = sc

        # Create a box #############
        bx = elementary.Box(self)
        bx.size_hint_weight_set(1.0, 1.0)
        bx.size_hint_align_set(-1.0, -1.0)
        bx.show()

        sc = elementary.Scroller(self)
        sc.size_hint_weight_set(1.0, 1.0)
        sc.size_hint_align_set(-1.0, -1.0)
        bx.pack_end(sc)

        # Create textbox for text entries
        en = elementary.Entry(self)
        en.line_wrap_set(False)
        en.editable_set(True)
        en.single_line_set(True)
        en.entry_set("")
        en.size_hint_weight_set(1.0, 1.0)
        en.size_hint_align_set(-1.0, -1.0)
        en.show()
        en.on_key_down_add(self._on_key_down)

        sc.content_set(en)
        sc.bounce_set(True, False)
        sc.policy_set(elementary.ELM_SCROLLER_POLICY_OFF,
            elementary.ELM_SCROLLER_POLICY_OFF)
        sc.show()
        self.chatentry = bx
        self.chatentry_en = en
        self.chatentry_sc = sc

        self.part_swallow("chatbox", self.chatbox)
        self.part_swallow("chatentry", self.chatentry)

    def model_updated(self, model):
        fix_symbols = elementary.c_elementary.Entry_utf8_to_markup

        msgs = model.get_messages()
        text = ""
        for id, alias, msg in msgs:
            msg = fix_symbols(msg.encode("utf-8"))
            #msg = "<br>".join(textwrap.wrap((len(alias) + 2) * ' ' + msg, 80))
            #msg = msg[(len(alias) + 2):]
            alias = fix_symbols(alias.encode("utf-8"))
            if id == "system":
                text += "<colorS>%s</>" % msg
            else:
                text += "%s%s:</> %s" % (self.ids_tags[id], alias, msg)
            if text[-2:] != "\n" and text[-4:] != "<br>":
                text += "<br>"
        self.chatbox_en.entry_set(text)

    def clear_entry_box(self):
        self.chatentry_en.entry_set("")

    def cb_send_clicked(self, *ignored):
        fix_symbols = elementary.c_elementary.Entry_markup_to_utf8

        msg = self.chatentry_en.entry_get()
        msg = unicode(fix_symbols(msg), "utf-8")
        msg = msg.rstrip(" \n")
        if self.callback_send_clicked:
            self.callback_send_clicked(msg)

    def cb_close_clicked(self, *ignored):
        if self.callback_close_clicked:
            self.callback_close_clicked()

    def cb_chatbox_scroll_down(self, *ignored):
        x, y, w, h = self.chatbox_sc.region_get()
        cw, ch = self.chatbox_sc.child_size_get()
        #log.debug("x=%d y=%d w=%d h=%d" % (x, y, w, h))
        #log.debug("cw=%d ch=%d (ch - h) = %d" % (cw, ch, ch - h))
        self.chatbox_sc.region_show(0, ch - h, w, h)

    def _on_key_down(self, obj, event):
        kn = event.keyname
        if kn == "Escape":
            self.cb_close_clicked()
        elif kn == "Return":
            self.cb_send_clicked()

    @evas.decorators.focus_in_callback
    def do_on_focus(self):
        self.chatentry.focus()

    def delete(self):
        self.chatbox.delete()
        self.chatentry.delete()
        NotifyWindow.delete(self)


class ChatDialogController(Controller):
    terra_type = "Controller/Dialog/Apps/IM/Chat"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        contact = self.contact = self.model.contact

        self._setup_view()
        self.model.changed_callback_add(self._update_ui)

        contact.notify_message_received_cb = self.model.add_contact_message
        contact.notify_status_changed_cb = self.cb_status_changed

    def _setup_view(self):
        m = self.model
        ids = (m.my_contact.id, m.contact.id)
        self.view = ChatDialogView(self.parent.view, self.model.message, ids)
        self.view.callback_send_clicked = self.cb_send_clicked
        self.view.callback_close_clicked = self.cb_close_clicked
        self._old_focus = self.view.evas.focus
        self.view.focus = True
        self._update_ui(self.model)

    def _update_ui(self, model):
        self.view.model_updated(model)

    def cb_send_clicked(self, msg):
        if self.model.offline:
            return
        if msg != "":
            self.model.add_self_message(msg)
            self.contact.send_message(msg)
        self.view.clear_entry_box()

    def cb_close_clicked(self):
        self.contact.log = list(self.model.get_messages())
        self.model.closed = True
        self.contact.close_textchannel()
        self.contact.notify_status_changed_cb = None
        self.contact.notify_message_received_cb = None
        self.parent.hide_notify(self, None)

    def cb_status_changed(self, contact):
        if not contact.online:
            self.model.user_offline()

    def delete(self):
        self._old_focus.focus = True
        self.view.delete()
