# Canola2 IM Plugin
# Authors: Thiago Borges Abdnur <bolaum@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import pdb

import logging

import edje
import evas.decorators
from terra.core.manager import Manager
from terra.ui.base import PluginThemeMixin
from efl_utils.callbacks import generate_multiple_wait

from model import AddAccountOptionsFolderModel, AccountOptionsFolderModel
from chat import ChatDialogModel
from backend import Backend

manager = Manager()
be = Backend()
BaseListController = manager.get_class("Controller/Folder")
OptionsControllerMixin = manager.get_class("OptionsControllerMixin")
ResizableRowRendererWidget = manager.get_class("Widget/ResizableRowRenderer")
DeletableRowRenderer = manager.get_class("Widget/BaseDeletableRowRenderer")
ActionButton = manager.get_class("Widget/ActionButton")
WaitNotify = manager.get_class("Model/WaitNotify")
ErrorNotify = manager.get_class("Model/Notify/Error")

log = logging.getLogger("plugins.canola-im.ui")

def calc_max_expand_size(w, h, max_w, max_h):
    if h == 0:
        log.error("Trying to expand image with size 0")
        return 0, 0

    aspect = float(w) / float(h)

    if w > max_w:
        w = max_w
        h = w / aspect

    if h > max_h:
        h = max_h
        w = h * aspect

    return int(w), int(h)

class BaseDeletableRowRenderer(PluginThemeMixin, DeletableRowRenderer):
    plugin = "im"

    def __init__(self, parent, theme=None):
        DeletableRowRenderer.__init__(self, parent, theme)
        self.delete_button.delete_text_set("Remove")
        self.bg_selection = self.PluginEdjeWidget("widget/list/bg_selection")
        self.part_swallow("selection", self.bg_selection)
        self.avatar = self.evas.FilledImage()
        self.part_swallow('avatar', self.avatar)

    def value_unset(self, model):
        model.renderer = None

    def value_set(self, model):
        if self._model and self._model.renderer is self:
            self.value_unset(self._model)

        self._model = model
        self._model.renderer = self

        if model.parent.is_removable_child(model):
            self.signal_emit("delete_button,show", "")
        else:
            self.signal_emit("delete_button,hide", "")

        # XXX: faster hash lookups
        if not hasattr(model, "_deletable_select_state"):
            model._deletable_select_state = False

        if model._deletable_select_state:
            self.delete_button.state_set(ActionButton.STATE_DELETE)
        else:
            self.delete_button.state_set(ActionButton.STATE_TRASH)

        self._model = model
        self.part_text_set("text", model.name.encode('utf-8'))
        self.part_text_set("subtext", model.desc.encode('utf-8'))
        self.state_set_model(model.activatable, model.icon, True)
        if model.desc == "":
            self.signal_emit("hidden", "subtext")
        if model.avatar:
            try:
                self.avatar.file_set(model.avatar)
            except:
                self.signal_emit('avatar,hide', '')
                return
            x, y, width, height = self.part_geometry_get("avatar_draw_area")
            iw, ih = self.avatar.image_size
            w, h = calc_max_expand_size(iw, ih, width, height)
            edje.extern_object_min_size_set(self.avatar, w, h)
            edje.extern_object_max_size_set(self.avatar, w, h)
            self.signal_emit('avatar,show', '')
        else:
            self.signal_emit('avatar,hide', '')

    def state_set_model(self, state, icon, model_changed=False):
        if state:
            signal = "active"
        else:
            signal = "inactive"

        if not icon:
            self.signal_emit("hidden", "icon")
        else:
            self.signal_emit(icon + "," + signal, "icon")

        self.signal_emit(signal, "text")
        self.signal_emit(signal, "subtext")

    @evas.decorators.del_callback
    def __on_delete(self):
        """Free internal data on delete."""
        self.bg_selection.delete()
        self.avatar.delete()


class ResizableDeletableRowRenderer(BaseDeletableRowRenderer,
                                    ResizableRowRendererWidget):
    row_group="list_item_im_resizable"
    #row_group="list_item_text_resizable"

    def __init__(self, parent, theme=None):
        BaseDeletableRowRenderer.__init__(self, parent, theme)


class DeletableRowRendererWidget(BaseDeletableRowRenderer):
    row_group="list_item_im"
    resizable_row_renderer = ResizableDeletableRowRenderer


class IMListController(BaseListController, OptionsControllerMixin):
    """IM Navigation List.

    This will show all accounts that are currently set.
    """
    terra_type = "Controller/Folder/Task/Apps/IM"
    row_renderer = DeletableRowRendererWidget


    def __init__(self, model, canvas, parent):
        self.empty_msg = model.empty_msg
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)
        self.parent = parent
        self.view._list.hold_cb = self.cb_click_hold
        self.network_status = manager.get_status_notifier("Network")
        if len(self.model.children) > 0:
            self.view.bottom_message_show("Click and hold to connect", 5)

    def cb_click_hold(self, index):
        model = self.model.children[index]
        account = model.account

        def update_model(*ignore):
            model.set_connection_status()
            self.model.notify_model_changed()

        def cb_dialog(*ignore):
            if not account.is_connected():
                log.debug("Connection timed out")
                account.disconnect()
            else:
                update_model()

        def cb_success():
            account.status_disconnected_cb = \
                self.account_disconnected_cb
            self._dialog.stop()

        def cb_error(account, reason):
            log.debug("Connection error!")
            self._dialog.stop()
            self.parent.show_notify(
                ErrorNotify("Connection error: %s" % reason))
        if not account.is_connected():
            if self.network_status.status == 0.0:
                self.parent.show_notify(
                    ErrorNotify("Connection error: no internet connection " \
                        "available"))
                return
            self._dialog = WaitNotify("Connecting...", 60, cb_dialog)
            self.parent.show_notify(self._dialog)
            self._dialog.start()
            account.connect(cb_success, cb_error)
        else:
            account.disconnect(update_model)

    def cb_on_clicked(self, view, index):
        model = self.model.children[index]
        account = self.model.children[index].account
        if account.is_connected():
            BaseListController.cb_on_clicked(self, view, index)

    def account_disconnected_cb(self, account, reason):
        log.warn("Account %s:%s was disconnected!" \
                 % (account.protocol_ref, account.username))
        if self.model:
            child = self.model.child_get_by_account(account)
            if child:
                child.set_connection_status()
            self.model.notify_model_changed()

    def options_model_get(self):
        return AddAccountOptionsFolderModel(None, self)

    def delete(self):
        BaseListController.delete(self)
        OptionsControllerMixin.delete(self)
        for account in be.accounts:
            account.status_disconnected_cb = None


class AccountListController(BaseListController, OptionsControllerMixin):
    """Contact List Controller.

    This will show all contacts for the current account.
    """
    terra_type = "Controller/Folder/Task/Apps/IM/Account"
    row_renderer = DeletableRowRendererWidget
    list_group = "list"

    def __init__(self, model, canvas, parent):
        self.empty_msg = model.empty_msg
        PluginThemeMixin.__init__(self)
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)
        self.parent = parent
        self.model.callback_offline = self.offline_cb
        self._chat = None

    def cb_on_clicked(self, view, index):
        if self.animating:
            log.info("click ignored: was animating")
            return

        model = self.model.children[index]
        if hasattr(model, "activatable"):
            if not model.activatable:
                return

        if not self.parent.can_use(model):
            self.parent.show_notify(model.state_reason)
            return

        self.animating = True
        self.view.select_item(index, self.cb_on_animation_ended)
        list = self.view._list
        renderer = list.renderer_for_index(index)
        if renderer is not None:
            renderer.state_default()

        # Chat modal dialog
        contact = model.contact
        my_contact = self.model.account.my_contact
        if not contact.open_textchannel():
            return
        self._chat = ChatDialogModel("Chat with %s" % contact.alias, contact)
        self.parent.show_notify(self._chat)

    def offline_cb(self):
        if self._chat and not self._chat.closed:
            self._chat.account_offline()
        if self.model.is_loaded:
            self.back()

    def options_model_get(self):
        return AccountOptionsFolderModel(self.model.account, None, self)

    def delete(self):
        BaseListController.delete(self)
        OptionsControllerMixin.delete(self)
        self.model.callback_offline = None
        self._chat = None

class ContactListController(BaseListController, OptionsControllerMixin):
    terra_type = "Controller/Folder/Task/Apps/IM/Account/Contact"
