# Canola2 IM Plugin
# Authors: Thiago Borges Abdnur <bolaum@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import pdb

import os
import logging
import dbus
import atexit
import time
import mimetypes
import imghdr
import socket

import telepathy
from telepathy.interfaces import *
from telepathy.constants import *

from terra.core.singleton import Singleton

from constants import *
from db import IMDataBase

log = logging.getLogger("plugins.canola-im.backend")

class TextChannel(telepathy.client.Channel):
    def __init__(self, parent, channel_path):
        parent.textchannel = self # Adds self to parent's text channel attr
        self.parent = parent
        conn = parent.conn
        self.sigmatch = {}
        self.state = CHANNEL_CHAT_STATE_INACTIVE

        telepathy.client.Channel.__init__(self, conn.service_name,
                                        channel_path)
        channel = self

        def cb(*ignore):
            channel[CHANNEL].GetInterfaces(
                            reply_handler = self.interfaces_cb,
                            error_handler = self.parent.error_cb)

        channel[dbus.PROPERTIES_IFACE].Get(CHANNEL, 'Interfaces',
                                    reply_handler = self.interfaces_cb,
                                    error_handler = cb)

    def interfaces_cb (self, interfaces):
        channel = self
        self.interfaces = interfaces

        if CHANNEL_INTERFACE_MESSAGES in interfaces:
            self.sigmatch['MessageReceived'] = \
                channel[CHANNEL_INTERFACE_MESSAGES].connect_to_signal(
                    'MessageReceived', self.message_received_cb)

            # find out if we have any pending messages
            channel[dbus.PROPERTIES_IFACE].Get(CHANNEL_INTERFACE_MESSAGES,
                'PendingMessages',
                reply_handler = self.get_pending_messages,
                error_handler = self.parent.error_cb)
        else:
            self.sigmatch['Received'] = \
                channel[CHANNEL_TYPE_TEXT].connect_to_signal(
                    'Received', self.message_received_deprecated_cb)
            # find out if we have any pending messages
            channel[CHANNEL_TYPE_TEXT].ListPendingMessages(
                True,
                reply_handler = self.get_pending_messages_deprecated,
                error_handler = self.parent.error_cb)

        if CHANNEL_INTERFACE_CHAT_STATE in interfaces:
            self.sigmatch['ChatStateChanged'] = \
                channel[CHANNEL_INTERFACE_CHAT_STATE].connect_to_signal(
                    'ChatStateChanged', self.update_chat_state)

    def update_chat_state(self, handle, state):
        if self.state == state:
            return
        self.state = state
        contact = self.parent
        if state == CHANNEL_CHAT_STATE_COMPOSING:
            log.debug("User %s is writing a message" % contact.id)
        elif state == CHANNEL_CHAT_STATE_PAUSED:
            log.debug("User %s stoped writing" % contact.id)
        elif state == CHANNEL_CHAT_STATE_INACTIVE:
            log.debug("User %s is idle" % contact.id)

    def get_pending_messages(self, messages):
        for message in messages:
            self.message_received_cb (message)

    def get_pending_messages_deprecated(self, messages):
        for id, timestamp, sender, type, flags, text in messages:
            self.message_received_deprecated_cb(id, timestamp, sender,
                                                type, flags, text)

    def message_received_cb(self, message):
        channel = self

        msg_id = message[0]['pending-message-id']
        channel[CHANNEL_TYPE_TEXT].AcknowledgePendingMessages([msg_id],
            reply_handler = lambda *stuff: None,
            error_handler = self.parent.error_cb)

        for part in message[1:]:
            if 'content-type' in part.keys() \
            and part['content-type'] == 'text/plain':
                msg = part['content']
                break

        rcvd_time = time.localtime(message[0]['message-received'])
        rcvd_time = time.strftime("%I:%M:%S %p", rcvd_time)
        log.debug("Message from %s received at %s: %s" %
            (self.parent.id, rcvd_time, msg))

        self.parent.message_received(msg)

    def message_received_deprecated_cb(self, id, timestamp, sender,
                                       type, flags, text):
        rcvd_time = time.localtime(timestamp)
        rcvd_time = time.strftime("%I:%M:%S %p", rcvd_time)
        log.debug("Message from %s received at %s: %s" %
            (self.parent.id, rcvd_time, text))
        self.parent.message_received(text)

    def send_message(self, msg):
        channel = self
        interfaces = self.interfaces

        log.debug("Sending message: %s" % msg)
        if CHANNEL_INTERFACE_MESSAGES in interfaces:
            # msg must be unicode!
            new_message = [
                {}, # let the CM fill in the headers
                {
                    'content': '%s' % msg,
                    'content-type': 'text/plain',
                },
            ]
            channel[CHANNEL_INTERFACE_MESSAGES].SendMessage(new_message, 0,
                reply_handler = self.send_message_cb,
                error_handler = self.parent.error_cb)
        else:
            channel[CHANNEL_TYPE_TEXT].Send(
                CHANNEL_TEXT_MESSAGE_TYPE_NORMAL,
                msg,
                reply_handler = lambda *ignore: None,
                error_handler = self.parent.error_cb)

    def send_message_cb (self, token):
        #log.debug("Sending message with token %s" % token)
        pass

    def close(self):
        for sig in self.sigmatch.values():
            sig.remove()
        log.debug("Closing text channel")
        try:
            self[CHANNEL].Close()
        except:
            pass
        self.parent.textchannel = None


class FileTransferChannel(telepathy.client.Channel):
    def __init__(self, parent, channel_path):
        self.parent = parent
        conn = parent.conn
        self.sigmatch = {}
        channel = self

        telepathy.client.Channel.__init__(self, conn.service_name,
                                        channel_path)
        channel[dbus.PROPERTIES_IFACE].Get(CHANNEL, 'Interfaces',
                                    reply_handler = self.interfaces_cb,
                                    error_handler = self.parent.error_cb)

    def interfaces_cb (self, interfaces):
        channel = self
        self.interfaces = interfaces

        self.filename = channel[dbus.PROPERTIES_IFACE].Get(
            CHANNEL_TYPE_FILE_TRANSFER, 'Filename')
        # TODO: test if file is acceptable
        self.size = channel[dbus.PROPERTIES_IFACE].Get(
            CHANNEL_TYPE_FILE_TRANSFER, 'Size')
        self.sigmatch['FileTransferStateChanged'] = \
            channel[CHANNEL_TYPE_FILE_TRANSFER].connect_to_signal(
                'FileTransferStateChanged', self.update_state)
        self.sigmatch['TransferredBytesChanged'] = \
            channel[CHANNEL_TYPE_FILE_TRANSFER].connect_to_signal(
                'TransferredBytesChanged', self.update_transferred_bytes)
        self.sigmatch['InitialOffsetDefined'] = \
            channel[CHANNEL_TYPE_FILE_TRANSFER].connect_to_signal(
                'InitialOffsetDefined', self.update_initial_offset)

    def accept_transfer(self):
        channel = self

        log.debug('Accepting transfer of %s' % self.filename)
        self.data_read = 0
        socks = channel[dbus.PROPERTIES_IFACE].Get(
                CHANNEL_TYPE_FILE_TRANSFER, 'AvailableSocketTypes')
        if socks.has_key(SOCKET_ADDRESS_TYPE_IPV4):
            addr_type = SOCKET_ADDRESS_TYPE_IPV4
            acc_ctrl = socks[SOCKET_ADDRESS_TYPE_IPV4][0]
            addr = channel[CHANNEL_TYPE_FILE_TRANSFER].AcceptFile(
                addr_type, acc_ctrl, '', 0)
            self.socket = socket.socket()
            self.socket.connect(addr)
            self.tmp = ''

    def update_state(self, state, reason):
        log.debug("%s: %s" % (state, reason))
        if state == FILE_TRANSFER_STATE_COMPLETED:
            f = open('/tmp/' + self.filename, 'wb')
            f.write(self.tmp)
            f.close
            self.close()

    def update_transferred_bytes(self, count):
        log.debug("Transferred %d bytes" % count)
        self.tmp += self.socket.recv(count - self.data_read)
        data_read = count

    def update_initial_offset(self, offset):
        log.debug("Initial offset defined: %d" % offset)

    def close(self):
        for sig in self.sigmatch.values():
            sig.remove()
        log.debug("Closing file transfer channel")
        try:
            self[CHANNEL].Close()
        except:
            pass

class Account(object):
    def __init__(self, protocol, username, password):
        config = config_defaults[protocol].copy()
        self.protocol_ref = protocol
        self.protocol = config['protocol']
        self.manager = config['manager']
        self.options = config['options'].copy()
        self.options['account'] = username
        self.options['password'] = password

        self._check_fix_options()
        self.init_attrs()

    def init_attrs(self):
        self.callback_status_update = None
        self.callback_contacts_fetched = None
        self.callback_contact_added = None
        self.callback_avatar_update = None
        self.status_connected_cb = None
        self.status_disconnected_cb = None

        self.contacts = {}
        self.my_contact = None
        self.conn = None
        self.registered = False
        self.contacts_fetched = False
        self.sigmatch = {}
        self.interfaces = None

    def __eq__(self, other):
        if self.protocol_ref == other.protocol_ref and \
            self.options['account'] == other.options['account']:
            return True
        else :
            return False

    def __str__(self):
        return "%s(username=%s, manager=%s, protocol_ref=%s, registered=%s)" \
                % (self.__class__.__name__, self.username, self.manager,
                self.protocol_ref, self.registered)

    __repr__ = __str__

    def _check_fix_options(self):
        protocol = self.protocol_ref
        user = self.options['account']

        if protocol == 'GTalk':
            if user.count('@') > 1:
                raise Exception('Invalid username! Too many @\'s')
            elif user.count('@') == 1:
                if not user.endswith('@gmail.com'):
                    raise Exception('Invalid username! Server must be ' + \
                                    'gmail.com')
            else:
                user += '@gmail.com'
        elif protocol == 'MSN':
            if user.count('@') > 1:
                raise Exception('Invalid username! Too many @\'s')
        elif protocol == 'ICQ':
            if not user.isdigit():
                raise Exception('UIN must be numbers only')

        self.options['account'] = user

    def get_username(self):
        return self.options['account']

    username = property(get_username)

    def get_password(self):
        return self.options['password']

    password = property(get_password)

    def get_contacts(self):
        return self.contacts.values()

    def is_connected(self):
        return self.registered

    def error_cb(self, error):
        log.error("Error: %s" % error)

    def error_friendly_msg(self, reason):
        error = "Unknown error"
        if reason in Connection_Status_Reason.keys():
            error = Connection_Status_Reason[reason]
        elif isinstance(reason, dbus.DBusException):
            exception = reason.get_dbus_name().split('.')[-1]
            if exception in Connection_Status_Reason.keys():
                error = Connection_Status_Reason[exception]
        return error

    def connect(self, status_connected_cb=None,
                status_disconnected_cb=None, test=False):

        def request_error_cb(error):
            if self.status_disconnected_cb:
                self.status_disconnected_cb(self,
                    self.error_friendly_msg(error))
            self.error_cb(error)

        self.test = test
        reg = telepathy.client.ManagerRegistry()
        reg.LoadManagers()

        self.status_connected_cb = status_connected_cb
        self.status_disconnected_cb = status_disconnected_cb

        mgr = reg.GetManager(self.manager)
        try:
            mgr[CONNECTION_MANAGER].RequestConnection(
                    self.protocol,
                    self.options,
                    reply_handler = self.request_connection_cb,
                    error_handler = request_error_cb)
        except:
            pass

    def request_connection_cb (self, bus_name, object_path):
        conn = self.conn = telepathy.client.Connection(bus_name, object_path)
        self.conn_ref = (bus_name, object_path)
        self.sigmatch['StatusChanged'] = \
            self.conn[CONNECTION].connect_to_signal('StatusChanged',
                self.status_changed_cb)
        try:
            self.conn[CONNECTION].Connect()
        except:
            pass

    def disconnect(self, status_disconnected_cb=None):
        self.status_disconnected_cb = status_disconnected_cb

        try:
            self.conn[CONNECTION].Disconnect()
        except:
            pass

    def status_changed_cb(self, state, reason):
        conn = self.conn
        be = Backend()

        if state == CONNECTION_STATUS_DISCONNECTED:
            log.debug("Account %s disconnected!" % self.username)

            be.remove_connection(self)
            for sig in self.sigmatch.values():
                sig.remove()
            self.registered = False
            self.contacts_fetched = False
            self.conn = None
            self.conn_ref = None
            self.signals = None

            if self.status_disconnected_cb:
                self.status_disconnected_cb(self,
                    self.error_friendly_msg(reason))

            self.status_connected_cb = None
            self.status_disconnected_cb = None
            self.callback_status_update = None
            self.callback_contacts_fetched = None
            self.callback_contact_added = None
            self.callback_avatar_update = None
            self.contacts = {}
            self.interfaces = None

            return

        elif state != CONNECTION_STATUS_CONNECTED:
            log.debug("Unhandled connection status: %d => %s" \
                      % (state, reason))
            return

        log.debug("Account %s connected!" % self.username)
        be.store_connection(self)
        self.registered = True

        if self.status_connected_cb:
            self.status_connected_cb()
            self.status_connected_cb = None

        if self.test:
            return

        conn[CONNECTION].GetInterfaces(reply_handler = self.get_interfaces_cb,
                                       error_handler = self.error_cb)

    def request_self_contact(self):
        conn = self.conn

        # Getting self handle
        my_handle = conn[dbus.PROPERTIES_IFACE].Get(
                                    CONNECTION, "SelfHandle")

        def get_self_attributes(attributes):
            self.my_contact = Contact(None,
                my_handle, attributes[my_handle])
            self.my_contact.parent = self
            self.my_contact.conn = conn
            log.debug('Self contact requested: %s' % self.my_contact)

        # Requesting self contact attributes
        conn[CONNECTION_INTERFACE_CONTACTS].GetContactAttributes(
                [my_handle], [
                    CONNECTION_INTERFACE_ALIASING,
                    CONNECTION_INTERFACE_SIMPLE_PRESENCE,
                ],
                True,
                reply_handler = get_self_attributes,
                error_handler = self.error_cb)

    def request_contact_list(self, *groups):
        conn = self.conn

        class ensure_channel_cb (object):
            def __init__ (self, parent, group):
                self.parent = parent
                self.group = group

            def __call__ (self, yours, path, properties):
                log.debug("got channel for %s -> %s, yours = %s" % (
                    self.group, path, yours))

                channel = telepathy.client.Channel(conn.service_name, path)
                self.channel = channel

                channel[dbus.PROPERTIES_IFACE].Get(CHANNEL_INTERFACE_GROUP,
                                         'Members',
                                         reply_handler = self.members_cb,
                                         error_handler = self.parent.error_cb)

            def members_cb (self, handles):
                conn[CONNECTION_INTERFACE_CONTACTS].GetContactAttributes(
                    handles, [
                        CONNECTION_INTERFACE_ALIASING,
                        CONNECTION_INTERFACE_SIMPLE_PRESENCE
                    ],
                    True,
                    reply_handler = self.get_contact_attributes_cb,
                    error_handler = self.parent.error_cb)

            def get_contact_attributes_cb (self, attributes):
                # Create members Contact objects
                for member, attr in attributes.iteritems():
                    Contact(self.parent, member, attr)
                self.parent.contacts_fetched_cb()

        for group in groups:
            log.debug("Ensuring channel to %s..." % group)
            conn[CONNECTION_INTERFACE_REQUESTS].EnsureChannel({
                CHANNEL + '.ChannelType'     : CHANNEL_TYPE_CONTACT_LIST,
                CHANNEL + '.TargetHandleType': HANDLE_TYPE_LIST,
                CHANNEL + '.TargetID'        : group,
                },
                reply_handler = ensure_channel_cb(self, group),
                error_handler = self.error_cb)

    def request_contact_list_deprecated(self, *groups):
        conn = self.conn

        channels = conn[CONNECTION].ListChannels()
        for path, channel_type, handle_type, handle in channels:
            if channel_type == CHANNEL_TYPE_CONTACT_LIST and \
               handle_type == HANDLE_TYPE_LIST:
                channel = telepathy.client.Channel(conn.service_name, path)
                try:
                    id = channel[dbus.PROPERTIES_IFACE].Get(
                        CHANNEL, 'TargetID')
                except:
                    id = path.split('/')[-1]
                if id not in groups:
                    continue
                # Get self handle
                my_handle = conn[dbus.PROPERTIES_IFACE].Get(
                                CONNECTION, "SelfHandle")
                # Get members handles
                try:
                    members = channel[dbus.PROPERTIES_IFACE].Get(
                        CHANNEL_INTERFACE_GROUP, 'Members')
                except:
                    members = \
                        channel[CHANNEL_INTERFACE_GROUP].GetAllMembers()[0]
                # Get members IDs
                ids = conn[CONNECTION].InspectHandles(
                    HANDLE_TYPE_CONTACT, [my_handle] + members)
                # Create self Contact object
                self.my_contact = Contact(None, my_handle, id=ids.pop(0))
                self.my_contact.parent = self
                self.my_contact.conn = conn
                # Create members Contact objects
                for member in members:
                    Contact(self, member, id=ids.pop(0))
                self.contacts_fetched_cb()

    def request_aliases(self):
        conn = self.conn

        def cb(aliases):
            self.my_contact.alias = aliases.pop(0)
            for member in sorted(self.contacts.keys()):
                self.contacts[member].alias = aliases.pop(0)

        conn[CONNECTION_INTERFACE_ALIASING].RequestAliases(
            [self.my_contact._handle] + sorted(self.contacts.keys()),
            reply_handler = cb,
            error_handler = self.error_cb)

    def update_aliases_cb(self, aliases):
        for member, alias in aliases:
            if self.contacts.has_key(member):
                self.contacts[member].alias = alias
                if self.callback_status_update:
                    self.callback_status_update(self.contacts[member])

    def request_presences(self):
        conn = self.conn
        interfaces = self.interfaces

        if CONNECTION_INTERFACE_SIMPLE_PRESENCE in interfaces:
            conn[CONNECTION_INTERFACE_SIMPLE_PRESENCE].GetPresences(
                [self.my_contact._handle] + self.contacts.keys(),
                reply_handler = self.update_presences_cb,
                error_handler = self.error_cb)
        elif CONNECTION_INTERFACE_PRESENCE in interfaces:
            conn[CONNECTION_INTERFACE_PRESENCE].GetPresence(
                [self.my_contact._handle] + self.contacts.keys(),
                reply_handler = self.update_presences_deprecated_cb,
                error_handler = self.error_cb)

    def update_presences_cb(self, members):
        for member, status in members.iteritems():
            if self.contacts.has_key(member):
                self.contacts[member].presence = status
                if self.callback_status_update:
                    self.callback_status_update(self.contacts[member])
            elif member == self.my_contact._handle:
                self.my_contact.presence = status

    def update_presences_deprecated_cb(self, members):
        members_conv = {}
        for member, presence in members.iteritems():
            status_str = presence[1].keys()[0]
            if presence[1].values()[0].has_key('message'):
                status_msg = presence[1].values()[0]['message']
            else:
                status_msg = ''
            if status_identifiers.has_key(status_str):
                status_type = status_identifiers[status_str]
            else:
                status_type = CONNECTION_PRESENCE_TYPE_UNKNOWN
            members_conv[member] = [status_type, status_str, status_msg]
        self.update_presences_cb(members_conv)

    def request_avatar_token(self, contact):
        be = Backend()
        conn = self.conn

        def got_tokens(tokens):
            if self.contacts.has_key(contact):
                c = self.contacts[contact]
                token = tokens[contact]
                if token:
                    self.update_avatar_cb(contact, token)
                else:
                    token, path = be.get_cached_avatar_by_user(c)
                    c.avatar_token = token
                    c.avatar = path
                    if self.callback_avatar_update:
                        self.callback_avatar_update(c)

        conn[CONNECTION_INTERFACE_AVATARS].GetKnownAvatarTokens(
            [contact],
            reply_handler = got_tokens,
            error_handler = self.error_cb)

    def request_avatars(self, contacts):
        conn = self.conn
        interfaces = self.interfaces

        if CONNECTION_INTERFACE_AVATARS in interfaces:
            conn[CONNECTION_INTERFACE_AVATARS].RequestAvatars(
                contacts)

    def update_avatar_cb(self, contact, token):
        be = Backend()

        if self.contacts.has_key(contact):
            c = self.contacts[contact]
            if token == c.avatar_token:
                return
            path = be.get_cached_avatar_by_token(c, token)
            if path:
                c.avatar_token = token
                c.avatar = path
                if self.callback_avatar_update:
                    self.callback_avatar_update(c)
            else:
                self.request_avatars([contact])

    def retrieved_avatar_cb(self, contact, token, avatar, type):
        be = Backend()

        if self.contacts.has_key(contact):
            c = self.contacts[contact]
            data = ''.join(chr(c) for c in avatar)
            path = be.cache_avatar_file(c, token, data, type)
            c.avatar_token = token
            c.avatar = path
            if self.callback_avatar_update:
                self.callback_avatar_update(c)

    def new_channels_cb (self, channels):
        for channel, props in channels:
            if props[CHANNEL + '.ChannelType'] == CHANNEL_TYPE_TEXT:
                log.debug('New chat from %s' % props[CHANNEL + '.TargetID'])
                # let's hook up to this channel
                contact = self.contacts[props[CHANNEL + '.TargetHandle']]
                if not contact.textchannel:
                    TextChannel(contact, channel)
                else:
                    log.debug('Text channel already created!')
            elif props[CHANNEL + '.ChannelType'] == CHANNEL_TYPE_FILE_TRANSFER:
                target_id = props[CHANNEL + '.TargetID']
                log.debug("A file transfer was requested by %s", target_id)
                ## Not stable yet ##
                '''
                chan = FileTransferChannel(self, channel)
                chan.accept_transfer()
                '''

    def new_channel_deprecated_cb(self, path, channel_type,
                                handle_type, handle, suppress):
        if channel_type == CHANNEL_TYPE_TEXT:
            if handle_type != HANDLE_TYPE_CONTACT:
                log.error('Wrong handle type: %s' % handle_type)
                return
            contact = self.contacts[handle]
            log.debug('New chat from %s' % contact.id)
            if not contact.textchannel:
                TextChannel(contact, path)
            else:
                log.debug('Text channel already created!')

    def set_status(self, status, message=''):
        conn = self.conn
        interfaces = self.interfaces
        # Setting self status
        if CONNECTION_INTERFACE_SIMPLE_PRESENCE in interfaces:
            statuses = conn[dbus.PROPERTIES_IFACE].Get(
                CONNECTION_INTERFACE_SIMPLE_PRESENCE,
                'Statuses')
            if status not in statuses.keys():
                log.error("Status '%s' not allowed" % status)
                return
            conn[CONNECTION_INTERFACE_SIMPLE_PRESENCE].SetPresence(
                status, message)
        elif CONNECTION_INTERFACE_PRESENCE in interfaces:
            statuses = conn[CONNECTION_INTERFACE_PRESENCE].GetStatuses().keys()
            if status not in statuses:
                if status == 'dnd':
                    status = 'busy'
                elif status in ('brb', 'xa'):
                    status = 'away'
                else:
                    log.error("Status '%s' not allowed" % status)
                    return
            conn[CONNECTION_INTERFACE_PRESENCE].SetStatus(
                { status : { 'message' : message}})

    def update_capabilities_cb(self, caps):
        pass

    def get_interfaces_cb (self, interfaces):
        conn = self.conn
        self.interfaces = interfaces

        if CONNECTION_INTERFACE_REQUESTS in interfaces:
            self.request_self_contact()
            self.request_contact_list('subscribe')
            self.sigmatch['NewChannels'] = \
                conn[CONNECTION_INTERFACE_REQUESTS].connect_to_signal(
                    'NewChannels', self.new_channels_cb)
        else:
            self.request_contact_list_deprecated('subscribe')
            self.sigmatch['NewChannel'] = \
                conn[CONNECTION].connect_to_signal(
                    'NewChannel', self.new_channel_deprecated_cb)

    def contacts_fetched_cb(self):
        conn = self.conn
        interfaces = self.interfaces
        be = Backend()

        if CONNECTION_INTERFACE_SIMPLE_PRESENCE in interfaces:
            self.sigmatch['PresencesChanged'] = \
                conn[CONNECTION_INTERFACE_SIMPLE_PRESENCE].connect_to_signal(
                    'PresencesChanged', self.update_presences_cb)
            self.set_status(be.status, be.status_message)
            self.request_presences()
        elif CONNECTION_INTERFACE_PRESENCE in interfaces:
            self.sigmatch['PresenceUpdate'] = \
                conn[CONNECTION_INTERFACE_PRESENCE].connect_to_signal(
                    'PresenceUpdate', self.update_presences_deprecated_cb)
            self.set_status(be.status, be.status_message)
            self.request_presences()

        if CONNECTION_INTERFACE_ALIASING in interfaces:
            self.request_aliases()
            self.sigmatch['AliasesChanged'] = \
                conn[CONNECTION_INTERFACE_ALIASING].connect_to_signal(
                    'AliasesChanged', self.update_aliases_cb)

        if CONNECTION_INTERFACE_AVATARS in interfaces:
            self.sigmatch['AvatarUpdated'] = \
                conn[CONNECTION_INTERFACE_AVATARS].connect_to_signal(
                    'AvatarUpdated', self.update_avatar_cb)
            self.sigmatch['AvatarRetrieved'] = \
                conn[CONNECTION_INTERFACE_AVATARS].connect_to_signal(
                    'AvatarRetrieved', self.retrieved_avatar_cb)

        if CONNECTION_INTERFACE_CAPABILITIES in interfaces:
            self.sigmatch['CapabilitiesChanged'] = \
                conn[CONNECTION_INTERFACE_CAPABILITIES].connect_to_signal(
                    'CapabilitiesChanged', self.update_capabilities_cb)

        self.contacts_fetched = True
        if self.callback_contacts_fetched:
            self.callback_contacts_fetched()


class BonjourAccount(Account):
    def __init__(self, firstname, lastname, nickname):
        config = config_defaults['Bonjour'].copy()
        self.protocol_ref = 'Bonjour'
        self.protocol = config['protocol']
        self.manager = config['manager']
        self.options = config['options'].copy()
        self.options['first-name'] = firstname
        self.options['last-name'] = lastname
        self.options['nickname'] = nickname

        self.init_attrs()

    def __eq__(self, other):
        if self.protocol_ref == other.protocol_ref and \
            self.options['first-name'] == other.options['first-name']:
            return True
        else:
            return False

    def get_username(self):
        return self.options['first-name'] + ' ' + self.options['last-name']

    username = property(get_username)

    def get_firstname(self):
        return self.options['first-name']

    firstname = property(get_firstname)

    def get_lastname(self):
        return self.options['last-name']

    lastname = property(get_lastname)

    def get_nickname(self):
        return self.options['nickname']

    nickname = property(get_nickname)

    def get_password(self):
        return ''

    password = property(get_password)

    def new_channels_cb(self, channels):
        Account.new_channels_cb(self, channels)
        conn = self.conn

        for channel, props in channels:
            if props[CHANNEL + '.ChannelType'] == CHANNEL_TYPE_CONTACT_LIST:
                target_id = props[CHANNEL + '.TargetID']
                log.debug('New contact list channel \'%s\'', target_id)
                if target_id != 'known':
                    return
                chan = telepathy.client.Channel(conn.service_name, channel)
                chan[dbus.PROPERTIES_IFACE].Get(CHANNEL_INTERFACE_GROUP,
                                        'Members',
                                        reply_handler = self.members_cb,
                                        error_handler = self.error_cb)
                self.sigmatch['MembersChanged'] = \
                    chan[CHANNEL_INTERFACE_GROUP].connect_to_signal(
                    'MembersChanged', self.members_changed_cb)

    def members_changed_cb(self, message, added, removed, *ignored):
        self.members_cb(added)

    def members_cb(self, handles):
        conn = self.conn

        notinlist = []
        for handle in handles:
            if handle not in self.contacts.keys():
                notinlist.append(handle)
        conn[CONNECTION_INTERFACE_CONTACTS].GetContactAttributes(
                    notinlist, [
                        CONNECTION_INTERFACE_ALIASING,
                        CONNECTION_INTERFACE_SIMPLE_PRESENCE,
                    ],
                    True,
                    reply_handler = self.get_contact_attributes_cb,
                    error_handler = self.error_cb)

    def get_contact_attributes_cb(self, attributes):
        for member, attr in attributes.iteritems():
            contact = Contact(self, member, attr)
            if self.callback_contact_added:
               self.callback_contact_added(contact)


class Backend(Singleton):
    accounts = []
    registered = False
    conn_ref = []
    _db = None

    # Notification callbacks
    callback_connected = None
    callback_new_message = None
    callback_message_read = None
    callback_self_status_changed = None

    def __init__(self):
        Singleton.__init__(self)
        atexit.register(self.disconnect_accounts_cb)
        im_path = os.path.expanduser("~/.canola/im")
        if not os.path.exists(im_path):
            os.makedirs(im_path, 0755)

        self.disconnect_zombies()

        self._db = IMDataBase()
        # Recovering last used status
        self.status = self._db.get_option('last_status')
        self.status_message = self._db.get_option('last_status_message')

        log.debug("Loading accounts from db")
        for protocol, username, password in self._db.get_accounts():
            if protocol == 'Bonjour':
                fn, ln, nn = username.split('#')
                self.accounts.append(BonjourAccount(fn, ln, nn))
            else:
                self.accounts.append(Account(protocol, username, password))

    def notify_buffered_message(self, contact):
        if self.callback_new_message:
           self.callback_new_message(contact)

    def notify_buffer_cleaned(self, contact):
        if self.callback_message_read:
           self.callback_message_read(contact)

    def notify_self_status_changed(self, status):
        if self.callback_self_status_changed:
           self.callback_self_status_changed(status)

    def set_global_status(self, status, message=''):
        if status not in [ "available", "away", "dnd", "busy", "offline" ]:
            log.error("Status not allowed!")
            return
        if self.status == status and self.status_message == message:
            # Status not changed
            return
        if status == 'offline':
            self.disconnect_accounts_cb()
            return
        for account in self.accounts:
            if account.is_connected():
                account.set_status(status, message)
        self.status = status
        self.status_message = message
        # Saving current status to DB
        self._db.set_option('last_status', status)
        self._db.set_option('last_status_message', unicode(message, "utf-8"))
        self.notify_self_status_changed(status)

    def disconnect_zombies(self):
        path = os.path.expanduser("~/.canola/im/connected.ref")
        if not os.path.exists(path): return
        f = open(path, 'rb')
        conn_ref = eval(f.read())
        for bus_name, object_path in conn_ref:
            def cb(*ignore):
                pass
            try:
                conn = telepathy.client.Connection(
                    bus_name, object_path,
                    error_handler = cb)
                conn[CONNECTION].Disconnect()
            except:
                pass
        f.close()
        os.remove(path)

    def get_available_networks(self):
        return config_defaults.keys()

    def store_connection(self, account):
        ref = account.conn_ref
        if not ref: return
        self.conn_ref.append(ref)
        path = os.path.expanduser("~/.canola/im/connected.ref")
        f = open(path, 'wb')
        f.write(str(self.conn_ref))
        f.close()
        if self.callback_connected:
           self.callback_connected(account, True)

    def remove_connection(self, account):
        ref = account.conn_ref
        if not ref or ref not in self.conn_ref:
            return
        self.conn_ref.remove(ref)
        path = os.path.expanduser("~/.canola/im/connected.ref")
        if self.callback_connected:
            self.callback_connected(account, False)
        if (len(self.conn_ref) == 0) and os.path.exists(path):
            os.remove(path)
        else:
            f = open(path, 'wb')
            f.write(str(self.conn_ref))
            f.close()

    def disconnect_accounts_cb(self):
        log.debug("Disconnecting all accounts")
        for account in self.accounts:
            if not account.is_connected():
                continue
            self.remove_connection(account)
            account.disconnect(account.status_disconnected_cb)

    def add_account(self, account):
        if account not in self.accounts:
            self.accounts.append(account)
            log.debug("Account added!")
            # Adding account to db
            self._db.add_account(account, replace=False)
            self._db.commit()
            return True
        else:
            log.debug("Account already in list. Not added again!")
            return False

    def remove_account(self, account):
        if account in self.accounts:
            self.accounts.remove(account)
            log.debug("Account removed!")
            # Removing account from the db
            self._db.remove_account(account)
            self._db.commit()
            return True
        else:
            log.debug("Account not in list!")
            return False

    def cache_avatar_file(self, contact, token, data, type):
        if not contact.parent:
            return None
        manager = contact.parent.manager
        protocol = contact.parent.protocol
        im_path = os.path.expanduser("~/.canola/im/avatars")
        im_path = os.path.join(im_path, manager, protocol)
        if not os.path.exists(im_path):
            os.makedirs(im_path, 0755)
        ext = imghdr.what('', data)
        if not ext:
            log.error("Image type unknown")
            return None
        path = os.path.join(im_path, token + '.' + ext)
        try:
            f = open(path, 'wb')
            f.write(data)
            f.close()
        except IOError , e:
            log.error(e)
            return None
        self._db.add_avatar(contact, token, path)
        self._db.commit()
        return path

    def get_cached_avatar_by_user(self, contact):
        token, path = self._db.get_avatar_by_user(contact)
        if path and not os.path.exists(path):
            self._db.remove_avatar_by_path(path)
            self._db.commit()
            return None, None
        return token, path

    def get_cached_avatar_by_token(self, contact, token):
        path = self._db.get_avatar_by_token(contact, token)
        if path and not os.path.exists(path):
            self._db.remove_avatar_by_path(path)
            self._db.commit()
            path = None
        return path


be = Backend()
class Contact(object):
    def __init__(self, parent, handle, attributes=None, id=""):
        self.parent = parent
        if parent:
            # Adds self to parent's contacts list
            parent.contacts[handle] = self
            self.conn = parent.conn
        else:
            self.conn = None

        self._handle = handle
        self.textchannel = None
        self.log = None
        self.online = False
        self.notify_status_changed_cb = None
        self.notify_message_received_cb = None
        self.msg_buffer = []

        self._avatar_token = None
        self._avatar = None
        if attributes:
            self.id = attributes[CONNECTION + '/contact-id']
            self.alias = unicode(
                attributes[CONNECTION_INTERFACE_ALIASING + '/alias'])
            self.presence = attributes[CONNECTION_INTERFACE_SIMPLE_PRESENCE +
                                        '/presence']
            '''
            if attributes.has_key(CONNECTION_INTERFACE_AVATARS + '/token'):
                self.avatar_token = attributes[CONNECTION_INTERFACE_AVATARS + \
                                        '/token']
            '''
        else:
            self.id = id
            self._alias = ""
            self._presence = [CONNECTION_PRESENCE_TYPE_UNKNOWN, u"unknown", ""]


    def __str__(self):
        return "%s(alias=%s, id=%s, status=%s)" \
                % (self.__class__.__name__, self.alias, self.id, self.status)

    __repr__ = __str__

    def presence_set(self, presence):
        self._presence = presence
        #log.debug("Presence changed for %s: %s" %
        #    (self.id, str(self._presence[1])))
        if self.status == "offline" or self.status == "unknown" \
          or self.status == "error":
            self.online = False
        else:
            self.online = True
            if not self.avatar_token and self.parent:
                self.parent.request_avatar_token(self._handle)
        if self.notify_status_changed_cb:
            self.notify_status_changed_cb(self)

    def presence_get(self):
        return self._presence

    presence = property(presence_get, presence_set)

    def alias_set(self, alias):
        self._alias = unicode(alias)
        #log.debug("Alias changed for %s: %s" % (self.id, alias))

    def alias_get(self):
        return self._alias

    alias = property(alias_get, alias_set)

    def status_get(self):
        return str(self._presence[1])

    status = property(status_get)

    def status_message_get(self):
        return unicode(self._presence[2])

    status_message = property(status_message_get)

    def avatar_set(self, avatar):
        self._avatar = avatar

    def avatar_get(self):
        return self._avatar

    avatar = property(avatar_get, avatar_set)

    def avatar_token_set(self, token):
        if not token:
            self._avatar_token = None
            self._avatar = None
            return
        self._avatar_token = token

    def avatar_token_get(self):
        return self._avatar_token

    avatar_token = property(avatar_token_get, avatar_token_set)

    def error_cb(self, error):
        log.error("Error: %s" % error)

    def get_buffered_messages(self):
        msgs = self.msg_buffer[:]
        self.msg_buffer = []
        be.notify_buffer_cleaned(self)
        return msgs

    def message_received(self, msg):
        if not self.notify_message_received_cb:
            log.debug("Buffering message")
            be.notify_buffered_message(self)
            self.msg_buffer.append(msg)
        else:
            self.notify_message_received_cb(msg)

    def send_message(self, msg):
        self.textchannel.send_message(msg)

    def open_textchannel(self):
        conn = self.conn

        if self.textchannel:
            return True

        if CONNECTION_INTERFACE_REQUESTS in self.parent.interfaces:
            #Ensuring channel
            conn[CONNECTION_INTERFACE_REQUESTS].EnsureChannel({
                    CHANNEL + '.ChannelType'     : CHANNEL_TYPE_TEXT,
                    CHANNEL + '.TargetHandleType': HANDLE_TYPE_CONTACT,
                    CHANNEL + '.TargetHandle'    : self._handle,
                    },
                    reply_handler = lambda *stuff: None,
                    error_handler = self.error_cb)
            return True
        else:
            try:
                conn[CONNECTION].RequestChannel(CHANNEL_TYPE_TEXT,
                    HANDLE_TYPE_CONTACT, self._handle, True,
                    reply_handler = lambda *stuff: None,
                    error_handler = self.error_cb)
            except:
                return False
            return True

    def close_textchannel(self):
        if not self.textchannel:
            return
        self.textchannel.close()

