#
# classes and functions to use the flickr api 
#

import flickrapi, helper, datetime, os

API_KEY     = 'a5a079f32c1ff7de4f56f5e71455a210'
API_SECRET  = 'bef25bdff383f762'

EXTRAS_OWNER_KNOWN  = "date_upload,date_taken,views"
EXTRAS_LONG         = "%s,owner_name" % EXTRAS_OWNER_KNOWN

class Flickr:
    def __init__(self, username="", api_key=API_KEY, api_secret=API_SECRET):
        helper.attributesFromDict(locals())

        self.flickr = flickrapi.FlickrAPI(api_key, api_secret, format='etree', cache=True)
        self.logged = False
        self.token  = None
        self.frob   = None
        self.user   = None

        self.users      = {}
        self.groups     = {}
        self.photos     = {}
        self.comments   = {}

        self.upload_status = None

        self.favorites      = []
        self.has_favorites  = True

    def check_login(func):
        """Used as decorator to validate login."""
        def new_def(*args, **kwds):
            self = args[0]
            if not self.logged:
                self.login()
            return func(*args, **kwds)
        return new_def

    def _parse_photolist(self, photolist, rootnode="photos", user="me"):
        photos = []

        if not photolist.attrib['stat'] == "ok":
            return None

        root = photolist.find(rootnode)

        try:
            page        = root.attrib['page']
            pages       = root.attrib['pages']
            per_page    = root.attrib['perpage']
            total       = root.attrib['total']
        except KeyError:
            page        = 1
            pages       = 1
            per_page    = 50
            total       = None

        photo_xml = root.findall('photo')

        for photo in photo_xml:
            id          = photo.attrib['id']
            title       = photo.attrib['title']
            server      = photo.attrib['server']
            icon_farm   = photo.attrib['farm']
            secret      = photo.attrib['secret']

            try:
                # filter out videos
                media = photo.attrib['media']
                if media == "video": continue
            except KeyError: pass

            try:
                realname = photo.attrib['ownername']
            except KeyError:
                realname = None

            try:
                owner_id = photo.attrib['owner']

                # use "cached" user info, if possible
                if not owner_id in self.users:
                    owner = User(nsid=owner_id, realname=realname)
                    self.users[owner_id] = owner
                else:
                    if not realname == None:
                        self.users[owner_id].realname = realname
                    owner = self.users[owner_id]
            except KeyError:
                if user == "me": owner = self.user
                else: owner = user

            try:
                date_posted = photo.attrib['dateupload']
                date_posted = datetime.datetime.fromtimestamp(int(date_posted))
            except KeyError:
                date_posted = None

            try:
                date_taken = photo.attrib['datetaken']
                formats = [ "%Y-%m-%d %H:%M:%S", "%Y-%m-00 %H:%M:%S" ]
                for format in formats:
                    try:
                        date_taken = datetime.datetime.strptime(date_taken, format)
                        break
                    except ValueError:
                        pass
                else:
                    date_taken = date_posted
            except KeyError:
                date_taken = date_posted

            try:
                views = int(photo.attrib['views'])
            except KeyError:
                views = 0

            photo_obj = Photo(id=id, owner=owner, title=title, secret=secret, views=views, date_posted=date_posted, date_taken=date_taken, server=server, icon_farm=icon_farm)

            if not id in self.photos:
                # cache photo info
                self.photos[id] = photo_obj
            else:
                # update cached photo info
                self.photos[id].owner       = owner
                self.photos[id].title       = title
                self.photos[id].secret      = secret
                self.photos[id].views       = views
                self.photos[id].date_posted = date_posted
                self.photos[id].date_taken  = date_taken
                self.photos[id].server      = server
                self.photos[id].icon_farm   = icon_farm

                photo_obj = self.photos[id]

            photos.append(photo_obj)

        if total == None:
            total = len(photos)

        return Photos(page=page, pages=pages, per_page=per_page, photos=photos, total=total)

    def _parse_grouplist(self, grouplist):
        groups = []

        if not grouplist.attrib['stat'] == "ok":
            return None

        grouplist_xml = grouplist.find('groups')

        try:
            page        = grouplist_xml.attrib['page']
            pages       = grouplist_xml.attrib['pages']
            per_page    = grouplist_xml.attrib['perpage']
            total       = grouplist_xml.attrib['total']
        except KeyError:
            pages = 1

        group_xml = grouplist_xml.findall('group')

        for group in group_xml:
            nsid            = group.attrib['nsid']
            name            = group.attrib['name']
            eighteenplus    = group.attrib['eighteenplus']

            try: admin = group.attrib['admin']
            except KeyError: admin = False

            group = Group(nsid=nsid, name=name, admin=admin, eighteenplus=eighteenplus)

            if not nsid in self.groups:
                # cache group info
                self.groups[nsid] = group
            else:
                # return cached group info (should have more infos available)
                group = self.groups[nsid]

            groups.append(group)

        if pages == 1:
            num_groups = len(groups)
            return Groups(groups=groups, total=num_groups, per_page=num_groups)
        else:
            return Groups(page=page, pages=pages, per_page=per_page, total=total, groups=groups)

    def get_photo_info(self, photo):
        try:
            res = self.flickr.photos_getInfo(photo_id=photo.id)
        except flickrapi.FlickrError, msg:
            print "FlickrError ocurred: %s" % msg
            return photo

        if not res.attrib['stat'] == "ok":
            return photo

        photo_xml   = res.find('photo')
        tags_xml    = photo_xml.find('tags').findall('tag')
        tags        = []

        for tag_xml in tags_xml:
            tags.append(tag_xml.text)

        dates_xml = photo_xml.find('dates')

        date_taken      = dates_xml.attrib['taken']
        date_taken      = datetime.datetime.strptime(date_taken, "%Y-%m-%d %H:%M:%S")
        date_posted     = dates_xml.attrib['posted']
        date_posted     = datetime.datetime.fromtimestamp(int(date_posted))
        date_lastupdate = dates_xml.attrib['lastupdate']
        date_lastupdate = datetime.datetime.fromtimestamp(int(date_lastupdate))

        owner_xml   = photo_xml.find('owner')
        username    = owner_xml.attrib['username']
        realname    = owner_xml.attrib['realname']
        nsid        = owner_xml.attrib['nsid']

        owner       = User(nsid=nsid, username=username, realname=realname)

        # "cache" user info
        if not nsid in self.users:
            self.users[nsid] = owner
        else:
            self.users[nsid].username = username
            self.users[nsid].realname = realname

        photo.title             = photo_xml.find('title').text
        photo.description       = photo_xml.find('description').text
        photo.date_taken        = date_taken
        photo.date_posted       = date_posted
        photo.date_lastupdate   = date_lastupdate
        photo.tags              = tags
        photo.owner             = owner
        photo.have_info         = True
        photo.server            = photo_xml.attrib['server']
        photo.icon_farm         = photo_xml.attrib['farm']
        photo.secret            = photo_xml.attrib['secret']

        if not photo.id in self.photos:
            # cache photo info
            self.photos[photo.id] = photo
        else:
            # update cached photo info
            self.photos[photo.id].title             = photo.title
            self.photos[photo.id].description       = photo.description
            self.photos[photo.id].date_taken        = photo.date_taken
            self.photos[photo.id].date_posted       = photo.date_posted
            self.photos[photo.id].date_lastupdate   = photo.date_lastupdate
            self.photos[photo.id].tags              = photo.tags
            self.photos[photo.id].owner             = photo.owner
            self.photos[photo.id].have_info         = photo.have_info
            self.photos[photo.id].server            = photo.server
            self.photos[photo.id].icon_farm         = photo.icon_farm
            self.photos[photo.id].secret            = photo.secret

        return self.photos[photo.id]

    def get_exif_info(self, photo):
        permission_denied = False
        try:
            res = self.flickr.photos_getExif(photo_id=photo.id)
        except flickrapi.FlickrError, msg:
            if msg.__str__() == "Error: 2: Permission denied":
                permission_denied = True
            else:
                print "FlickrError ocurred: %s" % msg
                return photo

        if not permission_denied:
            if not res.attrib['stat'] == "ok":
                return photo

            photo_xml   = res.find('photo')
            exifs_xml   = photo_xml.findall('exif')
            exif        = []

            for exif_xml in exifs_xml:
                exif_label  = exif_xml.attrib['label']
                exif_raw    = exif_xml.find('raw').text
                exif_clean  = exif_xml.find('clean')

                if exif_clean == None:
                    exif_text = "%s: %s" % (exif_label, exif_raw)
                else:
                    exif_text = "%s: %s" % (exif_label, exif_clean.text)

                exif.append(exif_text)

            photo.exif = exif

        # mark that we fetched exif information
        photo.have_exif = True

        if not photo.id in self.photos:
            # cache photo info
            self.photos[photo.id] = photo
        else:
            # update cached photo info
            self.photos[photo.id].exif      = photo.exif
            self.photos[photo.id].have_exif = photo.have_exif

        return self.photos[photo.id]

    def test_login(self):
        try:
            # call flickr.test.login to see if the token we already have is valid
            res = self.flickr.test_login()
            if not res.attrib['stat'] == "ok":
                return None

            user_xml    = res.find('user')
            nsid        = user_xml.attrib['id']
            username    = user_xml.find('username').text

            user = User(username=username, nsid=nsid)

            # cache user
            if not nsid in self.users:
                self.users[nsid] = user
            else:
                self.users[nsid].username = username

            return user
        except flickrapi.FlickrError, e:
            return None

    def login(self):
        user = self.test_login()
        if not user == None:
            self.user   = user
            self.logged = True
            return True

        if not self.frob and not self.token:
            token, self.frob = self.flickr.get_token_part_one(perms='write')
            if not token:
                self.token = None
                return False
            else: self.token = token

        self.token = self.flickr.get_token_part_two((self.token, self.frob))
        if not self.token:
            self.logged = False
            # TODO: raise Exception here, it seems that we can not login at all
        else:
            user = self.test_login()
            if not user == None:
                self.user   = user
                self.logged = True
                return True

        return self.logged

    def delete_token(self):
        import os
        token = os.path.join(os.path.expanduser("~"), ".flickr", self.api_key, "auth.token")

        if os.path.exists(token):
            os.unlink(token)
            return True
        else:
            return False

    def get_userinfo(self, user):
        try:
            nsid = user.nsid
        except AttributeError:
            return False

        res = self.flickr.people_getInfo(user_id=nsid)

        if not res.attrib['stat'] == "ok":
            return False

        person_xml = res.find('person')

        ispro = person_xml.attrib['ispro']

        if ispro == "0": user.ispro = False
        else: user.ispro = True

        user.realname   = person_xml.find('realname').text
        user.username   = person_xml.find('username').text
        user.iconserver = person_xml.attrib['iconserver']
        user.iconfarm   = person_xml.attrib['iconfarm']
        user.location   = person_xml.find('location').text
        user.have_info  = True

        photos_xml = person_xml.find('photos')

        user.photo_count = int(photos_xml.find('count').text)

        firstdate = photos_xml.find('firstdate').text
        if not firstdate == None:
            user.first_upload = datetime.datetime.fromtimestamp(int(firstdate))

        # "cache" user info
        if not user.nsid in self.users:
            self.users[nsid] = user
        else:
            self.users[nsid].ispro          = user.ispro
            self.users[nsid].realname       = user.realname
            self.users[nsid].username       = user.username
            self.users[nsid].iconfarm       = user.iconfarm
            self.users[nsid].iconserver     = user.iconserver
            self.users[nsid].have_info      = user.have_info
            self.users[nsid].location       = user.location
            self.users[nsid].photo_count    = user.photo_count
            self.users[nsid].first_upload   = user.first_upload

        return True

    @check_login
    def get_uploadstatus(self):
        res = self.flickr.people_getUploadStatus()

        if not res.attrib['stat'] == "ok":
            return False

        user_xml = res.find('user')

        ispro = user_xml.attrib['ispro']

        if ispro == "0": ispro = False
        else: ispro = True

        bandwidth_xml   = user_xml.find('bandwidth')
        filesize_xml    = user_xml.find('filesize')
        sets_xml        = user_xml.find('sets')

        upload_max          = int(bandwidth_xml.attrib['maxbytes'])
        upload_used         = int(bandwidth_xml.attrib['usedbytes'])
        upload_remaining    = int(bandwidth_xml.attrib['remainingbytes'])

        max_filesize = int(filesize_xml.attrib['maxbytes'])

        sets_created    = int(sets_xml.attrib['created'])
        sets_remaining  = sets_xml.attrib['remaining']

        upload_status = UploadStatus(
                upload_max=upload_max,
                upload_used=upload_used,
                upload_remaining=upload_remaining,
                max_filesize=max_filesize,
                sets_created=sets_created,
                sets_remaining=sets_remaining,
                ispro=ispro
            )

        self.upload_status = upload_status

        return self.upload_status

    def get_groupinfo(self, group):
        try:
            nsid = group.nsid
        except AttributeError:
            return False

        res = self.flickr.groups_getInfo(group_id=nsid)

        if not res.attrib['stat'] == "ok":
            return False

        group_xml = res.find('group')

        ispoolmoderated = group_xml.attrib['ispoolmoderated']

        if ispoolmoderated == "0": group.ispoolmoderated  = False
        else: group.ispoolmoderated = True

        group.name          = group_xml.find('name').text
        group.description   = group_xml.find('description').text
        group.members       = int(group_xml.find('members').text)
        group.privacy       = int(group_xml.find('privacy').text)
        group.iconserver    = group_xml.attrib['iconserver']
        group.iconfarm      = group_xml.attrib['iconfarm']
        group.have_info     = True

        # "cache" group info
        if not group.nsid in self.groups:
            self.groups[nsid] = group
        else:
            self.groups[nsid].ispoolmoderated   = group.ispoolmoderated
            self.groups[nsid].name              = group.name
            self.groups[nsid].description       = group.description
            self.groups[nsid].members           = group.members
            self.groups[nsid].privacy           = group.privacy
            self.groups[nsid].iconserver        = group.iconserver
            self.groups[nsid].iconfarm          = group.iconfarm
            self.groups[nsid].have_info         = group.have_info

        return True

    def get_buddyicon_url(self, user, group=False):
        try:
            nsid = user.nsid
        except AttributeError:
            return None

        if not user.iconfarm or not user.iconserver:
            if group:
                if self.get_groupinfo(user):
                    user = self.groups[nsid]
                else:
                    return None
            else:
                if self.get_userinfo(user):
                    user = self.users[nsid]
                else:
                    return None

        iconfarm   = user.iconfarm
        iconserver = user.iconserver

        return "http://farm%s.static.flickr.com/%s/buddyicons/%s.jpg" % (iconfarm, iconserver, nsid)

    def get_photosets(self, user=None):
        photosets = []

        if not user:
            sets = self.flickr.photosets_getList()
            user = self.user
        else:
            try:
                user_id = user.nsid
            except AttributeError:
                return None
            sets = self.flickr.photosets_getList(user_id=user_id)
        
        if not sets.attrib['stat'] == "ok":
            return None

        sets_xml = sets.find('photosets').findall('photoset')

        for set in sets_xml:
            id          = set.attrib['id']
            title       = set.find('title').text
            description = set.find('description').text
            photos      = int(set.attrib['photos'])
            primary     = set.attrib['primary']
            secret      = set.attrib['secret']
            server      = set.attrib['server']
            farm        = set.attrib['farm']

            photoset = Photoset(id=id, owner=user, title=title, description=description, photos=photos, primary=primary, secret=secret, server=server, farm=farm)
            photosets.append(photoset)

        return photosets

    def get_hottags(self, period="day", count=20):
        tags = []

        res = self.flickr.tags_getHotList(period=period, count=count)

        if not res.attrib['stat'] == "ok":
            return None

        tags_xml = res.find('hottags').findall('tag')

        for tag_xml in tags_xml:
            score       = tag_xml.attrib['score']
            name        = tag_xml.text

            tags.append((score, name))

        return Hottags(period=period, count=count, tags=tags)

    @check_login
    def get_favorites(self, user=None, page=1, per_page=10, extras=EXTRAS_LONG):
        if user:
            try:
                userid = user.nsid
            except AttributeError:
                return None
            res = self.flickr.favorites_getList(user_id=userid, page=page, per_page=per_page, extras=extras)
        else:
            res = self.flickr.favorites_getList(page=page, per_page=per_page, extras=extras)

        return self._parse_photolist(res)

    def is_favorite(self, photo):
        if not self.favorites and self.has_favorites:
            favorites = self.get_favorites(per_page=500)
            if favorites and favorites.photos:
                for image in favorites.photos:
                    self.favorites.append(image.id)
            else: self.has_favorites = False

        return photo.id in self.favorites

    @check_login
    def get_photos_notinset(self, page=1, per_page=10, extras=EXTRAS_OWNER_KNOWN):
        res = self.flickr.photos_getNotInSet(page=page, per_page=per_page, extras=extras)
        return self._parse_photolist(res)

    @check_login
    def get_photos_latest(self, user=None, page=1, per_page=10, extras=EXTRAS_OWNER_KNOWN):
        if not user: user_id = "me"
        else: user_id = user.nsid

        res = self.flickr.photos_Search(user_id=user_id, page=page, per_page=per_page, media="photos", extras=extras)
        return self._parse_photolist(res)

    def get_photos_interesting(self, page=1, per_page=100, extras=EXTRAS_LONG + ",media", date=None):
        if date == None:
            res = self.flickr.interestingness_getList(page=page, per_page=per_page, extras=extras)
        else:
            date_string = date.strftime("%Y-%m-%d")
            try:
                res = self.flickr.interestingness_getList(page=page, per_page=per_page, extras=extras, date=date_string)
            except flickrapi.FlickrError, msg:
                print "FlickrError ocurred: %s" % msg
                return None
        return self._parse_photolist(res)

    def get_photos_recent(self, page=1, per_page=100, extras=EXTRAS_LONG + ",media"):
        res = self.flickr.photos_getRecent(page=page, per_page=per_page, extras=extras)
        return self._parse_photolist(res)

    @check_login
    def get_photos_contacts(self, extras=EXTRAS_LONG, count=1, just_friends=False, include_self=False):
        if just_friends: friends_only = "1"
        else: friends_only = "0"

        if include_self: inc_self = "1"
        else: inc_self = "0"

        res = self.flickr.photos_getContactsPhotos(extras=extras, include_self=inc_self, just_friends=friends_only, count=count)
        return self._parse_photolist(res)

    def get_photos_grouppool(self, group, page=1, per_page=10, extras=EXTRAS_LONG):
        res = self.flickr.groups_pools_getPhotos(page=page, per_page=per_page, group_id=group.nsid, extras=extras)
        return self._parse_photolist(res)

    @check_login
    def search_tags(self, tags, all=False, contacts=False, page=1, per_page=100, user_id="me", extras=EXTRAS_OWNER_KNOWN):
        tag_mode    = "all"
        media       = "photos"
        if all:
            extras = EXTRAS_LONG
            res = self.flickr.photos_Search(tags=tags, tag_mode=tag_mode, page=page, per_page=per_page, media=media, extras=extras)
        elif contacts:
            res = self.flickr.photos_Search(user_id="me", tags=tags, contacts="all", tag_mode=tag_mode, page=page, per_page=per_page, media=media, extras=extras)
        else:
            res = self.flickr.photos_Search(user_id=user_id, tags=tags, tag_mode=tag_mode, page=page, per_page=per_page, media=media, extras=extras)
        return self._parse_photolist(res)

    def search_photos_location(self, lon, lat, radius=5, page=1, per_page=100, user_id=None, min_taken_date="2000-01-01", extras=EXTRAS_LONG):
        if user_id == None: res = self.flickr.photos_Search(lon=lon, lat=lat, radius=radius, page=page, per_page=per_page, media="photos", min_taken_date=min_taken_date, extras=extras)
        else: res = self.flickr.photos_Search(user_id=user_id, lon=lon, lat=lat, radius=radius, page=page, per_page=per_page, media="photos", min_taken_date=min_taken_date, extras=extras)
        return self._parse_photolist(res)

    @check_login
    def search_groups(self, groups, page=1, per_page=20):
        res = self.flickr.groups_search(text=groups, page=page, per_page=per_page)
        return self._parse_grouplist(res)

    def get_photos_photoset(self, photoset, user=None, page=1, per_page=500, extras=EXTRAS_LONG):
        try:
            photoset_id = photoset.id
        except AttributeError:
            return None

        if not user == None:
            res = self.flickr.photosets_getPhotos(photoset_id=photoset_id, page=page, per_page=per_page, extras=extras)
            return self._parse_photolist(res, rootnode="photoset", user=user)
        else:
            res = self.flickr.photosets_getPhotos(photoset_id=photoset_id, page=page, per_page=per_page, extras=extras)
            return self._parse_photolist(res, rootnode="photoset")

    def get_photosizes(self, photo):
        try:
            photo_id = photo.id
        except AttributeError:
            return None

        sizes = []

        res = self.flickr.photos_getSizes(photo_id=photo_id)

        if not res.attrib['stat'] == "ok":
            return None

        sizes_xml = res.find('sizes').findall('size')

        for size in sizes_xml:
            label       = size.attrib['label']
            width       = size.attrib['width']
            height      = size.attrib['height']
            url         = size.attrib['source']
            photopage   = size.attrib['url']

            sizes.append(Size(label=label, width=width, height=height, url=url, photopage=photopage))

        return sizes

    def get_photo_comments(self, photo):
        try:
            photo_id = photo.id
        except AttributeError:
            return None

        comments = []

        res = self.flickr.photos_comments_getList(photo_id=photo_id)

        if not res.attrib['stat'] == "ok":
            return None

        comments_xml = res.find('comments').findall('comment')

        for comment in comments_xml:
            id          = comment.attrib['id']
            date_create = comment.attrib['datecreate']
            author      = comment.attrib['author']
            authorname  = comment.attrib['authorname']
            permalink   = comment.attrib['permalink']
            text        = comment.text

            date_create = datetime.datetime.fromtimestamp(int(date_create))

            user = User(nsid=author, username=authorname)
            if not user.nsid in self.users:
                self.users[user.nsid] = user
            else:
                self.users[user.nsid].username = authorname

            comments.append(Comment(id=id, photo_id=photo.id, author=self.users[user.nsid], permalink=permalink, text=text, date_create=date_create))

        # "cache" comments
        self.comments[photo_id] = comments
        return self.comments[photo_id]

    @check_login
    def get_contacts(self, user=None, page=1, per_page=20):
        contacts = []

        if user == None: res = self.flickr.contacts_getList(page=page, per_page=per_page)
        else: res = self.flickr.contacts_getPublicList(user_id=user.nsid, page=page, per_page=per_page)

        if not res.attrib['stat'] == "ok":
            return None

        contacts_xml = res.find('contacts')

        page        = contacts_xml.attrib['page']
        pages       = contacts_xml.attrib['pages']
        per_page    = contacts_xml.attrib['perpage']
        total       = contacts_xml.attrib['total']

        contact_xml = contacts_xml.findall('contact')

        for contact in contact_xml:
            realname    = None
            nsid        = contact.attrib['nsid']
            username    = contact.attrib['username']
            try:
                realname = contact.attrib['realname']
                if realname == '': realname = None

                isfriend = contact.attrib['friend']
                isfamily = contact.attrib['family']

                if isfriend == "0": friend = False
                else: friend = True

                if isfamily == "0": family = False
                else: family = True

                iscontact = True
            except KeyError:
                family      = False
                friend      = False
                iscontact   = False

            contact = User(username=username, nsid=nsid, realname=realname, friend=friend, family=family, iscontact=iscontact)
            if not nsid in self.users:
                self.users[nsid] = contact
            else:
                self.users[nsid].username   = username
                self.users[nsid].realname   = realname
                self.users[nsid].friend     = friend
                self.users[nsid].family     = family
                self.users[nsid].iscontact  = iscontact

            contacts.append(self.users[nsid])

        return Contacts(page=page, pages=pages, per_page=per_page, total=total, contacts=contacts)

    @check_login
    def get_groups(self, user=None):
        if user == None:
            user = self.user

        try:
            user_id = user.nsid
        except AttributeError:
            return None

        res = self.flickr.people_getPublicGroups(user_id=user_id)
        return self._parse_grouplist(res)

    @check_login
    def add_favorite(self, photo):
        try:
            photo_id = photo.id
        except AttributeError:
            return False

        try:
            res = self.flickr.favorites_add(photo_id=photo_id)
        except flickrapi.FlickrError, e:
            return False
        self.favorites.append(photo_id)
        self.has_favorites = True
        return True

    @check_login
    def remove_favorite(self, photo):
        try:
            photo_id = photo.id
        except AttributeError:
            return False

        try:
            res = self.flickr.favorites_remove(photo_id=photo_id)
        except flickrapi.FlickrError, e:
            return False
        try: self.favorites.remove(photo_id)
        except ValueError: pass

        if not self.favorites: self.has_favorites = False
        return True

    @check_login
    def upload_photo(self, filename, title=None, description="", tags="", is_public=True, is_friend=False, is_family=False, callback=None):
        if title == None:
            title = os.path.basename(filename)

        if is_public: public=1
        else: public = 0

        if is_friend: friend=1
        else: friend=0

        if is_family: family=1
        else: family=0

        try:
            res = self.flickr.upload(
                    filename=filename,
                    description=description,
                    tags=tags,
                    is_public=public,
                    is_friend=friend,
                    is_family=family,
                    callback=callback
                )
        except flickrapi.FlickrError, e:
            return False

        if not res.attrib['stat'] == "ok":
            return False

        photo_id = res.find('photoid').text
        return photo_id

class User:
    def __init__(self,
                nsid,
                username        = None,
                realname        = None,
                iconserver      = None,
                iconfarm        = None,
                ispro           = False,
                location        = None,
                photo_count     = 0,
                first_upload    = None,
                family          = False,
                friend          = False,
                iscontact       = False,
                have_info       = False
            ):
        helper.attributesFromDict(locals())

    def __eq__(self, otheruser):
        try:
            othernsid = otheruser.nsid
        except AttributeError:
            return False

        return self.nsid == othernsid

    def __repr__(self):
        if self.username and self.realname:
            return "%s (%s)" % (self.username, self.realname)
        elif self.realname:
            return self.realname
        elif self.username:
            return self.username
        else:
            return self.nsid

    def __str__(self):
        return "User <nsid=%s, username=%s, realname=%s, iconserver=%s, iconfarm=%s, " \
                "ispro=%s, location=%s, photo_count=%s, first_upload=%s, family=%s, " \
                "friend=%s, iscontact=%s, have_info=%s>" % (
                self.nsid,
                self.username,
                self.realname,
                self.iconserver,
                self.iconfarm,
                self.ispro,
                self.location,
                self.photo_count,
                self.first_upload,
                self.family,
                self.friend,
                self.iscontact,
                self.have_info
            )

class Contacts:
    def __init__(self, page=1, pages=1, per_page=10, total=1, contacts=[]):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Contacts <page=%s, pages=%s, per_page=%s, contacts=%s, total=%s>" % (
                self.page, self.pages, self.per_page, len(self.contacts), self.total)

class Groups:
    def __init__(self, page=1, pages=1, per_page=10, total=1, groups=[]):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Groups <page=%s, pages=%s, per_page=%s, groups=%s, total=%s>" % (
                self.page, self.pages, self.per_page, len(self.groups), self.total)

class Group:
    def __init__(self,
                nsid,
                name,
                admin=False,
                eighteenplus=False,
                iconfarm=None,
                iconserver=None,
                ispoolmoderated=False,
                members=0,
                privacy=0,
                description="",
                have_info=False
            ):
        helper.attributesFromDict(locals())

    def __repr__(self):
        return self.name

    def __str__(self):
        return "Group <nsid=%s, name=%s, admin=%s, eighteenplus=%s, iconserver=%s, iconfarm=%s, ispoolmoderated=%s, members=%s, " \
                "privacy=%s, description=%s, have_info=%s>" % (
                self.nsid,
                self.name,
                self.admin,
                self.eighteenplus,
                self.iconserver,
                self.iconfarm,
                self.ispoolmoderated,
                self.members,
                self.privacy,
                self.description,
                self.have_info
            )

class Photos:
    def __init__(self, page=1, pages=1, per_page=10, total=1, photos=[]):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Photos <page=%s, pages=%s, per_page=%s, photos=%s, total=%s>" % (
                self.page, self.pages, self.per_page, len(self.photos), self.total)

class Photoset:
    def __init__(self, owner, id, title, description, photos, primary, secret, server, farm):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Photoset <id=%s, owner=%s, title=%s, description=%s, photos=%s, primary=%s, secret=%s, server=%s, farm=%s>" % (
                    self.id,
                    self.owner,
                    self.title,
                    self.description,
                    self.photos,
                    self.primary,
                    self.secret,
                    self.server,
                    self.farm
                )

class Photo:
    def __init__(self,
                id,
                title,
                description     = None,
                owner           = None,
                sizes           = None,
                secret          = None,
                server          = None,
                icon_farm       = None,
                tags            = [],
                have_info       = False,
                date_taken      = None,
                date_posted     = None,
                date_lastupdate = None,
                exif            = [],
                have_exif       = False,
                views           = 0
            ):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Photo <id=%s, owner: %s, secret: %s, title=%s, description=%s, sizes=%s, server=%s, icon_farm=%s, tags=%s, have_info=%s, date_taken=%s, " \
                "date_posted=%s, date_lastupdate=%s, exif: %s, have_exif=%s, views=%s>" % (
                self.id,
                self.owner,
                self.secret,
                self.title,
                self.description,
                self.sizes,
                self.server,
                self.icon_farm,
                self.tags,
                self.have_info,
                self.date_taken,
                self.date_posted,
                self.date_lastupdate,
                self.exif,
                self.have_exif,
                self.views
            )

    def __eq__(self, otherphoto):
        return self.id == otherphoto.id

class Size:
    def __init__(self, label, width, height, url, photopage):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Size <label=%s, width=%s, height=%s, url=%s, photopage=%s>" % (
                self.label, self.width, self.height, self.url, self.photopage)

    def __repr__(self):
        return "Size <label=%s, width=%s, height=%s, url=%s, photopage=%s>" % (
                self.label, self.width, self.height, self.url, self.photopage)

class Hottags:
    def __init__(self, period="day", count=20, tags=[]):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "Hottags <period=%s, count=%s, tags=%s>" % (
                self.period, self.count, self.tags)

class UploadStatus:
    def __init__(self,
                upload_max,
                upload_used,
                upload_remaining,
                max_filesize,
                sets_created,
                sets_remaining,
                ispro
            ):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "UploadStatus <upload_max=%s, upload_used=%s, upload_remaining=%s, "\
                "max_filesize=%s, sets_created=%s, sets_remaining=%s, ispro=%s>" % (
                        self.upload_max,
                        self.upload_used,
                        self.upload_remaining,
                        self.max_filesize,
                        self.sets_created,
                        self.sets_remaining,
                        self.ispro
                    )


class Comment:
    def __init__(self, id, photo_id, author, date_create, permalink, text):
        helper.attributesFromDict(locals())

    def __str__(self):
        return "comment for photo-id %s from %s, author: %s: %s" % (
                self.photo_id,
                self.date_create,
                self.author.__repr__(),
                self.text
            )

def main():
    flickr = Flickr()
    res = flickr.login()
    if not res == True:
        raw_input("Please switch to your browser window and permit Canola-Flickr access to your flickr account!")
        res = flickr.login()
        if not res == True:
            print "Error, could not get access to a flickr account!"
            return -1

    #print "User: %s" % flickr.user
    #flickr.get_userinfo(flickr.user)

    #print "User: %s" % flickr.user
    #print "Buddyicon: %s" % flickr.get_buddyicon_url(flickr.user)

    #contactlist = flickr.get_contacts()

    #if not contactlist:
    #    print "Error, no contacts found!"
    #    return 1

    #print "contactlist: %s\n" % contactlist

    #for contact in contactlist.contacts:
    #    print contact

    #contact = contactlist.contacts[1]
    #print "Contact: %s" % contact
    #print "Buddyicon: %s" % flickr.get_buddyicon_url(contact)

    #print "Contacts of: %s" % contact
    #contacts = flickr.get_contacts(contact)
    #for contact in contacts:
    #    print contact

    #res = flickr.delete_token()
    #print "res(delete_token()): %s" % res

    #print "Contact[1]: %s" % contacts[1]

    #photosets = flickr.get_photosets()
    #photosets = flickr.get_photosets(user=contact)
   
    #if not photosets:
    #    print "Error, no photosets found!"
    #    return 1

    #for photoset in photosets:
    #    try: print photoset
    #    except UnicodeEncodeError: print "could not print info of photoset %s (UnicodeEncodeError)" % photoset.id

    #print photosets[0]

    #photolist = flickr.get_photos_photoset(photoset=photosets[0])

    #print "\nFavorites of: %s" % contact
    #photolist = flickr.get_favorites(user=contact)
    #photolist = flickr.get_favorites()

    #photolist = flickr.get_photos_notinset()
    #photolist = flickr.search_tags(tags="n810")
    #photolist = flickr.search_photos_location(lat="51.02280", lon="11.125373", radius=5, per_page=20)
    #photolist = flickr.search_tags(tags="beben", contacts="all")
    #photolist = flickr.get_photos_interesting(per_page=3)
    #photolist = flickr.get_photos_contacts(count=20)

    #grouplist = flickr.get_groups()
    #print "\ngrouplist: %s\n" % grouplist

    #group = grouplist.groups[0]
    #print "Group: %s" % group
    #print "Buddyicon: %s" % flickr.get_buddyicon_url(group, group=True)

    #for group in grouplist.groups:
    #    try: print group
    #    except UnicodeEncodeError: pass

    #photolist = flickr.get_photos_grouppool(grouplist.groups[0])

    #print "\nphotolist: %s\n" % photolist

    #print "photos from %s:" % photosets[0]
    #if not photolist == None:
    #    for photo in photolist.photos:
    #        #photo = flickr.get_photo_info(photo)
    #        #photo = flickr.get_exif_info(photo)
    #        try: print photo
    #        except UnicodeEncodeError: print "could not print photo (id: %s) (UnicodeEncodeError)" % photo.id

    #        comments = flickr.get_photo_comments(photo)
    #        for comment in comments:
    #            try: print comment
    #            except UnicodeEncodeError: print "could not print comment (id: %s) (UnicodeEncodeError)" % comment.id

            #photo.sizes = flickr.get_photosizes(photo)
    #else: print "no photos found!"

    #print "\nflickr.users:"
    #for nsid in flickr.users:
    #    try:
    #        print flickr.users[nsid]
    #    except UnicodeEncodeError:
    #        print "could not print user: %s" % nsid

    #print "\nflickr.groups:"
    #for nsid in flickr.groups:
    #    try:
    #        print flickr.groups[nsid]
    #    except UnicodeEncodeError:
    #        print "could not print group: %s" % nsid

    #photo = photolist.photos[0]
    #print "\nis favorite (%s): %s" % (photo, flickr.is_favorite(photo))
    #print "favorites:\n"
    #for favorite in flickr.favorites:
    #    try: print favorite
    #    except UnicodeEncodeError: pass

    #hottags = flickr.get_hottags()
    #hottags = flickr.get_hottags(period="week", count=30)
    #print "\nhot tags:\n"
    #for score, tag in hottags.tags:
    #    print "score: %s, tag: %s" % (score, tag)

    # add photo to favourites
    #result = flickr.add_favorite(photos[0])
    #print "Result (favourites): %s" % result

    # remove photo from favourites
    #result = flickr.remove_favorite(photos[0])
    #print "Result (favourites): %s" % result

    #testfile = "/home/chelli/dsc_3246.jpg"
    #if not os.path.exists(testfile):
    #    print "Error, tesfile %s does not exist!" % testfile
    #    return 1

    def callback_func(progress, done):
        if done:
            print "Done uploading"
        else:
            print "At %s%%" % progress

    #res = flickr.upload_photo(filename=testfile, is_public=False, callback=callback_func)
    #if res == False:
    #    print "Error while uploading file %s!" % testfile
    #    return 1
    #else:
    #    print "Sucessfully uploaded testfile %s, photo_id is %s" % (testfile, res)

    upload_status = flickr.get_uploadstatus()
    print upload_status

    return 0

if __name__ == "__main__":
    import sys
    sys.exit(main())

