# Canola2 Flickr Plugin
# Copyright (C) 2008 Thomas Schmidt <tschmidt@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import evas
import ecore
import locale
import logging

from terra.core.manager import Manager
from terra.ui.base import PluginThemeMixin
from terra.utils.encoding import to_utf8

from model import SearchModelFolder, SearchByTagModelFolder, SearchGroupsModelFolder, \
        MainModelFolder, PhotosetModelFolder, GroupModelFolder, ContactModelFolder, \
        UploadsModelFolder, ViewerOptionsModel, PhotosetOptionModel, GroupOptionModel, \
        UserOptionModel, UploadInfoOptionModel, FavoriteOptionModel

manager = Manager()
CanolaError                 = manager.get_class("Model/Notify/Error")
BaseListController          = manager.get_class("Controller/Folder")
BaseImageViewerController   = manager.get_class("Controller/Media/Image")
ModalController             = manager.get_class("Controller/Modal")
EntryDialogModel            = manager.get_class("Model/EntryDialog")
OptionsControllerMixin      = manager.get_class("OptionsControllerMixin")
BaseRowRenderer             = manager.get_class("Widget/RowRenderer")
ResizableRowRenderer        = manager.get_class("Widget/ResizableRowRenderer")

log = logging.getLogger("plugins.canola-flickr.ui")

class ImageListRowRenderer(PluginThemeMixin, BaseRowRenderer):
    """Flickr Base List Item Renderer.

    This renderer is applied on ServiceController. Providing
    a personalized list item for images. It shows the following image
    properties: title, thumbnail upload_date, view count and owner.

    @note: This renderer extends BaseRowRenderer, overloading
    some methods.

    @note: An instance of this class can be reused to show properties
    of diferent models due to optimizations. So, to take full advantages
    of this feature, you can load heavy data like images, on value_set method,
    only keeping in the models the path to load the data. This has a nice
    effect with memory usage and is very scalable. For example if you have a
    list with 6 renderers, and your list has 800 images, only 6 images are
    stored in memory, not 800 (generally the number of renderers is close to
    the number of visible items on the list).

    @note: To tell Canola to look first on the plugin theme you have to
    add PluginThemeMixin class and set the plugin variable with the plugin
    name.

    @see: ServiceController, BaseRowRenderer, PluginThemeMixin
    """
    plugin = "flickr"

    def __init__(self, parent, theme=None):
        BaseRowRenderer.__init__(self, parent, theme)
        self.image = self.evas.FilledImage()
        self.part_swallow("contents", self.image)
        self.signal_emit("thumb,hide", "")

        self.bg_selection = self.PluginEdjeWidget("widget/list/bg_selection")
        self.part_swallow("selection", self.bg_selection)

    def theme_changed(self, end_callback=None):

        def cb(*ignored):
            self.part_swallow("selection", self.bg_selection)
            self.part_swallow("contents", self.image)
            if end_callback is not None:
                end_callback(self)

        self.bg_selection.theme_changed()

        BaseRowRenderer.theme_changed(self, cb)

    def cb_box_expanded(self, *ignored):
        self._model.selected_state = True

    def cb_box_collapsed(self, *ignored):
        self._model.selected_state = False

    def cb_load_thumbnail(self):
        if self._model == None or self._model.thumbnail == None:
            self.signal_emit("thumb,hide", "")
            return

        try:
            self.image.file_set(self._model.thumbnail)
            self.signal_emit("thumb,show", "")
        except Exception, e:
            log.error("could not load image %r: %s", self._model.thumbnail, e)
            self.signal_emit("thumb,hide", "")

    def value_set(self, model):
        """Apply the model properties to the renderer."""
        if not model or model is self._model:
            return

        if not hasattr(model, "selected_state"):
            model.selected_state = False

        self._model = model
        self.part_text_set("text", model.name)
        self.part_text_set("from", "From: " + to_utf8(model.owner.__repr__()))

        views = model.photo.views
        if views == 1: views_text = "view"
        else: views_text = "views"

        self.part_text_set("views", locale.format("%d", views, True) + " " + views_text)
        self.part_text_set("uploaded", "Uploaded on " + model.photo.date_posted.strftime("%b %d, %Y"))

        model.request_thumbnail(self.cb_load_thumbnail)

    @evas.decorators.del_callback
    def __on_delete(self):
        """Free internal data on delete."""
        self.image.delete()
        self.bg_selection.delete()

class PhotosetsRowRenderer(ImageListRowRenderer):
    def value_set(self, model):
        """Apply the model properties to the renderer."""
        if not model or model is self._model:
            return

        if not hasattr(model, "selected_state"):
            model.selected_state = False

        self._model = model
        self.part_text_set("text", model.name)
        self.part_text_set("from", "From: " + to_utf8(model.photoset.owner.__repr__()))

        num_photos = model.photoset.photos
        if num_photos == 1: num_photos_text = "photo"
        else: num_photos_text = "photos"

        self.part_text_set("photos", locale.format("%d", num_photos, True) + " " + num_photos_text)

        description = to_utf8(model.photoset.description)
        if len(description) == 0:
            description_text = "No description available."
        else:
            description = description.replace("&quot;", "\"")
            description = description.replace("\n", " ")
            description_text = "Description: " + description
        self.part_text_set("description", description_text)

        model.request_thumbnail(self.cb_load_thumbnail)

class UsersRowRenderer(ImageListRowRenderer):
    def value_set(self, model):
        """Apply the model properties to the renderer."""
        if not model or model is self._model:
            return

        if not hasattr(model, "selected_state"):
            model.selected_state = False

        self._model = model
        self.part_text_set("text", model.name)
        model.request_thumbnail(self.cb_load_thumbnail)

class ResizableImageListRowRendererWidget(ImageListRowRenderer, ResizableRowRenderer):
    row_group="list_item_flickr_resizeable"

    def __init__(self, parent, theme=None):
        ImageListRowRenderer.__init__(self, parent, theme)

class ResizablePhotosetsRowRendererWidget(PhotosetsRowRenderer, ResizableRowRenderer):
    row_group="list_item_photosets_flickr_resizeable"

    def __init__(self, parent, theme=None):
        ImageListRowRenderer.__init__(self, parent, theme)

class ResizableUsersRowRendererWidget(UsersRowRenderer, ResizableRowRenderer):
    row_group="list_item_users_flickr_resizeable"

    def __init__(self, parent, theme=None):
        ImageListRowRenderer.__init__(self, parent, theme)

class ImageListRowRendererWidget(ImageListRowRenderer):
    row_group="list_item_flickr"
    resizable_row_renderer = ResizableImageListRowRendererWidget

class PhotosetsRowRendererWidget(PhotosetsRowRenderer):
    row_group="list_item_photosets_flickr"
    resizable_row_renderer = ResizablePhotosetsRowRendererWidget

class UsersRowRendererWidget(UsersRowRenderer):
    row_group="list_item_users_flickr"
    resizable_row_renderer = ResizableUsersRowRendererWidget

class ListController(BaseListController, OptionsControllerMixin):
    """Flickr Navigation List.

    This list is identical to Canola Text List, with a special treatment for
    the Search item, that popup a Search Modal before go to the next screen.

    @see: BaseListController
    """
    terra_type = "Controller/Folder/Task/Image/Flickr"

    def __init__(self, model, canvas, parent):
        BaseListController.__init__(self, model, canvas, parent)
        OptionsControllerMixin.__init__(self)
        self.model.callback_info = self.cb_info
        self.model.callback_notify = self.cb_notify

    def cb_info(self, msg):
        """Display a message in the screen."""
        self.view.part_text_set("message", msg)

    def cb_notify(self, err):
        """Display a message in a notify window."""
        self.parent.show_notify(err)

    def cb_on_clicked(self, view, index):
        model = self.model.children[index]

        search_types = [ SearchGroupsModelFolder, SearchByTagModelFolder ]

        if (not type(model) in search_types) or (not model.query == None):
            BaseListController.cb_on_clicked(self, view, index)
            return

        def do_search(ignored, text):
            if text is not None:
                model.query = text
                BaseListController.cb_on_clicked(self, view, index)

        if type(model) is SearchGroupsModelFolder:
            title       = "Search groups"
            description = "Enter keywords or group-names:"
        else:
            title       = "Search for tags"
            description = "Enter tags:"

        dialog = EntryDialogModel(title, description, answer_callback=do_search)
        self.parent.show_notify(dialog)

    def options_model_get(self):
        if isinstance(self.model, PhotosetModelFolder):
            return PhotosetOptionModel(None, self)
        elif isinstance(self.model, GroupModelFolder):
            return GroupOptionModel(None, self)
        elif isinstance(self.model, ContactModelFolder):
            return UserOptionModel(None, self)
        elif isinstance(self.model, UploadsModelFolder):
            return UploadInfoOptionModel(None, self)

    def delete(self):
        self.model.callback_info = None
        self.model.callback_notify = None
        BaseListController.delete(self)
        OptionsControllerMixin.delete(self)

class ImageListController(ListController):
    """
        Flickr Image Listing
    """
    terra_type = "Controller/Folder/Task/Image/Flickr/Service/Image"
    row_renderer = ImageListRowRendererWidget

class PhotosetListController(ListController):
    """
        Flickr Photoset Listing
    """
    terra_type = "Controller/Folder/Task/Image/Flickr/Service/Photoset"
    row_renderer = PhotosetsRowRendererWidget

class UserListController(ListController):
    """
        Flickr User Listing
    """
    terra_type = "Controller/Folder/Task/Image/Flickr/Service/User"
    row_renderer = UsersRowRendererWidget

class GroupListController(ListController):
    """
        Flickr Group Listing
    """
    terra_type = "Controller/Folder/Task/Image/Flickr/Service/Group"
    # TODO: create special GroupsRowRenderer (currently UsersRowRendererWidget does what we want)
    #row_renderer = GroupssRowRendererWidget
    row_renderer = UsersRowRendererWidget

class ModalMessageController(ModalController):
    terra_type = "Controller/Modal"

    def __init__(self, model, canvas, parent, message):
        ModalController.__init__(self, model, canvas, parent)

        self.view = ModalMessage(parent.last_panel.view,
                                 message, hborder=60, vborder=120)
        self.callback_leave = None
        self.view.callback_escape = self.close
        self.view.callback_clicked = self.close
        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back(self.callback_leave)
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()

class ImageViewerController(BaseImageViewerController):
    """Flickr Image viewer Controller.

    This class extends the Canola Image viewer."""
    terra_type = "Controller/Media/Image/Flickr"

    def __init__(self, model, canvas, parent):
        BaseImageViewerController.__init__(self, model, canvas, parent)
        self.width, self.height = canvas.size

    def options_model_get(self):
        return ViewerOptionsModel(None, self)

    def reload_favorite_option_model(self):
        # reload FavoriteOptionModel to show the correct title
        for child in self.options_model.children:
            if isinstance(child, FavoriteOptionModel):
                log.debug("reloading FavoriteOptionModel")
                child.do_load()

    def load_prev(self, end_callback=None):
        log.debug("switching to previous photo!")
        res = BaseImageViewerController.load_prev(self, end_callback=end_callback)
        self.reload_favorite_option_model()
        return res

    def load_next(self, loop=False, end_callback=None):
        log.debug("switching to next photo (loop: %s)!" % loop)
        res = BaseImageViewerController.load_next(self, loop=loop, end_callback=end_callback)
        self.reload_favorite_option_model()
        return res
