# Canola2 Flickr Plugin
# Copyright (C) 2008 Thomas Schmidt <tschmidt@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import evas
import logging

from terra.core.manager import Manager
from terra.ui.base import PluginEdjeWidget
from terra.ui.base import PluginThemeMixin
from terra.utils.encoding import to_utf8

from manager import FlickrManager

manager = Manager()

from terra.ui.modal import ModalMessage

ModalController         = manager.get_class("Controller/Modal")
BasicPanel              = manager.get_class("Controller/BasicPanel")
BaseScrollableText      = manager.get_class("Widget/ScrollableTextBlock")
ControllerOptionsFolder = manager.get_class("Controller/Options/Folder")
PanelButtonWidget       = manager.get_class("Widget/PanelButton")

BaseSlideshowModeOptionsCheckedController   = manager.get_class("Controller/Options/Folder/Image/Fullscreen/Submenu/SlideshowMode")
BaseSlideshowTimeOptionController           = manager.get_class("Controller/Options/Folder/Image/Fullscreen/Submenu/SlideshowTime")

log = logging.getLogger("plugins.canola-flickr.options")

class ModalMessageController(ModalController):
    def __init__(self, model, canvas, parent, message):
        ModalController.__init__(self, model, canvas, parent)

        self.view = ModalMessage(parent.last_panel.view,
                                 message, hborder=60, vborder=120)
        self.callback_leave = None
        self.view.callback_escape = self.close
        self.view.callback_clicked = self.close
        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back(self.callback_leave)
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()

class ScrollableTextBlock(PluginThemeMixin, BaseScrollableText):
    plugin = "flickr"
    group = "textblock_description"

class InfoOptionController(BasicPanel):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/Info"

    def __init__(self, model, canvas, parent, theme=None, edje_group="panel_info_flickr"):
        BasicPanel.__init__(self, model, canvas, parent)

        self.thumbnail = evas.FilledImage(canvas)

        self._body = PluginEdjeWidget(self.view.evas, edje_group, self.view, plugin="flickr")

        self.description = ScrollableTextBlock(self.view.evas, self.view)
        self._body.part_swallow("description", self.description)

        self.model = model

        self.inner_contents_set(self._body)
        self.setup_information()

    def setup_information(self):
        photo = self.model.get_photo()
        if not photo.have_info:
            photo = self.model.flickr_manager.get_photo_info(photo)
            self.model.set_photo(photo)

        self._body.part_text_set("title", to_utf8(photo.title))
        self._body.part_text_set("author", "From " + to_utf8(photo.owner.__repr__()))
        self._body.part_text_set("date_taken", "Taken on " + photo.date_taken.strftime("%b %d, %Y"))
        self._body.part_swallow("contents", self.thumbnail)

        if photo.description:
            text = "Description:<br><br>"
            text = text + photo.description.replace("\n", "<br>") + "<br><br>"
        else:
            text = ""

        if len(photo.tags) > 0:
            text = text + "Tags:<br><br>"
            for tag in photo.tags:
                text = text + tag + "<br>"
            text = text + "<br>"

        dim = "Dimensions: %sx%s px" % (
                self.model.get_image_model().width,
                self.model.get_image_model().height
            )
        text = text + dim

        self.description.text_set(to_utf8(text))

        try:
            thumbnail_path = self.model.get_image_model().get_thumbnail()
            self.thumbnail.file_set(thumbnail_path)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

    def delete(self):
        self._body.delete()
        self.thumbnail.delete()
        self.description.delete()
        BasicPanel.delete(self)

class ExifInfoOptionController(InfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/ExifInfo"

    def setup_information(self):
        photo = self.model.get_photo()
        if not photo.have_exif:
            photo = self.model.flickr_manager.get_exif_info(photo)
            self.model.set_photo(photo)

        self._body.part_text_set("title", to_utf8(photo.title))
        self._body.part_text_set("author", "From " + to_utf8(photo.owner.__repr__()))
        self._body.part_text_set("date_taken", "Taken on " + photo.date_taken.strftime("%b %d, %Y"))
        self._body.part_swallow("contents", self.thumbnail)

        if len(photo.exif) > 0:
            text = ""
            for exif_tag in photo.exif:
                text = text + exif_tag + "<br><br>"
            text = text.replace("\n", " ")
        else:
            text = "No Exif Tags found!"

        self.description.text_set(to_utf8(text))

        try:
            thumbnail_path = self.model.get_image_model().get_thumbnail()
            self.thumbnail.file_set(thumbnail_path)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

class FavoriteOptionController(ModalMessageController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/Favorite"

    def __init__(self, model, canvas, parent):
        if model.execute():
            if model.is_favorite:
                msg = "This image has been removed from your favorites"
            else:
                msg = "This image has been added to your favorites"
            model.is_favorite = not model.is_favorite
            model.set_title()
        else:
            if model.is_favorite:
                msg = "Could not remove this image from your favorites"
            else:
                msg = "Could not add this image to your favorites"
        ModalMessageController.__init__(self, model, canvas, parent, msg)

    def close(self):
        ModalMessageController.close(self)

class PhotosetInfoOptionController(InfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Photoset/Info"

    def setup_information(self):
        photoset_model  = self.model.screen_controller.model
        photoset        = photoset_model.photoset

        self._body.part_text_set("title", to_utf8(photoset.title))
        self._body.part_text_set("author", "From " + to_utf8(photoset.owner.__repr__()))
        self._body.part_text_set("date_taken", "Photos: %s" % photoset.photos)
        self._body.part_swallow("contents", self.thumbnail)

        if photoset.description == None:
            text = "No description available."
        else:
            text = photoset.description.replace("\n", "<br>")

        self.description.text_set(to_utf8(text))

        try:
            thumbnail_path = photoset_model.get_thumbnail()
            self.thumbnail.file_set(thumbnail_path)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

class UserInfoOptionController(InfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/User/Info"

    def __init__(self, model, canvas, parent, theme=None):
        InfoOptionController.__init__(self, model, canvas, parent, theme=None, edje_group="panel_info_user_flickr")

    def setup_information(self):
        user_model  = self.model.screen_controller.model
        user        = user_model.user

        if not user.have_info:
            self.model.flickr_manager.get_userinfo(user)

        self._body.part_text_set("username", to_utf8(user.__repr__()))

        if user.ispro: ispro_text = "Yes"
        else: ispro_text = "No"

        self._body.part_text_set("ispro", to_utf8("Pro user: %s" % ispro_text))

        self._body.part_text_set("photos", to_utf8("Photos: %s" % user.photo_count))

        self._body.part_swallow("contents", self.thumbnail)

        if user.location == None: location_text = "unknown"
        else: location_text = user.location

        text = "Location:<br><br>%s<br><br>" % location_text

        if user.first_upload == None:
            upload_text = "%s has not uploaded anything yet." % user.__repr__()
        else:
            upload_text = "First upload on %s." % user.first_upload.strftime("%b %d, %Y")

        text = text + upload_text + "<br><br>"

        if user.iscontact:
            if user.family and user.friend: text = text + "Marked as family and friend."
            elif user.family: text = text + "Marked as family."
            elif user.friend: text = text + "Marked as friend."

        self.description.text_set(to_utf8(text))

        try:
            thumbnail_path = user_model.get_thumbnail()
            self.thumbnail.file_set(thumbnail_path)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

class GroupInfoOptionController(UserInfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Group/Info"

    def setup_information(self):
        group_model = self.model.screen_controller.model
        group       = group_model.group

        if not group.have_info:
            self.model.flickr_manager.get_groupinfo(group)

        self._body.part_text_set("username", to_utf8(group.name))

        member_text = to_utf8("Members: %s" % group.members)
        self._body.part_text_set("ispro", member_text)

        if group.ispoolmoderated: moderated_text = "Moderated group pool."
        else: moderated_text = "Unmoderated group pool."

        self._body.part_text_set("photos", to_utf8(moderated_text))
        self._body.part_swallow("contents", self.thumbnail)

        if group.description == None:
            text = "No description available."
        else:
            text = group.description.replace("\n", "<br>")

        self.description.text_set(to_utf8(text))

        try:
            thumbnail_path = group_model.get_thumbnail()
            self.thumbnail.file_set(thumbnail_path)
            self._body.signal_emit("thumb,show", "")
        except Exception, e:
            self._body.signal_emit("thumb,hide", "")

class CommentItemController(InfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/CommentsList/Item"

    def __init__(self, model, canvas, parent, theme=None):
        BasicPanel.__init__(self, model, canvas, parent)
        self._body = PluginEdjeWidget(self.view.evas, "panel_comment_flickr",
                                     self.view, plugin="flickr")
        self.description = ScrollableTextBlock(self.view.evas, self.view)
        self._body.part_swallow("description", self.description)
        self.inner_contents_set(self._body)
        self.setup_information()

    def setup_information(self):
        model = self.model.screen_controller.model
        self._body.part_text_set("author", to_utf8("From: %s" % self.model.authorname))
        self._body.part_text_set("date", to_utf8("Date: %s" % self.model.published.strftime("%c")))

        comment_panel = self.model.content
        comment_panel = comment_panel.replace("\n", "<br>")
        self.description.text_set(to_utf8(comment_panel))

    def delete(self):
        self._body.delete()
        self.description.delete()
        BasicPanel.delete(self)

class CommentsListOptionController(ControllerOptionsFolder):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/CommentsList"

    def page_ground(self):
        page_ground = (self.model.page - 1) * self.model.comments_per_page + 1
        return page_ground

    def set_msg_tit(self):
        if self.model.num_comments > (self.page_ground() + self.model.comments_per_page - 1):
            total_comments = (self.page_ground() + self.model.comments_per_page - 1)
        else:
            total_comments = self.model.num_comments

        self.msg_tit = "Comments %s to %s of %s" % (self.page_ground(), total_comments, self.model.num_comments)

    def _cb_button_clicked_prev(self, *args):
        def cb_finished(*ignored):
            self.set_msg_tit()
            self.view.header_text_set(self.msg_tit)
            if self.model.page == 1:
                self.prev_button.hide()
            if self.model.page*self.model.comments_per_page >= self.model.num_comments:
                self.next_button.hide()
            else:
                self.next_button.show()

        if not self.model.is_loading:
            log.debug('show previous comments')
            self.view.header_text_set("Loading previous comments...")
            self.model.do_prev(cb_finished)

    def _cb_button_clicked_next(self, *args):
        def cb_finished(*ignored):
            self.set_msg_tit()
            self.view.header_text_set(self.msg_tit)
            self.prev_button.show()
            if self.model.page * self.model.comments_per_page >= self.model.num_comments:
                self.next_button.hide()

        if not self.model.is_loading:
            log.debug('show next comments')
            self.view.header_text_set("Loading next comments...")
            self.model.do_next(cb_finished)

    def _setup_view(self):
        ControllerOptionsFolder._setup_view(self)

        if self.model.num_comments > 0:
            self.set_msg_tit()
            self.view.header_text_set(self.msg_tit)
            backbt = PanelButtonWidget(self.evas, "Previous",  \
                   clicked_callback=self._cb_button_clicked_prev, \
                   parent=self.view)
            nextbt = PanelButtonWidget(self.evas, "Next",  \
                   clicked_callback=self._cb_button_clicked_next, \
                   parent=self.view)
            self.view._setup_button_box(left=backbt, right=nextbt)
            self.prev_button = backbt
            self.next_button = nextbt
            if self.model.page == 1:
                self.prev_button.hide()
            if self.model.page*self.model.comments_per_page >= self.model.num_comments:
                self.next_button.hide()
        else:
            self.msg_tit = "No comments for this photo"
            self.view.header_text_set(self.msg_tit)

class UploadStatusOptionController(InfoOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Upload/Status"

    def setup_information(self):
        if self.model.flickr_manager.upload_status == None:
            self.model.flickr_manager.get_uploadstatus()

        upload_status = self.model.flickr_manager.upload_status

        self._body.part_text_set("title", "")
        self._body.part_text_set("author", to_utf8(self.model.flickr_manager.user.__repr__()))
        self._body.part_text_set("date_taken", "")
        self._body.part_swallow("contents", self.thumbnail)

        text = "Max bytes/month: %s" % upload_status.upload_max
        text = text + "<br>Used bytes/month: %s" % upload_status.upload_used
        text = text + "<br>Sets created: %s" % upload_status.sets_created
        text = text + "<br>Sets remaining: %s" % upload_status.sets_remaining

        self.description.text_set(to_utf8(text))

        #try:
        #    thumbnail_path = photoset_model.get_thumbnail()
        #    self.thumbnail.file_set(thumbnail_path)
        #    self._body.signal_emit("thumb,show", "")
        #except Exception, e:
        #    self._body.signal_emit("thumb,hide", "")

class UploadOptionController(UserInfoOptionController):
    terra_type = "Controller/Options/Folder/Image/Fullscreen/Upload"

    def setup_information(self):

        self._body.part_text_set("username", to_utf8("Upload-Test"))

        self._body.part_text_set("ispro", "")

        self._body.part_text_set("photos", "")
        self._body.part_swallow("contents", self.thumbnail)

        text = "Image: %s" % self.model.get_image_model().path
        self.description.text_set(to_utf8(text))

        #try:
        #    thumbnail_path = group_model.get_thumbnail()
        #    self.thumbnail.file_set(thumbnail_path)
        #    self._body.signal_emit("thumb,show", "")
        #except Exception, e:
        #    self._body.signal_emit("thumb,hide", "")

class SlideshowModeOptionsCheckedController(BaseSlideshowModeOptionsCheckedController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/SlideshowMode"

class SlideshowTimeOptionController(BaseSlideshowTimeOptionController):
    terra_type = "Controller/Options/Folder/Player/Flickr/Image/SlideshowTime"

