# Canola2 Flickr Plugin
# Copyright (C) 2008 Thomas Schmidt <tschmidt@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os, logging, urllib, ecore, datetime, Image, shlex, locale

from terra.core.task import Task
from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction

from manager import FlickrManager, DownloadManager
from flickr import Photo

mger = Manager()

flickr_manager      = FlickrManager()
download_manager    = DownloadManager(flickr_manager=flickr_manager)

from helper import Xlator

html_replace_dict = {
        "<br>" : " ",
        "<i>" : "",
        "</i>" : "",
        "<b>" : "",
        "</b>" : "",
        "&lt;" : "<",
        "&gt;" : ">",
        "&amp;" : "&",
        "&hearts;" : "",
        "<strong>" : "",
        "</strong>" : "",
        "&quot;" : "\"",
        "\n" : " ",
        "</img>" : "",
        "</a>" : "",
        "<em>" : "",
        "</em>" : "",
        "<u>" : "",
        "</u>" : "",
    }

html_remover = Xlator(html_replace_dict)

network             = mger.get_status_notifier("Network")
PluginDefaultIcon   = mger.get_class("Icon/Plugin")
CanolaError         = mger.get_class("Model/Notify/Error")
ImageLocalModel     = mger.get_class("Model/Media/Image")
OptionsModelFolder  = mger.get_class("Model/Options/Folder")
SystemProps         = mger.get_class("SystemProperties")

BaseSlideshowModeOptionsModel       = mger.get_class("Model/Options/Folder/Image/Fullscreen/Submenu/SlideshowMode")
BaseSlideshowTimeOptionModelFolder  = mger.get_class("Model/Options/Folder/Image/Fullscreen/Submenu/SlideshowTime")
BaseSlideshowRandomOptionModel      = mger.get_class("Model/Options/Action/Image/Fullscreen/SlideshowMode/SlideshowRandom")
BaseSlideshowLoopOptionModel        = mger.get_class("Model/Options/Action/Image/Fullscreen/SlideshowMode/SlideshowLoop")

sysprops            = SystemProps()

locale.setlocale(locale.LC_ALL, "C")
log = logging.getLogger("plugins.canola-flickr.model")

def space_available(path, bytes_required=1024**2):
    try:
        bytes_avail = sysprops.bytes_available(path)
        return (bytes_avail >= bytes_required)
    except OSError:
        return False

class NoSpaceException(Exception):
   def __init__(self, value):
        self.dir = value
   def __str__(self):
       return "Not enough space available in directory %s" % repr(self.dir)

class ImageModel(ImageLocalModel):
    terra_type = "Model/Media/Image/Flickr"

    def __init__(self, photo, parent, no_download=False):
        self.thumbnail      = None
        self.photo          = photo
        self.no_download    = no_download
        self.flickr_manager = flickr_manager

        try:
            self.owner = self.flickr_manager.users[self.photo.owner.nsid]
        except KeyError:
            log.debug("Error, owner of %s not cached!" % self.photo)
            self.owner = self.photo.owner

        res = self.get_photo()
        if res == None: return
        else: path, width, height = res

        title = to_utf8(self.photo.title)
        ImageLocalModel.__init__(self, title, parent)
        self.path   = path
        self.width  = width
        self.height = height

    def get_photo(self):
        # check if we already have a image-file for this
        # photo in the download-folder
        photopath = self.flickr_manager.search_imagefile(self.photo)

        if photopath:
            path = photopath
            log.debug("found image in local cache! (%s)" % path)
            try:
                width, height = Image.open(path).size

                # TODO: find a clever way to avoid the "this photo is
                #       currently unavailable image", the code below is
                #       just a workaround
                if ((width == 500) and (height == 375) and (os.stat(path).st_size == 2737)):
                    log.debug("deleting %s, seems to be the \"this photo is currently unavailable image\"" % path)
                    os.unlink(path)
                    # TODO: this might lead to an endless loop
                    return self.get_photo()
            except IOError:
                # try to re-download file!!!
                log.debug("Error opening image file %s, deleting this one!!!" % path)
                if os.path.exists(old_path): os.unlink(path)

                # TODO: this might lead to an endless loop
                return self.get_photo()
        else:
            # do not try to download more images
            if self.no_download:
                return None

            image_dir = self.flickr_manager.get_image_dir()
            if space_available(image_dir):
                photo = self.photo

                url_prefix = "http://farm%s.static.flickr.com/%s/%s_%s" % (photo.icon_farm, photo.server, photo.id, photo.secret)

                url_large       = "%s_b.jpg" % url_prefix
                url_medium      = "%s.jpg" % url_prefix
                url_small       = "%s_m.jpg" % url_prefix
                url_thumbnail   = "%s_t.jpg" % url_prefix

                for url in [ url_large, url_medium, url_small, url_thumbnail ]:
                    log.debug("trying to download %s" % url)
                    path = download_manager.download_image(url)

                    # abort downloading is self.no_download is True, delete downloaded image, because
                    # we can not be certain that it has the size we want
                    if self.no_download:
                        if not path == None:
                            if os.path.exists(path): os.unlink(path)
                            return None

                    if not path == None:
                        width, height = Image.open(path).size
                        # TODO: find a clever way to avoid the "this photo is
                        #       currently unavailable image", the code below is
                        #       just a workaround
                        if ((width == 500) and (height == 375) and (os.stat(path).st_size == 2737)):
                            log.debug("deleting %s, seems to be the \"this photo is currently unavailable image\"" % path)
                            os.unlink(path)
                            continue
                        else: break
                else:
                    log.error("could not get valid url for photo: %s" % photo)
                    return None

                try:
                    width, height = Image.open(path).size
                except IOError:
                    # try to re-download file!!!
                    log.debug("Error opening image file %s, deleting this one!!!" % path)
                    if os.path.exists(old_path): os.unlink(path)

                    # TODO: this might lead to an endless loop
                    return self.get_photo()

                # compare width, height with desired res, if it is less than threshold_min, try to get the original image
                device_res      = 800*480
                threshold_min   = int(0.7*device_res)
                threshold_max   = 3*device_res
                photo_res       = width*height

                if photo_res < threshold_min:
                    old_path = path

                    log.debug("getting sizes-info of photo (title: %s) (photo_res: %s px, theshold_min: %s px)" % (photo.title, photo_res, threshold_min))
                    sizes   = self.flickr_manager.get_photosizes(photo)
                    size    = sizes[-1]

                    if size.label == "Video Player":
                        # Seems we got an video, use the size we already downloaded
                        log.debug("\"photo\": %s is a video, ignoring other sizes!" % photo.title)
                        return path, width, height

                    orig_res = int(size.width)*int(size.height)
                    if orig_res > photo_res and orig_res < threshold_max:
                        log.debug("downloading original size of photo (title: %s) (orig_res: %s px)" % (photo.title, orig_res))
                        path = download_manager.download_image(size.url)
                        if path == None:
                            log.error("could not download original size of photo (title: %s) (url: %s)" % (photo.title, size.url))

                            # return the smaller image
                            return path, width, height
                        else:
                            # remove smaller image, as we now have the image in the original format
                            if os.path.exists(old_path): os.unlink(old_path)

                        width   = int(size.width)
                        height  = int(size.height)
                    elif orig_res > photo_res:
                        log.debug("ignoring original size of photo (title: %s), original resolution (%s px) is bigger than threshold_max (%s px)" % (photo.title, orig_res, threshold_max))
                    else:
                        log.debug("original size of photo (title: %s) is not bigger than already downloaded version (%s px / %s px)" % (photo.title, orig_res, photo_res))
            else:
                # raise exception to abort adding more images in ImagesListing
                raise NoSpaceException(image_dir)
                return None

        return path, width, height

    def get_thumbnail(self):
        if not self.thumbnail == None:
            return self.thumbnail

        filename = "%s_%s_s.jpg" % (self.photo.id, self.photo.secret)
        thumbnail = os.path.join(self.flickr_manager.get_image_dir(thumbnails=True), filename)
        if os.path.exists(thumbnail):
            self.thumbnail = thumbnail
            return thumbnail
        else:
            image_dir = self.flickr_manager.get_image_dir(thumbnails=True)
            if space_available(image_dir, bytes_required=1024*50):
                photo = self.photo
                url = "http://farm%s.static.flickr.com/%s/%s" % (photo.icon_farm, photo.server, filename)
                self.thumbnail = download_manager.download_image(url, thumbnail=True)
                return self.thumbnail
            else:
                # raise exception to abort adding more images in ImagesListing
                raise NoSpaceException(image_dir)
                return None

    def request_thumbnail(self, end_callback=None):
        def request(*ignored):
            self.get_thumbnail()

        def request_finished(exception, retval):
            if end_callback:
                end_callback()

        ThreadedFunction(request_finished, request).start()

class MainModelFolder(Task, ModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr"
    terra_task_type = "Task/Image/Flickr"
    title = "Flickr"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "Flickr", parent)

    def do_load(self):
        def refresh():
            # Try to log in to Flickr
            if not flickr_manager.is_logged():
                res = flickr_manager.login()
                if res == True: pass
                elif res == False: return

            res = flickr_manager.is_logged()

            if res:
                # get favorites
                if not flickr_manager.favorites:
                    favorites = flickr_manager.get_favorites(per_page=500)
                    if favorites and favorites.photos:
                        for photo in favorites.photos:
                            flickr_manager.favorites.append(photo.id)
                    else: flickr_manager.has_favorites = False

            return res

        def refresh_finished(exception, retval):
            if not retval or exception:
                self.inform_loaded()
                message = "You are not logged in.<br>"\
                          "Please switch to your browser and allow Canola-flickr access to your flickr account!"
                self.callback_info(message)
                return

            LatestImagesListModelFolder(self)
            PhotosetsListModelFolder(self)
            FavoritesModelFolder(self)
            ContactsModelFolder(self)
            GroupsModelFolder(self)
            ExploreModelFolder(self)
            SearchesModelFolder(self)
            #UploadsModelFolder(self)

            self.inform_loaded()

        self.is_loading = True
        ThreadedFunction(refresh_finished, refresh).start()

class ServiceModelFolder(Task, ModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr/Service"
    terra_task_type = "Task/Image/Flickr"

    threaded_search = True

    def __init__(self, name, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, name, parent)
        self.changed = False
        self.callback_search_finished = None
        self.callback_notify = None
        self.empty_msg = None
        self.flickr_manager = flickr_manager

    def do_load(self):
        self.search()

    def search(self, end_callback=None):
        def refresh():
            return self.do_search()

        def refresh_finished(exception, retval):
            self.inform_loaded()
            if not retval or exception:
                if not self.empty_msg == None:
                    self.callback_info(self.empty_msg)
            return

        self.is_loading = True
        ThreadedFunction(refresh_finished, refresh).start()

    def do_search(self):
        raise NotImplementedError("must be implemented by subclasses")

class ContactsModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr/Service/User"

    def __init__(self, parent, user=None, per_page=100):
        name = to_utf8("Contacts")
        ServiceModelFolder.__init__(self, name, parent)

        self.empty_msg  = "No contacts found."
        self.user       = user
        self.per_page   = per_page

    def do_search(self):
        contactlist = self.flickr_manager.get_contacts(self.user, per_page=self.per_page)

        if len(contactlist.contacts) > 0:
            # only add ContactsRecentPhotosModelFolder in
            # case we show our own contacts
            #if self.user == None:
            #    ContactsRecentPhotosModelFolder(self)

            for contact in contactlist.contacts:
                ContactModelFolder(self, contact=contact)
            return True
        else:
            return False

class ContactModelFolder(ServiceModelFolder):
    def __init__(self, parent, contact):
        self.user       = contact
        self.thumbnail  = None
        self.has_avatar = True

        title = contact.__repr__()
        ServiceModelFolder.__init__(self, to_utf8(title), parent)

    def do_search(self):
        LatestImagesListModelFolder(self, user=self.user)
        PhotosetsListModelFolder(self, user=self.user)
        FavoritesModelFolder(self, user=self.user)
        ContactsModelFolder(self, user=self.user)
        GroupsModelFolder(self, user=self.user)

    def get_thumbnail(self):
        if not self.thumbnail == None:
            return self.thumbnail

        # do not try to download the avatar again, if has_avatar = False
        if not self.has_avatar:
            return None

        self.thumbnail = flickr_manager.search_buddyicon(self.user)
        if not self.thumbnail == None:
            return self.thumbnail
        else:
            image_dir = flickr_manager.get_image_dir(buddyicons=True)
            if space_available(image_dir, bytes_required=1024*50):
                url = flickr_manager.get_buddyicon_url(self.user)
                self.thumbnail = download_manager.download_image(url, buddyicon=True)

                # if we did not receive an buddyicon at the first try,
                # do not try to download it again
                if self.thumbnail == None:
                    self.has_avatar = False

                return self.thumbnail
            else:
                raise NoSpaceException(image_dir)
                self.has_avatar = False
                return None

    def request_thumbnail(self, end_callback=None):
        def request(*ignored):
            self.get_thumbnail()

        def request_finished(exception, retval):
            if end_callback:
                end_callback()

        ThreadedFunction(request_finished, request).start()

class TagsModelFolder(ServiceModelFolder):
    def __init__(self, parent):
        title = "Tags"
        ServiceModelFolder.__init__(self, to_utf8(title), parent)

    def do_search(self):
        HottagModelFolder(self)
        HottagModelFolder(self, period="week")

class HottagModelFolder(ServiceModelFolder):
    def __init__(self, parent, period="day", count=20):
        if period == "day": title = "Hot tags (today)"
        else: title = "Hot tags (this week)"

        self.period = period
        self.count  = count

        ServiceModelFolder.__init__(self, to_utf8(title), parent)

    def do_search(self):
        hottags = self.flickr_manager.get_hottags(period=self.period, count=self.count)
        for score, tag in hottags.tags:
            SearchByTagModelFolder(self, all=True, query=tag, auto_reset=False)

class GroupsModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr/Service/Group"

    def __init__(self, parent, user=None, title=None):
        if title: name = to_utf8(title)
        else: name = to_utf8("Groups")
        ServiceModelFolder.__init__(self, name, parent)

        self.empty_msg  = "No groups found."
        self.user       = user

    def add_groups(self, grouplist):
        if len(grouplist.groups) > 0:
            for group in grouplist.groups:
                GroupModelFolder(self, group=group)
            return True
        else:
            return False

    def do_search(self):
        grouplist = self.flickr_manager.get_groups(self.user)
        return self.add_groups(grouplist)

class ImageListModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr/Service/Image"

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)

        self.empty_msg  = "No photos found."
        self.unloaded   = False

    def add_photos(self, photolist):
        no_download = False

        if photolist == None or len(photolist.photos) == 0:
            return False

        for photo in photolist.photos:
            if self.unloaded:
                # TODO: set no_download attribute in all children to True (needs Thread)
                break

            try:
                ImageModel(photo=photo, parent=self, no_download=no_download)
            except NoSpaceException:
                self.empty_msg = "No space left, please check your<br>memory card and delete something! :)"

                # do not try to download more images
                no_download = True

                return False

        return True

    def do_load(self):
        self.unloaded = False
        ServiceModelFolder.do_load(self)

    def do_search(self):
        return []

    def do_unload(self):
        self.unloaded = True
        ServiceModelFolder.do_unload(self)

class GroupModelFolder(ImageListModelFolder):
    def __init__(self, parent, group, per_page=100):
        self.group      = group
        self.per_page   = per_page
        self.thumbnail  = None
        self.has_avatar = True

        title = to_utf8(group.name)
        ImageListModelFolder.__init__(self, title, parent)

    def do_search(self):
        photolist = self.flickr_manager.get_photos_grouppool(group=self.group, per_page=self.per_page)
        return self.add_photos(photolist)

    def get_thumbnail(self):
        if not self.thumbnail == None:
            return self.thumbnail

        # do not try to download the avatar again, if has_avatar = False
        if not self.has_avatar:
            return None

        self.thumbnail = flickr_manager.search_buddyicon(self.group)
        if not self.thumbnail == None:
            return self.thumbnail
        else:
            image_dir = flickr_manager.get_image_dir(buddyicons=True)
            if space_available(image_dir, bytes_required=1024*50):
                url = flickr_manager.get_buddyicon_url(self.group, group=True)
                self.thumbnail = download_manager.download_image(url, buddyicon=True)

                # if we did not receive an buddyicon at the first try,
                # do not try to download it again
                if self.thumbnail == None:
                    self.has_avatar = False

                return self.thumbnail
            else:
                raise NoSpaceException(image_dir)
                self.has_avatar = False
                return None

    def request_thumbnail(self, end_callback=None):
        def request(*ignored):
            self.get_thumbnail()

        def request_finished(exception, retval):
            if end_callback:
                end_callback()

        ThreadedFunction(request_finished, request).start()

class FavoritesModelFolder(ImageListModelFolder):
    def __init__(self, parent, user=None, per_page=50):
        name = to_utf8("Favorites")
        ImageListModelFolder.__init__(self, name, parent)

        self.empty_msg  = "No favorites found."
        self.user       = user
        self.per_page   = per_page

    def do_search(self):
        photolist = self.flickr_manager.get_favorites(user=self.user, per_page=self.per_page)
        return self.add_photos(photolist)

class LatestImagesListModelFolder(ImageListModelFolder):
    def __init__(self, parent, user=None, per_page=30):
        title = to_utf8("Photostream")
        ImageListModelFolder.__init__(self, title, parent)

        self.user       = user
        self.per_page   = per_page

    def do_search(self):
        photolist = self.flickr_manager.get_photos_latest(user=self.user, per_page=self.per_page)
        return self.add_photos(photolist)

class ExploreModelFolder(ServiceModelFolder):
    def __init__(self, parent):
        title = "Explore"
        ServiceModelFolder.__init__(self, to_utf8(title), parent)

    def do_search(self):
        timedelta = datetime.timedelta(days=1)
        today = datetime.datetime.now()

        ExplorePhotosModelFolder(self)
        ExplorePhotosModelFolder(self, date=today-timedelta)
        ExplorePhotosModelFolder(self, date=today-2*timedelta)
        ExplorePhotosModelFolder(self, date=today-3*timedelta)
        ExplorePhotosModelFolder(self, date=today-4*timedelta)
        ExplorePhotosModelFolder(self, date=today-5*timedelta)
        ExplorePhotosModelFolder(self, date=today-6*timedelta)
        RecentPhotosModelFolder(self)
        TagsModelFolder(self)

class ExplorePhotosModelFolder(ImageListModelFolder):
    def __init__(self, parent, page=1, per_page=100, date=None):
        if date == None:
            title = to_utf8("Recent interesting photos")
        else:
            today       = datetime.datetime.now()
            timedelta   = today - date

            if timedelta.days == 1:
                title = to_utf8("Yesterday's interesting photos")
            else:
                weekday = date.strftime("%A")
                title   = to_utf8("Interesting photos from %s" % weekday)

        ImageListModelFolder.__init__(self, title, parent)

        self.page       = page
        self.per_page   = per_page
        self.date       = date

        if not self.date == None:
            self.empty_msg  = "No interesting photos from<br>%s" % date.strftime("%A, %x")

    def do_search(self):
        photolist = self.flickr_manager.get_photos_interesting(page=self.page, per_page=self.per_page, date=self.date)
        return self.add_photos(photolist)

class RecentPhotosModelFolder(ImageListModelFolder):
    def __init__(self, parent, page=1, per_page=100, date=None):
        title = to_utf8("Recently uploaded photos")
        ImageListModelFolder.__init__(self, title, parent)

        self.page       = page
        self.per_page   = per_page

    def do_search(self):
        photolist = self.flickr_manager.get_photos_recent(page=self.page, per_page=self.per_page)
        return self.add_photos(photolist)

class ContactsRecentPhotosModelFolder(ImageListModelFolder):
    def __init__(self, parent, count=20, include_self=False, just_friends=False):
        title = to_utf8("Recent photos from contacts")
        ImageListModelFolder.__init__(self, title, parent)

        self.count          = count
        self.include_self   = include_self
        self.just_friends   = just_friends

    def do_search(self):
        photolist = self.flickr_manager.get_photos_contacts(count=self.count, include_self=self.include_self, just_friends=self.just_friends)
        return self.add_photos(photolist)

    def request_thumbnail(self, end_callback=None):
        return

class SearchesModelFolder(ServiceModelFolder):
    def __init__(self, parent):
        name = to_utf8("Search")
        ServiceModelFolder.__init__(self, name, parent)

    def do_search(self):
        SearchByTagModelFolder(self)
        SearchByTagModelFolder(self, contacts=True)
        SearchByTagModelFolder(self, all=True)
        SearchGroupsModelFolder(self)

class SearchModelFolder:
    """This model implements the Search option."""
    terra_type = "Model/Folder/Task/Image/Flickr/Service/Search"

    def convert_query(self, query):
        '''Convert query, respecting double and single quotes.'''
        item_list = list(shlex.shlex(query, posix=True))

        result = ""
        for item in item_list:
            result = result + item + ","
        result = result[:-1]
        return result

class SearchByTagModelFolder(ImageListModelFolder, SearchModelFolder):
    def __init__(self, parent, all=False, contacts=False, per_page=100, query=None, auto_reset=True):
        if not query == None:
            title = query
        elif all:
            title = "Search by tag (global)"
        elif contacts:
            title = "Search by tag (contacts)"
        else:
            title = "Search by tag"

        ImageListModelFolder.__init__(self, to_utf8(title), parent)
        self.query      = query
        self.all        = all
        self.contacts   = contacts
        self.per_page   = per_page
        self.auto_reset = auto_reset

    def do_search(self):
        if self.query:
            # replace spaces in query with commas, search_tags expects a comma-separated list
            query = self.convert_query(self.query)
            photolist = self.flickr_manager.search_tags(query, all=self.all, contacts=self.contacts, per_page=self.per_page)
            if self.auto_reset:
                self.query = None
            return self.add_photos(photolist)

class SearchGroupsModelFolder(GroupsModelFolder, SearchModelFolder):
    def __init__(self, parent, per_page=100):
        title = "Search groups"
        GroupsModelFolder.__init__(self, parent=parent, title=title)

        self.query = None
        self.per_page = per_page

    def do_search(self):
        if self.query:
            # replace spaces in query with commas, search_tags expects a comma-separated list
            query = self.convert_query(self.query)
            grouplist = self.flickr_manager.search_groups(query, per_page=self.per_page)
            self.query = None
            return self.add_groups(grouplist)

class PhotosetModelFolder(ImageListModelFolder):
    def __init__(self, parent, photoset, per_page=500, user="me"):
        name = to_utf8(photoset.title)

        self.photoset       = photoset
        self.per_page       = per_page
        self.user           = user
        self.thumbnail      = None
        self.primary_photo  = Photo(id=photoset.primary, title="")

        ImageListModelFolder.__init__(self, name, parent)

    def do_search(self):
        photolist = self.flickr_manager.get_photos_photoset(photoset=self.photoset, user=self.user, per_page=self.per_page)
        return self.add_photos(photolist)

    def get_thumbnail(self):
        if not self.thumbnail == None:
            return self.thumbnail

        self.thumbnail = flickr_manager.search_imagefile(self.primary_photo, thumbnail=True)
        if not self.thumbnail == None:
            return self.thumbnail
        else:
            image_dir = flickr_manager.get_image_dir(thumbnails=True)
            if space_available(image_dir, bytes_required=1024*50):
                try:
                    photo = flickr_manager.photos[self.photoset.primary]
                except KeyError:
                    # photo not (yet) cached, get photo info
                    photo = flickr_manager.get_photo_info(self.primary_photo)

                if photo.icon_farm == None or photo.server == None or photo.secret == None:
                    return None

                url = "http://farm%s.static.flickr.com/%s/%s_%s_s.jpg" % (photo.icon_farm, photo.server, photo.id, photo.secret)
                self.thumbnail = download_manager.download_image(url, thumbnail=True)
                return self.thumbnail
            else:
                raise NoSpaceException(image_dir)
                return None

    def request_thumbnail(self, end_callback=None):
        def request(*ignored):
            self.get_thumbnail()

        def request_finished(exception, retval):
            if end_callback:
                end_callback()

        ThreadedFunction(request_finished, request).start()

class PhotosetsListModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr/Service/Photoset"

    def __init__(self, parent, user=None):
        title = to_utf8("Photosets")
        ServiceModelFolder.__init__(self, title, parent)

        self.empty_msg  = "No photosets found."
        self.user       = user

    def do_search(self):
        photosets = self.flickr_manager.get_photosets(user=self.user)

        if photosets:
            for photoset in photosets:
                PhotosetModelFolder(self, photoset, user=self.user)
            return True
        else:
            return False

class UploadsModelFolder(ServiceModelFolder):
    def __init__(self, parent):
        name = to_utf8("Uploads")
        ServiceModelFolder.__init__(self, name, parent)
        self.empty_msg = "No uploads running!"

    def do_search(self):
        for number in range(5):
            UploadModel(self, file="testfile%s.jpg" % number)

class UploadModel(ServiceModelFolder):
    def __init__(self, parent, file):
        name = to_utf8(file)

        self.alt_name   = to_utf8("%s - test" % name)
        self.def_name   = to_utf8(name)

        ServiceModelFolder.__init__(self, name, parent)

        #def request_finished(exception, retval):
        #    return
        #
        #ThreadedFunction(request_finished, self.do_test).start()

    def do_test(self):
        for test in range(5):
            self.title = self.name = self.alt_name
            log.debug("self.title = %s" % self.title)
            time.sleep(random.randint(2,5))
            self.title = self.name = self.def_name
            log.debug("self.title = %s" % self.title)

    def do_search(self):
        return False

class OptionsModel(OptionsModelFolder):
    def __init__(self, parent, screen_controller=None):
        OptionsModelFolder.__init__(self, parent, screen_controller)
        self.flickr_manager = self.screen_controller.model.flickr_manager

    def get_image_model(self):
        model = self.screen_controller.model
        return model.children[model.current]

    def get_photo(self):
        image_model = self.get_image_model()
        photo = image_model.photo
        return photo

    def set_photo(self, photo):
        image_model = self.get_image_model()
        image_model.photo = photo
        return True

class ViewerOptionsModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image"
    title = "Photo options"

    def __init__(self, parent, screen_controller=None):
        OptionsModel.__init__(self, parent, screen_controller)

        self.children_order = [
                    "/SlideshowMode",
                    "/SlideshowTime",
                    "/Info",
                    "/ExifInfo",
                    "/CommentsList",
                    "/Favorite",
                ]

class InfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/Info"
    title = "Info"

class ExifInfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/ExifInfo"
    title = "Exif"

class FavoriteOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/Favorite"

    def __init__(self, parent):
        photo = parent.get_photo()

        # show Favorite option only in case the photo owner is not the authenticated user
        if photo.owner.nsid == parent.flickr_manager.user.nsid:
            return

        OptionsModel.__init__(self, parent)
        self.photo          = photo
        self.is_favorite    = self.flickr_manager.is_favorite(photo)
        self.checked        = self.is_favorite
        self.set_title()

    def set_title(self):
        if self.is_favorite:
            self.title = self.name = "Remove from favorites"
        else:
            self.title = self.name = "Add to favorites"

    def do_load(self):
        if not self.photo == self.get_photo():
            # set new photo as the current photo
            self.photo = self.get_photo()

        self.is_favorite = self.flickr_manager.is_favorite(self.photo)
        self.set_title()

    def execute(self):
        if self.is_favorite:
            return self.flickr_manager.remove_favorite(self.photo)
        else:
            return self.flickr_manager.add_favorite(self.photo)

class CommentOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/CommentsList/Item"

class CommentsListOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/CommentsList"
    title = "Comments"
    comments_per_page = 25

    def __init__(self, parent, screen_controller=None):
        OptionsModel.__init__(self, parent, screen_controller)
        self.is_loading         = False
        self.page               = 1
        self.comments           = None
        self.num_comments       = 0
        self.title = self.name  = "Comments"
        self.photo = self.get_photo()

    def get_comments(self):
        if not self.photo.id in self.flickr_manager.comments:
            self.comments = self.flickr_manager.get_photo_comments(self.photo)
        else:
            self.comments = self.flickr_manager.comments[self.photo.id]
        self.num_comments = len(self.comments)

    def get_index_start(self):
        return (self.page - 1) * self.comments_per_page

    def get_index_end(self):
        index_end = (self.page * self.comments_per_page)
        if index_end > self.num_comments:
            index_end = self.num_comments
        return index_end

    def do_load(self):
        if self.comments == None or not self.photo == self.get_photo():
            # set new photo as the current photo
            self.photo  = self.get_photo()

            # (re-)set page value
            self.page   = 1

            # get comments for self.photo
            self.get_comments()

        if self.num_comments > 0:
            index_start = self.get_index_start()
            index_end   = self.get_index_end()

            for comment in self.comments[index_start:index_end]:
                c = CommentOptionModel(self, self.screen_controller)
                c.authorname = comment.author.__repr__()
                c.content = comment.text
                c.published = comment.date_create
                title = html_remover.xlat(c.content[:40])
                c.title = to_utf8(title)
                c.name = c.title

    def reload(self):
        self.is_loading = False
        self.children.freeze()
        self.unload()
        self.load()
        self.children.thaw()

    def next_page(self):
        self.page += 1
        return True

    def prev_page(self):
        self.page -= 1
        return True

    def do_next(self, end_callback=None):
        self.is_loading = True

        def cb_finished(*ignored):
            self.reload()
            if end_callback is not None:
                end_callback()
        ThreadedFunction(cb_finished, self.next_page).start()

    def do_prev(self, end_callback=None):
        self.is_loading = True

        def cb_finished(*ignored):
            self.reload()
            if end_callback is not None:
                end_callback()
        ThreadedFunction(cb_finished, self.prev_page).start()

class SlideshowModeOptionsModel(BaseSlideshowModeOptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/SlideshowMode"

class SlideshowTimeOptionModelFolder(BaseSlideshowTimeOptionModelFolder):
    terra_type = "Model/Options/Folder/Player/Flickr/Image/SlideshowTime"

class SlideshowLoopOptionModel(BaseSlideshowLoopOptionModel):
    terra_type = "Model/Options/Action/Player/Flickr/Image/SlideshowMode/" \
        "SlideshowLoop"

class SlideshowRandomOptionModel(BaseSlideshowRandomOptionModel):
    terra_type = "Model/Options/Action/Player/Flickr/Image/SlideshowMode/" \
        "SlideshowRandom"

class UploadOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Image/Fullscreen/Upload"
    title = "Upload to flickr"

    def __init__(self, parent):
        OptionsModel.__init__(self, parent)
        self.checked = False

class UploadInfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Upload"
    title = "Upload options"

class UploadStatusOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Upload/Status"
    title = "Status"

class PhotosetOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Photoset"
    title = "Photoset options"

class PhotosetInfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Photoset/Info"
    title = "Info"

class GroupOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Group"
    title = "Group options"

class GroupInfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/Group/Info"
    title = "Info"

class UserOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/User"
    title = "User options"

class UserInfoOptionModel(OptionsModel):
    terra_type = "Model/Options/Folder/Player/Flickr/User/Info"
    title = "Info"

class Icon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Image/Flickr"
    icon = "icon/main_item/flickr"
    plugin = "flickr"

