# Canola2 Flickr Plugin
# Copyright (C) 2008 Thomas Schmidt <tschmidt@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os, logging, urllib, ecore, time, Image

from terra.core.task import Task
from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction

from manager import FlickrManager, DownloadManager

mger = Manager()

flickr_manager      = FlickrManager()
download_manager    = DownloadManager()

network             = mger.get_status_notifier("Network")
PluginDefaultIcon   = mger.get_class("Icon/Plugin")
CanolaError         = mger.get_class("Model/Notify/Error")
ImageLocalModel     = mger.get_class("Model/Media/Image")
OptionsModelFolder  = mger.get_class("Model/Options/Folder")

log = logging.getLogger("plugins.canola-flickr.model")

class ImageModel(ImageLocalModel):
    terra_type = "Model/Media/Image/Flickr/Images"

    def __init__(self, photo, parent):
        # check if we already have a image-file for this
        # photo in the download-folder
        photopath = flickr_manager.search_imagefile(photo)

        # get the correct screen resolution from the controller (see ListController class)
        screen_resolution = 800*480

        if photopath:
            path            = photopath
            width, height   = Image.open(photopath).size
        else:
            photo.sizes = flickr_manager.get_photosizes(photo)
            size = photo.get_preferred_size(screen_resolution)

            path   = download_manager.download_image(size.url)
            width  = int(size.width)
            height = int(size.height)

        title = to_utf8(photo.title)
        ImageLocalModel.__init__(self, title, parent)
        self.path   = path
        self.width  = width
        self.height = height

class MainModelFolder(Task, ModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr"
    terra_task_type = "Task/Image/Flickr"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "Flickr", parent)

    def do_load(self):
        def refresh():
            # Try to log in to Flickr
            if not flickr_manager.is_logged():
                res = flickr_manager.login()
                if res == True: pass
                elif res == False: return

            return flickr_manager.is_logged()

        def refresh_finished(exception, retval):
            if not retval or exception:
                self.inform_loaded()
                message = "You are not logged in.<br>"\
                          "Please switch to your browser and allow Canola-flickr access to your flickr account!"
                self.callback_info(message)
                return

            LatestImagesListModelFolder(self)
            PhotosetsListModelFolder(self)
            FavoritesModelFolder(self)
            ContactsModelFolder(self)
            SearchModelFolder(self)
            InterestingImagesListModelFolder(self)

            self.inform_loaded()

        self.is_loading = True
        ThreadedFunction(refresh_finished, refresh).start()

class ServiceModelFolder(Task, ModelFolder):
    terra_type = "Model/Folder/Task/Image/Flickr"
    terra_task_type = "Task/Image/Flickr"

    threaded_search = True

    def __init__(self, name, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, name, parent)
        self.changed = False
        self.callback_search_finished = None
        self.callback_notify = None

    def do_load(self):
        self.search()

    def search(self, end_callback=None):
        def refresh():
            return self.do_search()

        def refresh_finished(exception, retval):
            self.inform_loaded()

        self.is_loading = True
        ThreadedFunction(refresh_finished, refresh).start()

    def do_search(self):
        raise NotImplementedError("must be implemented by subclasses")

class ContactsModelFolder(ServiceModelFolder):
    def __init__(self, parent, user=None):
        name = to_utf8("Contacts")
        ServiceModelFolder.__init__(self, name, parent)
        self.user = user

    def do_search(self):
        contacts = flickr_manager.get_contacts(self.user)

        if contacts:
            for contact in contacts:
                ContactModelFolder(self, contact=contact)

class ContactModelFolder(ServiceModelFolder):
    def __init__(self, parent, contact):
        if not contact.realname == None:
            title = "%s (%s)" % (contact.username, contact.realname)
        else: title = contact.username

        ServiceModelFolder.__init__(self, to_utf8(title), parent)
        self.contact = contact

    def do_search(self):
        LatestImagesListModelFolder(self, user=self.contact)
        PhotosetsListModelFolder(self, user=self.contact)
        ContactsModelFolder(self, user=self.contact)

class ImageListModelFolder(ServiceModelFolder):
    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)

    def add_photos(self, photos):
        for photo in photos:
            ImageModel(photo=photo, parent=self)

    def do_search(self):
        return []

class FavoritesModelFolder(ImageListModelFolder):
    def __init__(self, parent):
        name = to_utf8("Favorites")
        ImageListModelFolder.__init__(self, name, parent)

    def do_search(self):
        photos = flickr_manager.get_favorites()
        if photos:
            self.add_photos(photos)

class LatestImagesListModelFolder(ImageListModelFolder):
    def __init__(self, parent, user=None, maximages=20):
        title = to_utf8("Latest Images")
        ImageListModelFolder.__init__(self, title, parent)

        self.user       = user
        self.maximages  = maximages

    def do_search(self):
        photos = flickr_manager.get_photos_latest(user=self.user, maximages=self.maximages)
        if photos:
            self.add_photos(photos)

class InterestingImagesListModelFolder(ImageListModelFolder):
    def __init__(self, parent, maximages=20):
        title = to_utf8("Interesting Images")
        ImageListModelFolder.__init__(self, title, parent)

        self.maximages  = maximages

    def do_search(self):
        photos = flickr_manager.get_photos_interesting(maximages=self.maximages)
        if photos:
            self.add_photos(photos)

class SearchModelFolder(ImageListModelFolder):
    """This model implements the Search option."""
    terra_type = "Model/Folder/Task/Image/Flickr/Service/Search"

    def __init__(self, parent):
        title = to_utf8("Search by tag")
        ImageListModelFolder.__init__(self, title, parent)
        self.query = None

    def do_search(self):
        if self.query:
            # replace spaces in query with commas, search_tags expects a comma-separated list
            query = self.query.replace(' ', ',')
            photos = flickr_manager.search_tags(query)
            if photos:
                self.add_photos(photos)
            self.query = None

class PhotosetModelFolder(ImageListModelFolder):
    def __init__(self, parent, photoset):
        name = to_utf8(photoset.title)
        ImageListModelFolder.__init__(self, name, parent)
        self.photoset = photoset

    def do_search(self):
        photos = flickr_manager.get_photos_photoset(self.photoset)
        if photos:
            self.add_photos(photos)

class PhotosetsListModelFolder(ServiceModelFolder):
    def __init__(self, parent, user=None):
        title = to_utf8("Photosets")

        ServiceModelFolder.__init__(self, title, parent)
        self.user = user

    def do_search(self):
        photosets = flickr_manager.get_photosets(user=self.user)

        if photosets:
            for photoset in photosets:
                PhotosetModelFolder(self, photoset)

class Icon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Image/Flickr"
    icon = "icon/main_item/flickr"
    plugin = "flickr"

class OptionsModel(ModelFolder):
    terra_type = "Model/Settings/Folder/InternetMedia/Flickr"
    title = "Flickr"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        LoginOptionsModel(self)

MixedListItemDual = mger.get_class("Model/Settings/Folder/MixedList/Item/Dual")
class LoginOptionsModel(MixedListItemDual):
    terra_type = "Model/Settings/Folder/InternetMedia/Flickr/Login"
    title = "Login to Flickr"

    def __init__(self, parent=None):
        MixedListItemDual.__init__(self, parent)

    def get_title(self):
        return "Login to Flickr"

    def get_left_button_text(self):
        return "Log on"

    def get_right_button_text(self):
        return "Log off"

    def on_clicked(self):
        if not self.is_logged():
            self.callback_use(self)

    def on_left_button_clicked(self):
        # TODO: do useful things here! ;-)
        if not self.is_logged():
            self.callback_use(self)

    def on_right_button_clicked(self):
        # TODO: delete credentials in callback!!!
        self.callback_use(self)

    def delete_credentials(self):
        # TODO: add method to delete the credentials
        flickr_manager.logout()

    def is_logged(self):
        return flickr_manager.is_logged()

class ConfirmOptionModel(OptionsModelFolder):
    terra_type = "Model/Settings/Folder/InternetMedia/Flickr/Confirm"
    title = "Confirm"

    def execute(self):
        return True

    def has_record(self):
        return True
