/*
 * boxar - a musical instrument for nokia internet tablet (n810 & n800).
 * Copyright (c) 2008 Sampath Jagannathan
 * Distributed under the GPL 2.0 license.
 * boxar is Free Software.
 * Based on ideas initially developed in din (http://code.google.com/p/din)
*/

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <esd.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>

// audio output
int speakers = 0;
pthread_t audio_thread;
int runme = 1;

// audio params
int sample_rate = 44100; // cd quality
short* samples_buffer = 0;
int samples_buffer_size;
int num_samples = 32;
int num_channels = 2; // ie stereo

double start_frequency = 65.4064; // deep C  - pitch of box 0 @ left,bottom of screen

double decay_time = 2.0; // decay time in seconds for each note
double delta_decay_time = 0.1; 

// will init these later
int noctaves = 0;
int ntones = 0;
int nboxes = 0;

typedef unsigned short rhythm_pattern;
#define DEFAULT_RHYTHM_PATTERN 0xffff;
#define MASK 1

// stereo rhythms
struct stereo {
	rhythm_pattern l, r;
} rhythm_pat, last_rhythm_pat, empty_pat, rhythm_bit;

// pulse width modulation
int do_pwm = 0;
double pwm_depth = 0.30, max_depth = 1.0, delta_pwm_depth = 0.01; 
int pwm_rate = 15, delta_pwm_rate = 1; // hz

// bit shifts per sec => rhythms
struct bps_t {
	int i;
	double bps;
	double delta_bps;
	int nsam;
	int nshift, maxshift;
} rhythm_bps;


// ui
GtkWidget *window, *vbox;
typedef void (*menu_callback)(gint*);
GtkWidget* menubar;
int show_menubar = 1;

GtkWidget* statusbar;
char mesg [256];
GtkLabel* statusbar_label;

struct win_props { // main window
	int width, height;
	int menubar_height;
} win;

struct box_globs { // box globals
	int left, bottom;
	int width, height, gap;
	int dksam;
} boxg;


#define MAX_OCTAVES 7
#define NUM_SCALES 10
#define MAX_TONES 8
#define MAX_BOXES (MAX_TONES * MAX_OCTAVES)

#define MAXAMP 32767
#define SIMUL_NOTES 16
#define DEFAULT_AMP (MAXAMP / SIMUL_NOTES)
#define DEFAULT_COLOR (DEFAULT_AMP << 2)
#define RED_SHIFT 5 // DEFAULT_AMP << ? = 65535

enum {SILENT, DECAY};

struct box_pos {
	int x1, y1;
	int x2, y2;
	int wipe;
} boxpos [MAX_BOXES];

struct box_snd {
	
	int err, cerr;
	int run, shift;

	short amp;
	int sign;

	int npuls[3], puls_id, puls_min;
	int pw_sam, pw_nsam, pws;
	
	int dkid;
	int state;
	
} boxsnd [MAX_BOXES];

int last_box = 0;

int iscale = 0; // current scale

struct scale { // musical scale
	
	int ntones;
	int noctaves;
	char* name;
	double intervals [MAX_TONES];
	
} scales [NUM_SCALES] = {
		{7, 6, "The Blues", 1, 1.18, 1.33333, 1.40625, 1.5, 1.80, 2}, // C Eb F F# G Bb C'
		{6, 6, "Chinese - Guzheng", 1, 1.125, 1.265625, 1.5, 1.66667, 2}, // C D E G A C'
		{6, 6, "Chinese - Guqin", 1, 1.125, 1.3333, 1.5, 1.66667, 2}, // C D F G A C'
		{7, 6, "Native American - 6 Hole Flute", 1, 1.20, 1.3333, 1.5, 1.60, 1.80, 2}, // C Eb F G Ab Bb B C'
		{6, 6, "Native American - 5 Hole Flute", 1, 1.20, 1.3333, 1.5, 1.80, 2}, // C Eb F G Bb C'
		{6, 6, "Indian - Raga Hamsadhwani", 1, 1.125, 1.265625, 1.5, 1.875, 2}, // C D E G B C'
		{6, 6, "Indian - Raga Shivaranjani", 1, 1.125, 1.20, 1.5, 1.66667, 2}, // C D Eb G A C'
		{6, 6, "Indian - Raga Malkauns", 1, 1.20, 1.3333, 1.60, 1.80, 2}, // C Eb F Ab Bb C'
		{8, 6, "Indian - Raga Bhairav", 1, 1.0667, 1.265625, 1.3333, 1.5, 1.60, 1.875, 2}, // C Db E F G Ab B C'
		{4, 6, "Tonic 4th 5th Octave", 1, 1.3333, 1.5, 2}, // C F G C'
};

#define ERROR_LIMIT 100


void calc_decay () {
	boxg.dksam =  decay_time * sample_rate * 1./ DEFAULT_AMP + 0.5;
	sprintf (mesg, "  Decay time = %2.2f secs", decay_time);
	gtk_label_set_text (statusbar_label, mesg);
}

void* audio_wanted (void* arg) {
			
	int i, j, k, p;
		
	short amp;
	
	while (runme) {
		
		i = j = 0;
		
		for (;i < num_samples; ++i, j+=2) {
			amp = 0;
			for (p = 0; p < nboxes; ++p) {
				struct box_snd b = boxsnd [p];
				if (b.state != SILENT) {
					if (b.run >= b.shift) {
						
						if (do_pwm) {
							if (++b.pw_sam >= b.pw_nsam) { // modulate now
								b.pw_sam = 0;
								b.npuls[0] += b.pws;
								b.npuls[1] -= b.pws;
								if (b.pws == 1) k = 1; else k = 0;
								if (b.npuls[k] <= b.puls_min) {
									b.npuls[k] = b.puls_min;
									b.pws = -b.pws;
								}
							}
						}
						
						b.puls_id = !b.puls_id;
						b.cerr += b.err;
						if (b.cerr >= ERROR_LIMIT) {
							b.cerr -= ERROR_LIMIT;
							b.shift = b.npuls[b.puls_id] + 1;
						} else b.shift = b.npuls[b.puls_id];
						
						b.run = 0;
						b.sign = -b.sign;
						
					}
					
					++b.run;
					amp += (b.sign * b.amp);
					boxsnd [p] = b;
					
				}
				
				
			}
			
			samples_buffer [j] = rhythm_bit.l * amp;
			samples_buffer [j+1] = rhythm_bit.r * amp;
			
		}
		
		// decay
		for (i = 0; i < nboxes; ++i) {
			struct box_snd b = boxsnd [i];
			if (b.state != SILENT) {
				b.dkid += num_samples;
				while (b.dkid >= boxg.dksam) {
					b.dkid -= boxg.dksam;
					if (--b.amp < 0) {
						b.npuls[0] = b.npuls[1] = b.npuls[2];
						b.run = 0;
						b.shift = b.npuls[0];
						b.cerr = b.err;
						b.pw_sam = 0;
						b.pws = 1;
						b.amp = 0;
						b.state = SILENT;
						b.dkid = 0;
						boxpos[i].wipe = 1;
						break;
					}
				}
			}
			boxsnd [i] = b;
		}
		
		// stereo rhythm
		rhythm_bps.i += num_samples;
		if (rhythm_bps.i >= rhythm_bps.nsam) {
			rhythm_bps.i -= rhythm_bps.nsam;
			rhythm_bit.l = (rhythm_pat.l >> rhythm_bps.nshift) & MASK;
			rhythm_bit.r = (rhythm_pat.r >> rhythm_bps.nshift) & MASK;
			if (++rhythm_bps.nshift > rhythm_bps.maxshift) rhythm_bps.nshift = 0;
		}
						
		write (speakers, (int *) samples_buffer, samples_buffer_size);
		
	}
	
	return (void *) 0;
	
}

static int start_esd ()
{

  esd_format_t format = ESD_BITS16 | ESD_STEREO | ESD_STREAM | ESD_PLAY;
  speakers = esd_play_stream_fallback(format, ESD_DEFAULT_RATE, NULL, "boxar_speakers");
	
	if (speakers <= 0) {
		printf ("couldnt output to speakers\n");
		return -1;
	} else {
		samples_buffer_size = num_samples * num_channels * sizeof (short);
		samples_buffer = (short* ) malloc (samples_buffer_size);
		pthread_create (&audio_thread, 0, audio_wanted, 0);
	}
	
  return 0;
}

static int stop_esd () {
	runme = 0;
	pthread_join (audio_thread, 0);
  close(speakers);
	if (samples_buffer) free (samples_buffer);
}

void get_frequency_params (double f, int* npuls, int* err) {
	double dpuls = (sample_rate / (2 * f));
	npuls[0] = (int) dpuls;
	npuls[1] = npuls[0];
	npuls[2] = npuls[0];
	*err = (dpuls - npuls[0]) * ERROR_LIMIT;
}

void init_box_globals (int _ntones, int _noctaves) {
	
	boxg.gap = 3;
	boxg.left = boxg.gap;
	boxg.bottom = win.height;
	boxg.width = (win.width - boxg.gap) / _ntones - boxg.gap;
	boxg.height = (win.height - win.menubar_height) / _noctaves - boxg.gap;
	nboxes = _ntones * _noctaves;
	calc_decay ();
	
}

void init_boxes () {
	
	int i, j, k = 0;
	int x = 0, y = win.height;
	double f = start_frequency, g = 0;
	
	for (i = 0; i < noctaves; ++i) {
		
		x = boxg.gap;
		
		for (j = 0; j < ntones; ++j, ++k) {

			struct box_snd b = boxsnd [k];
			struct box_pos bp = boxpos [k];

			bp.x1 = x; bp.x2 = x + boxg.width;
			bp.y2 = y; bp.y1 = y - boxg.height;
			bp.wipe = 0;

			g = f * scales[iscale].intervals[j];
			get_frequency_params (g, b.npuls, &b.err);
			
			b.puls_id = 0;
			b.run = 0; 
			b.shift = b.npuls[b.puls_id];
			b.cerr = b.err;
			b.puls_min = pwm_depth * b.npuls[2];
			
			b.pw_sam = 0;
			b.pw_nsam = (int)(2 * g / pwm_rate);
			b.pws = 1;
			
			b.sign = 1;
			b.amp = 0;
			
			b.state = SILENT;
			b.dkid = 0;
			
			boxsnd [k] = b;
			boxpos [k] = bp;
			
			x = x + boxg.width + boxg.gap;
			
		}
		y = y - boxg.height - boxg.gap; 
		f = g;
		
	}
	
}

void set_scale (int i) {
	iscale = i;
	ntones = scales [iscale].ntones;
	noctaves = scales[iscale].noctaves;
	init_box_globals (ntones, noctaves);
	init_boxes ();

	sprintf (mesg, "  Loaded: %s", scales[iscale].name);
	gtk_label_set_text (statusbar_label, mesg);

}

int skip_draw = -1;

gboolean refresh_boxes (gpointer data) {
		
	GdkGC* gc = window->style->black_gc;
	GdkColor c; c.red = 0; c.green = DEFAULT_COLOR; c.blue = c.green;
	int i;
	
	for (i = 0; i < nboxes; ++i) {
		if (boxsnd [i].state != SILENT) {
			if (skip_draw != i) {
				c.red = boxsnd [i].amp << RED_SHIFT;
				gdk_gc_set_rgb_fg_color (gc, &c);	
				gdk_draw_rectangle (window->window, gc, 1, boxpos[i].x1, boxpos[i].y1, boxg.width, boxg.height);
			} else skip_draw = -1;
		} else if (boxpos[i].wipe) {
			boxpos[i].wipe = 0;
			c.red = 0;
			gdk_gc_set_rgb_fg_color (gc, &c);	
			gdk_draw_rectangle (window->window, gc, 1, boxpos[i].x1, boxpos[i].y1, boxg.width, boxg.height);
		}
	}
		
	return 1;
	
}

void draw_empty_boxes () {
	
	GdkGC* gc = window->style->black_gc;
	GdkColor color; color.red = color.green = color.blue = 0;
	int i;

	gdk_gc_set_rgb_fg_color (gc, &color);
  gdk_draw_rectangle (window->window, gc, TRUE, 0, 0, win.width, win.height);
	
	color.red = 0; color.green = DEFAULT_COLOR; color.blue = color.green;
	gdk_gc_set_rgb_fg_color (gc, &color);	
	for (i = 0; i < nboxes; ++i) {
		gdk_draw_rectangle (window->window, gc, 1, boxpos[i].x1, boxpos[i].y1, boxg.width, boxg.height);
	}
	
}

void select_boxes (double x, double y, double pressure) {
	
	int i;
	GdkGC* gc;
	GdkColor c;
			
	i = last_box;
	if ( (x >= boxpos[i].x1 && x <= boxpos[i].x2) && (y >= boxpos[i].y1 && y <= boxpos[i].y2) ) {
		boxsnd[i].amp = DEFAULT_AMP;
		boxsnd[i].state = DECAY;
		return;
	}
	
	for (i = 0; i < nboxes; ++i) {
		if (i != last_box) {
			if ( (x >= boxpos[i].x1 && x <= boxpos[i].x2) && (y >= boxpos[i].y1 && y <= boxpos[i].y2) ) {
				boxsnd[i].amp = DEFAULT_AMP;
				boxsnd[i].state = DECAY;
				last_box = i;
				skip_draw = i;
				
				// draw now for quick feedback. do not wait for refresh
				gc = window->style->black_gc;
				c.red = boxsnd[i].amp << RED_SHIFT; c.green = DEFAULT_COLOR; c.blue = c.green;
				gdk_gc_set_rgb_fg_color (gc, &c);	
				gdk_draw_rectangle (window->window, gc, 1, boxpos[i].x1, boxpos[i].y1, boxg.width, boxg.height);
				
				return;
			}
		}
	}
		
}

rhythm_pattern rnd1, rnd2, rnd;

void init_rand () {
	srand (clock());
	rnd1 = rand();
	rnd2 = rand();
}

rhythm_pattern get_rand () {
	/*
	 * simple random number generator
	 * fibonacci summation
	 * used in elite game on the amiga
	*/
	rnd = rnd1 + rnd2;
	rnd1 = rnd2;
	rnd2 = rnd;
	return rnd;
}

static gboolean expose_event (GtkWidget *widget, GdkEventExpose *event) {
	draw_empty_boxes ();
  return TRUE;
}

static gboolean button_press_event (GtkWidget *widget, GdkEventButton *event) {  
	static gdouble px = -1, py = -1;
	if ((event->x == px) && (event->x == py)) return TRUE; else {px = event->x; py = event->y;}
	gdouble pressure;
	gdk_event_get_axis ((GdkEvent *)event, GDK_AXIS_PRESSURE, &pressure);
	select_boxes (event->x, event->y, pressure);
  return TRUE;
}

static gboolean motion_notify_event (GtkWidget *widget, GdkEventMotion *event) {
  gdouble x, y;
  gdouble pressure;
  GdkModifierType state;
	
  if (event->is_hint) 
  {
		gdk_device_get_state (event->device, event->window, NULL, &state);
		gdk_event_get_axis ((GdkEvent *)event, GDK_AXIS_X, &x);
		gdk_event_get_axis ((GdkEvent *)event, GDK_AXIS_Y, &y);
		gdk_event_get_axis ((GdkEvent *)event, GDK_AXIS_PRESSURE, &pressure);
  }else
  {
		x = event->x;
		y = event->y;
		gdk_event_get_axis ((GdkEvent *)event, GDK_AXIS_PRESSURE, &pressure);
		state = event->state;
	}
	select_boxes (x, y, pressure);
  return TRUE;
}

void toggle_menu () {
	show_menubar = !show_menubar;
	if (show_menubar) gtk_widget_show (menubar); else gtk_widget_hide (menubar);
}

void change_bps (struct bps_t* bps, int d) {
	bps->bps += (d * bps->delta_bps);
	if (bps->bps < 0) bps->bps = bps->delta_bps;
	bps->nsam = sample_rate / bps->bps;
	sprintf (mesg, "  Throb speed = %.2f", bps->bps);
	gtk_label_set_text (statusbar_label, mesg);
}

void change_rhythm (gint* i) {
	
	rhythm_pat.l = get_rand ();
	rhythm_pat.r = get_rand ();
	sprintf (mesg, " Throb L, R : %04X, %04X", rhythm_pat.l, rhythm_pat.r);
	gtk_label_set_text (statusbar_label, mesg);
	
}

void toggle_rhythm (gint* i) {
	
	if ((rhythm_pat.l == empty_pat.l) && (rhythm_pat.r == empty_pat.r)) {
		rhythm_pat = last_rhythm_pat; 
		sprintf (mesg, " Throb L, R: %04X, %04X", rhythm_pat.l, rhythm_pat.r);
	} else {
		last_rhythm_pat = rhythm_pat;
		rhythm_pat = empty_pat;
		sprintf (mesg, " Stopped throbbing");
	}	
	gtk_label_set_text (statusbar_label, mesg);
		
}

void inc_rhythm_bps (gint* i) {
	change_bps (&rhythm_bps, +1);
}

void dec_rhythm_bps (gint* i) {
	change_bps (&rhythm_bps, -1);
}

void toggle_pwm (gint* i) {
	do_pwm = !do_pwm;
	char* str[] = {"Off", "On"};
	sprintf (mesg, "  Sway is %s", str[do_pwm]);
	gtk_label_set_text (statusbar_label, mesg);
}

void calc_pwm () {
	int i, j, k = 0;
	double f = start_frequency, g = 0;
	for (i = 0; i < noctaves; ++i) {
		for (j = 0; j < ntones; ++j, ++k) {
			struct box_snd b = boxsnd [k];
			g = f * scales[iscale].intervals[j];
			b.pw_nsam = (int)(2 * g / pwm_rate);
			boxsnd [k] = b;
		}
		f = g;
	}
	sprintf (mesg, "  Sway rate = %d Hz", pwm_rate);
	gtk_label_set_text (statusbar_label, mesg);

	
}

void inc_rate (gint* p) {
	pwm_rate += delta_pwm_rate;
	calc_pwm ();
}

void dec_rate (gint* p) {
	if (pwm_rate <= delta_pwm_rate) return;
	pwm_rate -= delta_pwm_rate;
	calc_pwm ();
}

void calc_depth () {
	int i;
	for (i = 0; i < nboxes; ++i) boxsnd[i].puls_min = pwm_depth * boxsnd[i].npuls[2];
	sprintf (mesg, "  Sway = %d %%", (int)((1 - pwm_depth) * 100));
	gtk_label_set_text (statusbar_label, mesg);
}

void inc_depth (gint* p) {
	
	pwm_depth += delta_pwm_depth;
	if (pwm_depth > max_depth) pwm_depth = max_depth;
	calc_depth ();
	
}

void dec_depth (gint* p) {

	if (pwm_depth <= 0) pwm_depth = 0; else pwm_depth -= delta_pwm_depth;
	calc_depth ();

}

void dec_decay (gint* p) {
	decay_time -= delta_decay_time;
	if (decay_time <= 0) decay_time = delta_decay_time;
	calc_decay ();
}

void inc_decay (gint* p) {
	decay_time += delta_decay_time;
	calc_decay ();
}

static gboolean key_press_event (GtkWidget * widget, GdkEventKey * event, void* ptr)
{
		
	switch (event->keyval) {
					
		case GDK_F7:
			inc_rhythm_bps (0);
			break;
			
		case GDK_F8:
			dec_rhythm_bps (0);
			break;
			
		case GDK_Escape:
			change_rhythm (0);
			break;
			
		case GDK_F6:
			toggle_rhythm (0);
			break;
			
		case GDK_F4:
			toggle_menu ();
			break;
			
		case GDK_Return:
			toggle_pwm (0);
			break;
			
		case GDK_Up:
			inc_rate (0);
			break;
			
		case GDK_Down:
			dec_rate (0);
			break;
			
		case GDK_Left:
			inc_depth (0);
			break;
			
		case GDK_Right:
			dec_depth (0);
			break;

	#define COMMA 44 
	#define FULLSTOP 46
	#define LESS_THAN 60
	#define GREATER_THAN 62
			
		case COMMA:
		case LESS_THAN:
			dec_decay (0);
			break;
			
		case FULLSTOP:
		case GREATER_THAN:
			inc_decay (0);
			break;
			
						
	}
	
	return TRUE;
}

void scale_changed ( gint*  i)
{
	set_scale ((int)i);
	draw_empty_boxes ();
}

void no_rhythm () {
	empty_pat.l = DEFAULT_RHYTHM_PATTERN;
	empty_pat.r = DEFAULT_RHYTHM_PATTERN;
	rhythm_pat = empty_pat;
	last_rhythm_pat = rhythm_pat;
}

void init_bps () {
	rhythm_bps.bps = 32;
	rhythm_bps.delta_bps = 1;
	rhythm_bps.i = 0;
	rhythm_bps.nshift = 0;
	rhythm_bps.maxshift = 8 * sizeof (rhythm_pattern) - 1;
	rhythm_bps.nsam = sample_rate / rhythm_bps.bps;
}

#define REFRESH_MSECS 100



int main (int argc, char *argv[])
{
	
	int tid;
	int i;
	
	GtkWidget *menu;
	GtkWidget *item_boxar, *item_scale, *item_rhythm, *dummy;
	GtkWidget* item;
	GSList* melgrp = 0;
	
	char* str;

	char* about [] = {
		"Hide menu",
		"Exit",
		"-",
		"Boxar 1.1",
		"Copyright (c) 2008 S Jagannathan",
		"Distributed under GPL 2.0 license",
		"E-mail me: jagernot@gmail.com",
		"My website: www.poojyum.com"
	};
	
	menu_callback about_cb [] = {
		toggle_menu,
		gtk_main_quit
	};
	
	char* rhythm_items [] = {
		"New Throb [ESC]",
		"Throb faster (+)",
		"Throb slower (-)",
		"Toggle throbbing [F6]",
		"-",
		"Toggle Sway [Select]",
		"Sway faster (Up)",
		"Sway slower (Down)",
		"Sway less (Left)",
		"Sway more (Right)",
		"-",
		"Increase decay (> or .)",
		"Decrease decay (< or ,)"
	};
	
	menu_callback rhythm_cb [] = {
		change_rhythm,
		inc_rhythm_bps,
		dec_rhythm_bps,
		toggle_rhythm,
		0,
		toggle_pwm,
		inc_rate,
		dec_rate,
		inc_depth,
		dec_depth,
		0,
		inc_decay,
		dec_decay
	};
			
  gtk_init (&argc, &argv);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (window, "boxar 1.1");
	gtk_window_fullscreen (window);
	
	menu = gtk_menu_new ();
	item_boxar = gtk_menu_item_new_with_label ("Boxar");
	gtk_widget_show (item_boxar);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_boxar), menu);	
	
	for (i = 0; i < 8; ++i) {
		str = about[i];
		if (strcmp (str, "-") == 0) {
			item = gtk_separator_menu_item_new ();
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			gtk_widget_show (item);
		} else {
			item = gtk_menu_item_new_with_label (about[i]);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			gtk_widget_show (item);
			if (i < 2) {
				g_signal_connect_swapped (G_OBJECT (item), "activate",  G_CALLBACK (about_cb[i]), 0);
			}
			
		}
		
	}
	
	menu = gtk_menu_new ();
	item_scale = gtk_menu_item_new_with_label ("Melody");
	gtk_widget_show (item_scale);
		
	for (i = 0; i < NUM_SCALES; ++i) {
		item = gtk_radio_menu_item_new_with_label (melgrp, scales[i].name);
		melgrp = gtk_radio_menu_item_get_group (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
		gtk_widget_show (item);
		g_signal_connect_swapped (G_OBJECT (item), "activate",
		                      G_CALLBACK (scale_changed), 
                                      i);
	}
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_scale), menu);	
	
	menu = gtk_menu_new ();
	item_rhythm = gtk_menu_item_new_with_label ("Rhythm");
	gtk_widget_show (item_rhythm);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_rhythm), menu);
	
	for (i = 0; i < 13; ++i) {
		if (strcmp (rhythm_items[i], "-") == 0) {
			item = gtk_separator_menu_item_new ();
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			gtk_widget_show (item);
		} else {
			item = gtk_menu_item_new_with_label (rhythm_items [i]);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			gtk_widget_show (item);
			g_signal_connect_swapped (G_OBJECT (item), "activate", G_CALLBACK (rhythm_cb[i]), i);
		}
	}
		
	menu = gtk_menu_new ();
	statusbar = gtk_menu_item_new_with_label ("");
	gtk_widget_show (statusbar);

	gtk_menu_item_set_submenu (GTK_MENU_ITEM(statusbar), menu);
	statusbar_label = GTK_LABEL( gtk_bin_get_child (GTK_BIN (statusbar)) );
  
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);
	gtk_widget_show (vbox);
	
	menubar = gtk_menu_bar_new ();
	gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 2);
	gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_boxar);
	gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_scale);
	gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_rhythm);
	gtk_menu_bar_append (GTK_MENU_BAR (menubar), statusbar);
	
	
	gtk_widget_show (menubar);
		
	// hardcode for maemo / n810 tablet.
	win.width = 800;
	win.height = 480;
	win.menubar_height = 0;
	
	init_rand ();
	no_rhythm ();
	init_bps ();
	set_scale (0);

  g_signal_connect (G_OBJECT (window), "destroy",
                    G_CALLBACK (gtk_main_quit), NULL);
										
										
  g_signal_connect (G_OBJECT (window), "expose_event",
                    G_CALLBACK (expose_event), NULL);

  g_signal_connect (G_OBJECT (window), "motion_notify_event",
                    G_CALLBACK (motion_notify_event), NULL);
										
  g_signal_connect (G_OBJECT (window), "button_press_event",
                    G_CALLBACK (button_press_event), NULL);
	
	g_signal_connect (G_OBJECT (window), "key_press_event", G_CALLBACK (key_press_event), NULL);

  gtk_widget_set_events (window, GDK_EXPOSURE_MASK
                         | GDK_LEAVE_NOTIFY_MASK
												 | GDK_KEY_PRESS_MASK
                         | GDK_BUTTON_PRESS_MASK
                         | GDK_POINTER_MOTION_MASK
                         | GDK_POINTER_MOTION_HINT_MASK);

  gtk_widget_set_extension_events (window, GDK_EXTENSION_EVENTS_CURSOR);

	gtk_widget_show_all (window);
	
	start_esd ();
			tid = g_timeout_add (REFRESH_MSECS, refresh_boxes, 0);
				gtk_main ();
			g_source_remove (tid);
	stop_esd ();
	
	
	
  return 0;
}
