//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#define ASUI_VERSION "0.6.0"

/*
/etc/init.d/hildon-desktop stop
su - user -c "maemo-summoner /usr/bin/hildon-desktop.launch"
*/

#include "main.h"
#include "config.h"
#include "dbus.h"
#include "battery.h"
#include "bme.h"
#include "cpu.h"

//////////////////////////////////////////////////////////////////////////

G_BEGIN_DECLS

typedef struct _AsuiBatteryPlugin AsuiBatteryPlugin;
typedef struct _AsuiBatteryPluginClass AsuiBatteryPluginClass;

#define ASUI_BATTERY_PLUGIN_PRIORITY 1

#define ASUI_BATTERY_PLUGIN_ICON_SIZE 40

#define ASUI_BATTERY_PLUGIN_TYPE			(asui_battery_plugin_get_type())
#define ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPlugin))
#define ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST((klass),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))
#define IS_ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_TYPE((obj),	ASUI_BATTERY_PLUGIN_TYPE))
#define IS_ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass),	ASUI_BATTERY_PLUGIN_TYPE))
#define ASUI_BATTERY_PLUGIN_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))

struct _AsuiBatteryPlugin { StatusbarItem parent; };
struct _AsuiBatteryPluginClass { StatusbarItemClass parent_class; };
GType asui_battery_plugin_get_type(void);

G_END_DECLS

HD_DEFINE_PLUGIN(AsuiBatteryPlugin, asui_battery_plugin, STATUSBAR_TYPE_ITEM);

#define ASUI_BATTERY_PLUGIN_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE((object), ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginPrivate))

//////////////////////////////////////////////////////////////////////////

static const gchar *asui_battery_icons[] = {
	"asui_power", "asui_power_pressed",
	"asui_battery100", "asui_battery88", "asui_battery76", "asui_battery64", "asui_battery52", "asui_battery40", "asui_battery28", "asui_battery16", "asui_battery4",
	"asui_charging100", "asui_charging66", "asui_charging33",
	"asui_charged"
};

static const gchar *asui_battery_wifi_icons[] = {
	NULL, "asui_wifi_disconnected", "asui_wifi_connected"
};

static const gchar *asui_battery_bluetooth_icons[] = {
	NULL, "asui_bluetooth", "asui_bluetooth_connected", "asui_bluetooth_visible", "asui_bluetooth_connected_visible"
};

static const gchar *asui_battery_drain_icons[] = {
	NULL, "asui_drain_charging", "asui_drain_low", "asui_drain_medium", "asui_drain_high", "asui_drain_veryhigh"
};

static const gchar *asui_battery_cpu_icons[] = {
	NULL, "asui_cpu25", "asui_cpu50", "asui_cpu75", "asui_cpu100"
};

void asui_battery_set_icon( AsuiBatteryPluginPrivate *p ){
	unsigned allow_tb_elements = 1, allow_lr_elements = 1;
	e_icon_type main_icon_type;

	if(!p->use_battery) p->new_icon_type = ICON_POWER; // disable
	if(!p->use_wifi) p->new_wifi_icon_type = ICON_WIFI_NONE; // disable
	if(!p->use_bluetooth) p->new_bluetooth_icon_type = ICON_BLUETOOTH_NONE; // disable
	if(!p->use_drain) p->new_drain_icon_type = ICON_DRAIN_NONE; // disable
	if(!p->use_cpu) p->new_cpu_icon_type = ICON_CPU_NONE; // disable

	if(p->press_timer){
		main_icon_type = ICON_POWER_PRESSED;
		allow_tb_elements = allow_lr_elements = 0; // disable all elements while pressed
	} else if(p->battery_initializing || !p->use_battery || p->new_icon_type == ICON_POWER){
		main_icon_type = ICON_POWER;
		allow_lr_elements = 0; // no space for side elements
	} else
		main_icon_type = p->new_icon_type;

	// only update if one of the elements has changed
	if(p->current_icon_type == main_icon_type
	&& p->current_wifi_icon_type == p->new_wifi_icon_type
	&& p->current_bluetooth_icon_type == p->new_bluetooth_icon_type
	&& p->current_drain_icon_type == p->new_drain_icon_type
	&& p->current_cpu_icon_type == p->new_cpu_icon_type
	) return;

	GtkIconTheme *icon_theme = gtk_icon_theme_get_default();

	// get capacity icon
	const gchar *name = asui_battery_icons[main_icon_type];
	GdkPixbuf *pixbuf = (name != NULL ? gtk_icon_theme_load_icon(icon_theme, name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);

	if(pixbuf){
		if(allow_tb_elements || allow_lr_elements){
			// original is immutable so make a copy for overlays
			GdkPixbuf *tmp_pixbuf = gdk_pixbuf_copy(pixbuf);
			g_object_unref(pixbuf);
			pixbuf = tmp_pixbuf;
		}

		// get drain icon
		if(allow_tb_elements){
			const gchar *drain_name = asui_battery_drain_icons[p->new_drain_icon_type];
			GdkPixbuf *drain_pixbuf = (drain_name != NULL ?
									gtk_icon_theme_load_icon(icon_theme, drain_name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
			if(drain_pixbuf){
				// overlay drain icon
				gdk_pixbuf_copy_area(drain_pixbuf, 0,0, 40,6, pixbuf, 0,0);
				g_object_unref(drain_pixbuf);
			}

			// get cpu icon
			const gchar *cpu_name = asui_battery_cpu_icons[p->new_cpu_icon_type];
			GdkPixbuf *cpu_pixbuf = (cpu_name != NULL ?
									gtk_icon_theme_load_icon(icon_theme, cpu_name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
			if(cpu_pixbuf){
				// overlay cpu icon
				gdk_pixbuf_copy_area(cpu_pixbuf, 0,0, 40,6, pixbuf, 0,34);
				g_object_unref(cpu_pixbuf);
			}
		}
		if(allow_lr_elements){
			// get wifi icon
			const gchar *wifi_name = asui_battery_wifi_icons[p->new_wifi_icon_type];
			GdkPixbuf *wifi_pixbuf = (wifi_name != NULL ?
									gtk_icon_theme_load_icon(icon_theme, wifi_name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
			if(wifi_pixbuf){
				// overlay wifi icon
				gdk_pixbuf_copy_area(wifi_pixbuf, 0,0, 12,28, pixbuf, 0,7);
				// flip wifi icon when bluetooth is off
				if(p->mirror_wifi && p->new_bluetooth_icon_type == ICON_BLUETOOTH_NONE){
					GdkPixbuf *flipped_pixbuf = gdk_pixbuf_flip(wifi_pixbuf, TRUE);
					if(flipped_pixbuf){
						gdk_pixbuf_copy_area(flipped_pixbuf, 0,0, 12,28, pixbuf, 28,7);
						g_object_unref(flipped_pixbuf);
					}
				}
				g_object_unref(wifi_pixbuf);
			}

			// get bluetooth icon
			const gchar *bluetooth_name = asui_battery_bluetooth_icons[p->new_bluetooth_icon_type];
			GdkPixbuf *bluetooth_pixbuf = (bluetooth_name != NULL ?
										gtk_icon_theme_load_icon(icon_theme, bluetooth_name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
			if(bluetooth_pixbuf){
				// overlay bluetooth icon
				gdk_pixbuf_copy_area(bluetooth_pixbuf, 0,0, 12,28, pixbuf, 28,7);

// TODO: last horizontal pixel column is being cut off, as if the statusbar icon is only 39 pixels wide

				g_object_unref(bluetooth_pixbuf);
			}
		}

		gtk_image_set_from_pixbuf(GTK_IMAGE(p->icon), pixbuf);
		g_object_unref(pixbuf);

		p->current_icon_type = main_icon_type;
		p->current_wifi_icon_type = p->new_wifi_icon_type;
		p->current_bluetooth_icon_type = p->new_bluetooth_icon_type;
		p->current_drain_icon_type = p->new_drain_icon_type;
		p->current_cpu_icon_type = p->new_cpu_icon_type;
	}
}

//////////////////////////////////////////////////////////////////////////

static gboolean asui_battery_reset_press( gpointer data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)data;

	p->press_timer = 0;
	asui_battery_set_icon(p); // revert to previous state
	return FALSE; // destroy timer
}

static gboolean map_asui_window( GtkButton *widget, GdkEventButton *button, gpointer data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)data;

	g_return_val_if_fail(data, FALSE);
	gtk_button_released(GTK_BUTTON(p->button));
	(void) button;

	p->press_timer = g_timeout_add(1000, asui_battery_reset_press, (void *)p);
	asui_battery_set_icon(p); // show power_pressed icon

	dbus_send_system_method(p, "com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "show");

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_plugin_finalize( GObject *object ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(object);

	if(p->use_cpu) cpu_disable(p, 1); // unhook
	if(p->use_drain) bme_disable(p, 1); // unhook
	if(p->use_battery) battery_disable(p, 1); // unhook
	if(p->press_timer) g_source_remove(p->press_timer);

	osso_deinitialize(p->osso);
	dbus_disconnect(p);
	config_disconnect(p);

	LOG_CLOSE();

	G_OBJECT_CLASS(g_type_class_peek_parent(G_OBJECT_GET_CLASS(object)))->finalize(object);
}

static void asui_battery_plugin_class_init( AsuiBatteryPluginClass *klass ){
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = asui_battery_plugin_finalize;
	g_type_class_add_private(klass, sizeof(AsuiBatteryPluginPrivate));
}

static void asui_battery_plugin_init( AsuiBatteryPlugin *plugin ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(plugin);

	ULOG_OPEN("asui-statusbar-battery");

	g_return_if_fail(p);

	if(!config_init(p)) return;
	if(!dbus_init(p)){ config_disconnect(p); return; }

	// initialize button
	p->icon = gtk_image_new_from_pixbuf(NULL);
	p->current_icon_type = ICON_POWER_PRESSED; p->new_icon_type = ICON_POWER; // force it
	p->current_wifi_icon_type = p->new_wifi_icon_type = ICON_WIFI_NONE;
	p->current_bluetooth_icon_type = p->new_bluetooth_icon_type = ICON_BLUETOOTH_NONE;
	p->current_drain_icon_type = p->new_drain_icon_type = ICON_DRAIN_NONE;
	p->current_cpu_icon_type = p->new_cpu_icon_type = ICON_CPU_NONE;
	asui_battery_set_icon(p);
	p->button = gtk_toggle_button_new();
	gtk_container_add(GTK_CONTAINER(p->button), GTK_WIDGET(p->icon));
	gtk_container_add(GTK_CONTAINER(plugin), p->button);
	g_signal_connect(G_OBJECT(p->button), "button-press-event", G_CALLBACK(map_asui_window), p);
	p->press_timer = 0;

	p->osso = osso_initialize("com.qzx.asui_statusbar_battery", ASUI_VERSION, TRUE, NULL);
	if(!p->osso) ULOG_WARN("%s: error while initializing osso\n", __FUNCTION__);

	gtk_widget_show_all(GTK_WIDGET(plugin));

	battery_init(p);
	bme_init(p);
	cpu_init(p);
}
