//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#define ALP_PATH "/etc/asui-audio-lock-profiles"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#include "window.h"
#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "hardware.h"
#include "button_audio_lock.h"

//////////////////////////////////////////////////////////////////////////

// when this button is active the keys will remain unlocked when the screen is locked
// each key will then be mapped to user-defined commands to control audio apps

typedef struct s_audio_profile {
	char *name;
	struct s_audio_profile *next;
} s_audio_profile;
static s_audio_profile *audio_profiles = NULL, *current_audio_profile = NULL;

void draw_audio_lock_button( unsigned force, int x, int y ){
	if(force){
		// background for all buttons is cleared by draw_window()
		audiolock_button_page = primary_widgets;
	} else return;

	if(current_audio_profile == NULL) load_audio_profiles();

	unsigned cx = x+40, cy = y+40;

	draw_text("audio", 16, fg2_color, cx, cy-14, ALIGN_CENTER, ALIGN_BASELINE, 0);
	draw_text("lock", 16, fg2_color, cx, cy+6, ALIGN_CENTER, ALIGN_BASELINE, 0);
	if(current_audio_profile)
		draw_text(current_audio_profile->name, 16, fg1_color, cx, cy+26, ALIGN_CENTER, ALIGN_BASELINE, 70);
	else
		draw_text("----", 16, fg2_color, cx, cy+26, ALIGN_CENTER, ALIGN_BASELINE, 0);

	draw_tap_markers(x, y, 80, 1, current_audio_profile ? 1 : 0); // short tap only, long tap if a profile is selected
}

//////////////////////////////////////////////////////////////////////////

typedef struct s_audio_key {
	unsigned keycode;
	const char *keyname;
	enum { AL_IGNORE, AL_DBUS, AL_EXEC, AL_EXEC_ROOT, AL_PASS } action;
	char *command;
} s_audio_key;

s_audio_key audio_keys[] = {
	{KEYCODE_ESCAPE, "ESCAPE", AL_IGNORE, NULL},
	{KEYCODE_MENU, "MENU", AL_IGNORE, NULL},
	{KEYCODE_HOME, "HOME", AL_IGNORE, NULL},
	{KEYCODE_ZOOM_PLUS, "PLUS", AL_IGNORE, NULL},
	{KEYCODE_ZOOM_MINUS, "MINUS", AL_IGNORE, NULL},
	{KEYCODE_FULLSCREEN, "FULLSCREEN", AL_IGNORE, NULL},
	{KEYCODE_DPAD_LEFT, "LEFT", AL_IGNORE, NULL},
	{KEYCODE_DPAD_RIGHT, "RIGHT", AL_IGNORE, NULL},
	{KEYCODE_DPAD_UP, "UP", AL_IGNORE, NULL},
	{KEYCODE_DPAD_DOWN, "DOWN", AL_IGNORE, NULL},
	{KEYCODE_ENTER, "ENTER", AL_IGNORE, NULL},
	{0, NULL, AL_IGNORE, NULL}
};

static void set_dbus_key( s_audio_key *ak, char *command ){
	ak->action = AL_DBUS;

// TODO: command = method/signal DEST PATH INTERFACE.METHOD args...
debug_write("unimplemented dbus action for audio lock key");

}
static void set_exec_key( s_audio_key *ak, char *command, unsigned as_root ){
	ak->action = as_root ? AL_EXEC_ROOT : AL_EXEC;
	if(ak->command) free(ak->command);
	ak->command = (char *)malloc(strlen(command)+2+strlen("su - user -c ''"));
	if(as_root)
		sprintf(ak->command, "%s&", command);
	else
		sprintf(ak->command, "su - user -c '%s'&", command);
	debug_write("audio lock: %s bound to %s \"%s\"", ak->keyname, as_root ? "EXEC-ROOT" : "EXEC", ak->command);
}
static unsigned load_audio_profile( ){
	// reset profile
	int i;
	for(i = 0; audio_keys[i].keyname != NULL; i++){
		audio_keys[i].action = AL_IGNORE;
		if(audio_keys[i].command){
			free(audio_keys[i].command);
			audio_keys[i].command = NULL;
		}
	}

	// load profile
	#ifdef ARCH_armel
	FILE *fp;
	char path[256];
	snprintf(path, 256, ALP_PATH "/%s", current_audio_profile->name);
	if((fp = fopen(path, "r")) == NULL) return 0;
	while(!feof(fp)){
		char key[11], action[5], command[1024];
		if(fscanf(fp, "%s %s ", key, action) != 2) break;
		fgets(command, 1024, fp);
		if(feof(fp)) break;
		int end = strlen(command)-1;
		if(command[end] == '\n') command[end] = '\0';

		if(!strcasecmp(action, "launch")){
			// key value is ignored for now
			char *user_command = (char *)malloc(strlen(command)+2+strlen("su - user -c ''"));
			sprintf(user_command, "su - user -c '%s'&", command);
			system(user_command);
			free(user_command);
		} else if(!strcasecmp(action, "launch-root")){
			// key is ignored for now
			char *root_command = (char *)malloc(strlen(command)+2);
			sprintf(root_command, "%s&", command);
			system(root_command);
			free(root_command);
		} else for(i = 0; audio_keys[i].keyname != NULL; i++){
			s_audio_key *ak = &audio_keys[i];
			if(!strcasecmp(key, ak->keyname)){
				if(!strcasecmp(action, "dbus")){ set_dbus_key(ak, command); break; }
				else if(!strcasecmp(action, "exec")){ set_exec_key(ak, command, 0); break; }
				else if(!strcasecmp(action, "exec-root")){ set_exec_key(ak, command, 1); break; }
				else if(!strcasecmp(action, "pass")){ ak->action = AL_PASS; break; }
			}
		}
	}
	fclose(fp);
	return 1;
	#else
	return 0;
	#endif
}

void click_audio_lock_button( ){
	if(click.type == CLICK_HOLD && current_audio_profile){

		// highlight to indicate long press
		highlight_long_press(widget_x, widget_y, 80,80, 1);

	} else if(click.type == CLICK_SHORT){

		highlight_selected_area(widget_x, widget_y, 80,80, 1);

		if(current_audio_profile){
			if(current_audio_profile->next) current_audio_profile = current_audio_profile->next;
			else current_audio_profile = audio_profiles;
			conf_set_string(conf_audio_lock_profile, current_audio_profile->name);
			set_string(&cfg_audio_lock_profile, current_audio_profile->name, 0);
		}

		// redraw button
		x_set_color(secondary_bg_color);
		x_fill_rectangle(widget_x, widget_y, 80, 80);
		draw_audio_lock_button(1, widget_x, widget_y);
		x_flush();

	} else if(click.type == CLICK_LONG && current_audio_profile){

		highlight_selected_area(widget_x, widget_y, 80,80, 1);

		if(load_audio_profile()){
			hw_audio_lock = 1;

			// disable devlock
			conf_set_int(conf_audio_lock_devlock_timeout, conf_get_int(conf_devlock_timeout));
			conf_set_int(conf_devlock_timeout, 1000);

// TODO: set governor

			window_unmap(0);
			window_open_blank();
			mce_lock_and_blank_screen();
		}

		unhighlight_selected_area();

	}
}

//////////////////////////////////////////////////////////////////////////

void load_audio_profiles( ){
	// clear profiles
	if(audio_profiles){
		s_audio_profile *ap = audio_profiles, *next;
		do {
			next = ap->next;
			free(ap->name);
			free(ap);
			ap = next;
		} while(ap);
		audio_profiles = current_audio_profile = NULL;
	}

	// load profiles
	#ifdef ARCH_armel
	DIR *profiles;
	s_audio_profile *tail = NULL;
	if((profiles = opendir(ALP_PATH)) != NULL){
		struct dirent *file;
		while((file = readdir(profiles))){
			if(file->d_name[0] == '.') continue;

			debug_write("found audio lock profile: \"%s\"", file->d_name);

			s_audio_profile *profile = (s_audio_profile *)malloc(sizeof(s_audio_profile));
			profile->name = malloc(strlen(file->d_name)+1);
			strcpy(profile->name, file->d_name);
			profile->next = NULL;
			if(audio_profiles)
				tail->next = profile;
			else
				audio_profiles = profile;
			tail = profile;

			if(!strcmp(file->d_name, cfg_audio_lock_profile))
				current_audio_profile = profile;
		}
		closedir(profiles);
	}
	#endif

	if(current_audio_profile == NULL){
		current_audio_profile = audio_profiles;
		if(current_audio_profile == NULL && strlen(cfg_audio_lock_profile)){
			conf_set_string(conf_audio_lock_profile, "");
			set_string(&cfg_audio_lock_profile, "", 0);
		}
	}
}

void audio_lock_events( XEvent *e ){
	int i;
	for(i = 0; audio_keys[i].keyname != NULL; i++){
		s_audio_key *ak = &audio_keys[i];
		if(ak->keycode == e->xkey.keycode){
			switch(ak->action){
			case AL_IGNORE:
				goto reblank;
			case AL_DBUS:
				// TODO: send dbus signal or method call
				break;
			case AL_EXEC:
			case AL_EXEC_ROOT:
				system(ak->command);
				break;
			case AL_PASS:
				// TODO: pass the key through to the top window
				break;
			}
			break;
		}
	}

	reblank:

	if(cfg_audio_lock_blank_timeout)
		set_timer(&audio_lock_blank_timer, cfg_audio_lock_blank_timeout);
}

void exit_audio_lock( ){
	hw_audio_lock = 0;

	// restore devlock
	conf_set_int(conf_devlock_timeout, conf_get_int(conf_audio_lock_devlock_timeout));
	conf_set_int(conf_audio_lock_devlock_timeout, 0);

// TODO: restore governor

}

////////////////////////////////////////////////////////////////////////// BUTTON

s_button button_audio_lock = {"*audio", "/apps/asui_state/bcell_audio_lock", draw_audio_lock_button, click_audio_lock_button, 0};
