//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#include <unistd.h>
#include <time.h>
#include <esd.h>

#include "main.h"
#include "config.h"
#include "dbus.h"
#include "battery.h"
#include "bme.h"
#include "cpu.h"

//////////////////////////////////////////////////////////////////////////

unsigned dbus_init( AsuiBatteryPluginPrivate *p ){
	// initialize dbus
	GError *error = NULL;
	if((p->connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error)) == NULL){
		ULOG_WARN("%s: can't connect to system method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return 0;
	}

// TODO: connect to session bus?

	p->display_hooks = 0;
	p->display_state = 1; // assume on
	p->current_low_battery_state = LOWBATT_NONE;

	return 1;
}

void dbus_disconnect( AsuiBatteryPluginPrivate *p ){
	dbus_g_connection_unref(p->connection);
}

void dbus_send_system_method( AsuiBatteryPluginPrivate *p, const char *destination, const char *path, const char *interface, const char *method ){
	DBusConnection *system_bus = dbus_g_connection_get_connection(p->connection);
	DBusMessage *message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

void dbus_send_session_method( DBusConnection *bus, const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	va_list ap;
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(bus, message, NULL) != FALSE)
		dbus_connection_flush(bus);
	dbus_message_unref(message);
}

static void dbus_display_notification( AsuiBatteryPluginPrivate *p, const char *msg, const char *sound ){
	// initialize dbus connection
	DBusGConnection *connection;
	GError *error = NULL;
	if((connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error)) == NULL){
		ULOG_WARN("%s: can't connect to session method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return;
	}
	DBusConnection *session_bus = dbus_g_connection_get_connection(connection);

	// play sound
	if(sound){
		unsigned system_alert_volume = conf_get_int("/apps/osso/sound/system_alert_volume");
		if(system_alert_volume){
			conf_set_bool("/apps/asui_state/ignore_master_volume", 1);
			unsigned master_volume = conf_get_int("/apps/osso/sound/master_volume");
			conf_set_int("/apps/osso/sound/master_volume", 50 * system_alert_volume);
			usleep(100000); // 100ms - give gconf time to set volume
			esd_play_file("com.qzx.asui_statusbar_battery", sound, 1);
			usleep(100000); // 100ms - let sound play before reseting volume
			conf_set_int("/apps/osso/sound/master_volume", master_volume);
			conf_set_bool("/apps/asui_state/ignore_master_volume", 0);
		}
	}

	// display notification
	dbus_send_session_method(session_bus, "org.freedesktop.Notifications", "/org/freedesktop/Notifications", "org.freedesktop.Notifications",
											"SystemNoteInfoprint", _STRING_, &msg, _END_);

	// close dbus connection
	dbus_g_connection_unref(connection);
}

//////////////////////////////////////////////////////////////////////////

static void increase_low_battery_state( AsuiBatteryPluginPrivate *p, unsigned capacity ){
	     if(capacity > 16) p->current_low_battery_state = LOWBATT_NONE;
	else if(capacity > 10) p->current_low_battery_state = LOWBATT_16;
	else if(capacity > 4) p->current_low_battery_state = LOWBATT_10;
}

static DBusHandlerResult dbus_battery_signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)user_data;
	DBusError dbus_error;

	if(dbus_message_is_signal(message, "com.qzx.asui", "charging_status")){
		unsigned status;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &status, _END_)){
			p->battery_initializing &= ~1;
			if(p->battery_status != status){
				p->battery_status = status;
				if(!p->battery_initializing){ battery_update_icon(p); bme_update_icon(p, 1); }
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.qzx.asui", "battery_capacity")){
		unsigned capacity;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &capacity, _END_)){
			p->battery_initializing &= ~2;
			if(p->battery_capacity != capacity){
				p->battery_capacity = capacity;
				if(!p->battery_initializing) battery_update_icon(p);

				// play warning sounds when battery drops below 16%, 10% and 4%
				unsigned play_sound = 0;
				switch(p->current_low_battery_state){
				case LOWBATT_NONE:
					     if(capacity <=  4){ p->current_low_battery_state = LOWBATT_4; play_sound = 1; }
					else if(capacity <= 10){ p->current_low_battery_state = LOWBATT_10; play_sound = 1; }
					else if(capacity <= 16){ p->current_low_battery_state = LOWBATT_16; play_sound = 1; }
					break;
				case LOWBATT_16:
					     if(capacity <=  4){ p->current_low_battery_state = LOWBATT_4; play_sound = 1; }
					else if(capacity <= 10){ p->current_low_battery_state = LOWBATT_10; play_sound = 1; }
					else if(capacity >  16) increase_low_battery_state(p, capacity);
					break;
				case LOWBATT_10:
					     if(capacity <= 4){ p->current_low_battery_state = LOWBATT_4; play_sound = 1; }
					else if(capacity > 10) increase_low_battery_state(p, capacity);
					break;
				case LOWBATT_4:
					if(capacity > 4) increase_low_battery_state(p, capacity);
					break;
				}
				if(play_sound){
					char buffer[20];
					snprintf(buffer, sizeof(buffer), "Low Battery - %u%%", capacity);
					dbus_display_notification(p, buffer, "/usr/share/sounds/ui-battery_low.wav");
				}
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.qzx.asui", "wifi_status")){
		unsigned status;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &status, _END_)){
			p->battery_initializing &= ~4;
			if(p->wifi_status != status){
				p->wifi_status = status;
				if(!p->battery_initializing) wifi_update_icon(p);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.qzx.asui", "bluetooth_status")){
		unsigned status;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &status, _END_)){
			p->battery_initializing &= ~8;
			if(p->bluetooth_status != status){
				p->bluetooth_status = status;
				if(!p->battery_initializing) bluetooth_update_icon(p);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_low")){
		dbus_display_notification(p, "Very Low Battery", "/usr/share/sounds/ui-recharge_battery.wav");
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_full")){
		dbus_display_notification(p, "Battery Full", NULL);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_connected")){
		dbus_display_notification(p, "Charging...", "/usr/share/sounds/ui-charging_started.wav");
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusHandlerResult dbus_display_signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)user_data;
	DBusError dbus_error;

	if(dbus_message_is_signal(message, "com.qzx.asui", "display_status")){
		unsigned state;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _BOOL_, &state, _END_)){
			p->display_state = state;
			if(p->use_drain){
				if(state)
					bme_enable(p, 0); // also updates icon, don't hook
				else
					bme_disable(p, 0); // don't unhook
			}
			if(p->use_cpu){
				if(state)
					cpu_enable(p, 0); // also updates icon, don't hook
				else
					cpu_disable(p, 0); // don't unhook
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

//////////////////////////////////////////////////////////////////////////

void dbus_hook_battery( AsuiBatteryPluginPrivate *p ){
	// add dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_add_filter(signal_bus, dbus_battery_signal_filter, (void *)p, NULL);
}

void dbus_unhook_battery( AsuiBatteryPluginPrivate *p ){
	// remove dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_remove_filter(signal_bus, dbus_battery_signal_filter, (void *)p);
}

void dbus_hook_display( AsuiBatteryPluginPrivate *p ){
	p->display_hooks++;

	if(p->display_hooks > 1) return;

	// add dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_add_filter(signal_bus, dbus_display_signal_filter, (void *)p, NULL);
}

void dbus_unhook_display( AsuiBatteryPluginPrivate *p ){
	p->display_hooks--;

	if(p->display_hooks) return;

	// remove dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_remove_filter(signal_bus, dbus_display_signal_filter, (void *)p);
}
