//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

// milliseconds
#define DRAIN_UPDATE_PERIOD 15000

// microseconds (100ms)
#define READ_TIMEOUT 100000

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>

#include "main.h"
#include "dbus.h"
#include "bme.h"

//////////////////////////////////////////////////////////////////////////

// timeout is in microseconds
static int read_with_timeout( int fd, void *buf, size_t count ){
	struct timeval tv;
	fd_set set;

	FD_ZERO(&set);
	FD_SET(fd, &set);
	tv.tv_sec = 0;
	tv.tv_usec = READ_TIMEOUT;
	if(select(fd+1, &set, NULL, NULL, &tv) < 1) return 0;

	return read(fd, buf, count);
}

static unsigned bme_connect( AsuiBatteryPluginPrivate *p ){
	struct sockaddr_un addr;

	p->bme_fd = socket(PF_UNIX, SOCK_STREAM, 0);
	if(p->bme_fd < 0){
		ULOG_WARN("%s: can't open BME socket\n", __FUNCTION__);
		p->bme_fd = 0;
		return 0;
	}
	memset(&addr, 0, sizeof(addr));
	addr.sun_family = PF_UNIX;
	strcpy(addr.sun_path, "/tmp/.bmesrv");
	if(connect(p->bme_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0){
		ULOG_WARN("%s: can't connect to BME socket\n", __FUNCTION__);
		close(p->bme_fd);
		p->bme_fd = 0;
		return 0;
	}
	fcntl(0, F_SETFL, fcntl(0, F_GETFL) | O_NONBLOCK);

	#define COOKIE "BMentity"
	if(write(p->bme_fd, (void *)COOKIE, sizeof(COOKIE)-1) != sizeof(COOKIE)-1){
		ULOG_WARN("%s: can't send cookie to BME server\n", __FUNCTION__);
		close(p->bme_fd);
		p->bme_fd = 0;
		return 0;
	}

	char reply;
	int len = read_with_timeout(p->bme_fd, (void *)&reply, 1);
	if(len == 1 && reply == '\n') return 1;
	ULOG_WARN("%s: connection to BME refused, received '%c'\n", __FUNCTION__, reply);
	close(p->bme_fd);
	p->bme_fd = 0;
	return 0;
}

//////////////////////////////////////////////////////////////////////////

static struct {
	uint32_t unknown1;
	uint32_t unknown2;

	uint32_t unknown3;
	uint16_t sw_status;	// Battery monitor SW status
	uint16_t instaneous_battery_voltage; // Instantaneous battery voltage (mV)

	uint16_t offset16;	// Remaining standby time to battery low (mins)
	uint16_t unknown4;
	uint16_t unknown5;
	uint16_t unknown6;

	uint16_t offset24;	// Battery monitor check voltage (mV)
	uint16_t offset26;	// Battery low warning interval counter
	uint16_t offset28;	// Double median filtered battery voltage
	uint16_t offset30;	// Initial battery monitor voltage (mV)

	uint16_t offset32;	// Time per battery bar (mins)
	uint16_t offset34;	// DMF voltage sampled at first battery low (mV)

	uint32_t drain;		// Average phone current (uA)

	uint16_t offset40;	// Most recent battery charge condition (mAh)
	uint16_t offset42;	// Lowest TX-Off voltage (mV)
	uint16_t offset44;	// Lowest TX-On voltage (mV)
	uint16_t offset46;	// Largest TX-Off/On voltage difference (mV)

	uint8_t  offset48;	// Battery bar level log mask
	uint8_t  offset49;	// Previous battery bar level
	uint8_t  offset50;	// Battery low reason
	uint8_t  offset51;	// CS state information
	uint16_t offset52;	// Number of battery bars
	uint16_t offset54;	// Battery type

	uint16_t offset56;	// Temperature, in kelvin
	uint16_t offset58;	// Battery capacity
	uint16_t offset60;	// Battery impedance (mOhm)
	uint16_t offset62;	// Present value of v_bat_full_level

	uint16_t offset64;	// Present value of v_bat_low_ths_mv
	uint16_t unknown7;
	uint16_t unknown8;
	uint16_t unknown9;

	uint16_t offset72;	// Load current estimated by Batmon4 (uA)
	uint16_t unknown10;
} bme_reply;

static unsigned bme_send_request( AsuiBatteryPluginPrivate *p ){
	if(!p->bme_fd) if(!bme_connect(p)) return 0; // problem connecting

	struct {
		uint16_t type;
		uint16_t subtype;
		uint32_t flags;
	} request = {0x42, 0, 0xFFFFFFFF};
	if(write(p->bme_fd, (void *)&request, sizeof(request)) != sizeof(request)){
		ULOG_WARN("%s: can't send request to BME server\n", __FUNCTION__);
		close(p->bme_fd);
		p->bme_fd = 0;
		return 0;
	}
	int len = read_with_timeout(p->bme_fd, (void *)&bme_reply, sizeof(bme_reply));
	if(len == sizeof(bme_reply)) return 1;
	ULOG_WARN("%s: received incorrect reply from BME server, got %d of %d bytes\n", __FUNCTION__, len, sizeof(bme_reply));
	close(p->bme_fd);
	p->bme_fd = 0;
	return 0;
}

static gboolean bme_update_drain( gpointer data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)data;

	if(!p->use_drain){
		p->bme_timer = 0;
		return FALSE; // destroy timer
	}

	bme_update_icon(p, 0);

	return TRUE; // repeat timer
}

void bme_update_icon( AsuiBatteryPluginPrivate *p, unsigned reset_timer ){
	switch(p->battery_status){
	case BATTERY_DRAINING:
		{
			unsigned status = bme_send_request(p);
			if(!status) status = bme_send_request(p); // try it a second time

			if(status){
				unsigned drain = bme_reply.drain/1000; // 5 second average drain rate (mA)

				     if(drain >= 375) p->new_drain_icon_type = ICON_DRAIN_VERYHIGH;		// less than 4 hours
				else if(drain >= 214) p->new_drain_icon_type = ICON_DRAIN_HIGH;			// 4-7 hours
				else if(drain >= 150) p->new_drain_icon_type = ICON_DRAIN_MEDIUM;		// 7-10 hours
				else p->new_drain_icon_type = ICON_DRAIN_LOW;							// greater than 10 hours
			} else
				p->new_drain_icon_type = ICON_DRAIN_NONE; // failed to get new value, turn off the icon to prevent stale readings
		}
		break;
	case BATTERY_CHARGING:
		p->new_drain_icon_type = ICON_DRAIN_CHARGING;
		break;
	case BATTERY_AC:
		p->new_drain_icon_type = ICON_DRAIN_NONE;
		break;
	}
	asui_battery_set_icon(p);

	if(reset_timer){
		if(p->bme_timer) g_source_remove(p->bme_timer);
		p->bme_timer = (p->display_state ? g_timeout_add(DRAIN_UPDATE_PERIOD, bme_update_drain, (void *)p) : 0);
	}
}

//////////////////////////////////////////////////////////////////////////

void bme_init( AsuiBatteryPluginPrivate *p ){
	p->bme_fd = 0;
	p->bme_timer = 0;
	if(p->use_drain) bme_enable(p, 1); // hook
}

void bme_disable( AsuiBatteryPluginPrivate *p, unsigned unhook ){
	if(p->bme_timer){ g_source_remove(p->bme_timer); p->bme_timer = 0; }

	if(unhook){
		dbus_unhook_display(p);
		if(p->bme_fd){ close(p->bme_fd); p->bme_fd = 0; }
	}
}

void bme_enable( AsuiBatteryPluginPrivate *p, unsigned hook ){
	bme_update_icon(p, 0);

	if(p->bme_timer) g_source_remove(p->bme_timer);
	p->bme_timer = (p->display_state ? g_timeout_add(DRAIN_UPDATE_PERIOD, bme_update_drain, (void *)p) : 0);

	if(hook) dbus_hook_display(p);
}
