//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <brand.huntsman@gmail.com>

// seconds (double)
#define INITIAL_REFRESH_PERIOD 1.0
#define REFRESH_PERIOD 5.0
#define SCROLL_UPDATE_PERIOD 0.100

#include <stdio.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>
#include <signal.h>

#include "window.h"
#include "main.h"
#include "draw.h"
#include "text.h"
#include "services.h"
#include "dialog.h"
 #include "process_viewer.h"

//////////////////////////////////////////////////////////////////////////

static int top_process, drag_top_process, selected_process_y;
static unsigned max_viewable, selected_process, disable_process_refresh;

//////////////////////////////////////////////////////////////////////////

static void draw_kill_confirm( unsigned force ){
	// force is not needed

	s_process *process;
	for(process = svc_process_list; process != NULL; process = process->next)
		if(process->pid == selected_process) break;
	if(process == NULL){
		dialog_close();
		return;
	}

	// dialog background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);

	// message
	{
		char buffer[30];

		draw_text(process->name, 22, fg1_color, window.width>>1, (window.width > window.height ? 125 : 200)-30, ALIGN_CENTER, ALIGN_MIDDLE, window.width);
		snprintf(buffer, sizeof(buffer), "%u", process->pid);
		draw_text(buffer, 24, fg2_color, window.width>>1, (window.width > window.height ? 125 : 200)+30, ALIGN_CENTER, ALIGN_MIDDLE, 0);
	}

	// buttons
	draw_cdlg_buttons("Kill", "Cancel");
}

static void kill_yes_handler( ){
	if(selected_process){
		// kill the process
		kill(selected_process, SIGKILL);
	}

	// reopen process viewer
	current_dialog_draw = NULL; // close kill confirmation dialog
	process_viewer_open_dialog();
}

static void kill_no_handler( ){
	// reopen process viewer
	current_dialog_draw = NULL; // close kill confirmation dialog
	process_viewer_open_dialog();
}

//////////////////////////////////////////////////////////////////////////

static void highlight_process( int y ){
	// remove bottom separator
	x_set_color(bg_color);
	x_draw_line(0, y+4, window.width-10, y+4);
	// highlight process
	x_set_color(selection_color);
	x_draw_rectangle(0, y-16, window.width-10,20);

	selected_process_y = y;
}

static void unhighlight_process( ){
	// unhighlight previous process
	x_set_color(bg_color);
	x_draw_rectangle(0, selected_process_y-16, window.width-10,20);
	// redraw separators
	x_set_color(line2_color);
	if(selected_process_y > 34)
		x_draw_line(0, selected_process_y-16, window.width-10, selected_process_y-16);
	if(selected_process_y < (int)window.height-82)
		x_draw_line(0, selected_process_y+4, window.width-10, selected_process_y+4);
}

static void _draw_process_list( unsigned force, unsigned top_offset, unsigned count ){
	s_process *process;
	float sb_ratio = (float)max_viewable/(float)svc_nr_processes;
	int sb_maxlen = window.height-(17+61), sb_len = sb_ratio*sb_maxlen, sb_y;
	unsigned selected_process_is_visible = 0;
	char buffer[30];

	if((unsigned)top_process + max_viewable > svc_nr_processes) top_process = svc_nr_processes - max_viewable;

	sb_y = 17 + (sb_maxlen - sb_len) * (float)top_process / (float)(svc_nr_processes - max_viewable);

	if(force){
		// clear background
		x_set_color(bg_color);
		x_fill_rectangle(0, 0, window.width, window.height);

		// column names
		draw_text("PID", 16, fg2_color, 0, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("Process", 16, fg2_color, 80, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("Size", 16, fg2_color, window.width-120, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("CPU", 16, fg2_color, window.width-50, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		x_set_color(line1_color);
		x_draw_line(0, 15, window.width, 15);

		// row separators
		x_set_color(line2_color);
		x_set_style(1, LineOnOffDash); // dotted line
		int y;
		for(y = 22+16; y < (int)window.height-62; y += 20)
			x_draw_line(0, y, window.width-10, y);
	} else
		x_set_style(1, LineOnOffDash); // dotted line

	// process rows
	int n = 0;
	for(process = svc_process_list; process != NULL && n - top_process < (int)max_viewable; process = process->next, n++){
		int y = 20*(n - top_process + 1) + 2 + 12;
		XColor *color;

		if(n < top_process) continue;

		// top rows are copied when scrolling
		if(top_offset == 0){
			// bottom rows are copied when scrolling
			if(count){
				if(!force){
					// row background
					x_set_color(bg_color);
					x_fill_rectangle(1, y-15, window.width-12+(process->pid == selected_process ? 0 : 2), 19);
				}

				// pid
				snprintf(buffer, sizeof(buffer), "%u", process->pid);
				draw_text(buffer, 16, fg2_color, 2, y, ALIGN_LEFT, ALIGN_BASELINE, 0);

				// name, rss and cpu color
				if(process->state == 'Z') color = red_color;
				else if(process->is_service) color = green_color;
				else if(process->is_other_service) color = scrollbar_color;
				else if(process->rss == 0) color = fg2_color;
				else color = fg1_color;

				// name
				draw_text(process->name, 16, color, 80, y, ALIGN_LEFT, ALIGN_BASELINE, window.width-(130+80));
//snprintf(buffer, sizeof(buffer), "%d", n);
//draw_text(buffer, 16, color, 80, y, ALIGN_LEFT, ALIGN_BASELINE, window.width-(130+80));

				if(process->state != 'Z'){
					// rss
					snprintf(buffer, sizeof(buffer), "%.1f", roundf(10.0*(float)process->rss/1024.0)/10.0);
					draw_text(buffer, 16, color, window.width-120, y, ALIGN_LEFT, ALIGN_BASELINE, 0);

					// cpu
					if(process->cpu){
						sprintf(buffer, "%u%%", process->cpu);
						draw_text(buffer, 16, color, window.width-50, y, ALIGN_LEFT, ALIGN_BASELINE, 0);
					}
				} else
					draw_text("zombie", 16, color, window.width-80, y, ALIGN_LEFT, ALIGN_BASELINE, 0);
			}
			count--;
		} else top_offset--;

		// selection box or row separator
		if(process->pid == selected_process){
			if(force || selected_process_y == 0){
				highlight_process(y);
			} else if(y != selected_process_y){
				unhighlight_process();
				highlight_process(y);
			}
			selected_process_is_visible = 1;
		}
	}
	if(!selected_process_is_visible && selected_process_y){
		unhighlight_process();
		selected_process_y = 0;
	}
	x_reset_style();

	if(sb_ratio < 1.0){
		if(!force){
			// clear scrollbar area
			x_set_color(bg_color);
			x_fill_rectangle(window.width-9, 17, 9, window.height-(17+61));
		}
		// draw scrollbar
		x_set_color(scrollbar_color);
		x_fill_rectangle(window.width-5, sb_y, 5, sb_len);
	}

	if(force){
		// button bar
		x_set_color(line1_color);
		x_draw_line(0, window.height-60, window.width, window.height-60);
		x_draw_line(window.width>>1, window.height-60, window.width>>1, window.height);

		// close button
		draw_text("Close", 24, button_color, window.width - (window.width>>2), window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
	}

	// kill button
	if(selected_process){
		if(!force){
			// kill button background
			x_set_color(bg_color);
			x_fill_rectangle(0, window.height-59, (window.width>>1)-1, 60);
		}

		// button label and PID
		snprintf(buffer, sizeof(buffer), "Kill Process (%u)", selected_process);
		draw_text(buffer, 24, button_color, window.width>>2, window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
	}
}

static void draw_process_list( unsigned force ){
	// refresh process list
	if(!disable_process_refresh)
		svc_refresh();

	if(!disable_process_refresh || force){
		max_viewable = (window.height-(20+60)) / 20;
		_draw_process_list(force, 0, max_viewable);
	}

	// change dialog refresh timer duration after first refresh
	if(dialog_refresh_timer.duration == INITIAL_REFRESH_PERIOD) dialog_refresh_timer.duration = REFRESH_PERIOD;
	// reset dialog refresh timer
	reset_timer(&dialog_refresh_timer);
}

static void reset_process_list( ){
	// free the process list maintained by svc_refresh()
	svc_process_viewer = 0;
}

static unsigned scroll_process_list( ){
	if(click.y + click.offset_y > 15 && click.y + click.offset_y <= (int)window.height-60){
		// scroll list
		int top = drag_top_process - click.offset_y/20;
		if(click.offset_y > 0 && top < 0) top = 0;
		if((unsigned)top + max_viewable > svc_nr_processes) top = svc_nr_processes - max_viewable;

		if(top != top_process){
			int row_shift = top_process - top, height, src_y, dst_y;
			top_process = top;
			if(row_shift > 0){
				// shift down
				height = 20*(max_viewable - row_shift - 1) + 18;
				src_y = 19;
				dst_y = window.height - height - 63;

				Pixmap pixmap = XCreatePixmap(window.display, window.window, window.width-10, height, window.depth);
				XCopyArea(window.display, window.window, pixmap, window.gc, 0, src_y, window.width-10, height, 0, 0);
				XCopyArea(window.display, pixmap, window.window, window.gc, 0, 0, window.width-10, height, 0, dst_y);
				XFreePixmap(window.display, pixmap);
				_draw_process_list(0, 0, row_shift);
			} else {
				// shift up
				height = 20*(max_viewable + row_shift - 1) + 18;
				src_y = window.height - height - 63;
				dst_y = 19; // top of first row

				XCopyArea(window.display, window.window, window.window, window.gc, 0, src_y, window.width-10, height, 0, dst_y);
				_draw_process_list(0, max_viewable + row_shift, -row_shift);
			}

			return 1; // motion timer is reset by main loop
		}
	}

	return 0;
}

static void click_process_list( ){
	if(click.y > (int)window.height-60){
		if(click.type != CLICK_SHORT) return; // no long press or drag support

		if(click.x > (int)window.width>>1){
			// close button
			highlight_selected_area((window.width>>1)+1, window.height-59, (window.width>>1)-1,59, 0);

			dialog_close();
		} else if(selected_process){
			// kill selected process
			highlight_selected_area(0, window.height-59, (window.width>>1)-1,59, 0);

			yes_color = red_color;
			cdlg_yes_handler = kill_yes_handler;
			cdlg_no_handler = kill_no_handler;
			current_dialog_draw = NULL; // close process viewer dialog
			dialog_open(draw_kill_confirm, click_cdlg, keypress_cdlg, reset_process_list, DLG_REDRAW_ON);
			// redraw is needed to close the dialog if the process terminates
		}
	} else if(click.y > 15){
		if(click.type == CLICK_SHORT){

			// select process
			unsigned n = top_process + (click.y-18)/20; // it doesn't matter that this can be negative if y is 16 or 17
			s_process *process;

			// clicking the last three pixels causes the process just offscreen to highlight, don't want that
			if(n == top_process + max_viewable) n--;

			for(process = svc_process_list; process != NULL && n; process = process->next, n--);
			if(process) selected_process = process->pid;
			else selected_process = 0;

			_draw_process_list(0, 0, max_viewable);

			disable_process_refresh = 0;
			motion_callback = NULL;

		} else if(click.type == CLICK_DOWN){

			// initialize a scroll
			drag_top_process = top_process;
			disable_process_refresh = 1;
			motion_callback = scroll_process_list;
			set_timer(&motion_timer, SCROLL_UPDATE_PERIOD);

		} else if(click.type == CLICK_DRAG || click.type == CLICK_LONG){

			scroll_process_list();

			disable_process_refresh = 0;
			motion_callback = NULL;

		}
	}
	// ignore clicks at top of screen
}

static void keypress_process_list( ){
	if(is_arrow(ARROW_UP)){
		if(keypress.type == KEY_SHORT){
			// scroll process list up
			if(top_process){
				top_process -= max_viewable * 8 / 10;
				if(top_process < 0) top_process = 0;
				_draw_process_list(0, 0, max_viewable);
				// just redraw it all, blitting won't help
			}
		}
	} else if(is_arrow(ARROW_DOWN)){
		if(keypress.type == KEY_SHORT){
			// scroll process list down
			if((unsigned)top_process + max_viewable < svc_nr_processes){
				top_process += max_viewable * 8 / 10;
				_draw_process_list(0, 0, max_viewable);
				// just redraw it all, blitting won't help
			}
		}
	} else switch(keypress.keycode){
	case KEYCODE_ESCAPE:
		if(keypress.type == KEY_SHORT){
			// close process list
			highlight_selected_area((window.width>>1)+1, window.height-59, (window.width>>1)-1,59, 0);
			dialog_close();
		}
		break;
	default:
		if(!dpadlock_keypress_handler())
			brv_keypress_handler();
				// ignore all other keys
		break;
	}
}

//////////////////////////////////////////////////////////////////////////

void process_viewer_open_dialog( ){
	// build a list of processes with service_refresh()
	svc_process_viewer = 1;
	// initialize scrollbar and selected process
	top_process = 0;
	selected_process = selected_process_y = 0;
	disable_process_refresh = 0;
	// open dialog
	dialog_open(draw_process_list, click_process_list, keypress_process_list, reset_process_list, DLG_REDRAW_ON);
	// set dialog refresh duration to INITIAL_REFRESH_PERIOD seconds to get cpu usage values faster
	set_timer(&dialog_refresh_timer, INITIAL_REFRESH_PERIOD);
}
