//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/input.h>

#include "window.h"
#include "main.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
#include "dbus.h"
#include "dsme.h"
#include "alarm.h"

////////////////////////////////////////////////////////////////////////// GLOBAL

#ifdef ARCH_armel
unsigned powerbutton_waiting_double;
#endif

//////////////////////////////////////////////////////////////////////////

#ifdef ARCH_armel
static unsigned screen_was_on, screen_was_locked;
static unsigned powerbutton_double;
static s_timer powerbutton_double_timer;
static double down_timestamp;

static void puk_unlock( unsigned release ){
	if(cfg_power_unlock_key){
		// secondary key required to unlock screen, keypad remains unlocked
		debug_write("    secondary key required to unlock, tc=1, kc=0");
		hw_lock_screen();
		hw_unlock_keys();
		if(!window.is_mapped) window_open_blank(); // open blank to catch keys
		if(release) notify_power_unlock_key();
	} else {
		// screen needs to be unlocked
		window_close_blank(); // power key caught, close the blank window
		if(!release) mce_unlock_screen();
	}
}

static void powerbutton_handler( unsigned release, unsigned power_action ){
	if(use_minimal_ui || current_dialog_draw == draw_alarm){
		// charging runlevel or alarm active, power button does nothing

	} else if(screen_was_on){
		// screen has been on for more than one second
		// CASE: power button used to blank screen (when locked)
		// CASE: power button used to lock, lock&blank or blank screen (when devlocked)
		// CASE: power button used to map/unmap ASUI

		// main loop doesn't need to be awakened, unless showing or hiding ASUI

		// these actions will only be performed on up events (short and second double tap)

		if(release){
			if(screen_was_locked){
				if(cfg_power_blanks_when_locked){
					hw_lock_keys(); // MCE unlocked the keys for SystemUI's secondary unlock key, relock them

					// screen is locked, power button will blank it
					int i;
					for(i = 0; i < 20; i++){
						if(hw_screen.dimmed) break;
						usleep(100000); // sleep 20 x 100ms until screen dims
					}
					mce_lock_and_blank_screen();
				} else
					puk_unlock(1);
			} else switch(power_action){
			case POWER_ACTION_NONE:
				// do nothing
				break;
			case POWER_ACTION_LOCK:
				// lock screen
				mce_lock_screen(1);
				break;
			case POWER_ACTION_LOCKBLANK:
				// lock&blank screen
				mce_lock_and_blank_screen();
				break;
			case POWER_ACTION_BLANK:
				// blank screen
				mce_blank_screen();
				break;
			case POWER_ACTION_OPEN:
				// map ASUI
				if(!window.is_mapped) dbus_send_wakeup(_WAKEUP_SHOW_);
				break;
			case POWER_ACTION_OPENCLOSE:
				// map or unmap ASUI
				if(window.is_mapped)
					dbus_send_wakeup(_WAKEUP_HIDE_);
				else
					dbus_send_wakeup(_WAKEUP_SHOW_);
				break;
			}
		} else if(screen_was_locked && !cfg_power_blanks_when_locked)
			puk_unlock(0);
	} else {
		// screen is in the process of turning on or has been on less than one second
		// CASE: power button used to turn on screen (locked or unlocked)

		// main loop doesn't need to be awakened, DSME socket will do it

		// these actions will be performed on both down and up events (short, long and first/second double tap)
		// notifications are only displayed on up events (short and second double tap)

		if(screen_was_locked){
			puk_unlock(release);
		} else {
			// screen wasn't locked
			window_close_blank(); // power key caught, close the blank window
			if(cfg_tkactive_notifications && release) notify_screen_unlocked();
		}
	}
}
#endif

//////////////////////////////////////////////////////////////////////////

static int powerbutton_fd;

int powerbutton_init( ){
	#ifdef ARCH_armel
	powerbutton_fd = open((is_810 ? "/dev/input/event1" : "/dev/input/event0"), O_RDONLY|O_NDELAY);
	if(powerbutton_fd < 0){
		error_write("can't open power button input device");
		return 0;
	}

	powerbutton_waiting_double = 0;
	powerbutton_double = 0;
	#else
	powerbutton_fd = 0;
	#endif

	return powerbutton_fd;
}

void powerbutton_disconnect( ){
	#ifdef ARCH_armel
	close(powerbutton_fd);
	#endif
}

//////////////////////////////////////////////////////////////////////////

void powerbutton_read( ){
	#ifdef ARCH_armel
	struct input_event ev;
	int rlen = read(powerbutton_fd, &ev, sizeof(ev));
	if(rlen != sizeof(ev)){
		error_write("could not read %d bytes from power button input device, got %d bytes", sizeof(ev), rlen);
		return;
	}

	if(ev.type == 1 && ev.code == 116){
		double timestamp = (double)ev.time.tv_sec + (double)ev.time.tv_usec / 1000000.0;
		if(ev.value){
			// power button down
			down_timestamp = timestamp;
			if(!powerbutton_waiting_double){
				// not updated for the second down event in a double-tap
				screen_was_on = (hw_screen.visible && current_time - hw_screen.visible_time > 1.0 ? 1 : 0);
				screen_was_locked = hw_screen.locked;
			}

			// power button will turn on the screen, if off
			dsme_display_state_waiting = 0; // reset blanking pause check

			debug_write("[POWERBUTTON] down event -- display was %s -- tklock=%u, asui=%u", (screen_was_on ? "on" : "off"), hw_screen.locked, window.is_mapped);

			// screen will turn on and needs to be relocked
			if(!powerbutton_waiting_double){
				powerbutton_handler(0, POWER_ACTION_NONE);
			} else {
				powerbutton_waiting_double = 0;
				powerbutton_double = 1;
			}
		} else {
			// power button up
			if(timestamp - down_timestamp > cfg_power_long_delay){
				if(powerbutton_double){
					// short tap followed by a long tap, execute single-tap action before longpress
					debug_write("[POWERBUTTON] short press - single-tap");
					powerbutton_handler(1, (hw_device_locked ? cfg_secure_power_action_single : cfg_power_action_single));

					powerbutton_double = 0;
				}

				// long press - let MCE handle it
				debug_write("[POWERBUTTON] long press -- %s", (hw_screen.locked ? (cfg_power_longpress_shutdown ? "ASUI shutdown" : "no action") : "MCE shutdown"));

				// MCE won't shutdown if screen is locked, this will leave the screen in a bad state so lets fix it
				screen_was_on = 0;
				powerbutton_handler(1, POWER_ACTION_NONE);

				// force a shutdown in case MCE doesn't
				if(cfg_power_longpress_shutdown && hw_screen.locked) mce_shutdown();
			} else if(powerbutton_double){
				// short press - double tap
				debug_write("[POWERBUTTON] short press - double-tap");
				powerbutton_handler(1, (hw_device_locked ? cfg_secure_power_action_double : cfg_power_action_double));

				powerbutton_double = 0;
			} else if(cfg_power_enable_double){
				// short press - wait for double
				debug_write("[POWERBUTTON] short press, waiting for double...");
				powerbutton_waiting_double = 1;
				set_timer(&powerbutton_double_timer, cfg_power_double_delay);
			} else {
				// short press - single tap
				debug_write("[POWERBUTTON] short press - single-tap");
				powerbutton_handler(1, (hw_device_locked ? cfg_secure_power_action_single : cfg_power_action_single));
			}
		}
	} else
		error_write("unknown power button event: type=%u, code=%u, value=%d", ev.type, ev.code, ev.value);
	#endif
}

double powerbutton_check_double_tap( ){
	#ifdef ARCH_armel
	if(is_timer_ready(&powerbutton_double_timer)){
		// timer has expired, execute single-tap action
		debug_write("[POWERBUTTON] short press - single-tap");
		powerbutton_handler(1, (hw_device_locked ? cfg_secure_power_action_single : cfg_power_action_single));

		powerbutton_waiting_double = 0;
		return 0.0;
	}
	return powerbutton_double_timer.remaining;
	#else
	return 0.0;
	#endif
}
