//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>
//
//	all dsme code ported from powerlaunch
//	Copyright (c) 2007-2008 Austin Che

#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "main.h"
#include "window.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
#include "dbus.h"
 #include "dsme.h"

//#define DISPLAY_BLANK_PAUSE_TIMEOUT 60000 /* 60 seconds */

#define DSME_GET_DEVICELOCK_CODE 0x1200
#define DSME_IN_GET_DEVICELOCK_CODE 0x1201
//#define DSME_SET_DEVICELOCK_CODE 0x1202
#define DSME_IN_SET_DEVICELOCK_CODE 0x1203

#define DSME_DISPLAY_BLANKING_PAUSE 0x0201
#define DSME_DISPLAY_BLANKING_ALLOW 0x0202
//#define DSME_SET_DISPLAY_BLANK_TIMEOUT 0x0203
//#define DSME_SET_DISPLAY_DIM_TIMEOUT 0x0204
//#define DSME_SET_DISPLAY_STATE 0x0280
#define DSME_IN_DISPLAY_STATE 0x0285
#define DSME_SET_DISPLAY_BRIGHTNESS 0x0289

//#define DSME_IN_SYSTEM_STATE 0x0301
//#define DSME_GET_SYSTEM_STATE 0x0302
//#define DSME_IN_SYSTEM_INACTIVITY 0x0303
//#define DSME_IN_SAVE_UNSAVED_DATA 0x0304
//#define DSME_REQUEST_POWERUP 0x0305
//#define DSME_REQUEST_SHUTDOWN 0x0306
//#define DSME_SET_ALARM_MODE 0x0307
//#define DSME_REQUEST_REBOOT 0x0308
//#define DSME_IN_SYSTEM_ACTIVITY 0x0311

// I've received the string /usr/sbin/ke-recv and /usr/bin/hildon-input-method with this message on shutdown
// these programs crash on shutdown. I think this is a message about that
// the above strings followed by \0 0x01 \0 and then 0x2c or 0x30
#define DSME_IN_PROCESS_UNKNOWN 0x0406

// PROCESS_PING 0x0504
//#define DSME_SET_SOFTPOWERON 0x0800
//#define DSME_SET_SOFTPOWEROFF 0x0801

// 0x0802 and 0x0803 might be related to MMC door as they happen everytime HAL sends an MMC door event

// soft power off doesn't appear to be any different than blanking the display

////////////////////////////////////////////////////////////////////////// GLOBALS

char *device_lock_code;
unsigned dsme_display_state_waiting;

//////////////////////////////////////////////////////////////////////////

static int dsme_fd;

static void dsme_write( unsigned type ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
	} message;
	message.len = sizeof(message);
	message.type = type;
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}

static void dsme_write_one_arg( unsigned type, unsigned arg ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
		unsigned arg;
	} message;
	message.len = sizeof(message);
	message.type = type;
	message.arg = arg;
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}

//////////////////////////////////////////////////////////////////////////

int dsme_init( ){
	struct sockaddr_un addr;

	dsme_fd = socket(PF_UNIX, SOCK_STREAM, 0);
	if(dsme_fd < 0){
		error_write("can't open dsme socket");
		return 0;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path, "/tmp/dsmesock");
	if(connect(dsme_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0){
		close(dsme_fd);
		error_write("can't connect to dsme socket");
		return 0;
	}

	device_lock_code = NULL;

	return dsme_fd;
}

void dsme_disconnect( ){
	#ifdef ARCH_armel
	dsme_allow_blanking(); // make sure blanking is enabled before exiting
	close(dsme_fd);
	#endif
}

//////////////////////////////////////////////////////////////////////////

#ifdef ARCH_armel
static int read_data( void *buffer, int len ){
	int rlen = read(dsme_fd, buffer, len);
	if(rlen != len){
		error_write("could not read %d bytes from dsme, got %d bytes", len, rlen);
		return -1;
	}
	return rlen;
}
#endif

static e_dsme_display_state display_state;

void dsme_read( ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
	} header;

	int rlen;
	if((rlen = read_data(&header, sizeof(header))) < 0){
		error_write("could not read dsme header");
		return;
	}

	char buffer[256];
	int data_size = header.len - rlen;
	if(data_size > (int)sizeof(buffer)){
		error_write("length(%d) in dsme header is too large", header.len);
		lseek(dsme_fd, 0, SEEK_END);
		return;
	}

	if((rlen = read_data(buffer, data_size)) < 0){
		error_write("could not read dsme data");
		return;
	}

	switch (header.type){
	case DSME_IN_GET_DEVICELOCK_CODE:
		{
			struct {
				char str[16];	// "lock_code"
				unsigned a1;	// 0x01
				unsigned a2;	// 0x10
				unsigned a3;	// 0x01
				char code[12];	// the device code
			} *message = (void *)buffer;

			if(rlen == sizeof(*message))
				if(!strcmp(message->str, "lock_code")){
					set_string(&device_lock_code, message->code, 10); // max of 10 digits
					debug_write("[DSME] got lock code **********");
				}
		}
		break;
	case DSME_IN_DISPLAY_STATE:
		if(rlen == 4){
			display_state = (e_dsme_display_state)(*(int *)buffer);
			unsigned visible = (display_state == DSME_DISPLAY_OFF ? 0 : 1);
			unsigned dimmed = (display_state == DSME_DISPLAY_DIM ? 1 : 0);

			if(!hw_screen.visible && visible){
				// screen was just turned on

				hw_screen.visible_time = current_time;
				hw_screen.brightness_mismatch = 1;
				hw_restore_cp_screen_settings();
				window_open_popup();

				debug_write("[DSME] screen on");
			} else if(hw_screen.visible && !visible){
				// screen was just turned off

				if(!hw_screen.locked){
					// screen is not locked
					if(!conf_get_screen_autolock_value()){
						// screen does not autolock
						if(!window.is_mapped)
							window_open_blank(); // screen is off but not locked, ASUI is not open -- open a blank window to catch the wakeup tap/key
						click.enabled = 0; // ignore first screen tap when unblanked
					}
				}
				window_close_popup();

				debug_write("[DSME] screen off");
			}
			if(hw_screen.visible != visible){
				hw_new_event("screen", &hw_screen.history_head, &hw_screen.history_tail, visible);
				hw_screen.visible = visible;
				dbus_send_display_status_signal();
			}
			// don't need to wakeup main loop for this, dbus unlock signal or X screen/key events will wake it

			if(hw_screen.dimmed != dimmed){
				hw_screen.brightness_mismatch = 1;
				hw_screen.dimmed = dimmed;
				if(is_value_visible(brightness_widget_page)){ hw_changes |= HW_BRIGHTNESS; dbus_send_wakeup(_WAKEUP_REFRESH_); }

				debug_write("[DSME] screen dimmed = %u", dimmed);
			}
		}
		break;
	default:
		debug_write("[DSME] unhandled message 0x%x with %d bytes of data", header.type, rlen);

		if(header.type == DSME_IN_PROCESS_UNKNOWN){
			char cmd[260];
			snprintf(cmd, ((unsigned)rlen > sizeof(cmd) ? sizeof(cmd) : (unsigned)rlen), "%s", (char *)buffer);
			debug_write("       a program may have crashed: %s", cmd);
		}

/*
		error_write("unhandled message 0x%x with %d bytes of data", header.type, rlen);
		int i;
		for(i = 0; i < rlen; i++){
			printf("%02x ", buffer[i]);
			if(i % 50 == 49) printf("\n");
		}
		if(rlen) printf("\n");
*/
		break;
	}
	#endif
}

////////////////////////////////////////////////////////////////////////// BLANKING PAUSE

void dsme_pause_blanking( ){
	dsme_write(DSME_DISPLAY_BLANKING_PAUSE);
}

void dsme_allow_blanking( ){
	dsme_write(DSME_DISPLAY_BLANKING_ALLOW);
}

// MCE's req_display_blanking_pause method somehow overrides DSME's allow blanking message
// the following code displays a notification when ASUI attempts to change the display state while MCE has blanking paused

static e_dsme_display_state expected_display_state;
static s_timer display_state_timer;

void dsme_expect_display_state( e_dsme_display_state state ){
	// if screen is expected to dim but blank timeout equals dim timeout then it will always transition from on to off without dimming
	if(conf_get_int(conf_blank_timeout) == conf_get_int(conf_dim_timeout) && state == DSME_DISPLAY_DIM)
		expected_display_state = DSME_DISPLAY_OFF;
	else
		expected_display_state = state;
	dsme_display_state_waiting = 2;
	set_timer(&display_state_timer, 6.0); // 6 seconds
	dbus_send_wakeup(_WAKEUP_REFRESH_);
}

double dsme_check_display_state( ){
	if(is_timer_ready(&display_state_timer)){
		if(display_state == expected_display_state){
			// got the expected state, end timer
			dsme_display_state_waiting = 0;
			return 0.0;
		} else if(dsme_display_state_waiting == 2){
			// MCE did not set the expected state yet, blanking is most likely paused
			// send first notification and set a second timer
			debug_write("blanking pause detected...");
			notify_blanking_pause1();
			dsme_display_state_waiting--;
			set_timer(&display_state_timer, 3.0); // 3 seconds
			hw_restore_cp_screen_settings();
		} else if(dsme_display_state_waiting){
			// send second notification and end timer
			notify_blanking_pause2();
			dsme_display_state_waiting--;
			return 0.0;
		}
	}
	return display_state_timer.remaining;
}

////////////////////////////////////////////////////////////////////////// BRIGHTNESS

void dsme_set_brightness( unsigned level ){
	dsme_write_one_arg(DSME_SET_DISPLAY_BRIGHTNESS, (level * 2) + 1);
}

////////////////////////////////////////////////////////////////////////// DEVICE LOCK CODE

void dsme_query_devicelock_code( ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
		char data[16];
		unsigned a1;
		unsigned a2; // this changes, not sure what this is
	} message;
	message.len = sizeof(message);
	message.type = DSME_GET_DEVICELOCK_CODE;
	strncpy(message.data, "lock_code", sizeof(message.data));
	message.a1 = 0x01;
	message.a2 = 0x00027ec8;

	// clear code
	set_string(&device_lock_code, NULL, 0);

	// this should cause dsme to send us back the lock code
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}
