//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>

#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "services.h"
#include "hardware.h"
#include "dialog.h"
#include "dbus.h"
 #include "alarm.h"

////////////////////////////////////////////////////////////////////////// GLOBAL

unsigned activate_alarm, resume_alarm;

//////////////////////////////////////////////////////////////////////////

static char *alarm_cb_service, *alarm_cb_path, *alarm_cb_iface, *alarm_cb_method, *alarm_description, *alarm_sound;//, *alarm_image, *alarm_title;
static unsigned alarm_mode, alarm_tklock;
static time_t alarm_timestamp;
static double alarm_volume, alarm_start_time;

static int alarm_original_volume;
static unsigned alarm_original_map_state;

//////////////////////////////////////////////////////////////////////////

#define DESC_LINE_HEIGHT 32
static void draw_description_line( char *description, unsigned line, unsigned width, unsigned height, unsigned base ){
	if(line >= height){
		draw_text("...", 24, fg1_color, window.width>>1, base+line*DESC_LINE_HEIGHT, ALIGN_CENTER, ALIGN_MIDDLE, 0);
	} else if(strlen(description) > 50){
		char buffer[51];
		snprintf(buffer, width+1, "%s", description);
// TODO: split lines on spaces
		draw_text(buffer, 24, fg1_color, window.width>>1, base+line*DESC_LINE_HEIGHT, ALIGN_CENTER, ALIGN_MIDDLE, 0);
		draw_description_line(description+width, line+1, width, height, base);
	} else {
		draw_text(description, 24, fg1_color, window.width>>1, base+line*DESC_LINE_HEIGHT, ALIGN_CENTER, ALIGN_MIDDLE, 0);
	}
}

typedef enum e_snooze { DISMISS, SNOOZE } e_snooze;
static void dismiss_alarm( e_snooze snooze ){
	int status;
	if(snooze){
		// snooze
		highlight_selected_area(0, window.height-59, (window.width>>1)-1,59, 0);
		status = ALARM_DIALOG_RESPONSE_SNOOZE;
	} else {
		// dismiss
		highlight_selected_area((window.width>>1)+1, window.height-59, (window.width>>1)-1,59, 0);
		status = ALARM_DIALOG_RESPONSE_DISMISS;
	}
	#ifdef ARCH_armel
	dbus_send_method(alarm_cb_service, alarm_cb_path, alarm_cb_iface, alarm_cb_method, _INT32_, &status, _END_);
	conf_set_bool(conf_alarm_open, 0);
	#endif
	dialog_close();
}

void draw_alarm( unsigned force ){
	if(!force){
		if(current_time - alarm_start_time >= cfg_alarm_duration){
			// snooze alarm after one minute
			dismiss_alarm(SNOOZE);
			return;
		}
		// gradually increase volume
		alarm_volume += (double)cfg_alarm_volume_step/1000.0; // milliseconds
		if(alarm_volume > (double)cfg_alarm_volume_max/100.0) alarm_volume = (double)cfg_alarm_volume_max/100.0;
		dbus_audio_set_volume(alarm_volume);
		reset_timer(&dialog_refresh_timer);
		return;
	}

	// clear background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);

	draw_text("Alarm", 64, red_color, 20, 20, ALIGN_LEFT, ALIGN_TOP, 0);

	// timestamp
	struct tm *tm = localtime(&alarm_timestamp);
	char buffer[20];
	strftime(buffer, sizeof(buffer), (cfg_clock_24h ? "%H:%M" : (cfg_clock_ampm ? "%l:%M %P" : "%l:%M")), tm); // time of day
	draw_text((*buffer == ' ' ? &buffer[1] : buffer), 64, red_color, window.width-20, 10, ALIGN_RIGHT, ALIGN_TOP, 0);
	strftime(buffer, sizeof(buffer), "%A", tm); // long day of week
	draw_text(buffer, 36, red_color, window.width-20, 65, ALIGN_RIGHT, ALIGN_TOP, 0);
	strftime(buffer, sizeof(buffer), "%e %B", tm); // day and long month
	draw_text((*buffer == ' ' ? &buffer[1] : buffer), 36, red_color, window.width-20, 105, ALIGN_RIGHT, ALIGN_TOP, 0);

	// description
	unsigned w, h, lines;
	if(window.width > window.height){ w = 50; h = 5; } else { w = 30; h = 9; }
	lines = ceil((float)strlen(alarm_description)/(float)w);
// TODO: split lines on spaces, will need to pre-scan the description, build each line and then calculate total lines for vertical centering
	if(lines > h) lines = h;
	draw_description_line(alarm_description, 0, w, h, 145+((window.height-205)>>1)-((lines*DESC_LINE_HEIGHT)>>1));

	x_set_color(line1_color);
	x_draw_line(0, window.height-60, window.width, window.height-60);
//	if(use_minimal_ui){
//		// power button message
//		draw_text(charging_boot_message, 14, fg2_color, window.width>>1, window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
//	} else {
		// close and snooze buttons
		x_draw_line(window.width>>1, window.height-60, window.width>>1, window.height);
		draw_text("Close", 24, button_color, window.width - (window.width>>2), window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
		if(alarm_mode == ALARM_MODE_NORMAL)
			draw_text("Snooze", 24, button_color, window.width>>2, window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
//	}
}

static void click_alarm( ){
	if(click.type != CLICK_SHORT) return;

	if(click.y > (int)window.height-60){
		if(click.x > (int)window.width>>1){
			// close button
			dismiss_alarm(DISMISS);
		} else {
			// snooze button
			dismiss_alarm(SNOOZE);
		}
	}
	// ignore clicks at top of screen
}

static void keypress_alarm( ){
	if(keypress.type != KEY_SHORT) return;

	if(keypress.keycode == KEYCODE_ESCAPE){
		// close button
		dismiss_alarm(DISMISS);
	} else if(keypress.keycode == KEYCODE_HOME){
		// snooze button
		dismiss_alarm(SNOOZE);
	}
}

static void reset_alarm( ){
	alarm_close(1);
}

//////////////////////////////////////////////////////////////////////////

void alarm_init( ){
	alarm_cb_service = alarm_cb_path = alarm_cb_iface = alarm_cb_method = alarm_description = alarm_sound /* = alarm_image = alarm_title */ = NULL;
	resume_alarm = 0;
}

void alarm_resume( unsigned boot ){
	if(conf_get_bool(conf_alarm_open)){
		// ASUI terminated while an alarm was open and screen could be in an unblankable state

		if(boot)
			sleep(5); // sleep 5 seconds to let system boot
		else
			usleep(500000); // sleep 500ms to let dbus acquire initial values

		// make sure screen is on
		mce_wake_screen();

		unsigned tklock = conf_get_bool(conf_alarm_tklock);
		if(tklock){
			resume_alarm = 1; // alarm_close() needs to relock tk
			hw_unlock_screen();
			hw_unlock_keys();
			if(hw_screen.locked || cfg_tkactive_notifications) notify_screen_unlocked();
			hw_screen.locked = 0;
		}

		alarm_open(tklock,
			conf_get_string(conf_alarm_cb_service),
			conf_get_string(conf_alarm_cb_path),
			conf_get_string(conf_alarm_cb_iface),
			conf_get_string(conf_alarm_cb_method),
			conf_get_string(conf_alarm_description),
			conf_get_string(conf_alarm_sound),
			conf_get_int(conf_alarm_mode),
			conf_get_int(conf_alarm_timestamp)
		);
	}
}

void alarm_open( unsigned tklock, char *cb_service, char *cb_path, char *cb_iface, char *cb_method, char *description, char *sound, unsigned mode, unsigned timestamp ){
	// called by the dbus alarm_open method

	set_string(&alarm_cb_service, cb_service, 0);
	set_string(&alarm_cb_path, cb_path, 0);
	set_string(&alarm_cb_iface, cb_iface, 0);
	set_string(&alarm_cb_method, cb_method, 0);
	set_string(&alarm_description, description, 0);
	set_string(&alarm_sound, sound, 0);
//	set_string(&alarm_image, image, 0);
//	set_string(&alarm_title, title, 0);
	alarm_mode = mode;
	alarm_tklock = tklock;
	alarm_timestamp = timestamp;
	alarm_volume = (double)cfg_alarm_volume_min/100.0;
	alarm_start_time = current_time;

	// main loop will open the alarm dialog
	activate_alarm = 1;

	// map ASUI if not already mapped and wake main loop to open dialog
	alarm_original_map_state = window.is_mapped;
	if(window.is_mapped)
		dbus_send_wakeup(_WAKEUP_REFRESH_);
	else
		dbus_send_wakeup(_WAKEUP_SHOW_); // map ASUI
}

void alarm_close( unsigned reset ){
	// called by the dbus alarm_close method and reset_alarm dialog callback

	// close dialog when dbus:alarm_close method received
	if(!reset && current_dialog_draw == draw_alarm)
		dialog_close();

	// stop sound
	dbus_audio_stop();

	// reset volume
	conf_set_int(conf_volume, alarm_original_volume);
	hw_volume = alarm_original_volume;

	// update screen lock status
	if(alarm_tklock){
		if(resume_alarm){
			hw_lock_screen();
			hw_lock_keys();
//			mce_lock_screen();
			resume_alarm = 0;
		}
		hw_screen.locked = 1;
		notify_screen_locked();
	}

	// reset map state
	if(alarm_original_map_state)
		dbus_send_wakeup(_WAKEUP_REFRESH_);
	else
		dbus_send_wakeup(_WAKEUP_HIDE_); // unmap ASUI
}

void alarm_open_dialog( ){
	// called by main loop to open the dialog

	dialog_reset();
	dialog_open(draw_alarm, click_alarm, keypress_alarm, reset_alarm, DLG_REDRAW_ON);
	set_timer(&dialog_refresh_timer, 10.0);

	// start audio services
	if(current_time - last_svc_refresh > 15.0) svc_refresh();
	svc_init_audio(SVC_START);

	// set 100% volume
	alarm_original_volume = conf_get_int(conf_volume);
	conf_set_int(conf_volume, 100);
	hw_volume = 100;

	// play sound
	dbus_audio_set_volume(alarm_volume);
	dbus_audio_play(alarm_sound, 1);
// TODO: not sure what the priority range is or what is the highest value but this pauses media player while sounding the alarm
	dbus_audio_set_loop(1);

	activate_alarm = 0;
}
