//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#define ASUI_VERSION "0.5.8"

/*
/etc/init.d/hildon-desktop stop
su - user -c "maemo-summoner /usr/bin/hildon-desktop.launch"
*/

#include <string.h>

#include <hildon/hildon-defines.h>
#include <libhildondesktop/libhildondesktop.h>
#include <libhildondesktop/statusbar-item.h>

#include <log-functions.h>
#include <libosso.h>
#include <osso-log.h>

#include <dbus/dbus.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>

#include <gconf/gconf-client.h>
#define conf_get_bool(key) gconf_client_get_bool(p->gconf_client, key, (GError **)NULL)
#define conf_get_int(key) gconf_client_get_int(p->gconf_client, key, (GError **)NULL)
#define KEYNAME_PATH "/apps/asui"
#define USE_BATTERY_KEYNAME "/enable_battery_applet"
static const char *conf_enable_battery_applet = KEYNAME_PATH USE_BATTERY_KEYNAME;

#define _STRING_ DBUS_TYPE_STRING
#define _BOOL_ DBUS_TYPE_BOOLEAN
#define _UINT32_ DBUS_TYPE_UINT32
#define _INT32_ DBUS_TYPE_INT32
#define _DOUBLE_ DBUS_TYPE_DOUBLE
#define _END_ DBUS_TYPE_INVALID

//////////////////////////////////////////////////////////////////////////

G_BEGIN_DECLS

typedef struct _AsuiBatteryPlugin AsuiBatteryPlugin;
typedef struct _AsuiBatteryPluginClass AsuiBatteryPluginClass;

#define ASUI_BATTERY_PLUGIN_PRIORITY 1

#define ASUI_BATTERY_PLUGIN_ICON_SIZE 40

#define ASUI_BATTERY_PLUGIN_TYPE			(asui_battery_plugin_get_type())
#define ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPlugin))
#define ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST((klass),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))
#define IS_ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_TYPE((obj),	ASUI_BATTERY_PLUGIN_TYPE))
#define IS_ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass),	ASUI_BATTERY_PLUGIN_TYPE))
#define ASUI_BATTERY_PLUGIN_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))

struct _AsuiBatteryPlugin { StatusbarItem parent; };
struct _AsuiBatteryPluginClass { StatusbarItemClass parent_class; };
GType asui_battery_plugin_get_type(void);

typedef struct {
	osso_context_t *osso; // osso
	DBusGConnection *connection; // system bus
	unsigned use_battery; // show battery status
	GConfClient *gconf_client; // gconf
	GtkWidget *icon; // icon in button
	GtkWidget *button; // button in statusbar
} AsuiBatteryPluginPrivate;

G_END_DECLS

HD_DEFINE_PLUGIN(AsuiBatteryPlugin, asui_battery_plugin, STATUSBAR_TYPE_ITEM);

#define ASUI_BATTERY_PLUGIN_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE((object), ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginPrivate))

//////////////////////////////////////////////////////////////////////////

static void asui_battery_plugin_finalize( GObject *object );

static void asui_battery_plugin_class_init( AsuiBatteryPluginClass *klass ){
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = asui_battery_plugin_finalize;
	g_type_class_add_private(klass, sizeof(AsuiBatteryPluginPrivate));
}

//////////////////////////////////////////////////////////////////////////

static const char *charging_notification = "Charging...";
static const char *battery_full_notification = "Battery Full";
static const char *battery_low_notification = "Low Battery";

static const char *charging_sound = "/usr/share/sounds/ui-charging_started.wav";
static const char *battery_low_sound = "/usr/share/sounds/ui-battery_low.wav";

static void asui_battery_dbus_send_session_method( DBusConnection *bus, const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	va_list ap;
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(bus, message, NULL) != FALSE)
		dbus_connection_flush(bus);
	dbus_message_unref(message);
}

static void asui_battery_display_notification( AsuiBatteryPluginPrivate *p, const char *msg, const char *sound ){
	// initialize dbus connection
	DBusGConnection *connection;
	GError *error = NULL;
	if((connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error)) == NULL){
		ULOG_WARN("%s: can't connect to session method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return;
	}
	DBusConnection *session_bus = dbus_g_connection_get_connection(connection);

	// play sound
	if(sound){
		unsigned system_alert_volume = conf_get_int("/apps/osso/sound/system_alert_volume");
		if(system_alert_volume){
			const char *mm_service = "com.nokia.osso_media_server";
			const char *mm_path = "/com/nokia/osso_media_server";
			const char *mm_interface = "com.nokia.osso_media_server.sound";
			double volume = 0.5 * system_alert_volume;
			unsigned loop = 0;
			int priority = 1;

			asui_battery_dbus_send_session_method(session_bus, mm_service, mm_path, mm_interface, "set_volume", _DOUBLE_, &volume, _END_);
			asui_battery_dbus_send_session_method(session_bus, mm_service, mm_path, mm_interface, "set_loop", _BOOL_, &loop, _END_);
			asui_battery_dbus_send_session_method(session_bus, mm_service, mm_path, mm_interface, "play_sound", _STRING_, &sound, _INT32_, &priority, _END_);
		}
	}

	// display notification
	asui_battery_dbus_send_session_method(session_bus, "org.freedesktop.Notifications", "/org/freedesktop/Notifications", "org.freedesktop.Notifications",
											"SystemNoteInfoprint", _STRING_, &msg, _END_);

	// close dbus connection
	dbus_g_connection_unref(connection);
}

//////////////////////////////////////////////////////////////////////////

static const gchar *asui_battery_icons[] = {
	"asui_power", "asui_power_pressed",
	"asui_battery100", "asui_battery88", "asui_battery76", "asui_battery64", "asui_battery52",
						"asui_battery40", "asui_battery28", "asui_battery16", "asui_battery4",
	"asui_charging100", "asui_charging66", "asui_charging33",
	"asui_charged"
};
typedef enum e_icon_type {
	ICON_POWER, ICON_POWER_PRESSED,
	ICON_BATTERY_100, ICON_BATTERY_88, ICON_BATTERY_76, ICON_BATTERY_64, ICON_BATTERY_52,
						ICON_BATTERY_40, ICON_BATTERY_28, ICON_BATTERY_16, ICON_BATTERY_4,
	ICON_CHARGING_100, ICON_CHARGING_66, ICON_CHARGING_33,
	ICON_CHARGED
} e_icon_type;
static e_icon_type asui_battery_current_icon_type;

static void asui_battery_set_icon( AsuiBatteryPluginPrivate *p, e_icon_type new_icon_type ) {
	GtkIconTheme *icon_theme;
	GdkPixbuf*pixbuf;

	if(asui_battery_current_icon_type == new_icon_type) return; // don't load the same icon more than once

	const gchar *name = asui_battery_icons[new_icon_type];
	icon_theme = gtk_icon_theme_get_default();
	pixbuf = (name != NULL ? gtk_icon_theme_load_icon(icon_theme, name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
	gtk_image_set_from_pixbuf(GTK_IMAGE(p->icon), pixbuf);

	if(pixbuf != NULL) g_object_unref(pixbuf);
	asui_battery_current_icon_type = new_icon_type;
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_dbus_send_method( AsuiBatteryPluginPrivate *p, const char *destination, const char *path, const char *interface, const char *method ){
	DBusConnection *system_bus = dbus_g_connection_get_connection(p->connection);
	DBusMessage *message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

static void asui_battery_update_icon( AsuiBatteryPluginPrivate *p ); // prototype

static gboolean asui_battery_reset_press( gpointer data ){
	asui_battery_update_icon((AsuiBatteryPluginPrivate *)data);
	return FALSE;
}

static gboolean asui_battery_open_asui( GtkButton *widget, GdkEventButton *button, gpointer data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)data;

	g_return_val_if_fail(data, FALSE);
	gtk_button_released(GTK_BUTTON(p->button));
	(void) button;

	asui_battery_dbus_send_method(p, "com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "show");

	asui_battery_set_icon(p, ICON_POWER_PRESSED);
	g_timeout_add(1000, asui_battery_reset_press, (void *)p);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////

typedef enum e_battery_status { BATTERY_DRAINING, BATTERY_CHARGING, BATTERY_AC } e_battery_status;
static e_battery_status asui_battery_status;
static unsigned asui_battery_capacity, asui_battery_initializing, asui_battery_init_timer, asui_battery_charging_cycle, asui_battery_charging_timer;
static unsigned asui_battery_gconf_notify;

static gboolean asui_battery_cycle_charging_icon( gpointer data ){
	if(asui_battery_status == BATTERY_CHARGING){
		switch(asui_battery_charging_cycle){
		case 0: asui_battery_set_icon((AsuiBatteryPluginPrivate *)data, ICON_CHARGING_33); asui_battery_charging_cycle = 1; break;
		case 1: asui_battery_set_icon((AsuiBatteryPluginPrivate *)data, ICON_CHARGING_66); asui_battery_charging_cycle = 2; break;
		case 2: asui_battery_set_icon((AsuiBatteryPluginPrivate *)data, ICON_CHARGING_100); asui_battery_charging_cycle = 0; break;
		}
		return TRUE;
	}
	asui_battery_charging_timer = 0;
	return FALSE;
}

static void asui_battery_update_icon( AsuiBatteryPluginPrivate *p ){
	if(asui_battery_initializing || !p->use_battery)
		asui_battery_set_icon(p, ICON_POWER);
	else switch(asui_battery_status){
	case BATTERY_DRAINING:
		     if(asui_battery_capacity > 88) asui_battery_set_icon(p, ICON_BATTERY_100);
		else if(asui_battery_capacity > 76) asui_battery_set_icon(p, ICON_BATTERY_88);
		else if(asui_battery_capacity > 64) asui_battery_set_icon(p, ICON_BATTERY_76);
		else if(asui_battery_capacity > 52) asui_battery_set_icon(p, ICON_BATTERY_64);
		else if(asui_battery_capacity > 40) asui_battery_set_icon(p, ICON_BATTERY_52);
		else if(asui_battery_capacity > 28) asui_battery_set_icon(p, ICON_BATTERY_40);
		else if(asui_battery_capacity > 16) asui_battery_set_icon(p, ICON_BATTERY_28);
		else if(asui_battery_capacity > 4) asui_battery_set_icon(p, ICON_BATTERY_16);
		else asui_battery_set_icon(p, ICON_BATTERY_4);
		break;
	case BATTERY_CHARGING:
		asui_battery_charging_cycle = 0;
		asui_battery_set_icon(p, ICON_CHARGING_100);
		if(!asui_battery_charging_timer)
			asui_battery_charging_timer = g_timeout_add(1000, asui_battery_cycle_charging_icon, (void *)p);
		break;
	case BATTERY_AC:
		asui_battery_set_icon(p, ICON_CHARGED);
		break;
	}
}

static DBusHandlerResult asui_battery_signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	DBusError dbus_error;
	if(dbus_message_is_signal(message, "com.qzx.asui", "charging_status")){
		unsigned status;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &status, _END_)){
			asui_battery_initializing &= 1;
			if(asui_battery_status != status){
				asui_battery_status = status;
				if(!asui_battery_initializing) asui_battery_update_icon((AsuiBatteryPluginPrivate *)user_data);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.qzx.asui", "battery_capacity")){
		unsigned capacity;
		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &capacity, _END_)){
			asui_battery_initializing &= 2;
			if(asui_battery_capacity != capacity){
				asui_battery_capacity = capacity;
				if(!asui_battery_initializing) asui_battery_update_icon((AsuiBatteryPluginPrivate *)user_data);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_low")){
		asui_battery_display_notification((AsuiBatteryPluginPrivate *)user_data, battery_low_notification, battery_low_sound);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_full")){
		asui_battery_display_notification((AsuiBatteryPluginPrivate *)user_data, battery_full_notification, NULL);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_connected")){
		asui_battery_display_notification((AsuiBatteryPluginPrivate *)user_data, charging_notification, charging_sound);
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean asui_battery_req_battery_status( gpointer data ){
	if(asui_battery_initializing){
		// send request every 2 seconds until initialized
		asui_battery_dbus_send_method((AsuiBatteryPluginPrivate *)data, "com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "req_battery_status");
		return TRUE;
	}
	// if charging status is received after capacity signal then icon won't be updated when draining
	// this forces an update to fix the problem
	asui_battery_update_icon((AsuiBatteryPluginPrivate *)data);

	asui_battery_init_timer = 0;
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_disable_battery( AsuiBatteryPluginPrivate *p ){
	// remove timers
	if(asui_battery_init_timer) g_source_remove(asui_battery_init_timer);
	if(asui_battery_charging_timer) g_source_remove(asui_battery_charging_timer);

	// remove dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_remove_filter(signal_bus, asui_battery_signal_filter, (void *)p);
}

static void asui_battery_enable_battery( AsuiBatteryPluginPrivate *p ){
	// add dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_add_filter(signal_bus, asui_battery_signal_filter, (void *)p, NULL);

	// initialize
	asui_battery_status = BATTERY_DRAINING;
	asui_battery_capacity = 0;
	asui_battery_initializing = 3;
	asui_battery_charging_timer = 0;
	asui_battery_init_timer = g_timeout_add(2000, asui_battery_req_battery_status, (void *)p);
	// 2 second wait before sending out first request
}

static void asui_battery_key_change_callback( GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer userData ){
	const gchar *keyname = gconf_entry_get_key(entry);
	if(keyname == NULL) return;
	if(strcmp(keyname, conf_enable_battery_applet)) return;

	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)userData;
	unsigned use_battery = conf_get_bool(conf_enable_battery_applet);
	if(use_battery != p->use_battery){
		p->use_battery = use_battery;
		if(use_battery){
			asui_battery_enable_battery(p);
		} else {
			asui_battery_disable_battery(p);
			asui_battery_update_icon(p);
		}
	}
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_plugin_finalize( GObject *object ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(object);
	GError *error = NULL;

	gconf_client_remove_dir(p->gconf_client, KEYNAME_PATH, &error);
	if(asui_battery_gconf_notify) gconf_client_notify_remove(p->gconf_client, asui_battery_gconf_notify);

	if(p->use_battery) asui_battery_disable_battery(p);

	dbus_g_connection_unref(p->connection);

	// remove osso context
	osso_deinitialize(p->osso);

	LOG_CLOSE();

	G_OBJECT_CLASS(g_type_class_peek_parent(G_OBJECT_GET_CLASS(object)))->finalize(object);
}

static void asui_battery_plugin_init( AsuiBatteryPlugin *plugin ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(plugin);

	ULOG_OPEN("asui-statusbar-battery");

	g_return_if_fail(p);

	// initialize dbus
	GError *error = NULL;
	if((p->connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error)) == NULL){
		ULOG_WARN("%s: can't connect to system method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return;
	}

	p->icon = gtk_image_new_from_pixbuf(NULL);
	asui_battery_current_icon_type = ICON_POWER_PRESSED;
	asui_battery_set_icon(p, ICON_POWER);
	p->button = gtk_toggle_button_new();
	gtk_container_add(GTK_CONTAINER(p->button), GTK_WIDGET(p->icon));
	gtk_container_add(GTK_CONTAINER(plugin), p->button);
	g_signal_connect(G_OBJECT(p->button), "button-press-event", G_CALLBACK(asui_battery_open_asui), p);

	// osso context
	p->osso = osso_initialize("com.qzx.asui_statusbar_battery", ASUI_VERSION, TRUE, NULL);
	if(!p->osso) ULOG_WARN("%s: error while initializing osso\n", __FUNCTION__);

	gtk_widget_show_all(GTK_WIDGET(plugin));

	// initialize gconf
	p->gconf_client = gconf_client_get_default();
	if(p->gconf_client){
		p->use_battery = conf_get_bool(conf_enable_battery_applet);
		gconf_client_add_dir(p->gconf_client, KEYNAME_PATH, GCONF_CLIENT_PRELOAD_NONE, &error);
		asui_battery_gconf_notify = gconf_client_notify_add(p->gconf_client, KEYNAME_PATH, asui_battery_key_change_callback, (void *)p, NULL, &error);
		if(p->use_battery) asui_battery_enable_battery(p);
	} else asui_battery_gconf_notify = 0;
}
