//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

// seconds (double)
#define REFRESH_PERIOD 60.0

#define CAPACITY_COLOR green_status_color
#define CHARGE_COLOR fg2_color
#define SCREEN_COLOR red_status_color
#define WIFI_COLOR yellow_status_color
#define BLUETOOTH_COLOR selection_color
#define HOURS_COLOR fg1_color

#include <stdio.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>

#include "window.h"
#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
 #include "battery_graph.h"

////////////////////////////////////////////////////////////////////////// LOCALS

static e_orientation original_orientation;
static double base_time;
static float units;
static unsigned event_size;

//////////////////////////////////////////////////////////////////////////

static void draw_event( s_event_history *head, s_event_history *tail, XColor *color, int y ){
	if(head == NULL || tail == NULL) return;
	unsigned nr_events = 0;
	int x1 = 0, x2;

	x_set_color(color);
	x_set_style(7, LineSolid); // 7 wide

	s_event_history *history;
	for(history = head; history != NULL; history = history->next, nr_events++){
		if(history->state){
			// on event -- update start point
			x1 = roundf((history->timestamp - base_time) / units);
		} else {
			// off event -- draw line from last on event
			x2 = roundf((history->timestamp - base_time) / units) + 1;
			x_draw_line(x1,y, x2,y);
		}
	}
	if(tail->state) x_draw_line(x1,y, window.width-1,y);

	x_reset_style();

	event_size += nr_events * sizeof(s_event_history);
}

static void draw_battery_graph( unsigned force ){
	// force is not needed

	char buffer[30];

	// dialog background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);
	// top and bottom graph lines
	x_set_color(line_color);
	x_draw_line(0,18, window.width,18);
	x_draw_line(0,window.height-18, window.width,window.height-18);

	int graph_bottom = window.height - 20;
	int graph_height = window.height - 40;

	event_size = 0;
	if(hw_battery.history_head){
		base_time = hw_battery.history_head->timestamp;
		units = (current_time - base_time) / window.width;
	} else {
		base_time = 0;
		units = 0;
	}
	if(hw_battery.history_head != hw_battery.history_tail && hw_battery.history_head != NULL){
		float prev_capacity = hw_battery.history_head->capacity;
		double prev_timestamp = hw_battery.history_head->timestamp;
		int xc = 0, x1 = 0;
		int y1 = graph_bottom - graph_height * (prev_capacity / 100.0);
		unsigned nr_events = 0;

		// horizontal dotted lines every 10%
		x_set_color(innerline_color);
		x_set_style(1, LineOnOffDash); // dotted line
		int n;
		for(n = 1; n < 10; n++){
			if(n == 5) continue;
			int y = graph_bottom - graph_height * ((float)n / 10.0);
			x_draw_line(0,y, window.width,y);
		}

		x_set_color(CAPACITY_COLOR);
		x_set_style(2, LineSolid); // 2 wide
		s_battery_history *history;
		for(history = hw_battery.history_head->next; history != NULL; history = history->next, nr_events++){
			int x2 = roundf((history->timestamp - base_time) / units);
			int y2 = graph_bottom - graph_height * (history->capacity / 100.0);
			x_draw_line(x1, y1, x2, y2);

			if(history == hw_battery.charge_point) xc = x2;

			x1 = x2; y1 = y2;
			prev_capacity = history->capacity;
			prev_timestamp = history->timestamp;
		}
		event_size += nr_events * sizeof(s_battery_history);
		x_reset_style();

		// last charging point
		if(hw_battery.charge_point){
			x_set_color(CAPACITY_COLOR);
			x_set_style(1, LineOnOffDash); // dotted line
			x_draw_line(xc,20+2, xc,graph_bottom);
			x_reset_style();
		}

		// battery labels
		draw_text("100%", 16, CAPACITY_COLOR, 0,0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("0% battery", 16, CAPACITY_COLOR, 0,window.height-3, ALIGN_LEFT, ALIGN_BASELINE, 0);

		// number of hours shown in graph
		float hours = (current_time - base_time) / 3600.0;
		if(hours >= 24.0){
			unsigned days = floorf(hours/24.0);
			snprintf(buffer, sizeof(buffer), "%u days   %.2f hours", days, hours - days*24);
		} else
			snprintf(buffer, sizeof(buffer), "%.1f hours", hours);
		draw_text(buffer, 16, HOURS_COLOR, window.width>>1,0, ALIGN_CENTER, ALIGN_TOP, 0);
	} else {
		// no battery data
		draw_text("waiting...", 18, CAPACITY_COLOR, window.width>>1,0, ALIGN_CENTER, ALIGN_TOP, 0);
	}

	int cy = (window.height>>1);

	if(cfg_track_usage_events && base_time){
		// charge events
		draw_event(hw_battery.charger_history_head, hw_battery.charger_history_tail, CHARGE_COLOR, cy-31);
		// screen events
		draw_event(hw_screen.history_head, hw_screen.history_tail, SCREEN_COLOR, cy-19);
		// wifi events
		draw_event(hw_wifi.history_head, hw_wifi.history_tail, WIFI_COLOR, cy+19);
		// bluetooth events
		draw_event(hw_bluetooth.history_head, hw_bluetooth.history_tail, BLUETOOTH_COLOR, cy+31);

// TODO: gps events

		// charge, screen, wifi and bluetooth event labels
		draw_text("charger", 16, CHARGE_COLOR, window.width/4,window.height-3, ALIGN_CENTER, ALIGN_BASELINE, 0);
		draw_text("screen", 16, SCREEN_COLOR, window.width>>1,window.height-3, ALIGN_CENTER, ALIGN_BASELINE, 0);
		draw_text("wifi", 16, WIFI_COLOR, window.width*3/4,window.height-3, ALIGN_CENTER, ALIGN_BASELINE, 0);
		draw_text("bluetooth", 16, BLUETOOTH_COLOR, window.width-1,window.height-3, ALIGN_RIGHT, ALIGN_BASELINE, 0);
	}

	// time markers
	if(current_time - base_time > 3600.0){
		unsigned h, hours = floorf((current_time - base_time) / 3600.0);
		float pixels = 3600.0 * (float)window.width / (float)(current_time - base_time);
		int d;
		x_set_color(HOURS_COLOR);
		for(h = 0, d = 0; h < hours; h++, d++){
			int x = window.width - (h+1)*pixels;
			if(d == 23){
				d = -1;
				x_draw_line(x,cy-8, x,cy+8);
			} else
				x_draw_line(x,cy-3, x,cy+3);
		}
	}
	x_set_color(line_color); x_draw_line(0,cy, window.width,cy);

	// memory used by events
	snprintf(buffer, sizeof(buffer), "%ukB", (unsigned)roundf((float)event_size/1024.0));
	draw_text(buffer, 16, line_color, window.width-1,0, ALIGN_RIGHT, ALIGN_TOP, 0);

	hw_changes = 0;
	reset_timer(&dialog_refresh_timer);
}

static void click_battery_graph( ){
	if(click.type == CLICK_DRAG && click.offset_y > 160){
		// close dialog
		highlight_selected_area(0, 0, window.width, window.height); // hack
		dialog_close();
	}
}

static void keypress_battery_graph( ){
	if(keypress.keycode == KEYCODE_ESCAPE && keypress.type == KEY_SHORT){
		// close dialog
		highlight_selected_area(0, 0, window.width, window.height); // hack
		dialog_close();
	}
}

static void reset_battery_graph( ){
	if(original_orientation != O_LANDSCAPE)
		rotate_screen(original_orientation, 1);
}

//////////////////////////////////////////////////////////////////////////

void battery_graph_open_dialog( ){
	original_orientation = window.orientation;
	if(window.orientation != O_LANDSCAPE && window.orientation != O_LANDSCAPE_INV)
		rotate_screen(O_LANDSCAPE, 1);
	dialog_open(draw_battery_graph, click_battery_graph, keypress_battery_graph, reset_battery_graph, DLG_REDRAW_ON);
	set_timer(&dialog_refresh_timer, REFRESH_PERIOD);
}
