//  Advanced System UI
//  Copyright (c) 2010-2011 Brand Huntsman <brand.huntsman@gmail.com>

// http://maemo.org/api_refs/4.1/dbus-1.0.2/
// http://dbus.freedesktop.org/doc/dbus/api/html/index.html
// http://dbus.freedesktop.org/doc/dbus-tutorial.html

#ifdef ARCH_armel
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>
#include <fcntl.h>
#include <pthread.h>

#include "window.h"
#include "main.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
 #include "dbus.h"
#include "dsme.h"
#include "alarm.h"

////////////////////////////////////////////////////////////////////////// GLOBALS

DBusError dbus_error;

unsigned dbus_listening;

int dbus_wakeup_fd;

unsigned original_screen_autolock, original_screen_stayslit;

////////////////////////////////////////////////////////////////////////// LOCAL GLOBALS

static pthread_t thread;

#ifdef ARCH_armel
static DBusConnection *signal_bus, *system_bus, *session_bus;

static const char *_notification_service_ = "org.freedesktop.Notifications";
static const char *_notification_path_ = "/org/freedesktop/Notifications";
static const char *_notification_interface_ = "org.freedesktop.Notifications";
static const char *_notification_method_ = "SystemNoteInfoprint"; // method

static const char *_osso_multimedia_service_ = "com.nokia.osso_media_server";
static const char *_osso_multimedia_path_ = "/com/nokia/osso_media_server";
static const char *_osso_multimedia_interface_ = "com.nokia.osso_media_server.sound";

static unsigned acting_dead_state;
//static unsigned powerkeymenu_open;
#endif

//////////////////////////////////////////////////////////////////////////

#ifdef ARCH_armel
/*
static void check_dimmed_state( ){
	if(hw_screen.dimmed && hw_battery.status != BATTERY_DRAINING && conf_get_screen_stayslit_value()){
		// unset dimmed state if on charger and screen stays lit
		hw_screen.brightness_mismatch = 1;
		hw_screen.dimmed = 0;
		hw_changes |= HW_BRIGHTNESS;
		dbus_send_wakeup(_WAKEUP_REFRESH_);
	}
}
*/

static void set_battery_status( e_battery_status status ){
	unsigned charging = (status == BATTERY_CHARGING ? 1 : 0);
	if(hw_battery.initialize){ hw_battery.initialize = 0; hw_battery.status = status; }
	else if((!charging && hw_battery.status == BATTERY_CHARGING) || (charging && hw_battery.status != BATTERY_CHARGING)){
		hw_battery.status = (charging ? BATTERY_CHARGING : BATTERY_AC);
		dbus_send_charging_status_signal();
		if(is_value_visible(battery_widget_page)){ hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		if(!charging) hw_query_battery(1); // flush
	}

//	check_dimmed_state();
}

static void fix_screen_and_keys_and_blank_window( ){
	// lock&blank doesn't properly lock the screen and connecting/disconnecting
	// the signal turns on and unlocks the screen without sending any tklock signals

	// this signal will turn on the screen, if off
	dsme_display_state_waiting = 0; // reset blanking pause check

	if(hw_screen.locked){
		if(cfg_power_unlock_key){
			// secondary key required to unlock screen, keypad remains unlocked
			hw_lock_screen();
			hw_unlock_keys();
			if(!window.is_mapped) window_open_blank(); // open blank to catch keys
			notify_power_unlock_key();
		} else {
			window_close_blank(); // signal turned screen on, close the blank window
			mce_lock_screen(0); // re-lock the screen, this won't send a tklock signal
			if(hw_screen.visible && current_time - hw_screen.visible_time < 1.0)
				notify_screen_locked(); // screen just turned on, show tklock state notification
		}
	} else {
		window_close_blank(); // signal turned screen on, close the blank window
		if(cfg_tkactive_notifications && hw_screen.visible && current_time - hw_screen.visible_time < 1.0)
			notify_screen_unlocked(); // screen just turned on, show tklock state notification
	}
}

static DBusHandlerResult signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	// needs to be updated here for debug_write() and hw_query_battery()
	current_time = get_utime();

	dbus_error_init(&dbus_error);

	if(dbus_message_is_method_call(message, "com.qzx.asui", "flush_calls")){

		debug_write("[DBUS] method: flush calls");

		// signal bus is now awake so any method calls sent over it can be flushed here
		dbus_connection_flush(signal_bus);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "reload_config")){

		debug_write("[DBUS] method: reload config");

		config_reload();

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "show")){

		debug_write("[DBUS] method: show");

		if(!window.is_mapped) dbus_send_wakeup(_WAKEUP_SHOW_);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "hide")){

		debug_write("[DBUS] method: hide");

		if(window.is_mapped) dbus_send_wakeup(_WAKEUP_HIDE_);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "rotate_left")){

		debug_write("[DBUS] method: rotate_left");

		dbus_send_wakeup(_WAKEUP_ROTATE_LEFT_);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "rotate_right")){

		debug_write("[DBUS] method: rotate_right");

		dbus_send_wakeup(_WAKEUP_ROTATE_RIGHT_);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "req_battery_status")){

		debug_write("[DBUS] method: req_battery_status");

		if(!(hw_initialized & HWIS_BME_CAPACITY)){
			dbus_send_charging_status_signal();
			dbus_send_battery_capacity_signal();
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// SYSTEM UI
//	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "tklock_open")){

		// signature = ssssubb

//	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "tklock_close")){

		// signature = b

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "devlock_open")){

		debug_write("[DBUS] method: com.nokia.system_ui.request.devlock_open");

		int response = 1;
		dbus_send_reply(signal_bus, message, _INT32_, &response, _END_);
		// reply to this so MCE will change state to locked and send out signal

		if(hw_screen.locked){
			// this signal unlocks screen and keys but MCE still thinks they are locked, relock them
			hw_lock_screen();
			hw_lock_keys();
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "devlock_close")){

		debug_write("[DBUS] method: com.nokia.system_ui.request.devlock_close");

		// don't need to do anything

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "alarm_open")){
		char *cb_service, *cb_path, *cb_iface, *cb_method, *description, *sound, *image, *title;
		unsigned mode, timestamp;

		debug_write("[DBUS] method: com.nokia.system_ui.request.alarm_open");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &cb_service, _STRING_, &cb_path, _STRING_, &cb_iface, _STRING_, &cb_method,
		   _UINT32_, &mode, _STRING_, &description, _STRING_, &sound, _STRING_, &image, _STRING_, &title, _UINT32_, &timestamp, _END_))
		{
			// save to gconf in case ASUI crashes
			conf_set_string(conf_alarm_cb_service, cb_service);
			conf_set_string(conf_alarm_cb_path, cb_path);
			conf_set_string(conf_alarm_cb_iface, cb_iface);
			conf_set_string(conf_alarm_cb_method, cb_method);
			conf_set_string(conf_alarm_description, description);
			conf_set_string(conf_alarm_sound, sound);
			conf_set_int(conf_alarm_mode, mode);
			conf_set_int(conf_alarm_timestamp, timestamp);
			conf_set_bool(conf_alarm_open, 1);
			conf_set_bool(conf_alarm_tklock, hw_screen.locked);

			// reply to the method
			int response = GTK_RESPONSE_ACCEPT;
			dbus_send_reply(signal_bus, message, _INT32_, &response, _END_); // or GTK_RESPONSE_REJECT

			if(use_minimal_ui){
				// just dismiss the alarm to keep it from breaking
				int status = ALARM_DIALOG_RESPONSE_DISMISS;
				dbus_send_method(cb_service, cb_path, cb_iface, cb_method, _INT32_, &status, _END_);
				conf_set_bool(conf_alarm_open, 0);
				mce_unlock_screen();
				// dismissing the alarm will cause the device to boot

// TODO: this needs a proper fix, can't just ignore silently dismiss alarms that go off in charging runlevel

			} else {
				// screen will be turned on if it was off
				dsme_display_state_waiting = 0; // reset blanking pause check

				// alarmd unlocks screen and keys, unable to relock them with MCE or manually
				unsigned tklock = hw_screen.locked;
				if(tklock){
					hw_screen.locked = 0;
					notify_screen_unlocked();
				} else if(cfg_tkactive_notifications && hw_screen.visible && current_time - hw_screen.visible_time < 1.0)
					notify_screen_unlocked(); // screen just turned on, show tklock state notification

				alarm_open(tklock, cb_service, cb_path, cb_iface, cb_method, description, sound, mode, timestamp);
			}

			return DBUS_HANDLER_RESULT_HANDLED;
		}

		// ignore unmatched signatures

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "alarm_close")){
		// not sure what invokes this method

		debug_write("[DBUS] method: com.nokia.system_ui.request.alarm_close");

		if(!use_minimal_ui){
			conf_set_bool(conf_alarm_open, 0);
			alarm_close(0);
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "acting_dead_open")){
		// not sure what invokes this method or what it expects

		debug_write("[DBUS] method: com.nokia.system_ui.request.acting_dead_open");

// TODO: inputs? reply value?
acting_dead_state = 1;

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "acting_dead_close")){
		// not sure what invokes this method or what it expects

		debug_write("[DBUS] method: com.nokia.system_ui.request.acting_dead_close");

// TODO:
acting_dead_state = 0;

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "acring_dead_getstate")){

		debug_write("[DBUS] method: com.nokia.system_ui.request.acring_dead_getstate");

// TODO: this assumes acring_dead_getstate is a typo for acting_dead_state
		dbus_send_reply(signal_bus, message, _BOOL_, &acting_dead_state, _END_);

		return DBUS_HANDLER_RESULT_HANDLED;

/*
	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "powerkeymenu_open")){
		char *cb_service, *cb_path, *cb_iface, *cb_method;
		unsigned unknown; // always zero

		debug_write("[DBUS] method: com.nokia.system_ui.request.powerkeymenu_open");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &cb_service, _STRING_, &cb_path, _STRING_, &cb_iface, _STRING_, &cb_method, _UINT32_, &unknown, _END_)){
			// SystemUI doesn't invoke the callback until it closes the menu
			// ASUI will tell MCE that the menu is immediately closed that way MCE will always call this method when power button is pressed
			int response = GTK_RESPONSE_CANCEL;
			dbus_send_method(cb_service, cb_path, cb_iface, cb_method, _INT32_, &response, _END_);
			powerkeymenu_open = 1;

			// this signal will turn on the screen, if off
			dsme_display_state_waiting = 0; // reset blanking pause check

			// this signal is not sent when:
			//		devlock
			//		alarm
			//		screen locked

			// it would complicate the code to have it split between the buttonpressed:power and powerkeymenu_open signals

			return DBUS_HANDLER_RESULT_HANDLED;
		}

		// ignore unmatched signatures

	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "powerkeymenu_close")){
		char *cb_service, *cb_path, *cb_iface, *cb_method;
		unsigned unknown; // always zero

		debug_write("[DBUS] method: com.nokia.system_ui.request.powerkeymenu_close");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &cb_service, _STRING_, &cb_path, _STRING_, &cb_iface, _STRING_, &cb_method, _UINT32_, &unknown, _END_)){
			if(powerkeymenu_open){
				// SystemUI invokes the callback after receiving the first powerkeymenu_close but not the second
				int response = GTK_RESPONSE_DELETE_EVENT;
				dbus_send_method(cb_service, cb_path, cb_iface, cb_method, _INT32_, &response, _END_);
				powerkeymenu_open = 0;
			}

			return DBUS_HANDLER_RESULT_HANDLED;
		}

		// ignore unmatched signatures
*/

	////////////////////
	// HAL
	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "Condition")){
		char *s1, *s2;

		debug_write("[DBUS] signal: org.freedesktop.Hal.Device.Condition");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _STRING_, &s2, _END_)){
			if(!strcmp(s1, "ButtonPressed")){
				if(!strcmp(s2, "power")){
					// path = /org/freedesktop/Hal/devices/computer_logicaldev_input
					// path = /org/freedesktop/Hal/devices/computer_logicaldev_input_0
					debug_write("[DBUS]         power button pressed (blank=%u)", blank_window_open);

					// this signal will turn on the screen, if off
					dsme_display_state_waiting = 0; // reset blanking pause check

					if(use_minimal_ui || current_dialog_draw == draw_alarm){
						// charging runlevel or alarm active, power button does nothing

					} else if(hw_screen.visible && current_time - hw_screen.visible_time > 1.0){
						// screen has been on for more than one second
						// CASE: power button used to blank screen (when locked)
						// CASE: power button used to lock, lock&blank or blank screen (when devlocked)
						// CASE: power button used to map/unmap ASUI

						// main loop doesn't need to be awakened, unless showing or hiding ASUI

debug_write("PB: screen has been on for more than one second, tklock=%u, asui=%u", hw_screen.locked, window.is_mapped);
						if(hw_screen.locked){
							hw_lock_keys(); // MCE unlocked the keys for SystemUI's secondary unlock key, relock them

							// screen is locked, power button will blank it
							int i;
							for(i = 0; i < 20; i++){
								if(hw_screen.dimmed) break;
								usleep(100000); // sleep 20 x 100ms until screen dims
							}
							mce_lock_and_blank_screen();
						} else if(cfg_power_action == POWER_ACTION_LOCKBLANK){
							// lock&blank screen instead of mapping or unmapping ASUI
							mce_lock_and_blank_screen();
						} else if(window.is_mapped){
							if(hw_device_locked){
								// device is locked, power button will invoke the secure_power_action
								press_lock_button(cfg_secure_power_action+1);
							} else if(cfg_power_action == POWER_ACTION_OPENCLOSE){
								// unmap ASUI unless n800_lock_mode is enabled
								dbus_send_wakeup(_WAKEUP_HIDE_);
							}
						} else {
							// map ASUI
							dbus_send_wakeup(_WAKEUP_SHOW_);
						}
					} else {
						// screen is in the process of turning on or has been on less than one second
						// CASE: power button used to turn on screen (locked or unlocked)

						// main loop doesn't need to be awakened, DSME socket will do it

debug_write("PB: screen is in the process of turning on, tklock=%u", hw_screen.locked);
						if(hw_screen.locked){
							if(cfg_power_unlock_key){
debug_write("-- secondary key required to unlock, tc=1, kc=0, asui=%u", window.is_mapped);
								// secondary key required to unlock screen, keypad remains unlocked
								hw_lock_screen();
								hw_unlock_keys();
								if(!window.is_mapped) window_open_blank(); // open blank to catch keys
								notify_power_unlock_key();
							} else {
								// screen needs to be unlocked
								window_close_blank(); // power key caught, close the blank window
								mce_unlock_screen();
							}
						} else {
							// screen wasn't locked
							window_close_blank(); // power key caught, close the blank window
							if(cfg_tkactive_notifications) notify_screen_unlocked();
						}
					}

					return DBUS_HANDLER_RESULT_HANDLED;
				} else if(!strcmp(s2, "connection")){
					// path = /org/freedesktop/Hal/devices/platform_headphone
					debug_write("[DBUS]         headphone connect/disconnect (blank=%u)", blank_window_open);

					// this signal will unlock the screen without sending an unlock signal, if locked
					fix_screen_and_keys_and_blank_window();

					hw_query_headphones();

					// this could have turned on the screen so main loop needs to be awakened
					dbus_send_wakeup(_WAKEUP_REFRESH_);

					return DBUS_HANDLER_RESULT_HANDLED;
				} else if(!strcmp(s2, "cover")){
					const char *signal_path = dbus_message_get_path(message) + 29; // strlen("/org/freedesktop/Hal/devices/")

					if(!strcmp(signal_path, "platform_slide") /* || !strcmp(signal_path, "platform_slide_0") */){
						debug_write("[DBUS]         n810 keyboard open/close (blank=%u)", blank_window_open);

						// this signal will turn on the screen, if off
						dsme_display_state_waiting = 0; // reset blanking pause check

						window_close_blank(); // signal turned screen on, close the blank window

//						if(conf_get_bool("/system/osso/af/slide-open")){
							// keyboard opened
// TODO: if screen was unlocked and off then display unlocked notification
							// tklock unlock signal is sent if screen was locked and notification is displayed
//						} else 

						if(hw_screen.locked) notify_screen_locked();

						// fix_screen_and_keys_and_blank_window() can't be used here because the tklock:locked signal re-locks the keys

						// this could have turned on the screen so main loop needs to be awakened
						dbus_send_wakeup(_WAKEUP_REFRESH_);

						return DBUS_HANDLER_RESULT_HANDLED;
//					} else if(!strcmp(signal_path, "platform_kb_lock_0")){
//						debug_write("[DBUS]         n810 lock key (blank=%u)", blank_window_open);

//						return DBUS_HANDLER_RESULT_HANDLED;
					} else if(!strcmp(signal_path, "platform_mmci_omap_1_mmc_host")){
						debug_write("[DBUS]         mmc open/close (blank=%u)", blank_window_open);

						// this signal will unlock the screen without sending an unlock signal, if locked
						fix_screen_and_keys_and_blank_window();

						// this could have turned on the screen so main loop needs to be awakened
						dbus_send_wakeup(_WAKEUP_REFRESH_);

						return DBUS_HANDLER_RESULT_HANDLED;
					}
				} else if(!strcmp(s2, "usb.cable")){
					// path = /org/freedesktop/Hal/devices/usb_device_0_0_musb_hdrc
					debug_write("[DBUS]         USB connect/disconnect (blank=%u)", blank_window_open);

					// this signal will unlock the screen without sending an unlock signal, if locked
					fix_screen_and_keys_and_blank_window();

					// this could have turned on the screen so main loop needs to be awakened
					dbus_send_wakeup(_WAKEUP_REFRESH_);

					return DBUS_HANDLER_RESULT_HANDLED;
				}
			}
		}

		// ignore unmatched signatures

	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){
		const char *signal_path = dbus_message_get_path(message) + 29; // strlen("/org/freedesktop/Hal/devices/")

		if(!strcmp(signal_path, "usb_device_0_0_musb_hdrc")){
			debug_write("[DBUS] signal: org.freedesktop.Hal.Device.PropertyModified (usb_device_0_0_musb_hdrc)");

			hw_query_usb();

			return DBUS_HANDLER_RESULT_HANDLED;
		} else if(!strcmp(signal_path, "bme")){
			debug_write("[DBUS] signal: org.freedesktop.Hal.Device.PropertyModified (bme)");

			hw_query_battery(1); // flush

			return DBUS_HANDLER_RESULT_HANDLED;
		}

		// ignore unmatched signatures

	////////////////////
	// MCE
	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "tklock_mode_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			if(!strcmp(s1, "unlocked")){
				debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind (unlock)");

				// the hardware unlock button will send this signal and it will turn on the screen, if off
				// assume screen is now on and reset blanking pause check
				dsme_display_state_waiting = 0; // reset blanking pause check

				if(hw_screen.locked){
					// screen was just unlocked
					notify_screen_unlocked();
				}

				// always make sure screen/keypad is unlocked
				hw_unlock_screen();
				hw_unlock_keys();
				hw_screen.locked = 0;
			} else if(!strcmp(s1, "locked")){
				debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind (lock)");

				if(!hw_screen.locked && hw_screen.visible){
					// screen was just locked but is still visible
					notify_screen_locked();
				}
				// always make sure screen/keypad is locked
				hw_lock_screen();
				hw_lock_keys();
				hw_screen.locked = 1;

				window_close_blank(); // close the blank window, if open
				click.enabled = 1; // don't ignore first tap since screen can't be unlocked by tapping it
			}
			dbus_send_wakeup(_WAKEUP_LOCK_BUTTON_);
		}

		return DBUS_HANDLER_RESULT_HANDLED;

/*
	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "display_status_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.display_status_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			// off, dimmed, on
			unsigned visible = (strcmp(s1, "off") ? 1 : 0);
			unsigned dimmed = (strcmp(s1, "dimmed") ? 0 : 1);

			debug_write("[DBUS] signal: com.nokia.mce.signal.display_status_ind (visible = %u)", visible);
//printf("com.nokia.mce.signal.display_status_ind -- %s\n", s1);

			if(!hw_screen.visible && visible){
				// screen was just turned on
				hw_screen.visible_time = current_time;
				hw_screen.brightness_mismatch = 1;
				if(original_screen_autolock){
					// restored scree_autolock setting
					conf_set_screen_autolock_value(original_screen_autolock-1);
					original_screen_autolock = 0;
				}
				if(original_screen_stayslit){
					// restore screen_stayslit setting
					conf_set_screen_stayslit_value(original_screen_stayslit-1);
					original_screen_stayslit = 0;
					check_dimmed_state();
				}
				window_open_popup();
			} else if(hw_screen.visible && !visible){
				// screen was just turned off
				if(!hw_screen.locked){
					// screen is not locked
					if(!conf_get_screen_autolock_value()){ // screen does not autolock
						if(!window.is_mapped)
							window_open_blank(); // screen is off but not locked, ASUI is not open -- open a blank window to catch the wakeup tap/key
						click.enabled = 0; // ignore first screen tap when unblanked
					}
				}
				window_close_popup();
			}
			hw_screen.visible = visible;
			// don't need to wakeup main loop for this, dbus unlock signal or X screen/key events will wake it


			if(hw_screen.dimmed != dimmed){
				if(!dimmed || !(hw_battery.status != BATTERY_DRAINING && conf_get_screen_stayslit_value())){
					// ignore dimmed signal if on charger and screen stays lit
					hw_screen.brightness_mismatch = 1;
					hw_screen.dimmed = dimmed;
					hw_changes |= HW_BRIGHTNESS;
					dbus_send_wakeup(_WAKEUP_REFRESH_);
				}
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;
*/

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "devicelock_mode_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.devicelock_mode_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			if(!strcmp(s1, "unlocked")){
				if(hw_device_locked){
					debug_write("[DBUS] signal: com.nokia.mce.signal.devicelock_mode_ind (code unlock)");
					test_write("signal: unlock device");
					hw_device_locked = 0;
					dbus_send_wakeup(_WAKEUP_DEVICE_LOCK_);
				}
			} else if(!strcmp(s1, "locked")){
				if(!hw_device_locked){
					debug_write("[DBUS] signal: com.nokia.mce.signal.device_mode_ind (code lock)");
					test_write("signal: lock device");
					hw_device_locked = 1;
					dbus_send_wakeup(_WAKEUP_DEVICE_LOCK_);
				}
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "shutdown_ind")){

		debug_write("[DBUS] signal: com.nokia.mce.signal.shutdown_ind");

		hw_restore_cp_screen_settings();

		dbus_send_wakeup(_WAKEUP_SHUTDOWN_);

		return DBUS_HANDLER_RESULT_HANDLED;

//	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "system_inactivity_ind")){

		// signature = b

//	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "save_unsaved_data_ind")){

		// signature = <none>

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "sig_device_mode_ind")){
		char *mode;

		debug_write("[DBUS] signal: com.nokia.mce.signal.sig_device_mode_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
			unsigned flight = (strcmp(mode, "normal") ? 1 : 0);
			if(hw_flight_mode != flight){
				hw_flight_mode = flight;
				if(is_value_visible(flightmode_button_page)){ hw_changes |= HW_FLIGHT_MODE; dbus_send_wakeup(_WAKEUP_REFRESH_); }
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// BME
//	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){

		// handled above

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_timeleft")){
		unsigned idle_minutes, active_minutes;

		debug_write("[DBUS] signal: com.nokia.bme.signal.battery_timeleft");

		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &idle_minutes, _UINT32_, &active_minutes, _END_)){
			if(hw_battery.active_minutes != active_minutes || hw_battery.idle_minutes != idle_minutes){
				hw_battery.active_minutes = active_minutes;
				hw_battery.idle_minutes = idle_minutes;
				if(is_value_visible(battery_widget_page)){ hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }

				hw_initialized &= ~HWIS_BME_ACTIVE;
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_connected")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_connected");

		// this signal will unlock the screen without sending an unlock signal, if locked
		fix_screen_and_keys_and_blank_window();

		if(hw_battery.status == BATTERY_DRAINING){
			hw_battery.status = BATTERY_AC;
			dbus_send_charging_status_signal();
			if(is_value_visible(battery_widget_page)){ hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }

			hw_new_event("charger", &hw_battery.charger_history_head, &hw_battery.charger_history_tail, 1);

			hw_query_battery(1); // flush
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_disconnected")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_disconnected");

		// this signal will unlock the screen without sending an unlock signal, if locked
		fix_screen_and_keys_and_blank_window();

		if(hw_battery.status != BATTERY_DRAINING){
			hw_battery.status = BATTERY_DRAINING;
			dbus_send_charging_status_signal();
			if(is_value_visible(battery_widget_page)){ hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }

			hw_new_event("charger", &hw_battery.charger_history_head, &hw_battery.charger_history_tail, 0);

			hw_query_battery(1); // flush
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_charging_on")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_charging_on");

		if(hw_battery.status == BATTERY_DRAINING)
			hw_new_event("charger", &hw_battery.charger_history_head, &hw_battery.charger_history_tail, 1);

		set_battery_status(BATTERY_CHARGING);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_charging_off")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_charging_off");

		set_battery_status(BATTERY_DRAINING);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_full")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.battery_full");

		set_battery_status(BATTERY_AC);

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_empty")){

		// signature = <none>
		// device shuts down after receiving this

		hw_restore_cp_screen_settings();

	////////////////////
	// BLUEZ
	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "ModeChanged")){
		char *mode;

		debug_write("[DBUS] signal: org.bluez.Adapter.ModeChanged");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
			unsigned enabled = (strcmp(mode, "off") ? 1 : 0);
			unsigned visible = (strcmp(mode, "discoverable") ? 0 : 1);

// TODO: connectable, discoverable or limited

			if(hw_bluetooth.enabled != enabled)
				hw_new_event("bluetooth", &hw_bluetooth.history_head, &hw_bluetooth.history_tail, enabled);

			if(hw_bluetooth.enabled != enabled || hw_bluetooth.visible != visible){
				hw_bluetooth.enabled = enabled;
				hw_bluetooth.visible = visible;
				if(is_value_visible(bluetooth_widget_page)){ hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }

				if(!hw_bluetooth.enabled && hw_entering_flight_mode) mce_flight_mode();
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "NameChanged")){
		char *name;

		debug_write("[DBUS] signal: org.bluez.Adapter.NameChanged");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _END_)){
			if(hw_bluetooth.name == NULL || strcmp(hw_bluetooth.name, name)){
				set_string(&hw_bluetooth.name, name, BLUETOOTH_NAME_LIMIT);
				if(is_value_visible(bluetooth_widget_page)){ hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "RemoteDeviceConnected")){

		debug_write("[DBUS] signal: org.bluez.Adapter.RemoteDeviceConnected");

		hw_bluetooth.nr_devices++;
		if(is_value_visible(bluetooth_widget_page)){ hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "RemoteDeviceDisconnected")){

		debug_write("[DBUS] signal: org.bluez.Adapter.RemoteDeviceDisconnected");

		if(hw_bluetooth.nr_devices){
			hw_bluetooth.nr_devices--;
			if(is_value_visible(bluetooth_widget_page)){ hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.audio.Headset", "Connected")){

		debug_write("[DBUS] signal: org.bluez.audio.Headset.org.bluez.audio.Headset");

		if(!hw_bluetooth.headset){
			hw_bluetooth.headset = 1;
			if(is_value_visible(volume_widget_page)){ hw_changes |= HW_VOLUME_OUTPUT; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.audio.Headset", "Disconnected")){

		debug_write("[DBUS] signal: org.bluez.audio.Headset.Disconnected");

		if(hw_bluetooth.headset){
			hw_bluetooth.headset = 0;
			if(is_value_visible(volume_widget_page)){ hw_changes |= HW_VOLUME_OUTPUT; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// ICD
	} else if(dbus_message_is_signal(message, "com.nokia.wlancond.signal", "connected")){

		debug_write("[DBUS] signal: com.nokia.wlancond.signal.connected");

		if(!hw_wifi.enabled){
			hw_wifi.enabled = 1;
			hw_new_event("wifi", &hw_wifi.history_head, &hw_wifi.history_tail, 1);
			if(is_value_visible(wifi_widget_page)){ hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.wlancond.signal", "disconnected")){

		debug_write("[DBUS] signal: com.nokia.wlancond.signal.disconnected");

		if(hw_wifi.enabled){
			hw_wifi.enabled = 0;
			hw_new_event("wifi", &hw_wifi.history_head, &hw_wifi.history_tail, 0);
			if(is_value_visible(wifi_widget_page)){ hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.icd", "status_changed")){
		char *name, *type, *state_string, *unknown;

		debug_write("[DBUS] signal: com.nokia.icd.status_changed");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _STRING_, &type, _STRING_, &state_string, _STRING_, &unknown, _END_)){

			debug_write("[DBUS] signal: com.nokia.icd.status_changed = %s", state_string);

			if(strncmp(state_string, "SCAN", 4)){
				// ignore SCAN_START and SCAN_STOP
				// get connection state (and name/IP if connected)
				e_hw_wifi_state state;
				unsigned name_changed = 0;
				if(!strcmp(state_string, "IDLE")) state = WIFI_DISCONNECTED;
				else if(!strcmp(state_string, "CONNECTED")){
					state = WIFI_CONNECTED;
					name_changed = hw_wifi_get_name(name);
					hw_wifi_get_ipinfo();
					dbus_connection_flush(signal_bus);
				} else if(!strcmp(state_string, "DISCONNECTING")) state = WIFI_DISCONNECTING;
				else state = WIFI_CONNECTING;
				// refresh if name or state changed
				if(name_changed || hw_wifi.state != state){
					if(hw_wifi.state != state && state == WIFI_DISCONNECTED){
						// free name/IP when disconnected
						set_string(&hw_wifi.name, NULL, WIFI_NAME_LIMIT);
						set_string(&hw_wifi.address, NULL, WIFI_ADDRESS_LIMIT);
					}
					hw_wifi.state = state;
					unsigned enabled = (state == WIFI_DISCONNECTED ? 0 : 1);
					if(hw_wifi.enabled != enabled)
						hw_new_event("wifi", &hw_wifi.history_head, &hw_wifi.history_tail, enabled);
					hw_wifi.enabled = enabled;
					if(is_value_visible(wifi_widget_page)){ hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }

					if(!hw_wifi.enabled && hw_entering_flight_mode) mce_flight_mode();
				}
			}
		}

		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// USB
//	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){

		// handled above

	////////////////////
	// one time signals
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_state_changed")){

		// signature = u
		// the value might be the number of bars in the default battery statusbar applet (1-4)

		debug_write("[DBUS] signal: com.nokia.bme.signal.battery_state_changed");

		// now get rid of this signal
		// this signal is sent along with charger state in response to the status_info_req signal
		// status_info_req signal will be sent until this signal is received
		dbus_bus_remove_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='battery_state_changed'", &dbus_error);
		dbus_connection_flush(signal_bus);
		hw_initialized &= ~HWIS_BME_CHARGER;

		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.icd2", "state_sig")){
		if(!strcmp(dbus_message_get_signature(message), "sussuaysu")){
			char *svc_type, *svc_id, *net_type, *net_id, *error;
			unsigned svc_attr, net_attr, net_state;
			int net_len;

			debug_write("[DBUS] signal: com.nokia.icd2.state_sig sussuaysu");

			if(dbus_message_get_args(message, &dbus_error,
				_STRING_, &svc_type, _UINT32_, &svc_attr, _STRING_, &svc_id,
				_STRING_, &net_type, _UINT32_, &net_attr,
				_ARRAY_, DBUS_TYPE_BYTE, &net_id, &net_len,
				_STRING_, &error, _UINT32_, &net_state, _END_))
			{
				// get connection state (and name/IP if connected)
				e_hw_wifi_state state;
				unsigned name_changed = 0;
				if(net_state == 0 /* DISCONNECTED */) state = WIFI_DISCONNECTED;
				else if(net_state == 2 /* CONNECTED */){
					state = WIFI_CONNECTED;
					name_changed = hw_wifi_get_name(net_id);
					hw_wifi_get_ipinfo();
					dbus_connection_flush(signal_bus);
				} else if(net_state == 3 /* DISCONNECTING */) state = WIFI_DISCONNECTING;
				else state = WIFI_CONNECTING;
				// refresh if name or state changed
				if(name_changed || hw_wifi.state != state){
					if(hw_wifi.state != state && state == WIFI_DISCONNECTED){
						// free name/IP when disconnected
						set_string(&hw_wifi.name, NULL, WIFI_NAME_LIMIT);
						set_string(&hw_wifi.address, NULL, WIFI_ADDRESS_LIMIT);
					}
					hw_wifi.state = state;
					hw_wifi.enabled = (state == WIFI_DISCONNECTED ? 0 : 1);
					if(is_value_visible(wifi_widget_page)){ hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
				}

				// now get rid of this signal
				// this signal is sent in response to the state_req method used to receive the initial wifi state
				dbus_bus_remove_match(signal_bus, "type='signal',path='/com/nokia/icd2',interface='com.nokia.icd2',member='state_sig'", &dbus_error);
				dbus_connection_flush(signal_bus);
				hw_initialized &= ~HWIS_WIFI_STATE;
			}
			return DBUS_HANDLER_RESULT_HANDLED;
		}

		// ignore unmatched signatures
		// signature = su
		// signature = u

	////////////////////
	// ERRORS
	} else if(dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_ERROR){
		char *s;

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s, _END_)){
			debug_write("[DBUS ERROR] %s", s);
			test_write("[DBUS ERROR] %s", s);
		}

		return DBUS_HANDLER_RESULT_HANDLED;
	}

	#ifdef TESTLOG
	// find unhandled signals that are waking the ASUI process
	switch(dbus_message_get_type(message)){
	case DBUS_MESSAGE_TYPE_METHOD_CALL: test_write("[DBUS] received unhandled METHOD CALL"); break;
	case DBUS_MESSAGE_TYPE_METHOD_RETURN: test_write("[DBUS] received unhandled METHOD RETURN"); break;
	case DBUS_MESSAGE_TYPE_SIGNAL: test_write("[DBUS] received unhandled SIGNAL"); break;
	default: test_write("[DBUS] received unhandled message type = UNKNOWN"); return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	test_write("       dest=%s %s %s.%s (%s)", dbus_message_get_destination(message), dbus_message_get_path(message), dbus_message_get_interface(message),
												dbus_message_get_member(message), dbus_message_get_signature(message));
	test_write("       wants reply? = %s", (dbus_message_get_no_reply(message) ? "no" : "yes"));
	return DBUS_HANDLER_RESULT_HANDLED;
	#else
		#ifdef DEBUGLOG
		switch(dbus_message_get_type(message)){
		case DBUS_MESSAGE_TYPE_METHOD_CALL: debug_write("[DBUS] received unhandled METHOD CALL"); break;
		case DBUS_MESSAGE_TYPE_METHOD_RETURN: debug_write("[DBUS] received unhandled METHOD RETURN"); break;
		case DBUS_MESSAGE_TYPE_SIGNAL: debug_write("[DBUS] received unhandled SIGNAL"); break;
		default: debug_write("[DBUS] received unhandled message type = UNKNOWN"); return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
		}
		debug_write("       dest=%s %s %s.%s (%s)", dbus_message_get_destination(message), dbus_message_get_path(message), dbus_message_get_interface(message),
													dbus_message_get_member(message), dbus_message_get_signature(message));
		debug_write("       wants reply? = %s", (dbus_message_get_no_reply(message) ? "no" : "yes"));
		return DBUS_HANDLER_RESULT_HANDLED;
		#else
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
		#endif
	#endif
}
#endif

//////////////////////////////////////////////////////////////////////////

#define	PARENT_READ  threadpipe[0]
#define	CHILD_WRITE  threadpipe[1]
static int threadpipe[2] = {-1,-1};

static void *dbus_thread( void *arg ){
	debug_write("[DBUS] thread starting...");

	#ifdef ARCH_armel
		dbus_wakeup_fd = CHILD_WRITE;

/*
		// ASUI methods
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/qzx/asui',interface='com.qzx.asui'", &dbus_error);

		// SystemUI methods
//		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='tklock_open'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='tklock_close'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='devlock_open'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='devlock_close'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='alarm_open'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='alarm_close'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='acting_dead_open'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='acting_dead_close'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='acring_dead_getstate'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='powerkeymenu_open'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request',member='powerkeymenu_close'", &dbus_error);
*/

		// power button signal
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/computer_logicaldev_input',"   // n800
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='power'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/computer_logicaldev_input_0'," // n810
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='power'", &dbus_error);
		// headphone jack
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_headphone',"
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='connection'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_headphone_0',"
//										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='connection'", &dbus_error);
		// n810 keyboard open/close
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_slide',"
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='cover'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_slide_0',"
//										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='cover'", &dbus_error);
		// n810 lock key
//		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_kb_lock_0',"
//										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='cover'", &dbus_error);
		// n810 mmc card
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_mmci_omap_1_mmc_host',"
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='cover'", &dbus_error);
		// usb jack
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/usb_device_0_0_musb_hdrc',"
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='usb.cable'", &dbus_error);

		// mce signals (screen lock, device lock and shutdown)
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='tklock_mode_ind'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='display_status_ind'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='devicelock_mode_ind'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='shutdown_ind'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='system_inactivity_ind'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='save_unsaved_data_ind'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal',member='sig_device_mode_ind'", &dbus_error);

		// bme signals (battery status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/bme',interface='org.freedesktop.Hal.Device',member='PropertyModified'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='battery_timeleft'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='charger_connected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='charger_disconnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='charger_charging_on'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='charger_charging_off'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='battery_full'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='battery_empty'", &dbus_error);

		// bluez signals (bluetooth status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='ModeChanged'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='NameChanged'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='RemoteDeviceConnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='RemoteDeviceDisconnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',interface='org.bluez.audio.Headset',member='Connected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',interface='org.bluez.audio.Headset',member='Disconnected'", &dbus_error);

		// icd signals (wifi status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/wlancond/signal',interface='com.nokia.wlancond.signal',member='connected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/wlancond/signal',interface='com.nokia.wlancond.signal',member='disconnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/icd',interface='com.nokia.icd',member='status_changed'", &dbus_error);

		// usb signals
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/usb_device_0_0_musb_hdrc',"
										"interface='org.freedesktop.Hal.Device',member='PropertyModified'", &dbus_error);

		// initialization only
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal',member='battery_state_changed'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/icd2',interface='com.nokia.icd2',member='state_sig'", &dbus_error);

		dbus_connection_add_filter(signal_bus, signal_filter, NULL, NULL);
		debug_write("[DBUS] filters ready");
		while(dbus_connection_read_write_dispatch(signal_bus, -1));
		debug_write("[DBUS] thread terminated");

		dbus_listening = 0; // inform main loop that signal bus was closed
		dbus_connection_close(signal_bus);
		close(dbus_wakeup_fd);
	#else
		while(1) sleep(10);
	#endif
	return NULL;
}

//////////////////////////////////////////////////////////////////////////

int dbus_init( ){
	#ifdef ARCH_armel
	original_screen_autolock = original_screen_stayslit = 0;
	acting_dead_state = 0;
//	powerkeymenu_open = 0;

	dbus_threads_init_default();
	dbus_error_init(&dbus_error);
	if((signal_bus = dbus_bus_get_private(DBUS_BUS_SYSTEM, &dbus_error)) == NULL){
		error_write("can't connect to system signal bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}
	if((system_bus = dbus_bus_get_private(DBUS_BUS_SYSTEM, &dbus_error)) == NULL){
		error_write("can't connect to system method bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}
	if((session_bus = dbus_bus_get_private(DBUS_BUS_SESSION, &dbus_error)) == NULL){
		error_write("can't connect to session method bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}

	if(dbus_bus_request_name(signal_bus, "com.qzx.asui", DBUS_NAME_FLAG_REPLACE_EXISTING, &dbus_error) == -1){
		error_write("dbus won't let me register as com.qzx.asui: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
	}
	if(dbus_bus_request_name(signal_bus, "com.nokia.system_ui", DBUS_NAME_FLAG_REPLACE_EXISTING, &dbus_error) == -1){
		error_write("dbus won't let me register as com.nokia.system_ui: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
	}
	#endif

	if(pipe(threadpipe) < 0){
		error_write("can't open dbus thread pipe");
		return 0;
	}

	dbus_listening = 1;
	pthread_create(&thread, 0, dbus_thread, NULL);

	return PARENT_READ;
}

void dbus_send_wakeup( char message ){
	write(dbus_wakeup_fd, (void *)&message, 1);
}

void dbus_disconnect( ){
	#ifdef ARCH_armel
	dbus_connection_close(system_bus);
	dbus_connection_close(session_bus);
	#endif
}

////////////////////////////////////////////////////////////////////////// SEND METHOD

#ifdef ARCH_armel
void dbus_send_method( const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	debug_write("-> dbus-send SYSTEM %s.%s(%s)", interface, method, (first_arg_type == _END_ ? "" : "..."));

	message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

void dbus_send_reply( DBusConnection *bus, DBusMessage *method_call, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	debug_write("-> dbus-send REPLY (%s)", (first_arg_type == _END_ ? "" : "..."));

	message = dbus_message_new_method_return(method_call);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(bus, message, NULL) != FALSE)
		dbus_connection_flush(bus);
	dbus_message_unref(message);
}

void dbus_send_signal( const char *path, const char *interface, const char *signal, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	debug_write("-> dbus-send SYSTEM SIGNAL %s.%s(%s)", interface, signal, (first_arg_type == _END_ ? "" : "..."));

	message = dbus_message_new_signal(path, interface, signal);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}
#endif

////////////////////////////////////////////////////////////////////////// SEND METHOD w/ RETURN

#ifdef ARCH_armel
void dbus_recv_method( DBusPendingCallNotifyFunction callback, const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;
	DBusPendingCall *pc;

	debug_write("-> dbus-recv SYSTEM %s.%s(%s)", interface, method, (first_arg_type == _END_ ? "" : "..."));

	message = dbus_message_new_method_call(destination, path, interface, method);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	dbus_connection_send_with_reply(signal_bus, message, &pc, 5000); // 5 second timeout
	if(pc)
		dbus_pending_call_set_notify(pc, callback, NULL, NULL);
	dbus_message_unref(message);
}

void dbus_flush_messages( ){
	dbus_send_method("com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "flush_calls", _END_);
}
#endif

////////////////////////////////////////////////////////////////////////// SEND SESSION METHOD

#ifdef ARCH_armel
void dbus_send_session_method( const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	debug_write("-> dbus-send SESSION %s.%s(%s)", interface, method, (first_arg_type == _END_ ? "" : "..."));

	message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(session_bus, message, NULL) != FALSE)
		dbus_connection_flush(session_bus);
	dbus_message_unref(message);
}
#endif

////////////////////////////////////////////////////////////////////////// NOTIFICATIONS

void notify_screen_locked( ){
	#ifdef ARCH_armel
	char *locked_message = "Touch screen and keys locked";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_, _STRING_, &locked_message, _END_);
	#endif
}

void notify_screen_unlocked( ){
	#ifdef ARCH_armel
	char *unlocked_message = "Touch screen and keys active";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_, _STRING_, &unlocked_message, _END_);
	#endif
}

void notify_power_unlock_key( ){
	#ifdef ARCH_armel
	if(cfg_power_unlock_key){
		dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_,
									_STRING_, &power_unlock_messages[cfg_power_unlock_key], _END_);
		if(conf_get_screen_blank_offset() > 10)
			mce_dim_screen(0);
	}
	#endif
}

void notify_blanking_pause1( ){
	#ifdef ARCH_armel
	char *paused_message   = "An application may have PAUSED screen blanking";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_, _STRING_, &paused_message, _END_);
	#endif
}
void notify_blanking_pause2( ){
	#ifdef ARCH_armel
	const char *locked_message   = "Screen is LOCKED and should dim in a minute...";
	const char *unlocked_message = "Screen is UNLOCKED and should dim in a minute...";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_,
								_STRING_, (hw_screen.locked ? &locked_message : &unlocked_message), _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// SOUNDS

void dbus_audio_set_volume( double volume ){
	#ifdef ARCH_armel
	dbus_send_session_method(_osso_multimedia_service_, _osso_multimedia_path_, _osso_multimedia_interface_, "set_volume", _DOUBLE_, &volume, _END_);
	#endif
}

void dbus_audio_set_loop( unsigned loop ){
	#ifdef ARCH_armel
	dbus_send_session_method(_osso_multimedia_service_, _osso_multimedia_path_, _osso_multimedia_interface_, "set_loop", _BOOL_, &loop, _END_);
	#endif
}

void dbus_audio_play( char *sound, int priority ){
	#ifdef ARCH_armel
	dbus_send_session_method(_osso_multimedia_service_, _osso_multimedia_path_, _osso_multimedia_interface_, "play_sound", _STRING_, &sound, _INT32_, &priority, _END_);
	#endif
}

void dbus_audio_stop( ){
	#ifdef ARCH_armel
	dbus_send_session_method(_osso_multimedia_service_, _osso_multimedia_path_, _osso_multimedia_interface_, "stop", _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// STATUSBAR APPLET SIGNALS

void dbus_send_charging_status_signal( ){
	#ifdef ARCH_armel
	unsigned status = (unsigned)hw_battery.status;
	dbus_send_signal("/com/qzx/asui", "com.qzx.asui", "charging_status", _UINT32_, &status, _END_);
	#endif
}

void dbus_send_battery_capacity_signal( ){
	#ifdef ARCH_armel
	unsigned capacity = (unsigned)roundf(hw_battery.capacity);
	dbus_send_signal("/com/qzx/asui", "com.qzx.asui", "battery_capacity", _UINT32_, &capacity, _END_);
	#endif
}
