//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

// seconds (double)
#define REFRESH_PERIOD 60.0

#define CAPACITY_COLOR green_status_color
#define SCREEN_COLOR red_status_color
#define WIFI_COLOR yellow_status_color
#define BLUETOOTH_COLOR selection_color

#include <stdio.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>

#include "window.h"
#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
 #include "battery_graph.h"

////////////////////////////////////////////////////////////////////////// LOCALS

static e_orientation original_orientation;
static double base_time;
static float units;
static unsigned event_size;

//////////////////////////////////////////////////////////////////////////

static void draw_event( s_event_history *head, s_event_history *tail, XColor *color, int y ){
	if(head == NULL || tail == NULL) return;
	unsigned nr_events = 0;
	int x1 = 0, x2;

	x_set_color(color);
	x_set_style(3, LineSolid); // 3 wide

	s_event_history *history;
	for(history = head; history != NULL; history = history->next, nr_events++){
		if(history->state){
			// on event -- update start point
			x1 = roundf((history->timestamp - base_time) / units);
		} else {
			// off event -- draw line from last on event
			x2 = roundf((history->timestamp - base_time) / units);
			x_draw_line(x1,y, x2,y);
		}
	}
	if(tail->state) x_draw_line(x1,y, window.width-1,y);

	x_reset_style();

	event_size += nr_events * sizeof(s_event_history);
}

static void draw_battery_graph( unsigned force ){
	// force is not needed

	char buffer[20];

	// dialog background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);
	// top and bottom graph lines
	x_set_color(line_color);
	x_draw_line(0,18, window.width,18);
	x_draw_line(0,window.height-18, window.width,window.height-18);

	event_size = 0;
	if(hw_battery.history_head){
		base_time = hw_battery.history_head->timestamp;
		units = (current_time - base_time) / window.width;
	} else {
		base_time = 0;
		units = 0;
	}
	if(hw_battery.history_head != hw_battery.history_tail && hw_battery.history_head != NULL){
		float prev_capacity = hw_battery.history_head->capacity;
		double prev_timestamp = hw_battery.history_head->timestamp;
		int graph_bottom = window.height - 20;
		int graph_height = window.height - 40;
		int xc = 0, x1 = 0;
		int y1 = graph_bottom - graph_height * (prev_capacity / 100.0);
		unsigned nr_events = 0;

		x_set_color(CAPACITY_COLOR);
		x_set_style(2, LineSolid); // 2 wide
		s_battery_history *history;
		for(history = hw_battery.history_head->next; history != NULL; history = history->next, nr_events++){
			int x2 = roundf((history->timestamp - base_time) / units);
			int y2 = graph_bottom - graph_height * (history->capacity / 100.0);
			x_draw_line(x1, y1, x2, y2);

			if(history == hw_battery.charge_point) xc = x2;

			x1 = x2; y1 = y2;
			prev_capacity = history->capacity;
			prev_timestamp = history->timestamp;
		}
		event_size += nr_events * sizeof(s_battery_history);
		x_reset_style();

		// last charging point
		if(hw_battery.charge_point){
			x_set_color(CAPACITY_COLOR);
			x_set_style(1, LineOnOffDash); // dotted line
			x_draw_line(xc,20+1, xc,graph_bottom);
			x_reset_style();
		}

		// battery labels
		draw_text("100%", 16, CAPACITY_COLOR, 0,0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("0% battery", 16, CAPACITY_COLOR, 0,window.height-1, ALIGN_LEFT, ALIGN_BOTTOM, 0);

		// number of hours shown in graph
		snprintf(buffer, sizeof(buffer), "%.1f hours", (current_time - base_time) / 3600.0);
		draw_text(buffer, 16, fg2_color, window.width>>1,0, ALIGN_CENTER, ALIGN_TOP, 0);
	} else {
		// no battery data
		draw_text("waiting...", 18, fg2_color, window.width>>1,0, ALIGN_CENTER, ALIGN_TOP, 0);
	}

	if(cfg_track_usage_events && base_time){
		// screen events
		draw_event(hw_screen.history_head, hw_screen.history_tail, SCREEN_COLOR, (window.height>>1)-10);
		// wifi events
		draw_event(hw_wifi.history_head, hw_wifi.history_tail, WIFI_COLOR, window.height>>1);
		// bluetooth events
		draw_event(hw_bluetooth.history_head, hw_bluetooth.history_tail, BLUETOOTH_COLOR, (window.height>>1)+10);

// TODO: gps events

		// screen, wifi and bluetooth event labels
		draw_text("screen", 16, SCREEN_COLOR, window.width/3,window.height-1, ALIGN_CENTER, ALIGN_BOTTOM, 0);
		draw_text("wifi", 16, WIFI_COLOR, window.width*2/3,window.height-1, ALIGN_CENTER, ALIGN_BOTTOM, 0);
		draw_text("bluetooth", 16, BLUETOOTH_COLOR, window.width-1,window.height-1, ALIGN_RIGHT, ALIGN_BOTTOM, 0);
	}

	snprintf(buffer, sizeof(buffer), "%ukB", (unsigned)roundf((float)event_size/1024.0));
	draw_text(buffer, 16, fg2_color, window.width-1,0, ALIGN_RIGHT, ALIGN_TOP, 0);

	hw_changes = 0;
	reset_timer(&dialog_refresh_timer);
}

static void click_battery_graph( ){
	if(click.type == CLICK_DRAG && click.offset_y > 160){
		// close dialog
		highlight_selected_area(0, 0, window.width, window.height); // hack
		dialog_close();
	}
}

static void keypress_battery_graph( ){
	if(keypress.keycode == KEYCODE_ESCAPE && keypress.type == KEY_SHORT){
		// close dialog
		highlight_selected_area(0, 0, window.width, window.height); // hack
		dialog_close();
	}
}

static void reset_battery_graph( ){
	if(original_orientation != O_LANDSCAPE)
		rotate_screen(original_orientation, 1);
}

//////////////////////////////////////////////////////////////////////////

void battery_graph_open_dialog( ){
	original_orientation = window.orientation;
	if(window.orientation != O_LANDSCAPE)
		rotate_screen(O_LANDSCAPE, 1);
	dialog_open(draw_battery_graph, click_battery_graph, keypress_battery_graph, reset_battery_graph, DLG_REDRAW_ON);
	set_timer(&dialog_refresh_timer, REFRESH_PERIOD);
}
