//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <brand.huntsman@gmail.com>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <linux/limits.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>

#include "main.h"
#include "draw.h"
#include "config.h"
 #include "services.h"
#include "hardware.h"

////////////////////////////////////////////////////////////////////////// GLOBALS

unsigned svc_changes;

unsigned svc_bluetooth_status, svc_gps_status, svc_wifi_status, svc_audio_status, svc_desktop_status, svc_ssh_status, svc_vpn_status;
unsigned svc_bluetooth_size, svc_gps_size, svc_wifi_size, svc_audio_size, svc_desktop_size, svc_ssh_size, svc_vpn_size, svc_self_size;

unsigned svc_bluetooth_HCI, svc_bluetooth_BTCOND, svc_bluetooth_OBEX;
unsigned svc_gps_DRIVER, svc_gps_SUPL, svc_gps_CLOCKD, svc_gps_GUARD;
unsigned svc_wifi_WLANCOND, svc_wifi_DNSMASQ, svc_wifi_CONNDLGS, svc_wifi_ICD, svc_wifi_NTPD;
unsigned svc_audio_MULTIMEDIA, svc_audio_DSP, svc_audio_ENGINE, svc_audio_ESD, svc_audio_SERVER;

unsigned svc_ssh_installed, svc_ssh_nr_clients;
unsigned svc_vpn_installed;

unsigned svc_process_viewer, svc_nr_processes;
s_process *svc_process_list;

double last_svc_refresh;

////////////////////////////////////////////////////////////////////////// LOCAL GLOBALS

typedef struct s_service {
	const char *name;
	unsigned *running;
	unsigned *rss;
} s_service;
typedef struct s_other_service {
	const char *name;
	unsigned match_length;
} s_other_service;

static unsigned ticks_per_second;
static double last_cpu_update;

//////////////////////////////////////////////////////////////////////////

static s_process *get_process( unsigned pid ){
	s_process *p;
	for(p = svc_process_list; p != NULL; p = p->next)
		if(p->pid == pid) return p;
	return NULL;
}

static unsigned sort_processes( ){
	s_process *prev = NULL, *process = svc_process_list;
	unsigned status = 1;

	while(process != NULL){
		if(process->next == NULL) return status;
		if((process->cpu < process->next->cpu) || (process->cpu == process->next->cpu && process->rss < process->next->rss)){
			s_process *next = process->next;

			// swap elements
			if(process == svc_process_list) svc_process_list = next;
			else prev->next = next;
			process->next = next->next;
			next->next = process;
			status = 0; // list changed
			prev = next;
		} else {
			prev = process;
			process = process->next;
		}
	}
	return status;
}

unsigned is_sshd_client( char *cmdline ){
	// sshd: user [priv]		<-- dont count this process as a client
	// sshd: user@pts/0
	while(*cmdline != '\0'){
		cmdline++; // don't bother testing first character, should be a username
		if(*cmdline == '@') return 1;
	}
	return 0;
}

//////////////////////////////////////////////////////////////////////////

void svc_init( ){
	svc_refresh(); // need to query now so hardware can request initial states from services
	svc_changes = 0;
	svc_process_viewer = 0;
	svc_process_list = NULL;
	svc_nr_processes = 0;
	ticks_per_second = sysconf(_SC_CLK_TCK);
	last_cpu_update = 0.0;
}

void svc_refresh( ){
	unsigned sc_bluetooth_size = 0, sc_gps_size = 0, sc_wifi_size = 0, sc_audio_size = 0;
	unsigned sc_desktop_size = 0, sc_desktop = 0;
	unsigned nr_ssh_clients = 0;
		unsigned sc_openssh_size = 0, sc_openssh = 0, sc_openssh_installed = 0;
		unsigned sc_dropbear_size = 0, sc_dropbear = 0, sc_dropbear_installed = 0;
	unsigned sc_vpn_size = 0, sc_vpn = 0;
	unsigned sc_self_size = 0, sc_self = 0;
	unsigned tmp_status;

	svc_bluetooth_HCI = svc_bluetooth_BTCOND = svc_bluetooth_OBEX = 0;
	svc_gps_DRIVER = svc_gps_SUPL = svc_gps_CLOCKD = svc_gps_GUARD = 0;
	svc_wifi_WLANCOND = svc_wifi_DNSMASQ = svc_wifi_CONNDLGS = svc_wifi_ICD = svc_wifi_NTPD = 0;
	svc_audio_MULTIMEDIA = svc_audio_DSP = svc_audio_ENGINE = svc_audio_ESD = svc_audio_SERVER = 0;

	s_service services[] = {
		// BT
		{"/usr/sbin/hcid", &svc_bluetooth_HCI, &sc_bluetooth_size},
		{"/usr/lib/bluetooth/bluetoothd-service-audio", NULL, &sc_bluetooth_size}, // just track memory, toggled with hcid
		{"/usr/lib/bluetooth/bluetoothd-service-input", NULL, &sc_bluetooth_size}, // just track memory, toggled with hcid
		{"/usr/bin/btcond", &svc_bluetooth_BTCOND, &sc_bluetooth_size},
		{"/usr/bin/obexsrv", &svc_bluetooth_OBEX, &sc_bluetooth_size},
		// GPS
		{"/usr/sbin/gpsdriver", &svc_gps_DRIVER, &sc_gps_size},
		{"/usr/sbin/supllistenerd", &svc_gps_SUPL, &sc_gps_size}, // optional
		{"/usr/sbin/gps-clockd", &svc_gps_CLOCKD, &sc_gps_size}, // optional
		{"(gpsdriver_guard)", &svc_gps_GUARD, &sc_gps_size}, // optional (/bin/sh /usr/local/sbin/gpsdriver_guard)
		// WIFI
		{"/usr/sbin/wlancond", &svc_wifi_WLANCOND, &sc_wifi_size},
		{"/usr/sbin/dnsmasq", &svc_wifi_DNSMASQ, &sc_wifi_size},
		{"/usr/bin/osso-connectivity-ui-conndlgs", &svc_wifi_CONNDLGS, &sc_wifi_size},
		{"/usr/sbin/icd2", &svc_wifi_ICD, &sc_wifi_size},
		{"/usr/sbin/ntpd", &svc_wifi_NTPD, &sc_wifi_size}, // optional
		// AUDIO
		{"/usr/sbin/multimediad", &svc_audio_MULTIMEDIA, &sc_audio_size},
		{"/usr/sbin/dsp_dld", &svc_audio_DSP, &sc_audio_size},
		{"/usr/bin/mediaplayer-engine", &svc_audio_ENGINE, &sc_audio_size},
		{"/usr/bin/esd", &svc_audio_ESD, &sc_audio_size},
		{"/usr/bin/osso-media-server", &svc_audio_SERVER, &sc_audio_size},
		// HILDON DESKTOP
		{"/usr/bin/hildon-desktop", &sc_desktop, &sc_desktop_size},
		// OpenSSH
		{"/usr/sbin/sshd", &sc_openssh, &sc_openssh_size}, // optional
		// Dropbear SSH
		{"/usr/sbin/dropbear", &sc_dropbear, &sc_dropbear_size}, // optional
		// OpenVPN
		{"/usr/sbin/openvpn", &sc_vpn, &sc_vpn_size}, // optional
		// SELF
		{"/usr/bin/advanced-systemui", &sc_self, &sc_self_size},
		{"/root/.advanced-systemui-test", &sc_self, &sc_self_size},
		{"/root/.advanced-systemui-debug", &sc_self, &sc_self_size},
		{NULL, NULL, NULL}
	};
	s_other_service other_services[] = {
		// optional
		{"/usr/bin/telescope",					0},
		// desktop
		{"/usr/bin/Xomap",						0},
		{"/usr/bin/matchbox-window-manager",	0},
		{"/usr/lib/sapwood/sapwood-server",		0},
		{"/usr/bin/hildon-input-method",		0},
		{"/usr/bin/clipboard-manager",			0},
		// services
		{"/usr/bin/alarmd",					0},
		{"/usr/sbin/browserd",				0},
		{"/usr/bin/metalayer-crawler",		0},
		// system
		{"/usr/libexec/gnome-vfs-daemon",	0},
		{"/usr/lib/gconf2/gconfd-2",		0},
		{"/usr/bin/dbus-daemon",			0},
		{"/sbin/mce",						0},
		{"dsme",							0},
		{"/usr/bin/bme_RX-",				16},
		{"/usr/lib/hal/hald-addon-",		24},
		{"hald-addon-",						11},
		{"hald-runner",						0},
		{"/usr/sbin/hald",					0},
		{"/usr/sbin/hulda",					0},
		{"/usr/sbin/temp-reaper",			0},
		{"/usr/sbin/kicker",				0},
		{"/usr/sbin/ke-recv",				0},
		{"/sbin/udevd",						0},
		{"init [2]",						0},
		{NULL, 0}
	};
	float interval = current_time - last_cpu_update;

	// free process list when viewer isn't open
	if(!svc_process_viewer && svc_process_list){
		s_process *process;
		for(process = svc_process_list; process != NULL;){
			s_process *next = process->next;
			free(process);
			process = next;
		}
		svc_process_list = NULL;
		svc_nr_processes = 0;
	}

	// set a valid ssh server
	if(cfg_ssh_server == SSH_SERVER_NULL) cfg_ssh_server = SSH_SERVER_OPENSSH;

	// is OpenSSH installed?
	tmp_status = file_exists("/etc/init.d/ssh");
	if(svc_ssh_installed != tmp_status) svc_changes |= SVC_SSH;
	sc_openssh_installed = tmp_status;

	// is Dropbear SSH installed?
	tmp_status = file_exists("/etc/init.d/dropbear");
	if(svc_ssh_installed != tmp_status) svc_changes |= SVC_SSH;
	sc_dropbear_installed = tmp_status;

	svc_ssh_installed = (cfg_ssh_server == SSH_SERVER_OPENSSH ? sc_openssh_installed : sc_dropbear_installed);

	// is OpenVPN installed?
	tmp_status = file_exists("/etc/init.d/openvpn");
	if(svc_vpn_installed != tmp_status) svc_changes |= SVC_VPN;
	svc_vpn_installed = tmp_status;

	// query running services
	DIR *proc;
	if((proc = opendir("/proc")) != NULL){
		struct dirent *file;
		s_process *process;

		// reset process list
		if(svc_process_viewer)
			for(process = svc_process_list; process != NULL; process = process->next)
				process->_alive = 0;

		while((file = readdir(proc))){
			if(file->d_name[0] > '0' && file->d_name[0] <= '9'){
				char filepath[30], command[PATH_MAX], cmdline[400], state;
				unsigned ticks = 0, rss = 0, is_service = 0, is_shell = 0, is_python = 0;
				FILE *p;

				// state, resident set size and cpu%
				snprintf(filepath, sizeof(filepath), "/proc/%s/stat", file->d_name);
				if((p = fopen(filepath, "r")) != NULL){
					unsigned utime, stime;
					fscanf(p, "%*d %s %c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %u %u %*d %*d %*d %*d %*d %*d %*u %*u %u",
						command, &state, &utime, &stime, &rss);
					ticks = utime + stime;
					rss *= 4;
					fclose(p);
				}

				// process name
				if(rss){
					snprintf(filepath, sizeof(filepath), "/proc/%s/cmdline", file->d_name);
					if((p = fopen(filepath, "r")) != NULL){
						fgets(cmdline, sizeof(cmdline), p);
						fclose(p);
						if(!strcmp(cmdline, "/bin/sh")) is_shell = 1;
						else if(!strncmp(cmdline, "/usr/bin/python", 15)) is_python = 1;
					} else cmdline[0] = '\0';
				} else cmdline[0] = '\0';

				// find service processes
				if(rss && cmdline[0] != '\0'){
					int i;
					for(i = 0; services[i].name != NULL; i++){
						s_service *s = &services[i];
						// no python services, add a check here if any become known
						if(is_shell){
							if(!strcmp(command, s->name)){
								if(s->running) *s->running = 1;
								*s->rss += rss;
								is_service = 1;
								break;
							}
						} else if(!strcmp(cmdline, s->name)){
							if(s->running){
								if(s->running == &sc_dropbear) sc_dropbear++; else *s->running = 1;
							}
							*s->rss += rss;
							is_service = 1;
							break;
						}
					}
					if(!is_service && sc_openssh_installed){
						if(!strncmp(cmdline, "sshd: ", 6)){
							if(is_sshd_client(&cmdline[6])) nr_ssh_clients++;
							sc_openssh_size += rss;
							is_service = 1;
						}
					}
				}

				if(svc_process_viewer){
					unsigned pid;
					int start, end;
					char *cmd;

					// pid
					sscanf(file->d_name, "%u", &pid);

					// strip path from process name
					if(cmdline[0] != '\0' && !is_shell && !is_python){
						for(end = 0; cmdline[end] != '\0' && cmdline[end] != ' '; end++);
						for(start = end; cmdline[start] != '/' && start > 0; start--);
						if(cmdline[start] == '/') start++;
						cmd = &cmdline[start];
					} else
						cmd = command; // probably a zombie process

					if((process = get_process(pid))){
						// process exists, update rss and cpu

						process->rss = rss;
						int cpu = round(100.0*(float)(ticks - process->_ticks) / (float)(ticks_per_second * interval));
						process->cpu = (cpu < 0 ? 0 : (cpu > 100 ? 100 : cpu));
						process->_ticks = ticks;
						process->_alive = 1;

// TODO: this value would need to be divided by the number of cores, if more than one

					} else if((process = (s_process *)malloc(sizeof(s_process)))){
						// add new process to list

						debug_write("malloc in svc_refresh()");

						// is it a known system service?
						process->is_other_service = 0;
						if(!is_service){
							int i;
							for(i = 0; other_services[i].name != NULL; i++){
								s_other_service *s = &other_services[i];
								if(s->match_length){
									if(!strncmp(cmdline, s->name, s->match_length)){
										process->is_other_service = 1;
										break;
									}
								} else if(!strcmp(cmdline, s->name)){
									process->is_other_service = 1;
									break;
								}
							}
						}

						if(is_shell)
							snprintf(process->name, sizeof(process->name), "sh: %s", cmd);
						else if(is_python)
							snprintf(process->name, sizeof(process->name), "python: %s", cmd);
						else
							snprintf(process->name, sizeof(process->name), "%s", cmd);
						process->state = state;
						process->pid = pid;
						process->rss = rss;
						process->cpu = 0; // no data yet
						process->_ticks = ticks;
						process->is_service = is_service;
						process->_alive = 1;

						process->next = svc_process_list;
						svc_process_list = process;
						svc_nr_processes++;
					}
				}
			}
		}
		closedir(proc);
	}

	if(svc_process_list){
		s_process *process, *prev = NULL;

		// remove terminated processes
		for(process = svc_process_list; process != NULL;){
			s_process *next = process->next;
			if(!process->_alive){
				if(prev == NULL) svc_process_list = next;
				else prev->next = next;
				free(process);
				svc_nr_processes--;
			} else prev = process;
			process = next;
		}

		// sort process list by CPU and RSS
		while(!sort_processes());
	}

	last_cpu_update = current_time;

	// SELF
	{
		if(svc_self_size != sc_self_size){ svc_changes |= SVC_SELF; debug_write("asui rss = %u", sc_self_size); }
		svc_self_size = sc_self_size;
	}

	// BLUETOOTH
		// hcid manages the bluetoothd-service-audio and bluetoothd-service-input services, ignoring them here
	{
		unsigned nr_bt_services = svc_bluetooth_HCI + svc_bluetooth_BTCOND + svc_bluetooth_OBEX;

		if(nr_bt_services == 3)
			tmp_status = 2;
		else if(nr_bt_services)
			tmp_status = 1;
		else
			tmp_status = 0;
		if(svc_bluetooth_status != tmp_status || svc_bluetooth_size != sc_bluetooth_size) svc_changes |= SVC_BLUETOOTH;
		svc_bluetooth_status = tmp_status;
		svc_bluetooth_size = sc_bluetooth_size;
	}

	// GPS
	{
		unsigned nr_gps_services = svc_gps_DRIVER + svc_gps_SUPL + svc_gps_CLOCKD + svc_gps_GUARD;
		unsigned nr_services = 1;

		if(file_exists("/etc/init.d/supllistenerd")) nr_services++; // optional service (A-GPS)
		if(file_exists("/etc/init.d/gpsclockd")) nr_services++; // optional service
		if(file_exists("/etc/init.d/gpsdriver_guard")) nr_services++; // optional service
		if(nr_gps_services == nr_services)
			tmp_status = 2;
		else if(nr_gps_services)
			tmp_status = 1;
		else
			tmp_status = 0;
		if(svc_gps_status != tmp_status || svc_gps_size != sc_gps_size) svc_changes |= SVC_GPS;
		svc_gps_status = tmp_status;
		svc_gps_size = sc_gps_size;
	}

	// WIFI
	{
		unsigned nr_wifi_services = svc_wifi_WLANCOND + svc_wifi_DNSMASQ + svc_wifi_CONNDLGS + svc_wifi_ICD + svc_wifi_NTPD;
		unsigned nr_services = 4;

		if(file_exists("/etc/init.d/openntpd")) nr_services++; // optional service (OpenNTPD)
		if(nr_wifi_services == nr_services)
			tmp_status = 2;
		else if(nr_wifi_services)
			tmp_status = 1;
		else
			tmp_status = 0;
		if(svc_wifi_status != tmp_status || svc_wifi_size != sc_wifi_size) svc_changes |= SVC_WIFI;
		svc_wifi_status = tmp_status;
		svc_wifi_size = sc_wifi_size;
	}

	// AUDIO
		// dsp_dld won't stop
		// esd won't stop with the default sound status bar applet
	{
		unsigned nr_audio_services = svc_audio_MULTIMEDIA + svc_audio_DSP + svc_audio_ENGINE + svc_audio_ESD + svc_audio_SERVER;

		if(nr_audio_services == 5)
			tmp_status = 2;
		else if(nr_audio_services > 2) // ignore esd and dsp_dld
			tmp_status = 1;
		else
			tmp_status = 0;
		if(svc_audio_status != tmp_status || svc_audio_size != sc_audio_size) svc_changes |= SVC_AUDIO;
		svc_audio_status = tmp_status;
		svc_audio_size = sc_audio_size;
	}

	// DESKTOP
	{
		tmp_status = (sc_desktop ? 2 : 0);
		if(svc_desktop_status != tmp_status || svc_desktop_size != sc_desktop_size) svc_changes |= SVC_DESKTOP;
		svc_desktop_status = tmp_status;
		svc_desktop_size = sc_desktop_size;
	}

	// SSH
	if(cfg_ssh_server == SSH_SERVER_OPENSSH){
		if(sc_openssh_installed){
			// nr_ssh_clients is set while scanning for services

			tmp_status = (sc_openssh ? 2 : 0);
			if(svc_ssh_status != tmp_status || svc_ssh_nr_clients != nr_ssh_clients || svc_ssh_size != sc_openssh_size) svc_changes |= SVC_SSH;
			svc_ssh_status = tmp_status;
			svc_ssh_nr_clients = nr_ssh_clients;
			svc_ssh_size = sc_openssh_size;
		}
	} else {
		if(sc_dropbear_installed){
			unsigned nr_ssh_services = (sc_dropbear ? 1 : 0);

			nr_ssh_clients = (sc_dropbear ? sc_dropbear-1 : 0);

			tmp_status = (nr_ssh_services ? 2 : 0);
			if(svc_ssh_status != tmp_status || svc_ssh_nr_clients != nr_ssh_clients || svc_ssh_size != sc_dropbear_size) svc_changes |= SVC_SSH;
			svc_ssh_status = tmp_status;
			svc_ssh_nr_clients = nr_ssh_clients;
			svc_ssh_size = sc_dropbear_size;
		}
	}

	// VPN
	if(svc_vpn_installed){
		tmp_status = (sc_vpn ? 2 : 0);
		if(svc_vpn_status != tmp_status || svc_vpn_size != sc_vpn_size) svc_changes |= SVC_VPN;
		svc_vpn_status = tmp_status;
		svc_vpn_size = sc_vpn_size;
	}

	last_svc_refresh = current_time;
}

//////////////////////////////////////////////////////////////////////////

void svc_init_bluetooth( e_service_action action, unsigned with_ticks ){
	// bluez-utils also handles the bluetoothd-service-audio and bluetoothd-service-input services

	switch(action){
	case SVC_STOP:
		if(svc_bluetooth_status){
			// turn off hardware before stopping services
			if(hw_bluetooth.enabled){
				hw_set_bluetooth(0);
				usleep(500000); // sleep 500ms to let bluez service process request
			}

			if(svc_bluetooth_OBEX) system("/etc/init.d/obexsrv stop");		// K20
			if(svc_bluetooth_BTCOND) system("/etc/init.d/btcond stop");		// K20
			if(svc_bluetooth_HCI) system("/etc/init.d/bluez-utils stop");	// K74

			sleep(1);
			svc_refresh();
		}
		break;
	case SVC_START:
		if(svc_bluetooth_status != 2){
			if(!svc_bluetooth_HCI) system("/etc/init.d/bluez-utils start");	// S56 - sends a "bt set on" notification
			if(!svc_bluetooth_BTCOND) system("/etc/init.d/btcond start");	// S57
			if(!svc_bluetooth_OBEX) system("/etc/init.d/obexsrv start");	// S70

			// takes a few seconds to start, wait up to 9 seconds
			int i;
			for(i = 0; i < 9; i++){
				if(with_ticks) draw_tick();
				sleep(1);
				svc_refresh();
				if(svc_bluetooth_status) break;
			}
		}
		break;
	case SVC_RESTART: error_write("svc_init_bluetooth(restart) -- not implemented"); break;
	}
}

void svc_init_gps( e_service_action action ){
	unsigned changed = 0;

	switch(action){
	case SVC_STOP:
		if(svc_gps_status){

// TODO: toggle hardware OFF

			if(svc_gps_GUARD) system("/etc/init.d/gpsdriver_guard stop");	// xxx - optional service
			if(svc_gps_SUPL) system("/etc/init.d/supllistenerd stop");		// K20 - optional service (A-GPS)
			if(svc_gps_CLOCKD) system("/etc/init.d/gpsclockd stop");		// xxx - optional service
			// libgpsbt														// K20 - doesn't use a process and therefore can't be stopped
			if(svc_gps_DRIVER) system("/etc/init.d/gpsdriver stop");		// K20
			changed = 1;
		}
		break;
	case SVC_START:
		if(svc_gps_status != 2){
			if(!svc_gps_DRIVER) system("/etc/init.d/gpsdriver start");														// S54
			system("/etc/init.d/libgpsbt start");																			// S54
			if(!svc_gps_SUPL && file_exists("/etc/init.d/supllistenerd")) system("/etc/init.d/supllistenerd start");		// S54 - optional service (A-GPS)
			if(!svc_gps_CLOCKD && file_exists("/etc/init.d/gpsclockd")) system("/etc/init.d/gpsclockd start");				// S99 - optional service
			if(!svc_gps_GUARD && file_exists("/etc/init.d/gpsdriver_guard")) system("/etc/init.d/gpsdriver_guard start");	// S99 - optional service
			changed = 1;
		}
		break;
	case SVC_RESTART: error_write("svc_init_gps(restart) -- not implemented"); break;
	}

	if(changed){
		sleep(1);
		svc_refresh();
	}
}

void svc_init_wifi( e_service_action action, unsigned with_ticks ){
	switch(action){
	case SVC_STOP:
		if(svc_wifi_status){
			// disconnect wifi before stopping services
			if(hw_wifi.enabled){
				hw_wifi_disconnect();
				usleep(500000); // sleep 500ms to let icd2 service process request
			}

			if(svc_wifi_NTPD) system("/etc/init.d/openntpd stop");								// K20 - optional service
			if(svc_wifi_CONNDLGS) system("/etc/osso-af-init/osso-connectivity-ui.sh stop");		// K20
			if(svc_wifi_ICD) system("/etc/init.d/icd2 stop");									// K15
			if(svc_wifi_WLANCOND) system("/etc/init.d/wlancond stop");							// K20
			// /etc/init.d/osso-ipv6 stop														// K20
			// /etc/init.d/wide-dhcpv6-client													// K40
			if(svc_wifi_DNSMASQ) system("/etc/init.d/dnsmasq stop");							// K85

			sleep(1);
			svc_refresh();
		}
		break;
	case SVC_START:
		if(svc_wifi_status != 2){
			if(!svc_wifi_DNSMASQ) system("/etc/init.d/dnsmasq start");										// S53
			// /etc/init.d/osso-ipv6 start																	// S54
			if(!svc_wifi_WLANCOND) system("/etc/init.d/wlancond start");									// S58
			// /etc/init.d/wide-dhcpv6-client start															// S58
			if(!svc_wifi_ICD) system("/etc/init.d/icd2 start");												// S59
			if(!svc_wifi_CONNDLGS) system("/etc/osso-af-init/osso-connectivity-ui.sh start");				// S60 - will only start when launched as root, sudo doesn't work
			if(!svc_wifi_NTPD && file_exists("/etc/init.d/openntpd")) system("/etc/init.d/openntpd start");	// S20 - optional service

			// takes a few seconds to start, wait up to 9 seconds
			int i;
			for(i = 0; i < 9; i++){
				if(with_ticks) draw_tick();
				sleep(1);
				svc_refresh();
				if(svc_wifi_status) break;
			}
		}
		break;
	case SVC_RESTART: error_write("svc_init_wifi(restart) -- not implemented"); break;
	}
}

void svc_init_audio( e_service_action action ){
	unsigned changed = 0;

	switch(action){
	case SVC_STOP:
		if(svc_audio_status){
			// /etc/init.d/metalayer-crawler0 stop											// K01
			if(svc_audio_ENGINE) system("/etc/init.d/mediaplayer-daemon stop");				// K01
			if(svc_audio_SERVER) system("/etc/osso-af-init/osso-media-server.sh stop");		// K20
//			if(svc_audio_ESD) system("/etc/init.d/esd stop");								// K26 - won't stop, sound applet restarts it
			if(svc_audio_MULTIMEDIA) system("/etc/init.d/multimediad stop");				// K75
//			if(svc_audio_DSP) system("/etc/init.d/dsp-init stop");							// K24 - won't stop
			changed = 1;
		}
		break;
	case SVC_START:
		if(svc_audio_status != 2){
			if(!svc_audio_DSP) system("/etc/init.d/dsp-init start");						// S24
			if(!svc_audio_MULTIMEDIA) system("/etc/init.d/multimediad start");				// S25
			if(!svc_audio_ESD) system("/etc/init.d/esd start");								// S26
			if(!svc_audio_SERVER) system("/etc/osso-af-init/osso-media-server.sh start");	// S60 will only start when launched as root, sudo doesn't work
			if(!svc_audio_ENGINE) system("/etc/init.d/mediaplayer-daemon start");			// S99
			// /etc/init.d/metalayer-crawler0 start											// S99
			changed = 1;
		}
		break;
	case SVC_RESTART: error_write("svc_init_audio(restart) -- not implemented"); break;
	}

	if(changed){
		sleep(1);
		svc_refresh();
	}
}

void svc_init_desktop( e_service_action action, unsigned with_ticks ){
	unsigned changed = 0;
	int i;

	switch(action){
	case SVC_STOP: if(svc_desktop_status){ system("/etc/init.d/hildon-desktop stop"); changed = 1; } break;
	case SVC_START: if(!svc_desktop_status){ system("/etc/init.d/hildon-desktop start"); changed = 1; } break;
	case SVC_RESTART: error_write("svc_init_desktop(restart) -- not implemented"); break;
	}

	if(changed){
		// takes a few seconds to start, wait up to 15 seconds
		for(i = 0; i < 15; i++){
			draw_tick();
			sleep(1);
			// wait 3 seconds before trying (after starting the service)
			if(i > 3 || action == SVC_STOP){
				svc_refresh();
				if(svc_desktop_status) break;
			}
		}
	}
}

void svc_init_ssh( e_service_action action, unsigned all_servers ){
	unsigned changed = 0;

	if(cfg_ssh_server == SSH_SERVER_NULL){
		error_write("svc_init_ssh() -- invalid ssh service");
		return;
	}

	switch(action){
	case SVC_STOP:
		if(all_servers){
			// svc_ssh_status only tracks current server so just start both
			system("/etc/init.d/ssh stop");
			system("/etc/init.d/dropbear stop");
			changed = 1;
		} else switch(cfg_ssh_server){
		case SSH_SERVER_OPENSSH: if(svc_ssh_status){ system("/etc/init.d/ssh stop"); changed = 1; } break;
		case SSH_SERVER_DROPBEAR: if(svc_ssh_status){ system("/etc/init.d/dropbear stop"); changed = 1; } break;
		default: break;
		}
		break;
	case SVC_START:
		if(all_servers){
			// svc_ssh_status only tracks current server so just start both
			system("/etc/init.d/ssh start");
			system("/etc/init.d/dropbear start");
			changed = 1;
		} else switch(cfg_ssh_server){
		case SSH_SERVER_OPENSSH: if(!svc_ssh_status){ system("/etc/init.d/ssh start"); changed = 1; } break;
		case SSH_SERVER_DROPBEAR: if(!svc_ssh_status){ system("/etc/init.d/dropbear start"); changed = 1; } break;
		default: break;
		}
		break;
	case SVC_RESTART: error_write("svc_init_ssh(restart) -- not implemented"); break;
	}

	if(changed){
		sleep(1);
		svc_refresh();
	}
}

void svc_init_vpn( e_service_action action ){
	unsigned changed = 0;
	switch(action){
	case SVC_STOP: if(svc_vpn_status){ system("/etc/init.d/openvpn stop"); changed = 1; } break;
	case SVC_START: if(!svc_vpn_status){ system("/etc/init.d/openvpn start"); changed = 1; } break;
	case SVC_RESTART: error_write("svc_init_vpn(restart) -- not implemented"); break;
	}

	if(changed){
		sleep(1);
		svc_refresh();
	}
}
