//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>
//
//	all dsme code ported from powerlaunch
//	Copyright (c) 2007-2008 Austin Che

#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "main.h"
#include "window.h"
#include "config.h"
#include "hardware.h"
#include "dbus.h"
 #include "dsme.h"

#define DSME_GET_DEVICELOCK_CODE 0x1200
#define DSME_IN_GET_DEVICELOCK_CODE 0x1201
//#define DSME_SET_DEVICELOCK_CODE 0x1202
#define DSME_IN_SET_DEVICELOCK_CODE 0x1203

#define DSME_DISPLAY_BLANKING_PAUSE 0x0201
#define DSME_DISPLAY_BLANKING_ALLOW 0x0202
//#define DSME_SET_DISPLAY_BLANK_TIMEOUT 0x0203
//#define DSME_SET_DISPLAY_DIM_TIMEOUT 0x0204
//#define DSME_SET_DISPLAY_STATE 0x0280
#define DSME_IN_DISPLAY_STATE 0x0285
#define DSME_SET_DISPLAY_BRIGHTNESS 0x0289

// soft power off doesn't appear to be any different than blanking the display

////////////////////////////////////////////////////////////////////////// GLOBALS

char *device_lock_code;
unsigned dsme_display_state_waiting;

//////////////////////////////////////////////////////////////////////////

static int dsme_fd;

static void dsme_write( unsigned type ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
	} message;
	message.len = sizeof(message);
	message.type = type;
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}

static void dsme_write_one_arg( unsigned type, unsigned arg ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
		unsigned arg;
	} message;
	message.len = sizeof(message);
	message.type = type;
	message.arg = arg;
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}

//////////////////////////////////////////////////////////////////////////

int dsme_init( ){
	struct sockaddr_un addr;

	dsme_fd = socket(PF_UNIX, SOCK_STREAM, 0);
	if(dsme_fd < 0){
		error_write("can't open dsme socket");
		return 0;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path, "/tmp/dsmesock");
	if(connect(dsme_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0){
		close(dsme_fd);
		error_write("can't connect to dsme socket");
		return 0;
	}

	device_lock_code = NULL;

	return dsme_fd;
}

void dsme_disconnect( ){
	#ifdef ARCH_armel
	dsme_allow_blanking(); // make sure blanking is enabled before exiting
	close(dsme_fd);
	#endif
}

//////////////////////////////////////////////////////////////////////////

#ifdef ARCH_armel
static int read_data( void *buffer, int len ){
	int rlen = read(dsme_fd, buffer, len);
	if(rlen != len){
		error_write("could not read %d bytes from dsme, got %d bytes", len, rlen);
		return -1;
	}
	return rlen;
}
#endif

static e_dsme_display_state display_state;

void dsme_read( ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
	} header;

	int rlen;
	if((rlen = read_data(&header, sizeof(header))) < 0){
		error_write("could not read dsme header");
		return;
	}

	char buffer[256];
	int data_size = header.len - rlen;
	if(data_size > (int)sizeof(buffer)){
		error_write("length(%d) in dsme header is too large", header.len);
		lseek(dsme_fd, 0, SEEK_END);
		return;
	}

	if((rlen = read_data(buffer, data_size)) < 0){
		error_write("could not read dsme data");
		return;
	}

	switch (header.type){
	case DSME_IN_GET_DEVICELOCK_CODE:
		{
			struct {
				char str[16];	// "lock_code"
				unsigned a1;	// 0x01
				unsigned a2;	// 0x10
				unsigned a3;	// 0x01
				char code[12];	// the device code
			} *message = (void *)buffer;

			if(rlen == sizeof(*message))
				if(!strcmp(message->str, "lock_code"))
					set_string(&device_lock_code, message->code, 10); // max of 10 digits
		}
		break;
	case DSME_IN_DISPLAY_STATE:
		if(rlen == 4){
			display_state = (e_dsme_display_state)(*(int *)buffer);
			unsigned visible = (display_state == DSME_DISPLAY_OFF ? 0 : 1);
			unsigned dimmed = (display_state == DSME_DISPLAY_DIM ? 1 : 0);

			if(!hw_screen.visible && visible){
				// screen was just turned on

				hw_screen.visible_time = current_time;
				hw_screen.brightness_mismatch = 1;
				if(original_screen_autolock){
					// restored scree_autolock setting
					conf_set_screen_autolock_value(original_screen_autolock-1);
					original_screen_autolock = 0;
				}
				if(original_screen_stayslit){
					// restore screen_stayslit setting
					conf_set_screen_stayslit_value(original_screen_stayslit-1);
					original_screen_stayslit = 0;
//					check_dimmed_state();
				}
				window_open_popup();

			} else if(hw_screen.visible && !visible){
				// screen was just turned off

				if(!hw_screen.locked){
					// screen is not locked
					if(!conf_get_screen_autolock_value()){
						// screen does not autolock
						if(!window.is_mapped)
							window_open_blank(); // screen is off but not locked, ASUI is not open -- open a blank window to catch the wakeup tap/key
						click.enabled = 0; // ignore first screen tap when unblanked
					}
				}
				window_close_popup();

			}
			hw_screen.visible = visible;
			// don't need to wakeup main loop for this, dbus unlock signal or X screen/key events will wake it

			if(hw_screen.dimmed != dimmed){
				hw_screen.brightness_mismatch = 1;
				hw_screen.dimmed = dimmed;
				hw_changes |= HW_BRIGHTNESS;
				dbus_send_wakeup(_WAKEUP_REFRESH_);
			}
		}
		break;
	default:
/*
		error_write("unhandled message from dsme of type 0x%x with %d bytes of data", header.type, rlen);
		int i;
		for(i = 0; i < rlen; i++){
			printf("%02x ", buffer[i]);
			if(i % 50 == 49) printf("\n");
		}
		if(rlen) printf("\n");
*/
		break;
	}
	#endif
}

////////////////////////////////////////////////////////////////////////// BLANKING PAUSE

void dsme_pause_blanking( ){
	dsme_write(DSME_DISPLAY_BLANKING_PAUSE);
}

void dsme_allow_blanking( ){
	dsme_write(DSME_DISPLAY_BLANKING_ALLOW);
}

// MCE's req_display_blanking_pause method somehow overrides DSME's allow blanking message
// the following code displays a notification when ASUI attempts to change the display state while MCE has blanking paused

static e_dsme_display_state expected_display_state;
static s_timer display_state_timer;

void dsme_expect_display_state( e_dsme_display_state state ){
	expected_display_state = state;
	dsme_display_state_waiting = 2;
	set_timer(&display_state_timer, 6.0); // 6 seconds
	dbus_send_wakeup(_WAKEUP_REFRESH_);
}

double dsme_check_display_state( ){
	if(is_timer_ready(&display_state_timer)){
		if(display_state == expected_display_state){
			// got the expected state, end timer
			dsme_display_state_waiting = 0;
			return 0.0;
		} else if(dsme_display_state_waiting == 2){
			// MCE did not set the expected state yet, blanking is most likely paused
			// send first notification and set a second timer
			notify_blanking_pause1();
			dsme_display_state_waiting--;
			set_timer(&display_state_timer, 3.0); // 3 seconds
		} else {
			// send second notification and end timer
			notify_blanking_pause2();
			dsme_display_state_waiting--;
			return 0.0;
		}
	}
	return display_state_timer.remaining;
}

////////////////////////////////////////////////////////////////////////// BRIGHTNESS

void dsme_set_brightness( unsigned level ){
	dsme_write_one_arg(DSME_SET_DISPLAY_BRIGHTNESS, (level * 2) + 1);
}

////////////////////////////////////////////////////////////////////////// DEVICE LOCK CODE

void dsme_query_devicelock_code( ){
	#ifdef ARCH_armel
	struct {
		unsigned len;
		unsigned type;
		char data[16];
		unsigned a1;
		unsigned a2; // this changes, not sure what this is
	} message;
	message.len = sizeof(message);
	message.type = DSME_GET_DEVICELOCK_CODE;
	strncpy(message.data, "lock_code", sizeof(message.data));
	message.a1 = 0x01;
	message.a2 = 0x00027ec8;

	// clear code
	set_string(&device_lock_code, NULL, 0);

	// this should cause dsme to send us back the lock code
	send(dsme_fd, (void *)&message, sizeof(message), 0);
	#endif
}
