//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <brand.huntsman@gmail.com>

#include "main.h"
#include "services.h"

#define ASUI_NAME "Advanced SystemUI"
#define SYSTEMUI_NAME "SystemUI"

//////////////////////////////////////////////////////////////////////////

typedef struct s_service_def {
	const char *init_name;
	unsigned allow_boot;
	unsigned allow_startstop;
	unsigned allow_restart;
	unsigned s0,s1,s2,s3,s4,s5,s6;
	unsigned k0,k1,k2,k3,k4,k5,k6;
	const char *description;
	const char *proc_name;
} s_service_def;
#define XX 0
s_service_def service_defs[] = {//	B S R	 0  1  2  3  4  5  6	 0  1  2  3  4  5  6
	{"advanced-systemui",			0,0,0,	XX,XX,61,61,61,61,XX,	45,45,XX,XX,XX,XX,45, ASUI_NAME,								"/usr/bin/advanced-systemui"},
	{"advanced-systemui-early",		0,0,0,	XX,XX,26,26,26,26,XX,	74,74,XX,XX,XX,XX,74, ASUI_NAME " (early)",						NULL},
	{"af-base-apps",				0,0,0,	XX,XX,60,60,60,XX,XX,	20,20,XX,XX,XX,20,20, "Application Framework: base apps",		NULL},								// NO RESTART -- keyboard, clipboard, media-server and connectivity-ui
	{"af-services",					0,0,0,	XX,XX,22,22,22,22,XX,	22,22,XX,XX,XX,XX,22, "Application Framework: services",		NULL},								// NO RESTART -- 
	{"af-startup",					0,0,0,	XX,XX,50,50,50,XX,XX,	21,21,XX,XX,XX,XX,21, "Application Framework: startup",			NULL},								// NO RESTART -- stops fb-progress.sh, starts startup-greeting.sh
	{"alarmd",						1,1,1,	XX,XX,99,99,99,99,XX,	01,01,XX,XX,XX,XX,01, "Alarm Dispatcher",						"/usr/bin/alarmd"},
	{"bluez-utils",					1,1,1,	XX,XX,56,56,56,XX,XX,	74,74,XX,XX,XX,74,74, "BLUETOOTH -- HCI Daemon",				"/usr/sbin/hcid"},
	{"btcond",						1,1,1,	XX,XX,57,57,57,XX,XX,	20,20,XX,XX,XX,15,20, "BLUETOOTH -- Connection Daemon",			"/usr/bin/btcond"},
	{"dbus",						0,1,1,	XX,XX,20,20,20,XX,XX,	20,20,XX,XX,XX,XX,20, "D-Bus",									"/usr/bin/dbus-daemon"},			// doesn't start in rc5?
	{"dnsmasq",						1,1,1,	XX,XX,53,53,53,53,XX,	85,85,XX,XX,XX,XX,85, "WIFI -- DNS forwarder and DHCP server",	"/usr/sbin/dnsmasq"},
	{"dropbear",					1,1,1,	XX,XX,20,20,20,20,XX,	20,20,XX,XX,XX,XX,20, "Dropbear SSH Server (optional)",			"/usr/sbin/dropbear"},				// optional - assuming it has restart
	{"dsp-init",					1,1,0,	XX,XX,24,24,24,24,XX,	24,24,XX,XX,XX,XX,24, "AUDIO -- DSP",							"/usr/sbin/dsp_dld"},				// won't stop
	{"esd",							1,1,1,	XX,XX,26,26,26,26,XX,	26,26,XX,XX,XX,XX,26, "AUDIO -- Enlightenment Sound Daemon",	"/usr/bin/esd"},
	{"fb-progress.sh",				0,0,0,	XX,XX,12,XX,XX,XX,XX,	XX,XX,XX,XX,XX,20,XX, NULL,										NULL},
	{"glibc.sh",					0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"gpsclockd",					1,1,0,	XX,XX,99,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, "GPS -- Clock Daemon (optional)",			"/usr/sbin/gps-clockd"},			// optional - needs K15 links in 0156
	{"gpsdriver",					1,1,1,	XX,XX,54,XX,XX,XX,XX,	20,20,XX,XX,XX,20,20, "GPS -- driver",							"/usr/sbin/gpsdriver"},
	{"gpsdriver_guard",				1,1,1,	XX,XX,99,99,99,99,XX,	XX,XX,XX,XX,XX,XX,XX, "GPS -- driver guard (optional)",			"(gpsdriver_guard)"},				// optional - needs K20 links in 0156
	{"hal",							0,1,1,	XX,XX,20,20,20,20,XX,	16,16,XX,XX,XX,XX,16, "Hardware Abstraction Layer",				"/usr/sbin/hald"},
	{"hildon-application-manager",	0,0,0,	XX,XX,52,52,52,XX,XX,	20,20,XX,XX,XX,20,20, "Hildon Application Manager",				NULL},
	{"hildon-desktop",				1,1,0,	XX,XX,51,51,51,XX,XX,	20,20,XX,XX,XX,20,20, "Hildon Desktop",							"/usr/bin/hildon-desktop"},
	{"hulda",						0,1,1,	XX,XX,99,99,99,XX,XX,	20,20,XX,XX,XX,20,20, "Kernel Events Proxy: hulda",				"/usr/sbin/hulda"},
	{"icd2",						1,1,1,	XX,XX,59,59,59,XX,XX,	15,15,XX,XX,XX,15,15, "WIFI -- Internet Connectivity Daemon 2",	"/usr/sbin/icd2"},					// rc5 networking
	{"ke-recv",						0,1,1,	XX,XX,30,30,30,XX,XX,	30,30,XX,XX,XX,30,30, "Kernel Events Proxy: ke-recv",			"/usr/sbin/ke-recv"},
	{"kernel-tunables",				0,0,1,	XX,XX,99,99,99,99,XX,	XX,XX,XX,XX,XX,XX,XX, "Diablo Turbo Kernel Tunables (optional)",NULL},								// optional
	{"libgpsbt",					1,1,0,	XX,XX,54,XX,XX,XX,XX,	20,20,XX,XX,XX,20,20, "GPS -- Bluetooth API",					NULL},								// doesn't stop
	{"maemo-launcher",				0,1,1,	XX,XX,45,45,45,XX,XX,	45,45,XX,XX,XX,45,45, "Maemo Launcher",							"/usr/bin/maemo-launcher"},
	{"mce",							0,1,1,	XX,XX,21,21,21,21,XX,	79,79,XX,XX,XX,XX,79, "Machine Control Entity",					"/sbin/mce"},
	{"mediaplayer-daemon",			1,1,1,	XX,XX,99,99,99,XX,XX,	01,01,XX,XX,XX,01,01, "AUDIO -- Media Player Daemon",			"/usr/bin/mediaplayer-engine"},
	{"metalayer-crawler0",			1,1,1,	XX,XX,99,99,99,99,XX,	01,01,XX,XX,XX,XX,01, "Metalayer Crawler",						"/usr/bin/metalayer-crawler"},
	{"minireboot",					0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"minishutdown",				0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"multimediad",					1,1,1,	XX,XX,25,25,25,25,XX,	75,75,XX,XX,XX,XX,75, "AUDIO -- Multimedia Daemon",				"/usr/sbin/multimediad"},
	{"networking",					0,1,1,	35,XX,XX,XX,XX,XX,35,	XX,XX,XX,XX,XX,XX,XX, NULL, 									NULL},
	{"obexsrv",						1,1,1,	XX,XX,70,70,70,XX,XX,	20,20,XX,XX,XX,20,20, "BLUETOOTH -- File Sharing",				"/usr/bin/obexsrv"},
	{"openntpd",					1,1,1,	XX,XX,20,20,20,20,XX,	20,20,XX,XX,XX,XX,20, "WIFI -- OpenNTP (optional)",				"/usr/sbin/ntpd"},					// optional
	{"openvpn",						1,1,1,	XX,XX,16,16,16,16,XX,	80,80,XX,XX,XX,XX,80, "OpenVPN (optional)",						"/usr/sbin/openvpn"},				// optional - assuming it has restart
	{"osso-applet-display",			0,0,0,	XX,XX,20,XX,XX,20,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"osso-ipv6",					1,1,1,	XX,XX,54,XX,XX,XX,XX,	20,20,XX,XX,XX,20,20, "WIFI -- IPv6",							NULL},								// rc5 networking
	{"osso-systemui",				0,0,0,	XX,XX,61,61,61,61,XX,	45,45,XX,XX,XX,XX,45, SYSTEMUI_NAME,							"/usr/bin/systemui"},
	{"osso-systemui-early",			0,0,0,	XX,XX,26,26,26,26,XX,	74,74,XX,XX,XX,XX,74, SYSTEMUI_NAME " (early)",					NULL},
	{"product-code",				0,0,0,	XX,XX,41,41,41,41,XX,	85,85,XX,XX,XX,XX,85, NULL,										NULL},
	{"ramzswap",					1,1,1,	XX,XX,52,52,52,52,XX,	XX,XX,XX,XX,XX,XX,XX, "ramzswap (optional)",					NULL},								// optional
	{"rc",							0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"rcS",							0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"ssh",							1,1,1,	XX,XX,55,55,55,55,XX,	20,20,XX,XX,XX,XX,20, "OpenSSH Server (optional)",				"/usr/sbin/sshd"},					// optional
	{"supllistenerd",				1,1,1,	XX,XX,54,XX,XX,XX,XX,	20,20,XX,XX,XX,20,20, "GPS -- A-GPS (optional)",				"/usr/sbin/supllistenerd"},			// optional
	{"tablet-browser-daemon",		1,1,1,	XX,XX,99,99,99,XX,XX,	15,15,XX,XX,XX,15,15, "MicroB Browser Daemon",					"/usr/sbin/browserd"},
	{"telescope-svc",				1,1,1,	XX,XX,99,99,99,99,XX,	99,99,XX,XX,XX,XX,99, "Telescope (optional)",					"/usr/bin/telescope"},				// optional
	{"ttyusb0",						0,0,0,	XX,XX,99,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL},
	{"wide-dhcpv6-client",			1,0,0,	XX,XX,58,58,58,58,XX,	40,40,XX,XX,XX,XX,40, "WIFI -- WIDE DHCPv6 client",				"/usr/sbin/dhcp6c"},
	{"wlancond",					1,1,1,	XX,XX,58,58,58,XX,XX,	20,20,XX,XX,XX,20,20, "WIFI -- WLAN Connection Daemon",			"/usr/sbin/wlancond"},				// rc5 networking
	{"x-server",					0,0,0,	XX,XX,21,21,21,21,XX,	23,23,XX,XX,XX,XX,23, "X11 Server",								"/usr/bin/Xomap"},
	{"zzinitdone",					0,0,0,	XX,XX,99,99,99,99,XX,	00,00,00,00,00,00,00, NULL,										NULL},
	{NULL,							0,0,0,	XX,XX,XX,XX,XX,XX,XX,	XX,XX,XX,XX,XX,XX,XX, NULL,										NULL}
};
typedef struct s_service {
	char *real_name;
	char *init_name;
	enum {IS_UNKNOWN, IS_STOPPED, IS_STARTED} running;
	unsigned boot;
	unsigned allow_boot;
	unsigned allow_startstop;
	unsigned allow_restart;
	s_service_def *service_def;
	const char *proc_name;
	GtkWidget *widget_checkbox, *widget_start, *widget_stop, *widget_restart;
	struct s_service *next;
} s_service;

static s_service *services, *services_tail;

//////////////////////////////////////////////////////////////////////////

static unsigned asui_running, systemui_running;
static GtkWidget *startstop_asui_button, *restart_asui_button, *startstop_systemui_button, *restart_systemui_button;

static void update_running_buttons( ){
	// enable restart button if service is running
	gtk_widget_set_sensitive(restart_asui_button, asui_running);
	gtk_widget_set_sensitive(restart_systemui_button, systemui_running);
	// set label on start/stop buttons
	gtk_button_set_label(GTK_BUTTON(startstop_asui_button), (asui_running ? "Stop" : "Start"));
	gtk_button_set_label(GTK_BUTTON(startstop_systemui_button), (systemui_running ? "Stop" : "Start"));
	// disable the other service start/stop button
	gtk_widget_set_sensitive(startstop_asui_button, !systemui_running);
	gtk_widget_set_sensitive(startstop_systemui_button, !asui_running);
}

static void update_startstop_buttons( s_service *service ){
	// start and stop buttons
	if(service->proc_name && service->allow_startstop)
		gtk_button_set_label(GTK_BUTTON(service->widget_start), (service->running == IS_STARTED ? "Stop" : "Start"));
	else if(service->allow_startstop){
		// do nothing
	} else if(service->running != IS_UNKNOWN)
		gtk_label_set_text(GTK_LABEL(service->widget_start), (service->running == IS_STARTED ? "r u n n i n g" : "s t o p p e d"));

	// restart button
	if(service->allow_restart)
		gtk_widget_set_sensitive(service->widget_restart, (service->running == IS_STARTED || service->proc_name == NULL ? TRUE : FALSE));
}

static void update_service_states( ){
	s_service *service;
	// reset running state for all services with known process names
	for(service = services; service != NULL; service = service->next)
		if(service->proc_name) service->running = IS_STOPPED;
	asui_running = systemui_running = 0;

	// query running services
	DIR *proc;
	if((proc = opendir("/proc")) != NULL){
		struct dirent *file;

		while((file = readdir(proc))){
			if(file->d_name[0] > '0' && file->d_name[0] <= '9'){
				char filepath[30], command[PATH_MAX];
				unsigned rss = 0;
				FILE *p;
				snprintf(filepath, sizeof(filepath), "/proc/%s/stat", file->d_name);
				if((p = fopen(filepath, "r")) != NULL){
					fscanf(p, "%*d %s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %*u %*u %*d %*d %*d %*d %*d %*d %*u %*u %u", command, &rss);
					fclose(p);
				}
				if(rss){
					char cmdline[400];
					snprintf(filepath, sizeof(filepath), "/proc/%s/cmdline", file->d_name);
					if((p = fopen(filepath, "r")) != NULL){
						unsigned is_shell;
						fgets(cmdline, sizeof(cmdline), p);
						fclose(p);

						if(!strcmp(cmdline, "/bin/sh")) is_shell = 1; else is_shell = 0;

						for(service = services; service != NULL; service = service->next){
							const char *proc_name = service->proc_name;

							if(proc_name){
								if(!strcmp((is_shell ? command : cmdline), proc_name)){
									service->running = IS_STARTED;
									if(!strcmp(proc_name, "/usr/bin/advanced-systemui")) asui_running = 1;
									else if(!strcmp(proc_name, "/usr/bin/systemui")) systemui_running = 1;
									break;
								}
							}
						}
					}
				}
			}
		}
		closedir(proc);
	}

	if(startstop_asui_button){
		update_running_buttons();
		for(service = services; service != NULL; service = service->next)
			update_startstop_buttons(service);
	}
}

static void click_startstop_asui_button_callback( GtkWidget *widget, gpointer data ){
	if(asui_running)
		system("/etc/init.d/advanced-systemui stop");
	else
		system("/etc/init.d/advanced-systemui start");
	asui_running = !asui_running;
	update_running_buttons();
}

static void click_restart_asui_button_callback( GtkWidget *widget, gpointer data ){
	system("/etc/init.d/advanced-systemui restart");
}

static void click_startstop_systemui_button_callback( GtkWidget *widget, gpointer data ){
	if(systemui_running)
		system("/etc/init.d/osso-systemui stop");
	else
		system("/etc/init.d/osso-systemui start");
	systemui_running = !systemui_running;
	update_running_buttons();
}

static void click_restart_systemui_button_callback( GtkWidget *widget, gpointer data ){
	system("/etc/init.d/osso-systemui restart");
}

static unsigned sort_services( ){
	s_service *prev = NULL, *service = services;
	unsigned status = 1;

	while(service != NULL){
		if(service->next == NULL) return status;
		if(strcasecmp(service->real_name, service->next->real_name) > 0){
			s_service *next = service->next;

			// swap elements
			if(service == services) services = next;
			else prev->next = next;
			service->next = next->next;
			next->next = service;
			status = 0; // list changed
			prev = next;
		} else {
			prev = service;
			service = service->next;
		}
	}
	return status;
}


//////////////////////////////////////////////////////////////////////////

static unsigned check_etc_rcd( const char *path, const char *name ){
	DIR *rcd;
	if((rcd = opendir(path)) != NULL){
		struct dirent *file;
		while((file = readdir(rcd))){
			if(!strcmp(name, &file->d_name[3]))
				if(file->d_name[0] == 'S'){ closedir(rcd); return 1; }
		}
		closedir(rcd);
	}
	return 0;
}

static unsigned is_started_at_boot( const char *name ){
	if(check_etc_rcd("/etc/rc0.d", name)) return 1;
	if(check_etc_rcd("/etc/rc1.d", name)) return 1;
	if(check_etc_rcd("/etc/rc2.d", name)) return 1;
	if(check_etc_rcd("/etc/rc3.d", name)) return 1;
	if(check_etc_rcd("/etc/rc4.d", name)) return 1;
	if(check_etc_rcd("/etc/rc5.d", name)) return 1;
	if(check_etc_rcd("/etc/rc6.d", name)) return 1;
	return 0;
}

static unsigned asui_boot, systemui_boot, in_boot_callback;
static GtkWidget *boot_label, *boot_asui_button, *boot_systemui_button;
static char *boot_message_both = "Both " ASUI_NAME " and " SYSTEMUI_NAME " are started at boot. This will cause problems and you really should click the one you don't want to start.";
static char *boot_message_none = "Neither " ASUI_NAME " nor " SYSTEMUI_NAME " are started at boot. Click one to start it at boot.";
static char *boot_message_one = "The selected service will be started at boot.";
static s_service *asui_service, *asui_early_service, *systemui_service, *systemui_early_service;

static void update_boot_checkboxes( ){
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(asui_service->widget_checkbox), asui_boot);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(asui_early_service->widget_checkbox), asui_boot);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(systemui_service->widget_checkbox), systemui_boot);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(systemui_early_service->widget_checkbox), systemui_boot);
}

static void click_boot_asui_button_callback( GtkWidget *widget, gpointer data ){
	if(in_boot_callback) return;

	gtk_widget_set_sensitive(boot_asui_button, FALSE);
	gtk_widget_set_sensitive(boot_systemui_button, FALSE);
//	gtk_button_set_label(GTK_BUTTON(boot_asui_button), ASUI_NAME "...");
	in_boot_callback = 1;
	if(asui_boot && systemui_boot){
		// both are set, unset asui
		system("rm -f /etc/rc?.d/S61advanced-systemui /etc/rc?.d/S26advanced-systemui-early");
		asui_boot = 0;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(boot_asui_button), FALSE);
	} else if(!asui_boot && !systemui_boot){
		// both are unset, add asui
		system("update-rc.d -f advanced-systemui remove >/dev/null ; update-rc.d -f advanced-systemui-early remove >/dev/null ; update-rc.d advanced-systemui defaults 61 45 >/dev/null ; update-rc.d advanced-systemui-early defaults 26 74 >/dev/null");
		asui_boot = 1;
	} else if(systemui_boot){
		// remove systemui, add asui
		system("rm -f /etc/rc?.d/S61osso-systemui /etc/rc?.d/S26osso-systemui-early ; "
				"update-rc.d -f advanced-systemui remove >/dev/null ; update-rc.d -f advanced-systemui-early remove >/dev/null ; update-rc.d advanced-systemui defaults 61 45 >/dev/null ; update-rc.d advanced-systemui-early defaults 26 74 >/dev/null");
		systemui_boot = 0;
		asui_boot = 1;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(boot_systemui_button), FALSE);
	} else {
		// do nothing - keep button pressed in
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
	}
	update_boot_checkboxes();
	in_boot_callback = 0;
	gtk_widget_set_sensitive(boot_asui_button, TRUE);
	gtk_widget_set_sensitive(boot_systemui_button, TRUE);
//	gtk_button_set_label(GTK_BUTTON(boot_asui_button), ASUI_NAME);
	gtk_label_set_text(GTK_LABEL(boot_label), boot_message_one);
}

static void click_boot_systemui_button_callback( GtkWidget *widget, gpointer data ){
	if(in_boot_callback) return;

	gtk_widget_set_sensitive(boot_asui_button, FALSE);
	gtk_widget_set_sensitive(boot_systemui_button, FALSE);
//	gtk_button_set_label(GTK_BUTTON(boot_systemui_button), SYSTEMUI_NAME "...");
	in_boot_callback = 1;
	if(asui_boot && systemui_boot){
		// both are set, unset systemui
		system("rm -f /etc/rc?.d/S61osso-systemui /etc/rc?.d/S26osso-systemui-early");
		systemui_boot = 0;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(boot_systemui_button), FALSE);
	} else if(!asui_boot && !systemui_boot){
		// both are unset, add systemui
		system("update-rc.d -f osso-systemui remove >/dev/null ; update-rc.d -f osso-systemui-early remove >/dev/null ; update-rc.d osso-systemui defaults 61 45 >/dev/null ; update-rc.d osso-systemui-early defaults 26 74 >/dev/null");
		systemui_boot = 1;
	} else if(asui_boot){
		// remove asui, add systemui
		system("rm -f /etc/rc?.d/S61advanced-systemui /etc/rc?.d/S26advanced-systemui-early ; "
				"update-rc.d -f osso-systemui remove >/dev/null ; update-rc.d -f osso-systemui-early remove >/dev/null ; update-rc.d osso-systemui defaults 61 45 >/dev/null ; update-rc.d osso-systemui-early defaults 26 74 >/dev/null");
		asui_boot = 0;
		systemui_boot = 1;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(boot_asui_button), FALSE);
	} else {
		// do nothing - keep button pressed in
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
	}
	update_boot_checkboxes();
	in_boot_callback = 0;
	gtk_widget_set_sensitive(boot_asui_button, TRUE);
	gtk_widget_set_sensitive(boot_systemui_button, TRUE);
//	gtk_button_set_label(GTK_BUTTON(boot_systemui_button), SYSTEMUI_NAME);
	gtk_label_set_text(GTK_LABEL(boot_label), boot_message_one);
}

//////////////////////////////////////////////////////////////////////////

unsigned network_enabled;

static void toggle_network_callback( GtkWidget *widget, gpointer data ){
	if(network_enabled){
		// disable rc5 networking
		system("ln -sf /etc/init.d/wlancond /etc/rc5.d/K20wlancond");
		system("ln -sf /etc/init.d/icd2 /etc/rc5.d/K15icd2");
		system("ln -sf /etc/init.d/osso-ipv6 /etc/rc5.d/K20osso-ipv6");
	} else {
		// enable rc5 networking
		system("rm -f /etc/rc5.d/K20wlancond /etc/rc5.d/K15icd2 /etc/rc5.d/K20osso-ipv6");
	}
	network_enabled = !network_enabled;
	gtk_button_set_label(GTK_BUTTON(widget), (network_enabled ? "RC5 networking is ENABLED" : "RC5 networking is DISABLED"));
}

//////////////////////////////////////////////////////////////////////////

static void click_boot_checkbox_callback( GtkToggleButton *widget, gpointer data ){
	s_service *service = (s_service *)data;
	if(gtk_toggle_button_get_active(widget) == TRUE){
		// enable service at boot
		if(service->service_def){
			s_service_def *def = service->service_def;
			unsigned *start = &def->s0;
			char buffer[400];
			unsigned i;

			for(i = 0; i < 7; i++){
				unsigned order = start[i];
				if(order){
					if(order >= 10)
						snprintf(buffer, sizeof(buffer), "ln -sf /etc/init.d/%s /etc/rc%u.d/S%u%s", service->init_name, i, order, service->init_name);
					else
						snprintf(buffer, sizeof(buffer), "ln -sf /etc/init.d/%s /etc/rc%u.d/S0%u%s", service->init_name, i, order, service->init_name);
					system(buffer);
				}
			}
			service->boot = 1;
		} else {

// TODO: restore start links

		}
	} else {
		// disable service at boot
		if(service->service_def){
			char buffer[200];
			snprintf(buffer, sizeof(buffer), "rm -f /etc/rc?.d/S??%s", service->init_name);
			system(buffer);
			service->boot = 0;
		} else {

// TODO: the above code will be common but wrapped in the IF for safety until link saving is ready

// TODO: save start orders for the service (/apps/asui_services/filename = list of 7 ints)

		}
	}
}
static GtkWidget *new_boot_checkbox( GtkWidget *parent, s_service *service ){
	GtkWidget *checkbox = gtk_check_button_new_with_label(service->real_name);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), service->boot);
	gtk_signal_connect(GTK_OBJECT(checkbox), "toggled", GTK_SIGNAL_FUNC(click_boot_checkbox_callback), (void *)service);
	gtk_widget_set_sensitive(checkbox, service->allow_boot);
	gtk_box_pack_start(GTK_BOX(parent), checkbox, FALSE, FALSE, 10);
	gtk_widget_show(checkbox);
	return checkbox;
}

static void click_startstop_service_callback( GtkWidget *widget, gpointer data ){
	s_service *service = (s_service *)data;
	char buffer[200];
	snprintf(buffer, sizeof(buffer), "/etc/init.d/%s %s", service->init_name, (service->running == IS_STARTED ? "stop" : "start"));
	system(buffer);
	// let auto-refresh update buttons
}
static void click_start_service_callback( GtkWidget *widget, gpointer data ){
	s_service *service = (s_service *)data;
	char buffer[200];
	snprintf(buffer, sizeof(buffer), "/etc/init.d/%s start", service->init_name);
	system(buffer);
	// let auto-refresh update buttons
}
static void click_stop_service_callback( GtkWidget *widget, gpointer data ){
	s_service *service = (s_service *)data;
	char buffer[200];
	snprintf(buffer, sizeof(buffer), "/etc/init.d/%s stop", service->init_name);
	system(buffer);
	// let auto-refresh update buttons
}
static void click_restart_service_callback( GtkWidget *widget, gpointer data ){
	s_service *service = (s_service *)data;
	char buffer[200];
	snprintf(buffer, sizeof(buffer), "/etc/init.d/%s restart", service->init_name);
	system(buffer);
	// auto-refresh will update buttons if something prevents a full restart
}

//////////////////////////////////////////////////////////////////////////

static unsigned service_refresh_timer;

static gboolean auto_refresh_service_states( gpointer data ){
	update_service_states();
	return TRUE;
}

static void services_cleanup_handler( ){
	g_source_remove(service_refresh_timer);

	s_service *service;
	for(service = services; service != NULL;){
		s_service *next = service->next;
		if(service->service_def == NULL) free(service->init_name);
		free(service);
		service = next;
	}
	startstop_asui_button = NULL;
}

//////////////////////////////////////////////////////////////////////////

void services_click_callback( GtkWidget *widget, gpointer data ){
	new_section_vbox("Services"); // add setting widgets to section_content_box

	GtkWidget *hbox, *label;

	// initialize service list

	DIR *initd;
	services = services_tail = NULL;
	if((initd = opendir("/etc/init.d")) != NULL){
		struct dirent *file;
		while((file = readdir(initd))){
			if(file->d_type == DT_REG){
				int i;
				for(i = 0; service_defs[i].init_name != NULL; i++)
					if(!strcmp(service_defs[i].init_name, file->d_name)) break;
				s_service *service = malloc(sizeof(s_service));
				if(service == NULL) exit(EXIT_FAILURE);
				if(service_defs[i].init_name == NULL){
					// unknown init file
					service->init_name = malloc(strlen(file->d_name)+strlen(" (OPTIONAL)")+1);
					if(service->init_name == NULL) exit(EXIT_FAILURE);
					sprintf(service->init_name, "%s (OPTIONAL)", file->d_name);
//					service->allow_boot = 1;
service->allow_boot = 0;
// TODO: enable boot checkbox when SK orders can be saved, disable for now
					service->allow_startstop = 1; // assume it can be start/stopped
					service->allow_restart = 1; // assume it can be restarted
					service->service_def = NULL;
					service->proc_name = NULL;
					service->real_name = service->init_name;
				} else {
					// known init file
					service->init_name = (char *)service_defs[i].init_name;
					service->allow_boot = service_defs[i].allow_boot;
					service->allow_startstop = service_defs[i].allow_startstop;
					service->allow_restart = service_defs[i].allow_restart;
					service->service_def = &service_defs[i];
					service->proc_name = service_defs[i].proc_name;
					if(service->service_def->description)
						service->real_name = (char *)service->service_def->description;
					else
						service->real_name = service->init_name;

					if(!strcmp(service->init_name, "advanced-systemui")) asui_service = service;
					else if(!strcmp(service->init_name, "advanced-systemui-early")) asui_early_service = service;
					else if(!strcmp(service->init_name, "osso-systemui")) systemui_service = service;
					else if(!strcmp(service->init_name, "osso-systemui-early")) systemui_early_service = service;
				}
				service->running = IS_UNKNOWN;
				service->boot = is_started_at_boot(service->init_name);
				service->next = NULL;
				if(services == NULL)
					services = service;
				else
					services_tail->next = service;
				services_tail = service;
			}
		}
		closedir(initd);
	}
	section_cleanup_handler = services_cleanup_handler;
	while(!sort_services());

	asui_running = systemui_running = 0;
	update_service_states();

	////////////////////

	// start, stop, restart buttons for ASUI and SystemUI

	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(section_content_box), hbox, FALSE, FALSE, 10);
	gtk_widget_show(hbox);
	label = gtk_label_new(ASUI_NAME);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 20);
	gtk_widget_show(label);
	startstop_asui_button = new_button(hbox, "", click_startstop_asui_button_callback, NULL);
	restart_asui_button = new_button(hbox, "Restart", click_restart_asui_button_callback, NULL);

	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(section_content_box), hbox, FALSE, FALSE, 10);
	gtk_widget_show(hbox);
	label = gtk_label_new(SYSTEMUI_NAME);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 20);
	gtk_widget_show(label);
	startstop_systemui_button = new_button(hbox, "", click_startstop_systemui_button_callback, NULL);
	restart_systemui_button = new_button(hbox, "Restart", click_restart_systemui_button_callback, NULL);

	update_running_buttons();

	new_separator();

	////////////////////

	// start at boot enable/disable buttons for ASUI and SystemUI

	asui_boot = is_started_at_boot("advanced-systemui") && is_started_at_boot("advanced-systemui-early");
	systemui_boot = is_started_at_boot("osso-systemui") && is_started_at_boot("osso-systemui-early");
	in_boot_callback = 0;

	if(asui_boot && systemui_boot)
		boot_label = new_message(boot_message_both);
	else if(!asui_boot && !systemui_boot)
		boot_label = new_message(boot_message_none);
	else
		boot_label = new_message(boot_message_one);

	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(section_content_box), hbox, FALSE, FALSE, 10);
	gtk_widget_show(hbox);
	boot_asui_button = new_toggle_button(hbox, ASUI_NAME, click_boot_asui_button_callback, (asui_boot ? TRUE : FALSE), TRUE,TRUE);
	boot_systemui_button = new_toggle_button(hbox, SYSTEMUI_NAME, click_boot_systemui_button_callback, (systemui_boot ? TRUE : FALSE), TRUE,TRUE);

	new_separator();

	////////////////////

	new_message("Some networking services are stopped when entering the shutdown charging runlevel. This feature will keep them enabled allowing SSH to continue working.");

	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(section_content_box), hbox, FALSE, FALSE, 10);
	gtk_widget_show(hbox);

	network_enabled = !file_exists("/etc/rc5.d/K20wlancond") && !file_exists("/etc/rc5.d/K15icd2") && !file_exists("/etc/rc5.d/K20osso-ipv6");
	new_toggle_button(hbox, (network_enabled ? "RC5 networking is ENABLED" : "RC5 networking is DISABLED"), toggle_network_callback, network_enabled, TRUE,FALSE);

	new_separator();

	////////////////////

	// start, stop, restart and start at boot enable/disable buttons for all services

	new_message("Checkboxes control the starting of services at boot."
				" Running state can't be determined for services with both start and stop buttons, have fun."
				" Services are refreshed every 5 seconds and tapped buttons will update during the next refresh."
				" Don't play around with these unless you know what you're doing!!!");

	s_service *service;
	for(service = services; service != NULL; service = service->next){
		// hbox
		hbox = gtk_hbox_new(FALSE, 10);
		gtk_box_pack_start(GTK_BOX(section_content_box), hbox, FALSE, FALSE, 10);
		gtk_widget_show(hbox);
		// boot checkbox and init name or description
		service->widget_checkbox = new_boot_checkbox(hbox, service);
		// restart button
		if(service->allow_restart){
			service->widget_restart = new_button_right(hbox, "Restart", click_restart_service_callback, (void *)service);
			gtk_widget_set_sensitive(service->widget_restart, (service->running == IS_STARTED || service->proc_name == NULL ? TRUE : FALSE));
		}
		// start, stop, state buttons
		if(service->proc_name && service->allow_startstop){
			// show start or stop button, depending on state
			service->widget_start = new_button_right(hbox, (service->running == IS_STARTED ? "Stop" : "Start"), click_startstop_service_callback, (void *)service);
		} else if(service->allow_startstop){
			// can't tell if service is running, show start and stop buttons and let user decide
			if(service->allow_restart)
				label = gtk_label_new("?"); gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0); gtk_widget_show(label);
			service->widget_stop = new_button_right(hbox, "Stop", click_stop_service_callback, (void *)service);
			label = gtk_label_new("?"); gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0); gtk_widget_show(label);
			service->widget_start = new_button_right(hbox, "Start", click_start_service_callback, (void *)service);
		} else if(service->running != IS_UNKNOWN){
			// state label
			service->widget_start = gtk_label_new((service->running == IS_STARTED ? "r u n n i n g" : "s t o p p e d"));
			gtk_box_pack_end(GTK_BOX(hbox), service->widget_start, FALSE, FALSE, 10);
			gtk_widget_show(service->widget_start);
		}
	}

	new_separator();

	////////////////////

	service_refresh_timer = g_timeout_add(5000, auto_refresh_service_states, NULL);
}
