//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <brand.huntsman@gmail.com>

// seconds (double)
#define INITIAL_REFRESH_PERIOD 1.0
#define REFRESH_PERIOD 5.0
#define SCROLL_UPDATE_PERIOD 0.100

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>
#include <signal.h>
#include <sys/types.h>

#include "window.h"
#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "services.h"
#include "hardware.h"
#include "dialog.h"
 #include "widget_battery.h"

//////////////////////////////////////////////////////////////////////////

static void draw_mem_indicator( int x, int y, unsigned used ){
	if(used){
		// red minus
		draw_text("-", 20, red_status_color, x-5, y, ALIGN_RIGHT, ALIGN_TOP, 0);
	} else {
		// green plus
		draw_text("+", 15, green_status_color, x-5, y, ALIGN_RIGHT, ALIGN_TOP, 0);
	}
}

// FORMAT: 0:percent free, 1:percent used, 2:free/total, 3:used/total
static void draw_mem_format( char *name, unsigned free, unsigned total, unsigned format, int x, int y ){
	char buffer[10];

	draw_text(name, 16, fg2_color, x+246, y+10, ALIGN_CENTER, ALIGN_TOP, 0);

	unsigned is_used = (format == 1 || format == 3 ? 1 : 0);
	unsigned numerator = (is_used ? total-free : free);
	XColor *ram_color;
	unsigned pct = 100*numerator/total;
	int w;

	if(is_used){
		// used
		if(pct < 70) ram_color = fg1_color;						// white  - 0-69% used
		else if(pct < 90) ram_color = yellow_status_color;		// yellow - 70-89% used
		else ram_color = red_status_color;						// red    - 90-100% used
	} else {
		// free
		if(pct > 30) ram_color = fg1_color;						// white  - 31-100% free
		else if(pct > 10) ram_color = yellow_status_color;		// yellow - 11-30% free
		else ram_color = red_status_color;						// red    - 0-10% free
	}
	#define UNDERLINE_OFFSET 3
	if(format == 0 || format == 1){
		// draw percentage free:0 or used:1
		snprintf(buffer, sizeof(buffer), "%u%%", pct);
		w = draw_text(buffer, 20, ram_color, x+246, y+30, ALIGN_CENTER, ALIGN_TOP, 0)>>1;
		draw_mem_indicator(x+246-w, y+30, is_used);
	} else {
		// draw free:2 or used:3 over total
		snprintf(buffer, sizeof(buffer), "%d", numerator/1024); w = draw_text(buffer, 20, ram_color, x+245, y+28, ALIGN_RIGHT, ALIGN_TOP, 0);
		draw_mem_indicator(x+245-w, y+28, is_used);
		snprintf(buffer, sizeof(buffer), "%d", total/1024); draw_text(buffer, 18, fg2_color, x+247, y+50, ALIGN_LEFT, ALIGN_TOP, 0);
		x_set_color(fg2_color);
		x_draw_line(x+220, y+55, x+272, y+35);
	}
}

//////////////////////////////////////////////////////////////////////////

static unsigned show_swap = 0;

static int battery_widget_x, battery_widget_y;

void draw_battery_widget( unsigned force, int x, int y ){
	if(force){
		// background for all widgets is cleared by draw_window()
		if(!drawing_popup_window){
			// don't save values if drawing the popup window
			battery_widget_page = primary_widgets;
			battery_widget_x = x;
			battery_widget_y = y;
		}
	} else if(hw_device_locked && (hw_changes & DEVLOCK_BATTERY_FLAGS)){
		x = battery_widget_x;
		y = battery_widget_y;
		// widget background on secure keypad
		x_set_color(bg_color);
		x_fill_rectangle(x, y, 199, 99);
	} else if(hw_changes & ALL_BATTERY_FLAGS){
		x = battery_widget_x;
		y = battery_widget_y;
		// widget background
		x_set_color(bg_color);
		x_fill_rectangle(x, y, 399, 99);
	} else return;

	char buffer[50];

	{
		char *title; XColor *title_color, *pct_color;
		int w, title_size, pct_size, xoff, yoff, y_base;
		switch(hw_battery.status){
		case BATTERY_DRAINING:	title = "Battery";  title_color = fg2_color;          pct_size = 26; pct_color = fg1_color; xoff = 4; yoff = 0; break;
		case BATTERY_CHARGING:	title = "Charging"; title_color = green_status_color; pct_size = 22; pct_color = fg2_color; xoff = 0; yoff = 4; break;
		default: /*BATTERY_AC*/	title = "Charged";  title_color = green_status_color; pct_size = 26; pct_color = fg1_color; xoff = 0; yoff = 0; break;
		}
		if(hw_device_locked){ title_size = 22; pct_size -= 4; y_base = y+20; } else { title_size = 26; y_base = y+16; }

		// status and percentage
		w = draw_text(title, title_size, title_color, x+12, y_base, ALIGN_LEFT, ALIGN_TOP, 0);
		if(hw_battery.capacity < 100.0)
			snprintf(buffer, sizeof(buffer), "%.1f%%", hw_battery.capacity);
		else
			sprintf(buffer, "100%%");
		draw_text(buffer, pct_size, pct_color, x+12+w+10+xoff, y_base+yoff, ALIGN_LEFT, ALIGN_TOP, 0);

		// green line under "Charged"
		if(hw_battery.status == BATTERY_AC){
			x_set_color(title_color);
			x_set_style(1, LineOnOffDash); // dotted line
			x_draw_rectangle(x+12-5,y_base-5, w+9,title_size+7);
			x_reset_style();
		}
	}
	if(!hw_device_locked){
		int minutes = hw_battery.active_minutes;

		// active time
		if(minutes > 60){
			int hours = minutes / 60;
			XColor *color = (hours > 2 ? fg1_color : yellow_status_color); // yellow @ 2 hours or less
			snprintf(buffer, sizeof(buffer), "%d hours", hours);
			draw_text("active:", 16, fg2_color, x+65, y+62, ALIGN_RIGHT, ALIGN_BASELINE, 0);
			draw_text(buffer, 16, color, x+75, y+62, ALIGN_LEFT, ALIGN_BASELINE, 0);
		} else {
			snprintf(buffer, sizeof(buffer), "less than %d minutes", minutes); // red @ 1 hour or less
			draw_text(buffer, 16, red_status_color, x+100, y+62, ALIGN_CENTER, ALIGN_BASELINE, 0);
		}
	}
	if(!hw_device_locked && cfg_battery_idle_time){
		int days, hours, minutes = hw_battery.idle_minutes;

		// idle time
		draw_text("idle:", 16, fg2_color, x+65, y+85, ALIGN_RIGHT, ALIGN_BASELINE, 0);
		if(minutes > 1440){
			days = minutes / 1440;
			hours = (minutes - days * 1440) / 60;
			if(hours)
				snprintf(buffer, sizeof(buffer), "%d day%s %d hour%s", days, (days > 1 ? "s" : ""), hours, (hours > 1 ? "s" : ""));
			else
				snprintf(buffer, sizeof(buffer), "%d day%s", days, (days > 1 ? "s" : ""));
		} else {
			hours = minutes / 60;
			snprintf(buffer, sizeof(buffer), "%d hour%s", hours, (hours > 1 ? "s" : ""));
		}
		draw_text(buffer, 16, fg2_color, x+75, y+85, ALIGN_LEFT, ALIGN_BASELINE, 0);
	}

//	draw_tap_markers(x, y, 99, 0, 0); // no taps

	////////////////////

	if(!hw_device_locked && !use_minimal_ui){
		x_set_color(line_color);
		x_draw_line(x+199, y, x+199, y+99);

		// memory or swap usage
		if(hw_swap.total && show_swap)
			draw_mem_format("swap", hw_swap.free, hw_swap.total, cfg_swap_format, x, y);
		else
			draw_mem_format("memory", hw_memory.free, hw_memory.total, cfg_memory_format, x, y);

		// cpu usage
		snprintf(buffer, sizeof(buffer), "%d%% cpu", hw_cpu.usage);
		draw_text(buffer, (hw_cpu.usage > 33 ? 20 : 18),
			(hw_cpu.usage > 50 ? red_status_color : (hw_cpu.usage ? yellow_status_color : fg2_color)),
			x+246, y+92, ALIGN_CENTER, ALIGN_BOTTOM, 0);

		if(!drawing_popup_window) draw_tap_markers(x+201, y, 99, 1, hw_swap.total); // short tap only (long tap if swap is enabled)

		x_set_color(line_color);
		x_draw_line(x+292, y, x+292, y+99);
	}

	////////////////////

	if(!hw_device_locked && !use_minimal_ui){
		// cpu frequency
		snprintf(buffer, sizeof(buffer), "%dMHz", hw_cpu.frequency);
		draw_text(buffer, 20, (hw_cpu.frequency == 165 ? fg2_color : (hw_cpu.frequency >= 400 ? red_status_color : yellow_status_color)),
			x+380, y+20, ALIGN_RIGHT, ALIGN_TOP, 0);

		if(hw_cpu.dsp_frequency){
			// DSP frequency
			snprintf(buffer, sizeof(buffer), "DSP %dMHz", hw_cpu.dsp_frequency);
			draw_text(buffer, 12, fg1_color, x+380, y+45, ALIGN_RIGHT, ALIGN_MIDDLE, 0);
		} else if(hw_cpu.current_governor){
			// governor name
			draw_text(hw_cpu.current_governor->name, 14, fg1_color, x+380, y+45, ALIGN_RIGHT, ALIGN_MIDDLE, 0);
		}

		if(!drawing_popup_window) draw_tap_markers(x+293, y, 99, 0, 1); // long tap only
	}
	{
		// temperature
		snprintf(buffer, sizeof(buffer), "%.1f %c", (cfg_temperature_F ? hw_sensors.temperature * 1.8 + 32 : hw_sensors.temperature),
				(cfg_temperature_F ? 'F' : 'C'));
		draw_text(buffer, 20, fg2_color, x+(hw_device_locked ? 12 : 380), y+(use_minimal_ui ? 55 : 80), (hw_device_locked ? ALIGN_LEFT : ALIGN_RIGHT), ALIGN_BOTTOM, 0);
	}
}

//////////////////////////////////////////////////////////////////////////

static int top_process, drag_top_process, selected_process_y;
static unsigned max_viewable, selected_process, disable_process_refresh;

////////////////////

static void draw_kill_confirm( unsigned force ){
	// force is not needed

	s_process *process;
	for(process = svc_process_list; process != NULL; process = process->next)
		if(process->pid == selected_process) break;
	if(process == NULL){
		dialog_close();
		return;
	}

	// dialog background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);

	// message
	{
		char buffer[30];

		draw_text(process->name, 22, fg1_color, window.width>>1, (window.width > window.height ? 125 : 200)-30, ALIGN_CENTER, ALIGN_MIDDLE, window.width);
		snprintf(buffer, sizeof(buffer), "%u", process->pid);
		draw_text(buffer, 24, fg2_color, window.width>>1, (window.width > window.height ? 125 : 200)+30, ALIGN_CENTER, ALIGN_MIDDLE, 0);
	}

	// buttons
	draw_cdlg_buttons("Kill", "Cancel");
}

static void init_process_list( ); // prototype
static void kill_yes_handler( ){
	if(selected_process){
		// kill the process
		kill(selected_process, SIGKILL);
	}

	// reopen process viewer
	current_dialog_draw = NULL; // close kill confirmation dialog
	init_process_list();
}
static void kill_no_handler( ){
	// reopen process viewer
	current_dialog_draw = NULL; // close kill confirmation dialog
	init_process_list();
}

////////////////////

static void highlight_process( int y ){
	// remove bottom separator
	x_set_color(bg_color);
	x_draw_line(0, y+4, window.width-10, y+4);
	// highlight process
	x_set_color(selection_color);
	x_draw_rectangle(0, y-16, window.width-10,20);

	selected_process_y = y;
}

static void unhighlight_process( ){
	// unhighlight previous process
	x_set_color(bg_color);
	x_draw_rectangle(0, selected_process_y-16, window.width-10,20);
	// redraw separators
	x_set_color(innerline_color);
	if(selected_process_y > 34)
		x_draw_line(0, selected_process_y-16, window.width-10, selected_process_y-16);
	if(selected_process_y < (int)window.height-82)
		x_draw_line(0, selected_process_y+4, window.width-10, selected_process_y+4);
}

static void _draw_process_list( unsigned force, unsigned top_offset, unsigned count ){
	s_process *process;
	float sb_ratio = (float)max_viewable/(float)svc_nr_processes;
	int sb_maxlen = window.height-(17+61), sb_len = sb_ratio*sb_maxlen, sb_y;
	unsigned selected_process_is_visible = 0;
	char buffer[30];

	if((unsigned)top_process + max_viewable > svc_nr_processes) top_process = svc_nr_processes - max_viewable;

	sb_y = 17 + (sb_maxlen - sb_len) * (float)top_process / (float)(svc_nr_processes - max_viewable);

	if(force){
		// clear background
		x_set_color(bg_color);
		x_fill_rectangle(0, 0, window.width, window.height);

		// column names
		draw_text("PID", 16, fg2_color, 0, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("Process", 16, fg2_color, 80, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("Size", 16, fg2_color, window.width-120, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		draw_text("CPU", 16, fg2_color, window.width-50, 0, ALIGN_LEFT, ALIGN_TOP, 0);
		x_set_color(line_color);
		x_draw_line(0, 15, window.width, 15);

		// row separators
		x_set_color(innerline_color);
		x_set_style(1, LineOnOffDash); // dotted line
		int y;
		for(y = 22+16; y < (int)window.height-62; y += 20)
			x_draw_line(0, y, window.width-10, y);
	} else
		x_set_style(1, LineOnOffDash); // dotted line

	// process rows
	int n = 0;
	for(process = svc_process_list; process != NULL && n - top_process < (int)max_viewable; process = process->next, n++){
		int y = 20*(n - top_process + 1) + 2 + 12;
		XColor *color;

		if(n < top_process) continue;

		// top rows are copied when scrolling
		if(top_offset == 0){
			// bottom rows are copied when scrolling
			if(count){
				if(!force){
					// row background
					x_set_color(bg_color);
					x_fill_rectangle(1, y-15, window.width-12+(process->pid == selected_process ? 0 : 2), 19);
				}

				// pid
				snprintf(buffer, sizeof(buffer), "%u", process->pid);
				draw_text(buffer, 16, fg2_color, 2, y, ALIGN_LEFT, ALIGN_BASELINE, 0);

				// name, rss and cpu color
				if(process->state == 'Z') color = red_status_color;
				else if(process->is_service) color = green_status_color;
				else if(process->is_other_service) color = scrollbar_color;
				else if(process->rss == 0) color = fg2_color;
				else color = fg1_color;

				// name
				draw_text(process->name, 16, color, 80, y, ALIGN_LEFT, ALIGN_BASELINE, window.width-(130+80));
//snprintf(buffer, sizeof(buffer), "%d", n);
//draw_text(buffer, 16, color, 80, y, ALIGN_LEFT, ALIGN_BASELINE, window.width-(130+80));

				if(process->state != 'Z'){
					// rss
					snprintf(buffer, sizeof(buffer), "%.1f", round(10.0*(float)process->rss/1024.0)/10.0);
					draw_text(buffer, 16, color, window.width-120, y, ALIGN_LEFT, ALIGN_BASELINE, 0);

					// cpu
					if(process->cpu){
						sprintf(buffer, "%u%%", process->cpu);
						draw_text(buffer, 16, color, window.width-50, y, ALIGN_LEFT, ALIGN_BASELINE, 0);
					}
				} else
					draw_text("zombie", 16, color, window.width-80, y, ALIGN_LEFT, ALIGN_BASELINE, 0);
			}
			count--;
		} else top_offset--;

		// selection box or row separator
		if(process->pid == selected_process){
			if(force || selected_process_y == 0){
				highlight_process(y);
			} else if(y != selected_process_y){
				unhighlight_process();
				highlight_process(y);
			}
			selected_process_is_visible = 1;
		}
	}
	if(!selected_process_is_visible && selected_process_y){
		unhighlight_process();
		selected_process_y = 0;
	}
	x_reset_style();

	if(sb_ratio < 1.0){
		if(!force){
			// clear scrollbar area
			x_set_color(bg_color);
			x_fill_rectangle(window.width-9, 17, 9, window.height-(17+61));
		}
		// draw scrollbar
		x_set_color(scrollbar_color);
		x_fill_rectangle(window.width-5, sb_y, 5, sb_len);
	}

	if(force){
		// button bar
		x_set_color(line_color);
		x_draw_line(0, window.height-60, window.width, window.height-60);
		x_draw_line(window.width>>1, window.height-60, window.width>>1, window.height);

		// close button
		draw_text("Close", 24, fg2_color, window.width - (window.width>>2), window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
	}

	// kill button
	if(selected_process){
		if(!force){
			// kill button background
			x_set_color(bg_color);
			x_fill_rectangle(0, window.height-59, (window.width>>1)-1, 60);
		}

		// button label and PID
		snprintf(buffer, sizeof(buffer), "Kill Process (%u)", selected_process);
		draw_text(buffer, 24, fg2_color, window.width>>2, window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);
	}
}

static void draw_process_list( unsigned force ){
	// refresh process list
	if(!disable_process_refresh)
		svc_refresh();

	if(!disable_process_refresh || force){
		max_viewable = (window.height-(20+60)) / 20;
		_draw_process_list(force, 0, max_viewable);
	}

	// change dialog refresh timer duration after first refresh
	if(dialog_refresh_timer.duration == INITIAL_REFRESH_PERIOD) dialog_refresh_timer.duration = REFRESH_PERIOD;
	// reset dialog refresh timer
	reset_timer(&dialog_refresh_timer);
}

static void reset_process_list( ){
	// free the process list maintained by svc_refresh()
	svc_process_viewer = 0;
}

static unsigned scroll_process_list( ){
	if(click.y + click.offset_y > 15 && click.y + click.offset_y <= (int)window.height-60){
		// scroll list
		int top = drag_top_process - click.offset_y/20;
		if(click.offset_y > 0 && top < 0) top = 0;
		if((unsigned)top + max_viewable > svc_nr_processes) top = svc_nr_processes - max_viewable;

		if(top != top_process){
			int row_shift = top_process - top, height, src_y, dst_y;
			top_process = top;
			if(row_shift > 0){
				// shift down
				height = 20*(max_viewable - row_shift - 1) + 18;
				src_y = 19;
				dst_y = window.height - height - 63;

				Pixmap pixmap = XCreatePixmap(window.display, window.window, window.width-10, height, window.depth);
				XCopyArea(window.display, window.window, pixmap, window.gc, 0, src_y, window.width-10, height, 0, 0);
				XCopyArea(window.display, pixmap, window.window, window.gc, 0, 0, window.width-10, height, 0, dst_y);
				XFreePixmap(window.display, pixmap);
				_draw_process_list(0, 0, row_shift);
			} else {
				// shift up
				height = 20*(max_viewable + row_shift - 1) + 18;
				src_y = window.height - height - 63;
				dst_y = 19; // top of first row

				XCopyArea(window.display, window.window, window.window, window.gc, 0, src_y, window.width-10, height, 0, dst_y);
				_draw_process_list(0, max_viewable + row_shift, -row_shift);
			}

			return 1; // motion timer is reset by main loop
		}
	}

	return 0;
}

static void click_process_list( ){
	if(click.y > (int)window.height-60){
		if(click.type != CLICK_SHORT) return; // no long press or drag support

		if(click.x > (int)window.width>>1){
			// close button
			highlight_selected_area((window.width>>1)+1, window.height-59, (window.width>>1)-1,59);

			dialog_close();
		} else if(selected_process){
			// kill selected process
			highlight_selected_area(0, window.height-59, (window.width>>1)-1,59);

			yes_color = red_status_color;
			cdlg_yes_handler = kill_yes_handler;
			cdlg_no_handler = kill_no_handler;
			current_dialog_draw = NULL; // close process viewer dialog
			dialog_open(draw_kill_confirm, click_cdlg, keypress_cdlg, reset_process_list, DLG_REDRAW_ON);
			// redraw is needed to close the dialog if the process terminates
		}
	} else if(click.y > 15){
		if(click.type == CLICK_SHORT){

			// select process
			unsigned n = top_process + (click.y-18)/20; // it doesn't matter that this can be negative if y is 16 or 17
			s_process *process;

			// clicking the last three pixels causes the process just offscreen to highlight, don't want that
			if(n == top_process + max_viewable) n--;

			for(process = svc_process_list; process != NULL && n; process = process->next, n--);
			if(process) selected_process = process->pid;
			else selected_process = 0;

			_draw_process_list(0, 0, max_viewable);

			disable_process_refresh = 0;
			motion_callback = NULL;

		} else if(click.type == CLICK_DOWN){

			// initialize a scroll
			drag_top_process = top_process;
			disable_process_refresh = 1;
			motion_callback = scroll_process_list;
			set_timer(&motion_timer, SCROLL_UPDATE_PERIOD);

		} else if(click.type == CLICK_DRAG || click.type == CLICK_LONG){

			scroll_process_list();

			disable_process_refresh = 0;
			motion_callback = NULL;

		}
	}
	// ignore clicks at top of screen
}

static void keypress_process_list( ){
	if(is_arrow(ARROW_UP)){
		if(keypress.type == KEY_SHORT){
			// scroll process list up
			if(top_process){
				top_process -= max_viewable * 8 / 10;
				if(top_process < 0) top_process = 0;
				_draw_process_list(0, 0, max_viewable);
				// just redraw it all, blitting won't help
			}
		}
	} else if(is_arrow(ARROW_DOWN)){
		if(keypress.type == KEY_SHORT){
			// scroll process list down
			if((unsigned)top_process + max_viewable < svc_nr_processes){
				top_process += max_viewable * 8 / 10;
				_draw_process_list(0, 0, max_viewable);
				// just redraw it all, blitting won't help
			}
		}
	} else switch(keypress.keycode){
	case KEYCODE_ESCAPE:
		if(keypress.type == KEY_SHORT){
			// close process list
			highlight_selected_area((window.width>>1)+1, window.height-59, (window.width>>1)-1,59);
			dialog_close();
		}
		break;
	default:
		if(!dpadlock_keypress_handler())
			brv_keypress_handler();
				// ignore all other keys
		break;
	}
}

////////////////////

static void init_process_list( ){
	// build a list of processes with service_refresh()
	svc_process_viewer = 1;
	// initialize scrollbar and selected process
	top_process = 0;
	selected_process = selected_process_y = 0;
	disable_process_refresh = 0;
	// open dialog
	dialog_open(draw_process_list, click_process_list, keypress_process_list, reset_process_list, DLG_REDRAW_ON);
	// set dialog refresh duration to INITIAL_REFRESH_PERIOD seconds to get cpu usage values faster
	set_timer(&dialog_refresh_timer, INITIAL_REFRESH_PERIOD);
}

//////////////////////////////////////////////////////////////////////////

static unsigned waiting_on_governors;

static void draw_governor_config( unsigned force ){
	// force is not needed

	// dialog background
	x_set_color(bg_color);
	x_fill_rectangle(0, 0, window.width, window.height);

	int y, cpu_x, dsp_x, dsp_y;
	if(window.width > window.height){
		// landscape
		cpu_x = 50+150;
		dsp_x = 450+150;
		dsp_y = 0;
	} else {
		// portrait
		cpu_x = dsp_x = window.width>>1;
		dsp_y = 400;
	}

	// governors
	s_hw_governor *g;
	draw_text("CPU Policy", 24, fg1_color, cpu_x, 20, ALIGN_CENTER, ALIGN_TOP, 0);
	if(hw_cpu.governor_list){
		for(g = hw_cpu.governor_list, y = 50; g != NULL; g = g->next, y += 70){
			x_set_color((g == hw_cpu.current_governor ? green_status_color : fg2_color));
			x_draw_rectangle(cpu_x-150, y, 300,60);
			draw_text(g->name, 18, (g == hw_cpu.current_governor ? green_status_color : fg2_color), cpu_x, y+30, ALIGN_CENTER, ALIGN_MIDDLE, 0);
		}
		waiting_on_governors = 0;
	} else {
		draw_text("...governor list hasn't loaded yet...", 18, red_status_color, cpu_x, 240, ALIGN_CENTER, ALIGN_MIDDLE, 0);
		waiting_on_governors = 1;
	}

	// DSP clock ratios
	s_hw_dsp_clock *clock;
	draw_text("CPU / DSP MHz", 24, fg1_color, dsp_x, dsp_y+20, ALIGN_CENTER, ALIGN_TOP, 0);
	if(hw_cpu.dsp_clocks){
		unsigned op_locked = get_file_value("/sys/power/op_locked");
		unsigned ratio = 0, current_ratio = get_file_value("/sys/power/op_dsp");
		for(clock = hw_cpu.dsp_clocks, y = dsp_y+50; clock != NULL; clock = clock->next, y += 70, ratio++){
			char buffer[30];
			snprintf(buffer, sizeof(buffer), "%u / %u", clock->cpu, clock->dsp);
			x_set_color((ratio == current_ratio ? (op_locked ? red_status_color : green_status_color) : fg2_color));
			x_draw_rectangle(dsp_x-150, y, 300,60);
			draw_text(buffer, 18, (ratio == current_ratio ? (op_locked ? red_status_color : green_status_color) : fg2_color),
						dsp_x, y+30, ALIGN_CENTER, ALIGN_MIDDLE, 0);
		}
	} else {
		draw_text("...install Diablo-Turbo...", 18, red_status_color, dsp_x, dsp_y+220, ALIGN_CENTER, ALIGN_MIDDLE, 0);
		draw_text("330 / 220", 18, green_status_color, dsp_x, dsp_y+260, ALIGN_CENTER, ALIGN_MIDDLE, 0);
	}

	// close button
	x_set_color(line_color);
	int x = window.width-240; y = window.height-60;
	x_draw_line(x, y, window.width, y);
	x_draw_line(x, y, x, window.height);
	draw_text("Close", 24, fg2_color, window.width-120, window.height-20, ALIGN_CENTER, ALIGN_BASELINE, 0);

	hw_changes = 0;
}

static void click_governor_config( ){
	if(click.type != CLICK_SHORT) return; // no long press or drag support

	int y, cpu_x, dsp_x, dsp_y;
	if(window.width > window.height){
		// landscape
		cpu_x = 50+150;
		dsp_x = 450+150;
		dsp_y = 0;
	} else {
		// portrait
		cpu_x = dsp_x = window.width>>1;
		dsp_y = 400;
	}

	// governors
	if(!waiting_on_governors){
		s_hw_governor *g;
		for(g = hw_cpu.governor_list, y = 50; g != NULL; g = g->next, y += 70){
			if(click.y >= y && click.y <= y+60 && click.x >= cpu_x-150 && click.x <= cpu_x+150){
				if(g != hw_cpu.current_governor){
					highlight_selected_area(cpu_x-150, y, 301,61);
					hw_set_governor(g);
					draw_governor_config(1);
				}
				return;
			}
		}
	}

	// DSP clock ratios
	if(!get_file_value("/sys/power/op_locked")){
		unsigned ratio = 0, current_ratio = get_file_value("/sys/power/op_dsp");
		s_hw_dsp_clock *clock;
		for(clock = hw_cpu.dsp_clocks, y = dsp_y+50; clock != NULL; clock = clock->next, y += 70, ratio++){
			if(click.y >= y && click.y <= y+60 && click.x >= dsp_x-150 && click.x <= dsp_x+150){
				if(ratio != current_ratio){
					highlight_selected_area(dsp_x-150, y, 301,61);
					hw_set_dsp_ratio(ratio);
					draw_governor_config(1);
				}
				return;
			}
		}
	}

	// close button
	if(click.y > (int)window.height-60 && click.x > (int)window.width-240){
		highlight_selected_area(window.width-240, window.height-60, 240,60);
		dialog_close();
	}
}

static void keypress_governor_config( ){
	switch(keypress.keycode){
	case KEYCODE_ESCAPE:
		// close dialog
		if(keypress.type == KEY_SHORT){
			highlight_selected_area(window.width-240, window.height-60, 240,60);
			dialog_close();
		}
		break;
	default:
		if(!dpadlock_keypress_handler())
			brv_keypress_handler();
				// ignore all other keys
		break;
	}
}

//////////////////////////////////////////////////////////////////////////

void click_battery_widget( ){
	if(click.x > 292){
		// open governor selector

		if(click.type == CLICK_HOLD){

			// highlight to indicate long press
			highlight_long_press(widget_x+293, widget_y, 106,99);

		} else if(click.type == CLICK_LONG){

			if(throttle_taps(&dialog_throttle)) return;

			highlight_selected_area(widget_x+293, widget_y, 106, 99);
			dialog_open(draw_governor_config, click_governor_config, keypress_governor_config, NULL, DLG_REDRAW_OFF); // no redraw

			finalize_tap(dialog_throttle);
		}

	} else if(click.x > 199){
		// open process list

		if(hw_swap.total && click.type == CLICK_HOLD){

			// highlight to indicate long press
			highlight_long_press(widget_x+200, widget_y, 92, 99);

		} else if(hw_swap.total && click.type == CLICK_LONG){

			if(throttle_taps(&dialog_throttle)) return;

			highlight_selected_area(widget_x+200, widget_y, 92, 99);
			show_swap = (show_swap ? 0 : 1); // toggle between swap and memory display

			hw_changes |= HW_MEMORY;
			draw_battery_widget(0, 0, 0);
			hw_changes &= ~ALL_BATTERY_FLAGS;
			x_flush();

			finalize_tap(dialog_throttle);

		} else if(click.type == CLICK_SHORT){

			if(throttle_taps(&dialog_throttle)) return;

			highlight_selected_area(widget_x+200, widget_y, 92, 99);
			init_process_list();

			finalize_tap(dialog_throttle);
		}

	} else {
		// battery stats -- do nothing
	}
}

////////////////////////////////////////////////////////////////////////// WIDGET

s_widget widget_battery = {"Battery", "/apps/asui_state/cell_battery", FULL_WIDGET, draw_battery_widget, click_battery_widget, 0};
