//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <brand.huntsman@gmail.com>
//
//	backlight code ported from advanced-backlight
//	(C) 2008 Jonas Hurrelmann <j@outpo.st>.
//	(C) 2008 Adam Harwell <aharwell@trinity.edu>.
//
//	keypad locking code ported from powerlaunch
//	Copyright (c) 2007-2008 Austin Che
//
//	MCE, BME and CPUFREQ d-bus calls from advanced-power-monitor package and various wiki pages
//
//	Bluetooth d-bus calls from 'switch on BT' package

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>

#include "window.h"
#include "main.h"
#include "config.h"
#include "services.h"
 #include "hardware.h"
#include "dbus.h"
#include "dsme.h"
#include "secure.h"

////////////////////////////////////////////////////////////////////////// GLOBALS

unsigned hw_initialized;

unsigned hw_changes;

int hw_volume;

s_hw_memory hw_memory;
s_hw_swap hw_swap;

s_hw_screen hw_screen;
unsigned hw_flight_mode;
unsigned hw_device_locked, hw_device_autolocked, hw_device_autolock_map_state;

s_hw_battery hw_battery;

s_hw_cpu hw_cpu;

s_hw_bluetooth hw_bluetooth;
s_hw_gps hw_gps;
s_hw_wifi hw_wifi;

s_hw_sensors hw_sensors;

s_hw_usb hw_usb;

unsigned hw_key_repeat, hw_screen_stays_lit, hw_screen_autolock;

double last_hw_refresh;

//////////////////////////////////////////////////////////////////////////

int get_file_value( const char *filename ){
	char line[50];
	FILE *in;
	int value;

	if((in = fopen(filename, "r")) == NULL) return 0;

	fgets(line, 50, in);
	if(feof(in)) return 0;
	sscanf(line, "%d\n", &value);
	fclose(in);
	return value;
}

////////////////////////////////////////////////////////////////////////// HAL

#ifdef ARCH_armel
static const char *_hal_service_ = "org.freedesktop.Hal";
static const char *_hal_interface_ = "org.freedesktop.Hal.Device";

static const char *_hal_GetPropertyInteger_ = "GetPropertyInteger"; // method
#endif

////////////////////////////////////////////////////////////////////////// BRIGHTNESS

void brightness_init( ){
	// called from config_reload()

	#ifdef ARCH_armel
	if(cfg_brightness_enhanced){
		hw_screen.brightness_max = get_file_value("/sys/devices/platform/omapfb/panel/backlight_max");
		hw_screen.brightness_level = conf_get_int(conf_brightness_level);
	} else {
		hw_screen.brightness_max = conf_get_int(conf_max_brightness);
		hw_screen.brightness_level = conf_get_int(conf_brightness);
	}

	if((!cfg_brightness_enhanced && hw_screen.brightness_level == 0) || hw_screen.brightness_level > hw_screen.brightness_max)
		hw_screen.brightness_current = hw_screen.brightness_level = hw_screen.brightness_max>>1; // 50% of max
	else
		hw_screen.brightness_current = hw_screen.brightness_level;
	// make sure stored brightness matches current brightness
	// this doesn't work when turning off enhanced brightness
	hw_set_brightness(hw_screen.brightness_level, 1);
	#else
	hw_screen.brightness_max = 100;
	hw_screen.brightness_level = 50;
	hw_screen.brightness_current = hw_screen.brightness_level;
	#endif
}

void hw_query_brightness( ){
	// called from hw_refresh() and main.c:main():brightness_mismatch_timer

	#ifdef ARCH_armel
	unsigned brightness;
	if(cfg_brightness_enhanced)
		brightness = get_file_value("/sys/devices/platform/omapfb/panel/backlight_level");
	else
		brightness = conf_get_int(conf_brightness);
	if(hw_screen.brightness_current != brightness){
		if(hw_screen.brightness_current != hw_screen.brightness_level && !hw_screen.dimmed) hw_screen.brightness_mismatch = 1;
		hw_screen.brightness_current = brightness;
		hw_changes |= HW_BRIGHTNESS;
	}
	#else
	hw_screen.brightness_current = 50;
	#endif
}

void hw_query_brightness_with_delay( ){
	if(hw_screen.visible && !hw_screen.dimmed){
		if(cfg_brightness_enhanced){
			// wait 750ms after turning on screen to let brightness stabilize
			unsigned backlight_time = 1000000 * (current_time - hw_screen.visible_time);
			if(hw_screen.visible_time != 0.0 && backlight_time < 750000)
				usleep(750000 - backlight_time);
		}
		hw_query_brightness();
	}
}

void hw_set_brightness( unsigned level, unsigned update ){
	#ifdef ARCH_armel
	if(level == 0 && !cfg_brightness_allow_zero) level = 1; // no zero brightness

	if(cfg_brightness_enhanced){
		dsme_set_brightness(level);
		if(update) conf_set_int(conf_brightness_level, (level == 0 ? 1 : (int)level)); // don't save a zero brightness
	} else
		conf_set_int(conf_brightness, (int)level);

	if(hw_screen.brightness_level != level){
		if(!hw_screen.dimmed) hw_screen.brightness_mismatch = 1;
		hw_screen.brightness_level = level;
		hw_changes |= HW_BRIGHTNESS;
	}
	#endif
}

////////////////////////////////////////////////////////////////////////// VOLUME

static void hw_query_volume( ){
	// called from hw_refresh()

	#ifdef ARCH_armel
	int volume = conf_get_int(conf_volume);
	if(hw_volume != volume){ hw_volume = volume; hw_changes |= HW_VOLUME; }
	#else
	hw_volume = 0;
	#endif
}

////////////////////////////////////////////////////////////////////////// MEMORY / SWAP

static void hw_query_memory( ){
	// called from hw_refresh()

	unsigned mem_total = 0, mem_free = 0, mem_buffers = 0, mem_cached = 0, swap_total = 0, swap_free = 0;
	FILE *in;

	if((in = fopen("/proc/meminfo", "r")) != NULL){
		while(!feof(in)){
			char line[400], label[30];
			int value;

			fgets(line, 400, in);
			if(feof(in)) break;

			sscanf(line, "%s %d kB\n", label, &value);

			if(!strcmp(label, "MemTotal:")) mem_total = value;
			else if(!strcmp(label, "MemFree:")) mem_free = value;
			else if(!strcmp(label, "Buffers:")) mem_buffers = value;
			else if(!strcmp(label, "Cached:")) mem_cached = value;
			else if(!strcmp(label, "SwapTotal:")) swap_total = value;
			else if(!strcmp(label, "SwapFree:")){ swap_free = value; break; }
		}
		fclose(in);
	}
	mem_free += mem_buffers + mem_cached;
	if(hw_memory.free != mem_free || hw_memory.total != mem_total){
		hw_memory.free = mem_free;
		hw_memory.total = mem_total;
		hw_changes |= HW_MEMORY;
	}
	if(hw_swap.free != swap_free || hw_swap.total != swap_total){
		hw_swap.free = swap_free;
		hw_swap.total = swap_total;
		hw_changes |= HW_MEMORY;
	}
}

////////////////////////////////////////////////////////////////////////// MCE

// http://maemo.org/api_refs/5.0/5.0-final/mce-dev/files.html

#ifdef ARCH_armel
static const char *_mce_service_ = "com.nokia.mce";
static const char *_mce_path_ = "/com/nokia/mce/request";
static const char *_mce_interface_ = "com.nokia.mce.request";

static const char *_mce_get_tklock_mode_ = "get_tklock_mode"; // method
static const char *_mce_req_tklock_mode_change_ = "req_tklock_mode_change"; // method
static const char *_mce_tklock_LOCKED_ = "locked";
static const char *_mce_tklock_UNLOCKED_ = "unlocked";

static const char *_mce_get_display_status_ = "get_display_status"; // method
static const char *_mce_req_display_state_off_ = "req_display_state_off"; // method

static const char *_mce_get_device_mode_ = "get_device_mode"; // method
static const char *_mce_req_device_mode_change_ = "req_device_mode_change"; // method
static const char *_mce_device_NORMAL_ = "normal";
static const char *_mce_device_FLIGHT_ = "flight";
static unsigned got_flight_mode, sl_ready;

static const char *_mce_get_devicelock_mode_ = "get_devicelock_mode"; // method
static const char *_mce_devlock_callback_ = "devlock_callback"; // method
static const char *_mce_device_LOCKED_ = "locked";

#define TOUCHSCREEN_CONTROL "/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_ts"

#define KEYPAD_CONTROL_N800 "/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_kp"
#define KEYPAD_CONTROL_N810 "/sys/devices/platform/i2c_omap.2/i2c-0/0-0045/disable_kp"
static const char *keypad_control;

//#define TOUCHSCREEN_CONTROL "/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_ts"

static void sl_init( ){
	if(!hw_screen.visible && !hw_screen.locked && !conf_get_screen_autolock_value()){
		if(cfg_start_visible)
			click.enabled = 0; // ignore first tap
		else
			window_open_blank(); // open blank window to catch first tap/key
	}
}

static void mce_get_display_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *mode;
	debug_write("[DBUS] callback: mce_get_display_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
		debug_write("[DBUS] callback: mce_get_display_callback (mode = %s)", mode);
		hw_screen.visible = (strcmp(mode, "off") ? 1 : 0);
		if(sl_ready) sl_init(); else sl_ready++;
		unsigned dimmed = (strcmp(mode, "dimmed") ? 0 : 1);
		if(hw_screen.dimmed != dimmed){
			if(!dimmed || !(hw_battery.status != BATTERY_DRAINING && conf_get_screen_stayslit_value())){
				// ignored dimmed signal if on charger and screen stays lit
				hw_screen.dimmed = dimmed;
				hw_changes |= HW_BRIGHTNESS;
			}
		}
		debug_write("[DBUS] callback: mce_get_display_callback (visible = %u)", hw_screen.visible);
		hw_initialized &= ~HWIS_MCE_DISPLAY;
	}
	dbus_message_unref(message);
}

static void mce_get_tklock_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *mode;
	debug_write("[DBUS] callback: mce_get_tklock_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
		hw_screen.locked = (strcmp(mode, _mce_tklock_LOCKED_) ? 0 : 1);
		if(sl_ready) sl_init(); else sl_ready++;
		hw_initialized &= ~HWIS_MCE_LOCK;
		unsigned ts_locked = get_file_value(TOUCHSCREEN_CONTROL);
		unsigned kp_locked = get_file_value(keypad_control);
		if(cfg_power_unlock_key && ts_locked && !kp_locked){
			if(!cfg_start_visible) window_open_blank();
			notify_power_unlock_key();
		}
//		else if(hw_screen.locked)
//			hw_lock_keys(); // lock the keypad
		debug_write("[DBUS] callback: mce_get_tklock_callback (locked = %u)", hw_screen.locked);
	}
	dbus_message_unref(message);
}

static void mce_get_flight_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *mode;
	debug_write("[DBUS] callback: mce_get_flight_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
		unsigned flight = (strcmp(mode, _mce_device_NORMAL_) ? 1 : 0);
		if(hw_flight_mode != flight){ hw_flight_mode = flight; hw_changes |= HW_FLIGHT_MODE; }
		got_flight_mode = 1;
		hw_initialized &= ~HWIS_MCE_FLIGHT;
	}
	dbus_message_unref(message);
}

static void mce_get_devicelock_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *mode;
	debug_write("[DBUS] callback: mce_get_devicelock_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
		unsigned locked = (strcmp(mode, _mce_device_LOCKED_) ? 0 : 1);
		if(hw_device_locked != locked){
			if(locked && !cfg_start_visible){
				hw_device_autolocked = 1;
				hw_device_autolock_map_state = 0; // ASUI started up in locked state but should unmap after code is entered
			}
			hw_device_locked = locked;
			dbus_send_wakeup(_WAKEUP_DEVICE_LOCK_);
		}
		hw_initialized &= ~HWIS_MCE_DEVICELOCK;
	}
	dbus_message_unref(message);
}
#endif

void mce_init( ){
	// this combo won't produce a "screen active" notification but will produce a "screen locked"
	// if the app is restarted while the screen is locked
	hw_screen.visible = 1;
	hw_screen.dimmed = 0;
	hw_screen.visible_time = 0.0;
	hw_screen.locked = 0;
	hw_flight_mode = 0;
	hw_device_locked = 0;
	hw_device_autolocked = 0;

	#ifdef ARCH_armel
	sl_ready = 0;
	#endif
	click.enabled = 1; // allow screen taps

	#ifdef ARCH_armel
	keypad_control = (file_exists(KEYPAD_CONTROL_N800) ? KEYPAD_CONTROL_N800 : KEYPAD_CONTROL_N810);
//	hw_unlock_keys(); // unlock the keypad to match the current hw_screen.locked state, will be relocked by mce_get_tklock_callback()

	if(!use_minimal_ui){
		hw_initialized |= HWIS_MCE_DISPLAY|HWIS_MCE_LOCK|HWIS_MCE_FLIGHT|HWIS_MCE_DEVICELOCK;
		dbus_recv_method(mce_get_display_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_display_status_, _END_);
		dbus_recv_method(mce_get_tklock_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_tklock_mode_, _END_);
		dbus_recv_method(mce_get_flight_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_device_mode_, _END_);
		dbus_recv_method(mce_get_devicelock_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_devicelock_mode_, _END_);
	} else
		hw_screen.locked = 1;
	#endif
}

////////////////////

void mce_dim_screen( ){
	#ifdef ARCH_armel
	if(hw_battery.status == BATTERY_DRAINING || !conf_get_screen_stayslit_value()){
		dsme_allow_blanking(); // make sure screen isn't paused
		dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, "req_display_state_dim", _END_);
		dsme_expect_display_state(DSME_DISPLAY_DIM);
	}
	#endif
}

void mce_blank_screen( ){
	#ifdef ARCH_armel
	unsigned screen_autolock = conf_get_screen_autolock_value();
	if(screen_autolock){
		// turn off autolock
		conf_set_screen_autolock_value(0);
		original_screen_autolock = 2; // save ON state
	}

	unsigned screen_stayslit = conf_get_screen_stayslit_value();
	if(screen_stayslit && hw_battery.status != BATTERY_DRAINING){
		// turn off stayslit
		conf_set_screen_stayslit_value(0);
		original_screen_stayslit = 2; // save ON state
	}

	// make sure screen isn't paused
	dsme_allow_blanking();

	usleep(100000); // sleep 100ms to let MCE update settings

	// blank screen
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_display_state_off_, _END_);
	dsme_expect_display_state(DSME_DISPLAY_OFF);
	#endif
}

void mce_lock_and_blank_screen( ){
	#ifdef ARCH_armel
	unsigned screen_autolock = conf_get_screen_autolock_value();
	if(!screen_autolock){
		// turn on autolock
		conf_set_screen_autolock_value(1);
		original_screen_autolock = 1; // save OFF state
	}

	unsigned screen_stayslit = conf_get_screen_stayslit_value();
	if(screen_stayslit && hw_battery.status != BATTERY_DRAINING){
		// turn off stayslit
		conf_set_screen_stayslit_value(0);
		original_screen_stayslit = 2; // save ON state
	}

	// make sure screen isn't paused
	dsme_allow_blanking();

	usleep(100000); // sleep 100ms to let MCE update settings

	// lock and blank screen
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_display_state_off_, _END_);
	dsme_expect_display_state(DSME_DISPLAY_OFF);
	#endif
}

void mce_lock_screen( ){
	#ifdef ARCH_armel
	// lock screen, dims and turns off after a minute
	if(hw_battery.status == BATTERY_DRAINING || !conf_get_screen_stayslit_value())
		dsme_allow_blanking(); // make sure screen isn't paused
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_tklock_mode_change_, _STRING_, &_mce_tklock_LOCKED_, _END_);
	if(hw_battery.status == BATTERY_DRAINING || !conf_get_screen_stayslit_value())
		dsme_expect_display_state(DSME_DISPLAY_DIM);
	#endif
}

void mce_unlock_screen( ){
	#ifdef ARCH_armel
	// unlock screen
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_tklock_mode_change_, _STRING_, &_mce_tklock_UNLOCKED_, _END_);
	#endif
}

void hw_lock_screen( ){
	#ifdef ARCH_armel
	FILE *out = fopen(TOUCHSCREEN_CONTROL, "w");
	if(out != NULL){
		fputc('1', out);
		fclose(out);
	} else {
		debug_write("failed to open file needed to lock screen");
	}
	#endif
}

void hw_lock_keys( ){
	#ifdef ARCH_armel
	FILE *out = fopen(keypad_control, "w");
	if(out != NULL){
		fputc('1', out);
		fclose(out);
	} else {
		debug_write("failed to open file needed to lock keypad");
	}
	#endif
}

void hw_unlock_keys( ){
	#ifdef ARCH_armel
	FILE *out = fopen(keypad_control, "w");
	if(out != NULL){
		fputc('0', out);
		fclose(out);
	} else {
		debug_write("failed to open file needed to unlock keypad");
	}
	#endif
}

////////////////////

static void hw_query_flight_mode( unsigned flush ){
	// called from hw_refresh(), mce_normal_mode() and mce_flight_mode()

	#ifdef ARCH_armel
	got_flight_mode = 0;
	dbus_recv_method(mce_get_flight_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_device_mode_, _END_);
	if(flush){
		dbus_flush_messages();
		int i; // wait 2 seconds for reply
		for(i = 0; i < 20; i++){
			usleep(100000); // 100ms sleep to let MCE process method
			if(got_flight_mode) break;
		}
	}
	#endif
}

void mce_normal_mode( ){
	#ifdef ARCH_armel
	// enable normal mode
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_device_mode_change_, _STRING_, &_mce_device_NORMAL_, _END_);

	usleep(100000); // 100ms sleep to let MCE process method
	hw_query_flight_mode(1);
	#endif
}

void mce_flight_mode( ){
	#ifdef ARCH_armel
	// enable flight mode
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_req_device_mode_change_, _STRING_, &_mce_device_FLIGHT_, _END_);

	debug_write("*** ENTER FLIGHT MODE ***");

	usleep(100000); // 100ms sleep to let MCE process method
	hw_query_flight_mode(1);
	#endif
}

////////////////////

void mce_lock_device( ){
	#ifdef ARCH_armel
	if(!hw_device_locked){
		int lock = 3;
		dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_devlock_callback_, _INT32_, &lock, _END_);

// TODO: the above call does absolutely nothing without systemui running
//		MCE's internal lock state remains unlocked and restarting ASUI or rebooting system will result in an unlocked device
//		if locked and then systemui starts, MCE will update its internal state

		// MCE doesn't send out a locked signal so send we need to send it for anything listening
		dbus_send_signal("/com/nokia/mce/signal", "com.nokia.mce.signal", "devicelock_mode_ind", _STRING_, &_mce_device_LOCKED_, _END_);
		// dbus thread will see this signal and put ASUI into secure mode
	}
	#endif
}

void mce_unlock_device( ){
	#ifdef ARCH_armel
	if(hw_device_locked){
		int unlock = 2;
		dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, _mce_devlock_callback_, _INT32_, &unlock, _END_);
		// MCE will send out an unlock signal which is seen by dbus thread and remove ASUI from secure mode
	}
	#endif
}

////////////////////

void mce_reboot( ){
	#ifdef ARCH_armel
	// reboot device
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, "req_reboot", _END_);
	#endif
}

void mce_shutdown( ){
	#ifdef ARCH_armel
	// shutdown device
	dbus_send_method(_mce_service_, _mce_path_, _mce_interface_, "req_shutdown", _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// BME

static int battery_reporting_design;

// bme (uses: hal_service and hal_interface)
#ifdef ARCH_armel
static const char *_bme_path_ = "/org/freedesktop/Hal/devices/bme";
static const char *_bme_request_path_ = "/com/nokia/bme/request";
static const char *_bme_request_interface_ = "com.nokia.bme.request";

static const char *_bme_reporting_design_ = "battery.reporting.design"; // method property
static const char *_bme_reporting_current_ = "battery.reporting.current"; // method property
static const char *_bme_remaining_time_ = "battery.remaining_time"; // method property

static void bme_get_capacity_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	int current;
	debug_write("[DBUS] callback: bme_get_capacity_callback");
	if(dbus_message_get_args(message, &dbus_error, _INT32_, &current, _END_)){
		float capacity = round(1000.0 * current / battery_reporting_design) / 10;
		if(capacity > 100) capacity = 100.0;
		if(hw_battery.capacity != capacity){ hw_battery.capacity = capacity; hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		hw_initialized &= ~HWIS_BME_CAPACITY;
		dbus_send_battery_capacity_signal();
	}
	dbus_message_unref(message);
}

static void bme_get_max_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	debug_write("[DBUS] callback: bme_get_max_callback");
	if(dbus_message_get_args(message, &dbus_error, _INT32_, &battery_reporting_design, _END_) && battery_reporting_design){
//		battery_reporting_design = (unsigned)(100.0*roundf((float)battery_reporting_design/100.0));
		dbus_recv_method(bme_get_capacity_callback, _hal_service_, _bme_path_, _hal_interface_, _hal_GetPropertyInteger_, _STRING_, &_bme_reporting_current_, _END_); // current mAh (capacity)
		dbus_flush_messages();
		hw_initialized &= ~HWIS_BME_MAX;
	}
	dbus_message_unref(message);
}

static void bme_get_active_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	int active_seconds;
	debug_write("[DBUS] callback: bme_get_active_callback");
	if(dbus_message_get_args(message, &dbus_error, _INT32_, &active_seconds, _END_)){
		unsigned active_minutes = active_seconds / 60;
		if(hw_battery.active_minutes != active_minutes){ hw_battery.active_minutes = active_minutes; hw_changes |= HW_BATTERY; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		hw_initialized &= ~HWIS_BME_ACTIVE;
	}
	dbus_message_unref(message);
}
#endif

static void bme_init( ){
	battery_reporting_design = 0;
	hw_battery.initialize = 1;
	hw_battery.status = (use_minimal_ui ? BATTERY_CHARGING : BATTERY_DRAINING); // assume charging when at runlevel 5, otherwise assume draining
	hw_battery.capacity = 0.0;
	hw_battery.active_minutes = 0;
	hw_battery.idle_minutes = 0;

	#ifdef ARCH_armel
	hw_initialized |= HWIS_BME_CHARGER|HWIS_BME_MAX|HWIS_BME_CAPACITY|HWIS_BME_ACTIVE;
	dbus_send_signal(_bme_request_path_, _bme_request_interface_, "status_info_req", _END_); // request charging status signal
	hw_query_battery(0); // flushed by hw_init()
	#else
	hw_battery.capacity = 50.0;
	#endif
}

void hw_query_battery( unsigned flush ){
	// called from bme_init() and the /org/freedesktop/Hal/devices/bme org.freedesktop.Hal.Device.PropertyModified signal handler

	#ifdef ARCH_armel
	if(battery_reporting_design)
		dbus_recv_method(bme_get_capacity_callback, _hal_service_, _bme_path_, _hal_interface_, _hal_GetPropertyInteger_, _STRING_, &_bme_reporting_current_, _END_); // current mAh (capacity)
	else
		dbus_recv_method(bme_get_max_callback, _hal_service_, _bme_path_, _hal_interface_, _hal_GetPropertyInteger_, _STRING_, &_bme_reporting_design_, _END_); // max mAh and current mAh (capacity)

	if(cfg_battery_idle_time){
		// this signal will generate a battery_timeleft signal that contains active and idle time in minutes
		// it will also cause the default battery statusbar applet to show its popup (only use this with the asui battery applet or advanced-power applet)
		dbus_send_signal(_bme_request_path_, _bme_request_interface_, "timeleft_info_req", _END_);
	} else {
		dbus_recv_method(bme_get_active_callback, _hal_service_, _bme_path_, _hal_interface_, _hal_GetPropertyInteger_, _STRING_, &_bme_remaining_time_, _END_); // active time in minutes
	}

	if(flush)
		dbus_flush_messages();
	#endif
}

////////////////////////////////////////////////////////////////////////// CPU

static double last_cpu_refresh, last_cpu_usage_refresh;
#ifdef ARCH_armel
static unsigned get_current_governor;
#endif
static unsigned ticks_per_second, ticks;

// governor (uses: hal_service)
#ifdef ARCH_armel
static const char *_governor_path_ = "/org/freedesktop/Hal/devices/computer";
static const char *_governor_interface_ = "org.freedesktop.Hal.Device.CPUFreq";

static const char *_governor_GetCPUFreqAvailableGovernors_ = "GetCPUFreqAvailableGovernors"; // method
static const char *_governor_GetCPUFreqGovernor_ = "GetCPUFreqGovernor"; // method

static void cpu_get_governor_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *governor;
	debug_write("[DBUS] callback: cpu_get_governor_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &governor, _END_)){
		s_hw_governor *g;
		for(g = hw_cpu.governor_list; g != NULL; g = g->next)
			if(!strcmp(g->name, governor)) break;
		if(hw_cpu.current_governor != g){ hw_cpu.current_governor = g; hw_changes |= HW_CPU; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		// don't free the string
	}
	dbus_message_unref(message);
}

static void cpu_get_all_governors_callback( DBusPendingCall *pending, void *user_data ){
	// it is possible for this to be queried a second time before the first query has a chance to reply
	// ignore if the first query has been handled
	if(hw_cpu.governor_list == NULL){
		DBusMessage *message = dbus_pending_call_steal_reply(pending);
		dbus_pending_call_unref(pending);
		char **governors;
		int len;
		debug_write("[DBUS] callback: cpu_get_all_governors_callback");
		if(dbus_message_get_args(message, &dbus_error, _ARRAY_, _STRING_, &governors, &len, _END_)){
			s_hw_governor *tail = NULL, *g;
			int i;
			for(i = 0; i < len; i++){
				g = malloc(sizeof(s_hw_governor));
				debug_write("[DBUS] callback: cpu_get_all_governors_callback (malloc)");
				if(g == NULL) break;
				g->name = malloc(strlen(governors[i])+1);
				if(g->name == NULL){ free(g); break; }
				strcpy(g->name, governors[i]);
				g->next = NULL;

				hw_cpu.nr_governors++;
				if(hw_cpu.governor_list == NULL)
					hw_cpu.governor_list = g;
				else
					tail->next = g;
				tail = g;

				//dbus_free(governors[i]);
			}

			dbus_free_string_array(governors);

			if(get_current_governor){
				dbus_recv_method(cpu_get_governor_callback, _hal_service_, _governor_path_, _governor_interface_, _governor_GetCPUFreqGovernor_, _END_);
				dbus_flush_messages();
			}
		}
		debug_write("[DBUS] callback: cpu_get_all_governors_callback (done)");

		dbus_message_unref(message);
	}
}
#endif

static void cpu_init( ){
	hw_cpu.nr_governors = 0;
	hw_cpu.governor_list = NULL;
	hw_cpu.current_governor = NULL;
	hw_cpu.frequency = 0;
	hw_cpu.usage = 0;
	hw_cpu.nr_dsp_clocks = 0;
	hw_cpu.dsp_clocks = NULL;
	hw_cpu.dsp_frequency = 0;
	last_cpu_refresh = 0.0;
	last_cpu_usage_refresh = 0.0;
	ticks_per_second = sysconf(_SC_CLK_TCK);
	ticks = 0;

	#ifdef ARCH_armel
	if(!use_minimal_ui){
		get_current_governor = 0;
		dbus_recv_method(cpu_get_all_governors_callback, _hal_service_, _governor_path_, _governor_interface_, _governor_GetCPUFreqAvailableGovernors_, _END_);
	}

	FILE *in;
	if((in = fopen("/sys/power/op_list_info", "r")) != NULL){
		s_hw_dsp_clock *tail = NULL, *clock;
		while(!feof(in)){
			char line[30];
			fgets(line, sizeof(line), in);
			if(feof(in)) break;

			unsigned cpu, dsp;
			sscanf(line, "%d/%d", &cpu, &dsp);
			clock = malloc(sizeof(s_hw_dsp_clock));
			clock->cpu = cpu/1000;
			clock->dsp = dsp/1000;
			clock->next = NULL;

			hw_cpu.nr_dsp_clocks++;
			if(hw_cpu.dsp_clocks == NULL)
				hw_cpu.dsp_clocks = clock;
			else
				tail->next = clock;
			tail = clock;
		}

		fclose(in);
	}
	#else
	const char *govs[] = {"conservative", "ondemand", "null (x86)", "powersave", "performance", NULL};
	s_hw_governor *tail = NULL, *g;
	int i;

	for(i = 0; govs[i] != NULL; i++){
		g = malloc(sizeof(s_hw_governor));
		debug_write("malloc in cpu_init()");
		g->name = (char *)govs[i];
		g->next = NULL;

		if(!strcmp(g->name, "null (x86)"))
			hw_cpu.current_governor = g;

		hw_cpu.nr_governors++;
		if(hw_cpu.governor_list == NULL)
			hw_cpu.governor_list = g;
		else
			tail->next = g;
		tail = g;
	}
	#endif
}

static void hw_query_cpu( unsigned force ){
	// called from hw_refresh() and hw_set_governor()

	// cpu frequency
	unsigned frequency = get_file_value("/sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq") / 1000;
	if(hw_cpu.frequency != frequency){ hw_cpu.frequency = frequency; hw_changes |= HW_CPU; }

	#ifdef ARCH_armel
	// dsp frequency (requires Diablo-Turbo kernel)
	frequency = 0;
	if(hw_cpu.nr_dsp_clocks){
		if(get_file_value("/sys/power/op_locked")){
			// /sys/devices/platform/dsp/state is 3 when dsp or pcm is active, or 0 when not
			// mplayer uses the PCM which set state=3 but the DSP isn't enabled so frequency=0
			unsigned op_dsp = get_file_value("/sys/power/op_dsp");
			s_hw_dsp_clock *clock = hw_cpu.dsp_clocks;
			for(; clock != NULL; clock = clock->next, op_dsp--)
				if(op_dsp == 0){
					frequency = clock->dsp;
					break;
				}
		}
	}
	if(hw_cpu.dsp_frequency != frequency){ hw_cpu.dsp_frequency = frequency; hw_changes |= HW_CPU; }

	// 30 second query interval
	if(force || last_cpu_refresh + 29.0 < current_time){
		if(hw_cpu.governor_list){
			dbus_recv_method(cpu_get_governor_callback, _hal_service_, _governor_path_, _governor_interface_, _governor_GetCPUFreqGovernor_, _END_);
		} else {
			get_current_governor = 1;
			dbus_recv_method(cpu_get_all_governors_callback, _hal_service_, _governor_path_, _governor_interface_, _governor_GetCPUFreqAvailableGovernors_, _END_);
		}

		dbus_flush_messages();

		last_cpu_refresh = current_time;
	}
	#endif
}

void hw_set_governor( s_hw_governor *governor ){
	#ifdef ARCH_armel
	if(governor == NULL) return;

	// set cpu governor
	dbus_send_method(_hal_service_, _governor_path_, _governor_interface_, "SetCPUFreqGovernor", _STRING_, &governor->name, _END_);

	usleep(100000); // 100ms sleep to let HAL process method
	hw_query_cpu(1);
	usleep(100000); // 100ms sleep to let HAL process method
	#endif
}

void hw_set_dsp_ratio( unsigned ratio ){
	#ifdef ARCH_armel
	if(hw_cpu.nr_dsp_clocks && ratio < hw_cpu.nr_dsp_clocks){
		if(!get_file_value("/sys/power/op_locked")){
			FILE *out;
			if((out = fopen("/sys/power/op_dsp", "w")) != NULL){
				fputc('0'+ratio, out);
				fclose(out);
				usleep(100000); // 100ms sleep to let kernel process value
				hw_query_cpu(0);
			}
		}
	}
	#endif
}

static void hw_query_cpu_usage( ){
	// called from hw_refresh_cpu()

	unsigned usage = 0, total = ticks;
	FILE *in;

	if((in = fopen("/proc/stat", "r")) != NULL){
		unsigned user, nice, sys, iowait, irq, softirq;
		float interval = current_time - last_cpu_usage_refresh;

		fscanf(in, "cpu  %u %u %u %*u %u %u %u", &user, &nice, &sys, /* idle, */ &iowait, &irq, &softirq);
		fclose(in);
		total = user + nice + sys + iowait + irq + softirq;
		if(ticks)
			usage = round(100.0*(float)(total - ticks) / (float)(ticks_per_second * interval));

// TODO: this value would need to be divided by the number of cores, if more than one

	}
	ticks = total;

	if(hw_cpu.usage != usage){ hw_cpu.usage = usage; hw_changes |= HW_CPU_USAGE; }

	last_cpu_usage_refresh = current_time;
}

////////////////////////////////////////////////////////////////////////// BLUETOOTH

// http://bluez.cvs.sourceforge.net/viewvc/bluez/utils/hcid/dbus-api.txt
// http://bluez.cvs.sourceforge.net/viewvc/bluez/utils/audio/audio-api.txt
// http://maemo.org/maemo_release_documentation/maemo4.1.x/node10.html
// http://wiki.maemo.org/Documentation/Maemo_5_Developer_Guide/Using_Connectivity_Components/Maemo_Connectivity#Bluetooth_D-Bus_UI_dialogs

#ifdef ARCH_armel
static const char *_bluez_service_ = "org.bluez";
static const char *_bluez_path_ = "/org/bluez/hci0";
static const char *_bluez_interface_ = "org.bluez.Adapter";

static const char *_bluez_GetMode_ = "GetMode"; // method
static const char *_bluez_SetMode_ = "SetMode"; // method
static const char *_bluez_mode_OFF_ = "off";
static const char *_bluez_mode_CONNECTABLE_ = "connectable";
static const char *_bluez_mode_DISCOVERABLE_ = "discoverable";

static const char *_bluez_GetName_ = "GetName"; // method
static const char *_bluez_ListConnections_ = "ListConnections"; // method

static const char *_btui_service_ = "com.nokia.bt_ui";
static const char *_btui_path_ = "/com/nokia/bt_ui";
static const char *_btui_interface_ = "com.nokia.bt_ui";

static const char *_bluez_audio_proxy_service_ = "com.nokia.bluez_audio_proxy";
static const char *_bluez_audio_proxy_path_ = "/com/nokia/bluez_audio_proxy";
static const char *_dbus_property_interface_ = "org.freedesktop.DBus.Properties";
static const char *_dbus_Get_ = "Get"; // method
static const char *_dbus_FIXME_ = "FIXME";
static const char *_dbus_DefaultHeadset_ = "DefaultHeadset";
static const char *_dbus_State_ = "State";

static void bluez_get_mode_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *mode;
	debug_write("[DBUS] callback: bluez_get_mode_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
		debug_write("[DBUS] callback: bluez_get_mode_callback (mode = %s)", mode);
		unsigned enabled = (strcmp(mode, _bluez_mode_OFF_) ? 1 : 0);
		unsigned visible = (strcmp(mode, _bluez_mode_DISCOVERABLE_) ? 0 : 1);
		if(hw_bluetooth.enabled != enabled || hw_bluetooth.visible != visible){
			hw_bluetooth.enabled = enabled;
			hw_bluetooth.visible = visible;
			hw_changes |= HW_BLUETOOTH;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
		}
		hw_initialized &= ~HWIS_BLUETOOTH_MODE;
	}
	dbus_message_unref(message);
}

static void bluez_get_name_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *name;
	debug_write("[DBUS] callback: bluez_get_name_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _END_)){
		if(hw_bluetooth.name == NULL || strcmp(hw_bluetooth.name, name)){
			set_string(&hw_bluetooth.name, name, BLUETOOTH_NAME_LIMIT);
			hw_changes |= HW_BLUETOOTH;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
		}
		hw_initialized &= ~HWIS_BLUETOOTH_NAME;
	}
	dbus_message_unref(message);
}

/*
static void bluez_device_connected_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	unsigned state;
	debug_write("[DBUS] callback: bluez_device_connected_callback");
	if(dbus_message_get_args(message, &dbus_error, _BOOL_, &state, _END_))
		if(state){ hw_bluetooth.nr_devices++; hw_changes |= HW_BLUETOOTH; }
	dbus_message_unref(message);
}
*/
static void bluez_get_devices_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char **devices;
	unsigned len;
	debug_write("[DBUS] callback: bluez_get_devices_callback");
	if(dbus_message_get_args(message, &dbus_error, _ARRAY_, _STRING_, &devices, &len, _END_)){
		hw_bluetooth.nr_devices = len;

//		unsigned i;
//		for(i = 0; i < len; i++){
//			char *id = devices[i];
//			dbus_recv_method(bluez_device_connected_callback, _bluez_service_, _bluez_path_, _bluez_interface_, "IsConnected", _STRING_, &id, _END_);

			//dbus_free(devices[i]);
//		}

		hw_initialized &= ~HWIS_BLUETOOTH_DEVICES;

		dbus_free_string_array(devices);
	}
	dbus_message_unref(message);
}

static void bluez_get_headset_state_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *state;
	debug_write("[DBUS] callback: bluez_get_headset_state_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &state, _END_)){
		// Disconnected, PlayingPowerManaged, Playing
		unsigned headset = (strcmp(state, "Disconnected") ? 1 : 0);
		if(hw_bluetooth.headset != headset){
			hw_bluetooth.headset = headset;
			hw_changes |= HW_VOLUME_OUTPUT;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
		}
		hw_initialized &= ~HWIS_BLUETOOTH_HEADSET;
	}
	dbus_message_unref(message);
}
static void bluez_get_headset_path_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *path;
	debug_write("[DBUS] callback: bluez_get_headset_path_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &path, _END_)){
		if(path[0] == '/'){
			const char *empty = "";
			dbus_recv_method(bluez_get_headset_state_callback, _bluez_audio_proxy_service_, path, _dbus_property_interface_, _dbus_Get_,
								_STRING_, &empty, _STRING_, &_dbus_State_, _END_); // headset state
		} else {
			if(hw_bluetooth.headset){ hw_bluetooth.headset = 0; hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }
			hw_initialized &= ~HWIS_BLUETOOTH_HEADSET;
		}
	}
	dbus_message_unref(message);
}
#endif

static void bluetooth_init( ){
	hw_bluetooth.enabled = 0;
	hw_bluetooth.visible = 0;
	hw_bluetooth.name = NULL;
	hw_bluetooth.nr_devices = 0;
	hw_bluetooth.headset = 0;

	if(!svc_bluetooth_HCI) return;

	#ifdef ARCH_armel
	if(!use_minimal_ui){
		hw_initialized |= HWIS_BLUETOOTH_MODE|HWIS_BLUETOOTH_NAME|HWIS_BLUETOOTH_DEVICES;
		dbus_recv_method(bluez_get_mode_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_GetMode_, _END_); // enabled
		dbus_recv_method(bluez_get_name_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_GetName_, _END_); // name
		dbus_recv_method(bluez_get_devices_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_ListConnections_, _END_); // nr_devices

		if(svc_bluetooth_status == 2 && svc_audio_status == 2){

// TODO: figure out the services required to query this information and fix the above condition

			hw_initialized |= HWIS_BLUETOOTH_HEADSET;
			dbus_recv_method(bluez_get_headset_path_callback, _bluez_audio_proxy_service_, _bluez_audio_proxy_path_, _dbus_property_interface_, _dbus_Get_,
								_STRING_, &_dbus_FIXME_, _STRING_, &_dbus_DefaultHeadset_, _END_); // headset path
		}
	}
	#endif
}

void hw_set_bluetooth( unsigned enable ){
	#ifdef ARCH_armel
	if(enable){
		// enable bluetooth
		dbus_send_method(_bluez_service_, _bluez_path_, _bluez_interface_, _bluez_SetMode_, _STRING_, &_bluez_mode_CONNECTABLE_, _END_);
	} else {
		// disable bluetooth
		dbus_send_method(_bluez_service_, _bluez_path_, _bluez_interface_, _bluez_SetMode_, _STRING_, &_bluez_mode_OFF_, _END_);
	}
	#endif
}

void hw_set_bluetooth_visible( unsigned visible ){
	#ifdef ARCH_armel
	if(!hw_bluetooth.enabled) return;

	if(visible){
		if(!hw_bluetooth.visible){
			// enable bluetooth visibility
			dbus_send_method(_bluez_service_, _bluez_path_, _bluez_interface_, _bluez_SetMode_, _STRING_, &_bluez_mode_DISCOVERABLE_, _END_);

			//unsigned timeout = 0;
			//dbus_send_method(_bluez_service_, _bluez_path_, _bluez_interface_, "SetDiscoverableTimeout", _UINT32_, &timeout, _END_);
		}
	} else {
		if(hw_bluetooth.visible){
			// disable bluetooth visibility
			dbus_send_method(_bluez_service_, _bluez_path_, _bluez_interface_, _bluez_SetMode_, _STRING_, &_bluez_mode_CONNECTABLE_, _END_);
		}
	}
	#endif
}

void hw_bluetooth_open_search( ){
	#ifdef ARCH_armel
	char *empty = "";
	unsigned len = 0;
	dbus_send_method(_btui_service_, _btui_path_, _btui_interface_, "show_search_dlg",
						_STRING_, &empty,
						_STRING_, &empty,
						_ARRAY_, _STRING_, &empty, &len,
						_STRING_, &empty,
						_END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// GPS

// http://maemo.org/api_refs/4.0/gpsbt/group__gpsbt__api.html
// http://andrew.triumf.ca/N810/gpstest/
// http://andrew.daviel.org/N810-FAQ.html#gpsd
// http://maemo.org/maemo_release_documentation/maemo4.1.x/node10.html

// TODO:

////////////////////////////////////////////////////////////////////////// WIFI

// http://maemo.org/api_refs/5.0/beta/icd2/group__dbus__api.html
// http://maemo.org/api_refs/4.1/libconic-0.16/
// http://maemo.org/maemo_release_documentation/maemo4.1.x/node10.html

#ifdef ARCH_armel
//#include <conic/coniciap.h>

static const char *_icd_service_ = "com.nokia.icd";
static const char *_icd_path_ = "/com/nokia/icd";
static const char *_icd_interface_ = "com.nokia.icd";

static const char *_icd_get_ipinfo_ = "get_ipinfo"; // method

static const char *_icdui_service_ = "com.nokia.icd_ui";
static const char *_icdui_path_ = "/com/nokia/icd_ui";
static const char *_icdui_interface_ = "com.nokia.icd_ui";

static const char *_icd2_service_ = "com.nokia.icd2";
static const char *_icd2_path_ = "/com/nokia/icd2";
static const char *_icd2_interface_ = "com.nokia.icd2";

static const char *_icd2_state_req_ = "state_req"; // method

static void icd2_state_req_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	unsigned enabled;
	debug_write("[DBUS] callback: icd2_state_req_callback");
	if(dbus_message_get_args(message, &dbus_error, _UINT32_, &enabled, _END_)){
		if(hw_wifi.enabled != enabled){ hw_wifi.enabled = enabled; hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		hw_initialized &= ~HWIS_WIFI_MODE;
	}
	dbus_message_unref(message);
}

static void icd_get_ipinfo_callback( DBusPendingCall *pending, void *user_data ){
	DBusMessage *message = dbus_pending_call_steal_reply(pending);
	dbus_pending_call_unref(pending);
	char *name, *addr, *mask, *gateway, *dns1, *dns2;
	debug_write("[DBUS] callback: icd_get_ipinfo_callback");
	if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _STRING_, &addr, _STRING_, &mask, _STRING_, &gateway, _STRING_, &dns1, _STRING_, &dns2, _END_)){
		if((hw_wifi.address ? strcmp(hw_wifi.address, addr) : 1)){
			set_string(&hw_wifi.address, addr, WIFI_ADDRESS_LIMIT);
			hw_changes |= HW_WIFI;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
		}
	}
	dbus_message_unref(message);
}
#endif

static void wifi_init( ){
	hw_wifi.enabled = 0;
	hw_wifi.name = NULL;
	hw_wifi.address = NULL;
	hw_wifi.state = WIFI_DISCONNECTED;

	if(!svc_wifi_ICD) return;

	#ifdef ARCH_armel
	if(!use_minimal_ui){
		hw_initialized |= HWIS_WIFI_MODE|HWIS_WIFI_STATE;
		dbus_recv_method(icd2_state_req_callback, _icd2_service_, _icd2_path_, _icd2_interface_, _icd2_state_req_, _END_); // enabled
		// this method call also generates a state_sig signal used to acquire initial name and state values
	}
	#endif
}

int hw_wifi_get_name( char *iap ){
//	char *last_iap = conf_get_string("/system/osso/connectivity/IAP/last_used_iap");
//	char *last_network = conf_get_string("/system/osso/connectivity/IAP/last_used_network");
	char *name = NULL;
	unsigned changed = 0;

	if(iap != NULL){
		char buffer[200];
		snprintf(buffer, sizeof(buffer), "/system/osso/connectivity/IAP/%s/name", iap);
		name = conf_get_string(buffer);
	}

	if(hw_wifi.name == NULL && name == NULL) return 0;

	if((hw_wifi.name != NULL && name != NULL ? strcmp(hw_wifi.name, name) : 1)){
		set_string(&hw_wifi.name, name, WIFI_NAME_LIMIT);
		changed = 1;
	}

	if(name != NULL) g_free(name);
	return changed;
}

void hw_wifi_get_ipinfo( ){
	#ifdef ARCH_armel
	dbus_recv_method(icd_get_ipinfo_callback, _icd_service_, _icd_path_, _icd_interface_, _icd_get_ipinfo_, _END_); // name and IP address
	// dbus message handler flushes this method call after issuing it
	#endif
}

void hw_wifi_open_selector( ){
	#ifdef ARCH_armel
	// open wifi selector if in normal mode
	// does nothing if in flight mode
	unsigned false_value = FALSE;
	dbus_send_method(_icdui_service_, _icdui_path_, _icdui_interface_, "show_conn_dlg", _BOOL_, &false_value, _END_);
	#endif
}

void hw_wifi_auto_connect( ){
	#ifdef ARCH_armel
	// auto-connects if in normal mode
	// enables normal mode and open wifi selector if in flight mode
	char *any_value = "[ANY]";
	unsigned zero_value = 0;
	dbus_send_method(_icd_service_, _icd_path_, _icd_interface_, "connect", _STRING_, &any_value, _UINT32_, &zero_value, _END_);
	#endif
}

void hw_wifi_disconnect( ){
	#ifdef ARCH_armel
	// disable wifi (icd2)
	unsigned disconnect_value = 0x8000;
	dbus_send_method(_icd2_service_, _icd2_path_, _icd2_interface_, "disconnect_req", _UINT32_, &disconnect_value, _END_);

	// disable wifi (icd - doesn't work on diablo)
//	unsigned true_value = TRUE;
//	dbus_send_method(_icd_service_, _icdui_path_, _icdui_interface_, "disconnect", _BOOL_, &true_value, _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// SENSORS

static double last_als_refresh;

void hw_query_temperature( ){
	#ifdef ARCH_armel
	// temperature
	float temperature = (float)get_file_value("/sys/devices/platform/i2c_omap.1/i2c-1/1-0048/temp1_input") / 1000.0;
	if(hw_sensors.temperature != temperature){ hw_sensors.temperature = temperature; hw_changes |= HW_TEMPERATURE; }
	#endif
}

void hw_query_als( ){
	#ifdef ARCH_armel
	// ambient light sensor (5 second query interval)
	if(is_810 && last_als_refresh + 4.9 < current_time){
		unsigned light = get_file_value("/sys/devices/platform/i2c_omap.2/i2c-0/0-0029/lux");
		if(hw_sensors.light != light){ hw_sensors.light = light; hw_changes |= HW_LIGHT; if(cfg_theme == 0) window_auto_set_theme(0); }
		last_als_refresh = current_time;
	}
	#endif
}

void hw_query_headphones( ){
	// called from hw_init() and the /org/freedesktop/Hal/devices/platform_headphone org.freedesktop.Hal.Device.Condition(ButtonPressed, connection) signal handler

	#ifdef ARCH_armel
	char status_str[50];
	FILE *in;
	unsigned status;

	if((in = fopen("/sys/devices/platform/gpio-switch/headphone/state", "r")) == NULL) return;
	fgets(status_str, 50, in);
	fclose(in);
	status = (strcmp(status_str, "connected\n") ? 0 : 1);
	if(hw_sensors.headphones != status){ hw_sensors.headphones = status; hw_changes |= HW_VOLUME_OUTPUT; }
	#endif
}

////////////////////////////////////////////////////////////////////////// USB

void hw_query_usb( ){
	// called from hw_init() and the /org/freedesktop/Hal/devices/usb_device_0_0_musb_hdrc org.freedesktop.Hal.Device.PropertyModified signal handler

	#ifdef ARCH_armel
	char mode_str[50];
	unsigned mode, connected = 0;
	FILE *in;

	if((in = fopen("/sys/devices/platform/musb_hdrc/mode", "r")) == NULL) return;
	fgets(mode_str, 50, in);
	fclose(in);
	mode = (mode_str[0] == 'a' ? 0 /* a = host */ : 1 /* b = OTG */);
	connected = (strncmp(&mode_str[2], "host", 4) ? (strncmp(&mode_str[2], "peripheral", 10) ? 0 : 2) : 1);
	if(hw_usb.mode != mode || hw_usb.connected != connected){
		hw_usb.mode = mode;
		hw_usb.connected = connected;
		hw_changes |= HW_USB;
		dbus_send_wakeup(_WAKEUP_REFRESH_);
	}

	// this code assumes a_host/peripheral and b_host/peripheral means connected

	// list of all modes are available in the kernel source -- drivers/usb/musb/musb_core.c

	#endif
}

void hw_toggle_usb_mode( ){
	if(hw_usb.mode == 0){
		// switch to OTG
		// I read something about device not properly switching to OTG unless peripheral is sent first
		system("echo peripheral > /sys/devices/platform/musb_hdrc/mode ; echo otg > /sys/devices/platform/musb_hdrc/mode");
	} else {
		// switch to host
		system("echo host > /sys/devices/platform/musb_hdrc/mode");
	}
	usleep(100000); // 100ms sleep to let HAL send PropertyModified signal
}

//////////////////////////////////////////////////////////////////////////

void hw_init( ){
	hw_initialized = 0;

	brightness_init();
//	volume
//	memory
	mce_init();
	bme_init();
	cpu_init();
	bluetooth_init();
//	gps_init();
	wifi_init();

	// sensors init
	hw_sensors.temperature = 0.0;
	hw_sensors.light = 0;
	hw_sensors.headphones = 0;
	hw_query_headphones();
	last_als_refresh = 0.0;

	hw_usb.mode = 0;
	hw_usb.connected = 0;
	hw_query_usb();

	hw_changes = 0;
	dbus_flush_messages();
}

void hw_refresh_cpu( ){
	if(use_minimal_ui) return;

	// grab this first so the other queries don't cause ondemand governor to increase cpu speed and scew the result
	hw_query_cpu(0);

	hw_query_cpu_usage();
}

void hw_refresh( ){
	if(!use_minimal_ui){
		if(brightness_widget_page == primary_widgets)
			hw_query_brightness_with_delay();

		if(volume_widget_page == primary_widgets)
			hw_query_volume();

		if(battery_widget_page == primary_widgets)
			hw_query_memory();

		if(flightmode_button_page == primary_widgets){
			// query flight mode status
			hw_query_flight_mode(0);
			dbus_flush_messages();
		}

		if(primary_widgets){
			// an external rotation resulting in a screen resize will produce an X event that is used to recreate window with update_orientation()
			// however, no event is generated if the screen merely flips orientation so it must be checked and corrected here so the rotation icons can be redrawn
			if(check_orientation())
				hw_changes |= HW_ROTATION;
		}

		if(keyrepeat_button_page == primary_widgets){
			// query key repeat status
			XKeyboardState kbd_state;
			XGetKeyboardControl(window.display, &kbd_state);
			unsigned key_repeat = (kbd_state.global_auto_repeat == AutoRepeatModeOn ? 1 : 0);
			if(hw_key_repeat != key_repeat){ hw_key_repeat = key_repeat; hw_changes |= HW_KEY_REPEAT; }
		}

		if(screenstayslit_button_page == primary_widgets){
			// query screen stays lit status
			unsigned screen_stays_lit = conf_get_screen_stayslit_value();
			if(hw_screen_stays_lit != screen_stays_lit){ hw_screen_stays_lit = screen_stays_lit; hw_changes |= HW_SCREEN_STAYS_LIT; }
		}

		if(screenautolock_button_page == primary_widgets){
			// query screen autolock status
			unsigned screen_autolock = conf_get_screen_autolock_value();
			if(hw_screen_autolock != screen_autolock){ hw_screen_autolock = screen_autolock; hw_changes |= HW_SCREEN_AUTOLOCK; }
		}
	}

	#ifdef ARCH_armel
	// poll initial values until all have been received
	if(hw_initialized){
		// MCE
		if(hw_initialized & HWIS_MCE_DISPLAY)
			dbus_recv_method(mce_get_display_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_display_status_, _END_);
		if(hw_initialized & HWIS_MCE_LOCK)
			dbus_recv_method(mce_get_tklock_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_tklock_mode_, _END_);
		if(hw_initialized & HWIS_MCE_FLIGHT)
			dbus_recv_method(mce_get_flight_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_device_mode_, _END_);
		if(hw_initialized & HWIS_MCE_DEVICELOCK)
			dbus_recv_method(mce_get_devicelock_callback, _mce_service_, _mce_path_, _mce_interface_, _mce_get_devicelock_mode_, _END_);

		// BME
		if(hw_initialized & HWIS_BME_CHARGER)
			dbus_send_signal(_bme_request_path_, _bme_request_interface_, "status_info_req", _END_); // request charging status signal
		if(hw_initialized & (HWIS_BME_MAX|HWIS_BME_CAPACITY|HWIS_BME_ACTIVE))
			hw_query_battery(0); // flushed below

		// BLUETOOTH
		if(hw_initialized & HWIS_BLUETOOTH_MODE)
			dbus_recv_method(bluez_get_mode_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_GetMode_, _END_); // enabled
		if(hw_initialized & HWIS_BLUETOOTH_NAME)
			dbus_recv_method(bluez_get_name_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_GetName_, _END_); // name
		if(hw_initialized & HWIS_BLUETOOTH_DEVICES)
			dbus_recv_method(bluez_get_devices_callback, _bluez_service_, _bluez_path_, _bluez_interface_, _bluez_ListConnections_, _END_); // nr_devices
		if(hw_initialized & HWIS_BLUETOOTH_HEADSET)
			dbus_recv_method(bluez_get_headset_path_callback, _bluez_audio_proxy_service_, _bluez_audio_proxy_path_, _dbus_property_interface_, _dbus_Get_,
								_STRING_, _dbus_FIXME_, _STRING_, _dbus_DefaultHeadset_, _END_); // headset path

		// WIFI
		if(hw_initialized & (HWIS_WIFI_MODE|HWIS_WIFI_STATE))
			dbus_recv_method(icd2_state_req_callback, _icd2_service_, _icd2_path_, _icd2_interface_, _icd2_state_req_, _END_); // enabled
			// this method call also generates a state_sig signal used to acquire initial name and state values

		dbus_flush_messages();
	}

	if(battery_widget_page == primary_widgets)
		hw_query_temperature();

	hw_query_als();
	#endif

	last_hw_refresh = current_time;
}
